// morse.cxx -- Morse code generation class
//
// Written by Curtis Olson, started March 2001.
//
// Copyright (C) 2001  Curtis L. Olson - http://www.flightgear.org/~curt
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation; either version 2 of the
// License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
//
// $Id: morse.cxx,v 1.20 2010/01/02 20:51:32 torsten Exp $


#include <simgear/constants.h>

#include "morse.hxx"


static const char alphabet[26][4] = {
    { DI, DAH, end, end },	/* A */ 
    { DA, DI, DI, DIT },	/* B */ 
    { DA, DI, DA, DIT },	/* C */ 
    { DA, DI, DIT, end },	/* D */ 
    { DIT, end, end, end },	/* E */ 
    { DI, DI, DA, DIT },	/* F */ 
    { DA, DA, DIT, end },	/* G */ 
    { DI, DI, DI, DIT },	/* H */ 
    { DI, DIT, end, end },	/* I */ 
    { DI, DA, DA, DAH },	/* J */ 
    { DA, DI, DAH, end },	/* K */ 
    { DI, DA, DI, DIT },	/* L */ 
    { DA, DAH, end, end },	/* M */ 
    { DA, DIT, end, end },	/* N */ 
    { DA, DA, DAH, end },	/* O */ 
    { DI, DA, DA, DIT },	/* P */ 
    { DA, DA, DI, DAH },	/* Q */ 
    { DI, DA, DIT, end },	/* R */ 
    { DI, DI, DIT, end },	/* S */ 
    { DAH, end, end, end },	/* T */ 
    { DI, DI, DAH, end },	/* U */ 
    { DI, DI, DI, DAH },	/* V */ 
    { DI, DA, DAH, end },	/* W */ 
    { DA, DI, DI, DAH },	/* X */ 
    { DA, DI, DA, DAH },	/* Y */ 
    { DA, DA, DI, DIT }		/* Z */ 
};

static const char numerals[10][5] = {
    { DA, DA, DA, DA, DAH },    // 0
    { DI, DA, DA, DA, DAH },    // 1
    { DI, DI, DA, DA, DAH },    // 2
    { DI, DI, DI, DA, DAH },    // 3
    { DI, DI, DI, DI, DAH },    // 4
    { DI, DI, DI, DI, DIT },    // 5
    { DA, DI, DI, DI, DIT },    // 6
    { DA, DA, DI, DI, DIT },    // 7
    { DA, DA, DA, DI, DIT },    // 8
    { DA, DA, DA, DA, DIT }     // 9
};


// constructor
FGMorse::FGMorse() {
}

// destructor
FGMorse::~FGMorse() {
}


// Make a tone of specified freq and total_len with trans_len ramp in
// and out and only the first len bytes with sound, the rest with
// silence
void make_tone( unsigned char *buf, int freq, 
		int len, int total_len, int trans_len )
{
    int i, j;

    for ( i = 0; i < trans_len; ++i ) {
	float level = ( sin( (double) i * SGD_2PI / (BYTES_PER_SECOND / freq) ) )
	    * ((double)i / trans_len) / 2.0 + 0.5;

	/* Convert to unsigned byte */
	buf[ i ] = (unsigned char) ( level * 255.0 ) ;
    }

    for ( i = trans_len; i < len - trans_len; ++i ) {
	float level = ( sin( (double) i * SGD_2PI / (BYTES_PER_SECOND / freq) ) )
	    / 2.0 + 0.5;

	/* Convert to unsigned byte */
	buf[ i ] = (unsigned char) ( level * 255.0 ) ;
    }
    j = trans_len;
    for ( i = len - trans_len; i < len; ++i ) {
	float level = ( sin( (double) i * SGD_2PI / (BYTES_PER_SECOND / freq) ) )
	    * ((double)j / trans_len) / 2.0 + 0.5;
	--j;

	/* Convert to unsigned byte */
	buf[ i ] = (unsigned char) ( level * 255.0 ) ;
    }
    for ( i = len; i < total_len; ++i ) {
	buf[ i ] = (unsigned char) ( 0.5 * 255.0 ) ;
    }
}


// allocate and initialize sound samples
bool FGMorse::init() {
    // Make Low DIT
    make_tone( lo_dit, LO_FREQUENCY, DIT_SIZE - COUNT_SIZE, DIT_SIZE,
	       TRANSITION_BYTES );

    // Make High DIT
    make_tone( hi_dit, HI_FREQUENCY, DIT_SIZE - COUNT_SIZE, DIT_SIZE,
	       TRANSITION_BYTES );

    // Make Low DAH
    make_tone( lo_dah, LO_FREQUENCY, DAH_SIZE - COUNT_SIZE, DAH_SIZE,
	       TRANSITION_BYTES );

    // Make High DAH
    make_tone( hi_dah, HI_FREQUENCY, DAH_SIZE - COUNT_SIZE, DAH_SIZE,
	       TRANSITION_BYTES );

    // Make SPACE
    int i;
    for ( i = 0; i < SPACE_SIZE; ++i ) {
	space[ i ] = (unsigned char) ( 0.5 * 255 ) ;
    }

    return true;
}


// allocate and initialize sound samples
bool FGMorse::cust_init(const int freq ) {
    int i;

    // Make DIT
    make_tone( cust_dit, freq, DIT_SIZE - COUNT_SIZE, DIT_SIZE,
	       TRANSITION_BYTES );

    // Make DAH
    make_tone( cust_dah, freq, DAH_SIZE - COUNT_SIZE, DAH_SIZE,
	       TRANSITION_BYTES );

    // Make SPACE
    for ( i = 0; i < SPACE_SIZE; ++i ) {
	space[ i ] = (unsigned char) ( 0.5 * 255 ) ;
    }

    return true;
}


// make a SGSoundSample morse code transmission for the specified string
SGSoundSample *FGMorse::make_ident( const string& id, const int freq ) {

    char *idptr = (char *)id.c_str();

    int length = 0;
    int i, j;

    // 0. Select the frequency.  If custom frequency, generate the
    // sound fragments we need on the fly.
    unsigned char *dit_ptr, *dah_ptr;

    if ( freq == LO_FREQUENCY ) {
	dit_ptr = lo_dit;
	dah_ptr = lo_dah;
    } else if ( freq == HI_FREQUENCY ) {
	dit_ptr = hi_dit;
	dah_ptr = hi_dah;
    } else {
	cust_init( freq );
	dit_ptr = cust_dit;
	dah_ptr = cust_dah;
    }

    // 1. Determine byte length of message
    for ( i = 0; i < (int)id.length(); ++i ) {
	if ( idptr[i] >= 'A' && idptr[i] <= 'Z' ) {
	    int c = (int)(idptr[i] - 'A');
	    for ( j = 0; j < 4 && alphabet[c][j] != end; ++j ) {
		if ( alphabet[c][j] == DIT ) {
		    length += DIT_SIZE;
		} else if ( alphabet[c][j] == DAH ) {
		    length += DAH_SIZE;
		}
	    }
	    length += SPACE_SIZE;
        } else if ( idptr[i] >= '0' && idptr[i] <= '9' ) {
            int c = (int)(idptr[i] - '0');
            for ( j = 0; j < 5; ++j) {
                if ( numerals[c][j] == DIT ) {
                    length += DIT_SIZE;
                } else if ( numerals[c][j] == DAH ) {
                    length += DAH_SIZE;
                }
            }
            length += SPACE_SIZE;
	} else {
	    // skip unknown character
	}
    }
    // add 2x more space to the end of the string
    length += 2 * SPACE_SIZE;

    // 2. Allocate space for the message
    const unsigned char* buffer = (const unsigned char *)malloc(length);

    // 3. Assemble the message;
    unsigned char *buf_ptr = (unsigned char*)buffer;

    for ( i = 0; i < (int)id.length(); ++i ) {
	if ( idptr[i] >= 'A' && idptr[i] <= 'Z' ) {
	    int c = (int)(idptr[i] - 'A');
	    for ( j = 0; j < 4 && alphabet[c][j] != end; ++j ) {
		if ( alphabet[c][j] == DIT ) {
		    memcpy( buf_ptr, dit_ptr, DIT_SIZE );
		    buf_ptr += DIT_SIZE;
		} else if ( alphabet[c][j] == DAH ) {
		    memcpy( buf_ptr, dah_ptr, DAH_SIZE );
		    buf_ptr += DAH_SIZE;
		}
	    }
	    memcpy( buf_ptr, space, SPACE_SIZE );
	    buf_ptr += SPACE_SIZE;
        } else if ( idptr[i] >= '0' && idptr[i] <= '9' ) {
            int c = (int)(idptr[i] - '0');
            for ( j = 0; j < 5; ++j ) {
                if ( numerals[c][j] == DIT ) {
                    memcpy( buf_ptr, dit_ptr, DIT_SIZE );
                    buf_ptr += DIT_SIZE;
                } else if ( numerals[c][j] == DAH ) {
                    memcpy( buf_ptr, dah_ptr, DAH_SIZE );
                    buf_ptr += DAH_SIZE;
                }
            }
	    memcpy( buf_ptr, space, SPACE_SIZE );
	    buf_ptr += SPACE_SIZE;
	} else {
	    // skip unknown character
	}
    }
    memcpy( buf_ptr, space, SPACE_SIZE );
    buf_ptr += SPACE_SIZE;
    memcpy( buf_ptr, space, SPACE_SIZE );
    buf_ptr += SPACE_SIZE;

    // 4. create the simple sound and return
    SGSoundSample *sample = new SGSoundSample( &buffer, length,
                                               BYTES_PER_SECOND );

    sample->set_reference_dist( 10.0 );
    sample->set_max_dist( 20.0 );

    return sample;
}
