/***************************************************
// mainwindow.cpp -- top level construction routines
//
// Written by Geoff R. McLane, started May 2011.
//
// Copyright (C) 2011  Geoff R. McLane - reports@geoffair.info
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation; either version 2 of the
// License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
//
// $Id: $
// ***************************************************/

#include "tgs_config.h"
#include "mainwindow.h"
#include "ui_mainwindow.h"
#include <QMessageBox>
#include <QFileDialog>
#include <QSettings>
#include <QStatusBar>
#include <QPalette>
#include <QUrl>
#include <QDesktopServices>
#include <QTextStream>
#include <QFile>
#include <QXmlStreamReader>
#include <QFileInfo>
#include <QVBoxLayout>
#include <QSize>
#include <QSizePolicy>

// more local includes
#include "newbucket.h"
#include "tggui_utils.h"
#include "tggui_srtm.h"

QString appname = APP_NAME " v" APP_VERS;
QString title = APP_TITLE;

// contents of the about box
QString about = APP_NAME "\n"
                "Version: " APP_VERS "\n\n"
                "A graphical user interface\nfor building TerrGear scenery.\n"
                "Part of the FlightGear project.\n\n"
                "GNU General Public License - GPL2 (or later)\n\n"
                "Commenced Geoff R. McLane 2011\n"
                "Thanks to Gijs de Rooy for TerraGUI";

// to change this, loses previous settings, so
QString setname = APP_SET;
QSettings settings("TerraGear",setname); // establish persistent settings
// WIN32: In registry - HKEY_CURRENT_USER/Software/TerraGear/<name>
// unix: $HOME/.config/<app name>.conf, in an INI file format

// TBA PAGES INDEXES
#define ind_Setup    0
#define ind_Elev     1
#define ind_Landuse  2
#define ind_Airports 3
#define ind_Constr   4
#define ind_Advanced 5

// bit flags
#define flg_AirObj  1
#define flg_AirArea 2
#define flg_Default 4
#define flg_SRTM    8
#define flg_Land   16
#define flg_AirObj_cnt  0x80000000
#define flg_AirArea_cnt 0x40000000
#define flg_Default_cnt 0x20000000
#define flg_SRTM_cnt    0x10000000
#define flg_Land_cnt    0x08000000

#define flg_ALL (flg_AirObj | flg_AirArea | flg_Default | flg_SRTM | flg_Land)
#define flg_ALL_cnt (flg_AirObj_cnt | flg_AirArea_cnt | flg_Default_cnt | flg_SRTM_cnt | flg_Land_cnt)

// primary directories
QString projectDir;  // MAIN project directory
QString prevprojDir; // Value last time showed CRITICAL ERROR mess!!!
QString toolDir;     // Location of TG suite of tools
QString fgrootDir;   // Location of FG_ROOT data
QString hgtDir;      // location of HGT elevation files
// tools
QString toolWget;
QString toolUnzip;

// Scenery Limits
QString m_north; // = max. lat
QString m_west;  // = min. lon
QString m_east;  // = max. lon
QString m_south; // = min. lat
// in SRTM form
QString maxElev;
QString minElev;
QString elevList;
QStringList hgturlList;
int m_Hgt_Cnt;
#ifdef USE_FILE_INTERFACE // use the FILE interface - depreciated
QMap<QString, QString> qmSRTM2URL;
#endif
QMap<QString, QString> qmSRTM2URL2;

// bunch of 'composite' directories
QString m_dataDir;
QString m_outpDir;
QString m_workDir;
QString m_tempDir;
QString m_srtmDir;
QString m_shapDir;
QString m_elevDir;

// files
QString m_aptFile;
QString m_logFile;
QString m_argFile;

// default tools
QString defWget = "wget";
QString defUnzip = "unzip";

bool good_to_go = false;
int m_MaxDiag = 512; // max. 'info' text to a confirmation dialog
//int m_stdMsgSecs = 10; // time to display a message
int m_waitSecs = 5;

// variables
QStringList m_zipList; // add zip downloads
bool m_Auto_Roll = false;
bool m_User_Break = false;

bool m_Verify_Tools = true;

// simple WAIT dialog
QDialog *m_pWaitDialog;
QLabel *m_pWaitLabel;
int m_waitCount = 0;

#define OPENWAITDIALOG     m_waitCount++; m_pWaitDialog->show(); m_pWaitDialog->repaint();

#define SETWAITDIALOG(a) if(m_pWaitDialog) { \
    m_pWaitLabel->setText(a); \
    m_pWaitLabel->repaint(); \
    OPENWAITDIALOG }

#define SHOWWAITDIALOG(a,b) if (m_pWaitDialog) { \
    m_pWaitDialog->setWindowTitle(a); \
    SETWAITDIALOG(b); }

#define CLOSEWAITDIALOG if (m_waitCount) { m_waitCount--; } \
    if ((m_waitCount == 0) && m_pWaitDialog) { m_pWaitDialog->hide(); qApp->processEvents(); }

#define NEWWAITMESSAGE(a) if (m_pWaitDialog && m_waitCount) { \
    m_pWaitLabel->setText(a); \
    m_pWaitLabel->repaint(); \
    m_pWaitDialog->repaint(); }

// status bar
QStatusBar *m_status;

MainWindow::MainWindow(QWidget *parent) :
    QMainWindow(parent),
    ui(new Ui::MainWindow)
{
    ui->setupUi(this);
    setUpdatesEnabled(true);
    setWindowTitle(appname);
    m_pWaitDialog = new QDialog(this);
    //m_pWaitDialog->setSizePolicy(SizePolicy::Expanding);
    //m_pWaitDialog->resize(QMIN(200, QApplication::desktop()->width() - 200),
    //    QMIN(200, QApplication::desktop()->height() - 200));
    m_pWaitDialog->resize(350,150);

    m_pWaitLabel = new QLabel(m_pWaitDialog);
    m_pWaitLabel->setWordWrap(true);
    //m_pWaitLabel->setSizePolicy(SizePolicy::Expanding);
    //m_pWaitLabel->setHorizontalPolicy (SizePolicy::Expanding);

    m_pWaitDialog->setWindowTitle("STARTUP");
    QVBoxLayout *l = new QVBoxLayout(m_pWaitDialog);
    //l->addStretch();
    l->addWidget(m_pWaitLabel);

    SHOWWAITDIALOG("DOING STARTUP","Moment... Doing startup of "+appname+"\n.\n")

    QDir d;
    QFile f;
    QString tmp;

    m_Auto_Roll = false;
    m_Hgt_Cnt = 0;
    // working directory set
    projectDir = settings.value("paths/projectDir").toString();
    ui->lineEdit_pd->setText(projectDir);
    toolDir = settings.value("paths/toolDir").toString();
    ui->lineEdit_td->setText(toolDir);
    fgrootDir = settings.value("paths/fgrootDir").toString();
    ui->lineEdit_rd->setText(fgrootDir);
    m_srtmDir = settings.value("paths/temphgt").toString();
    ui->lineEdit_hgt->setText(m_srtmDir);
    m_shapDir = settings.value("paths/tempshp").toString();
    ui->lineEdit_shp->setText(m_shapDir);

    // log files
    m_logFile = settings.value("files/log").toString();
    ui->lineEdit_lf->setText(m_logFile);
    m_argFile = settings.value("files/args").toString();
    ui->lineEdit_lf_2->setText(m_argFile);

    NEWWAITMESSAGE("Moment... Doing startup of "+appname+"\nupdateDirectories()...\n");
    updateDirectories();

    // scenery limits
    m_north = settings.value("limits/north").toString();
    ui->lineEdit_maxlat->setText(m_north);
    m_west = settings.value("limits/west").toString();
    ui->lineEdit_minlon->setText(m_west);
    m_east = settings.value("limits/east").toString();
    ui->lineEdit_maxlon->setText(m_east);
    m_south = settings.value("limits/south").toString();
    ui->lineEdit_minlat->setText(m_south);
    NEWWAITMESSAGE("Moment... Doing startup of "+appname+"\nupdateCenter()...\n");
    updateCenter();

    // files
    m_aptFile = settings.value("files/airport").toString();
    if ((m_aptFile.size() == 0) && fgrootDir.size() && d.exists(fgrootDir)) {
        QString af = fgrootDir+"/Airports/apt.dat.gz";
        if (f.exists(af)) {
            m_aptFile = af;
            settings.setValue("files/airport",m_aptFile);
        }
    }
    ui->lineEdit_ap->setText(m_aptFile);

    // other tools
    toolWget = settings.value("tools/wget").toString();
    if (toolWget.size() == 0) {
        toolWget = defWget;
        settings.setValue("tools/wget",toolWget);
    }
    ui->lineEdit_wg->setText(toolWget);
    toolUnzip = settings.value("tools/unzip").toString();
    if (toolUnzip.size() == 0) {
        toolUnzip = defUnzip;
        settings.setValue("tools/unzip",toolUnzip);
    }
    ui->lineEdit_uz->setText(toolUnzip);

    NEWWAITMESSAGE("Moment... Doing startup of "+appname+"\nupdateMaterials()...\n");
    updateMaterials(); // populate the materials combo

    // return to the last tab page
    int index = settings.value("settings/index",0).toInt();
    ui->tabWidget->setCurrentIndex(index);
    on_tabWidget_currentChanged(index); // seems not called on startup

    m_MaxDiag = settings.value("settings/maxtext",512).toInt();
    tmp.sprintf("%d",m_MaxDiag);
    ui->lineEdit_maxtext->setText(tmp);

    m_waitSecs = settings.value("settings/waitsecs",3).toInt();
    if (m_waitSecs < 1) m_waitSecs = 1;
    tmp.sprintf("%d",m_waitSecs);
    ui->lineEdit_secs->setText(tmp);

    // re-apply the check boxes (for construct)
    bool chk;
    chk = settings.value("check/const_no_overwrite").toBool();
    ui->checkBox_noovr->setCheckState(chk ? Qt::Checked : Qt::Unchecked);
    chk = settings.value("check/const_ignore_landmass").toBool();
    ui->checkBox_lm->setCheckState(chk ? Qt::Checked : Qt::Unchecked);
    chk = settings.value("check/const_no_data").toBool();
    ui->checkBox_nodata->setCheckState(chk ? Qt::Checked : Qt::Unchecked);
    chk = settings.value("check/const_ign_errors").toBool();
    ui->checkBox_skiperr->setCheckState(chk ? Qt::Checked : Qt::Unchecked);
    chk = settings.value("check/const_skip_cfm_2").toBool();
    ui->checkBox_skipcfm_2->setCheckState(chk ? Qt::Checked : Qt::Unchecked);

    NEWWAITMESSAGE("Moment... Doing startup of "+appname+"\nupdate status bar...\n");

    // STATUSBAR -  maybe ??? - have not got this right yet???
    //ui->statusBar->show(); // maybe I need to add a widget
    //ui->statusBar->showMessage("Ready"); // hmmm, this did not show???
    //statusBar()->showMessage("Ready"); // maybe this - still no show???
    m_status = statusBar();
    m_status->show();
    m_status->showMessage("Ready");
    m_status->repaint();    // STILL NO SHOW???


    //util_uselessWait(10);
    CLOSEWAITDIALOG; // remove dialog
    setUpdatesEnabled(true);
}

MainWindow::~MainWindow()
{
    QString dt = util_getDTstg();
    outLog(dt+" - Application exit.\n");
    delete ui;
}

// Toggle between the following TABS/PAGES
// Setup Elevations Landuse Airports Construction Advanced
void MainWindow::on_tabWidget_currentChanged(int index)
{
    QFile f;
    QDir d;
    QFileInfo finfo;
    QString tmp;
    QString info;
    int i, res;
    double eastInt     = m_east.toDouble();
    double northInt    = m_north.toDouble();
    double southInt    = m_south.toDouble();
    double westInt     = m_west.toDouble();
    bool got_projdir = false;
    bool got_tooldir = false;
    bool got_fgroot = false;
    bool got_limits = false;
    bool enabled = true;
    tmp = "\n\n";
    switch (index)
    {
    case ind_Setup: // Setup
        SETWAITDIALOG("Changing to Setup..."+tmp);
        break;
    case ind_Elev: // Elevations - ACTIONS: Download-Unzip-Decode-Terrafit
        SETWAITDIALOG("Changed to Elevations..."+tmp);
        break;
    case ind_Landuse: // Landuse / shapefile download, and chopping
        SETWAITDIALOG("Changed to Landuse..."+tmp);
        break;
    case ind_Airports: // Airports
        SETWAITDIALOG("Changed to Airports..."+tmp);
        break;
    case ind_Constr: // Construction
        SETWAITDIALOG("Changed to Construction..."+tmp);
        break;
    case ind_Advanced:
        SETWAITDIALOG("Changed to Advanced..."+tmp);
        break;
    default:
        SETWAITDIALOG("Changed to WHAT!!!"+tmp);
        break;
    }

    if (projectDir.size() && d.exists(projectDir))
        got_projdir = true;

    // could/should also TEST tools
    if (toolDir.size() && d.exists(toolDir))
        got_tooldir = true;

    if (fgrootDir.size() && d.exists(fgrootDir))
        got_fgroot = true;

    if ((westInt < eastInt) && (northInt > southInt) &&
         latlonInWorld(northInt,eastInt) &&
         latlonInWorld(southInt,westInt)) {
        got_limits = true;
    } else {
        got_limits = false;
    }
    // set a general enable
    if (got_limits && got_projdir && got_tooldir && got_fgroot)
        enabled = true;
    else
        enabled = false;

    res = 0; // clear any flag bits
    switch (index)
    {
    case ind_Setup: // Setup
        outLog("Changed to Setup\n");
        if (got_limits && got_projdir && got_tooldir &&
            got_fgroot && good_to_go) {
            enabled = true;
        } else {
            enabled = false;
        }
        ui->pushButton_auto->setEnabled(enabled);
        break;
    case ind_Elev: // Elevations -
        // ACTIONS: Download-Unzip-Decode-Terrafit
        outLog("Changed to Elevations\n");
        // To down load
        if (got_limits && got_projdir && got_tooldir)
            enabled = true;
        else
            enabled = false;
        ui->pushButton_srtm->setEnabled(enabled); // download HGT
        if (!enabled) {
            ui->pushButton_srtmuz->setEnabled(enabled); // unzip HGT
            ui->pushButton_srtmdc->setEnabled(enabled); // Decode HGT
            ui->pushButton_srtmtf->setEnabled(enabled); // Run Terrafit
        } else {
            // need to check unzip tool, and files to UNZIP
            ui->pushButton_srtmuz->setEnabled(enabled); // unzip HGT
            // need to check decode tool, and unzipped files
            ui->pushButton_srtmdc->setEnabled(enabled); // Decode HGT
            // need to check Terrafit tool, and decoded hgt files done
            ui->pushButton_srtmtf->setEnabled(enabled); // Run Terrafit
        }

        info.sprintf("Of %d HGT, %d avail. for download.",m_Hgt_Cnt,hgturlList.size());
        if (enabled) {
            // basically enabled to go do elevations
            QStringList list1 = findFiles(m_srtmDir, QStringList() << "*.zip", false);
            QStringList list2 = findFiles(m_srtmDir, QStringList() << "*.hgt", false);
            QStringList list3, list4;
            int zip_cnt = list1.size();
            int hgt_cnt = list2.size();
            int arr_cnt = 0;
            int fit_cnt = 0;
            if ( m_elevDir.size() && d.exists(m_elevDir) ) {
                //list1 = findFiles(m_elevDir,QStringList() << "*",true);
                list3 = findFiles(m_elevDir,QStringList() << "*.arr.gz",true);
                list4 = findFiles(m_elevDir,QStringList() << "*.fit.gz",true);
                arr_cnt = list3.size();
                fit_cnt = list4.size();
            }
            if (zip_cnt) {
                tmp.sprintf("\nFound %d zips.", zip_cnt);
                info += tmp;
            } else {
                info += "\nNo zips found. Maybe need to do 'Download HGT'";
            }
            if (hgt_cnt) {
                tmp.sprintf("\nFound %d hgt.", hgt_cnt);
                info += tmp;
            }
            if (zip_cnt  && (zip_cnt != hgt_cnt)) {
                    if (hgturlList.size() < hgt_cnt) {
                        info += "\nLooks like 'download HGT' needs to be completed";
                        info += "\nand 'Unzip HGT' can be run.";
                    } else {
                        info += "\nLooks like 'Unzip HGT' needs to be run.";
                    }
            } else if (zip_cnt && (zip_cnt == hgt_cnt)) {
                if (arr_cnt) {
                    info += "\nLooks like decode HGT has been done.";
                } else {
                    info += "\nLooks like decode HGT needs to be done.";
                }
            }

            if ( m_elevDir.size() && d.exists(m_elevDir) ) {
                tmp.sprintf("\nFound work directory with %d array files, and %d fit files.",
                     arr_cnt, fit_cnt);
                info += tmp;
                if (arr_cnt) {
                    if (arr_cnt == fit_cnt) {
                        info += "\nLook like elevations have been completed, ready for airport building, and final construction.";
                    } else if (fit_cnt == 0) {
                        info += "\nLook like 'Terrafit' needs to be run.";
                    }
                } else if (hgt_cnt) {
                    info += "\nLook like 'Decode HGT' needs to be run.";
                }
            } else {
                // do not yet have an work/SRTM-30 elevation directory
                if (hgt_cnt)
                    info += "\nLook like 'Decode HGT' needs to be run.";
            }
        }
        ui->plainTextEdit_hgtinfo->setPlainText(info);
        break;
    case ind_Landuse: // Landuse / shapefile download, and chopping
        outLog("Changed to Landuse\n");
        // =========================================
        // enable/disable shape download - just on on valid area
        ui->pushButton_shp->setEnabled(enabled);
        // =========================================
        // enable/disable fill - any shape files in data,
        // or any landuse directories in data
        enabled = false;
        if (m_dataDir.size() && d.exists(m_dataDir)) {
            QDir sdir(m_dataDir);
            sdir.setFilter(QDir::Files | QDir::NoDotAndDotDot);
            QFileInfoList fList = sdir.entryInfoList();
            res = 0;
            for (i = 0; i < fList.size(); i++) {
                finfo = fList.at(i);
                if (finfo.suffix() == "shp") {
                    enabled = true;
                    res |= 1;
                    break;
                }
            }
            sdir.setFilter(QDir::Dirs | QDir::NoDotAndDotDot);
            QFileInfoList dList = sdir.entryInfoList();
            for (i = 0; i < dList.size(); i++) {
                finfo = dList.at(i);
                tmp = finfo.fileName();
                if (!srtm_isElevationDir(tmp)) {
                    enabled = true;
                    res |= 2;
                    break;
                }
            }
        }
        // enable/disable shape fill list
        ui->pushButton_fill->setEnabled(enabled);
        // if we got land use directories, and the rowCount == 0
        // the auto roll to do it now...
        if ((res & 2) && (ui->tblShapesAlign->rowCount() == 0)) {
            m_Auto_Roll = true;
            on_pushButton_fill_clicked();
            m_Auto_Roll = false;
        }
        // =========================================
        // enable/disable unzip depends on at least 1 of
        // 2 zip files being present in temp/shape download area
        enabled = false;
        if (d.exists(m_shapDir)) {
            if ((f.exists(m_shapDir+"/landuse.zip"))||
                (f.exists(m_shapDir+"/landmass.zip"))) {
                enabled = true;
            }
        }
        ui->pushButton_shpuz->setEnabled(enabled);

        // =========================================
        // enable/disable shape decode - on shapes assigned in list
        enabled = false;
        if (ui->tblShapesAlign->rowCount() > 0)
            enabled = true;
        ui->pushButton_shpdec->setEnabled(enabled);

        // =======================================
        // check for tool existance
        tmp = toolDir+"/ogr-decode";
#ifdef Q_OS_WIN
        tmp += ".exe";
#endif
        if ( !f.exists(tmp)) {
            tmp = toolDir+"/shape-decode";
#ifdef Q_OS_WIN
            tmp += ".exe";
#endif
            if (f.exists(tmp)) {
                ui->checkBox_useshp->setCheckState(Qt::Checked);
            }
        }
        break;
    case ind_Airports: // Airports
        outLog("Changed to Airports\n");
        if (m_aptFile.size() && f.exists(m_aptFile))
            enabled = true;
        else
            enabled = false;
        ui->pushButton_genair->setEnabled(enabled);
        // *TBD* for the moment, disable the big auto land use button
        ui->pushButton_autolu->setEnabled(false);
        break;
    case ind_Constr: // Construction
        outLog("Changed to Construction\n");
        // could check
        // 1 - Elevations available - with *.arr and *.fit files
        // 2 - Landuse directories - with <index>.<counter> arrays
        if (enabled) {
            if (ui->listWidget_2->count() == 0) {
                on_pushButton_updt_clicked();
            }
        }
        ui->pushButton_gen->setEnabled(enabled);
        break;
    case ind_Advanced:
        outLog("Changed to Advanced\n");
        break;
    default:
        outLog("Changed to WHAT!!!\n");
        break;
    }
    settings.setValue("settings/index",index);
    CLOSEWAITDIALOG; // remove dialog
}

void MainWindow::updateCenter()
{
    bool okn,okw,oke,oks;
    double dnorth = m_north.toDouble(&okn);
    double dwest = m_west.toDouble(&okw);
    double deast = m_east.toDouble(&oke);
    double dsouth = m_south.toDouble(&oks);
    QString msg;
    QString tmp;
    QString url;
    QString hgt;
    QString path;
    QStringList list;
    int buck_cnt, buck_srtm;
    int dx, dy, i, j;
    QPalette p;
    int inorth = (int)dnorth;
    int iwest = (int)dwest;
    int ieast = (int)deast;
    int isouth = (int)dsouth;
    if ( okn && okw && oke && oks &&
       (dnorth > dsouth)&&(deast > dwest)&&
        (latlonInWorld(dnorth,deast))&&
        (latlonInWorld(dsouth,dwest))&&
        ((double)inorth == dnorth)&&
        ((double)iwest == dwest)&&
        ((double)ieast == deast)&&
        ((double)isouth == dsouth)) {
        // ***************************************
        // all is well with the min/max world ;=))
        // ***************************************
        p.setColor(QPalette::WindowText, Qt::black);
        msg = "Scenery range has been set.";

        // get it as a center and span
        double clat = (dnorth + dsouth) / 2;
        double clon = (deast + dwest) / 2;
        QString lat = QString::number(clat);
        QString lon = QString::number(clon);
        QString xdist = QString::number(deast - clon);
        QString ydist = QString::number(dnorth - clat);
        QString index;
        // Setup Page
        ui->label_clat->setText(lat);
        ui->label_clon->setText(lon);
        ui->label_xdist->setText(xdist);
        ui->label_ydist->setText(ydist);
        // Construction Page
        ui->label_clat_2->setText(lat);
        ui->label_clon_2->setText(lon);
        ui->label_xdist_2->setText(xdist);
        ui->label_ydist_2->setText(ydist);

        // max. elevation SRTM - north and east
        maxElev = srtm_latlonToHGTBase( dnorth, deast );
        // min. elevation SRTM - south and west
        minElev = srtm_latlonToHGTBase( dsouth, dwest );

        // Setup - set labels
        ui->label_min->setText(minElev);
        ui->label_max->setText(maxElev);
        // Elevations - set labels
        ui->label_min_2->setText(minElev);
        ui->label_max_2->setText(maxElev);
        // set elevation range
        ui->label_maxlat_2->setText(m_north);
        ui->label_maxlon_2->setText(m_east);
        ui->label_minlat_2->setText(m_south);
        ui->label_minlon_2->setText(m_west);
        // landuse range
        ui->label_maxlat_3->setText(m_north);
        ui->label_maxlon_3->setText(m_east);
        ui->label_minlat_3->setText(m_south);
        ui->label_minlon_3->setText(m_west);
        // airports range
        ui->label_maxlat_4->setText(m_north);
        ui->label_maxlon_4->setText(m_east);
        ui->label_minlat_4->setText(m_south);
        ui->label_minlon_4->setText(m_west);

        SGBucket b_min(dwest,dsouth);
        SGBucket b_max(deast,dnorth);
        SGBucket b_cur;
        buck_cnt = 0;
        buck_srtm = 0;
        list.clear();
        if (b_min == b_max) {
            buck_cnt++;
            hgt = srtm_bucketToHGTBase(b_cur); // get the HGTbase string
            if ( srtm_getUrlForHgtBase( hgt, url ) ) {
                if (list.indexOf(url) == -1)
                    list += url;    // accumulate SRTM url's
                buck_srtm++;
            }
        } else {
            sgBucketDiff(b_min,b_max,&dx,&dy);
            for (j = 0; j <= dy; j++) {
                for (i = 0; i <= dx; i++) {
                    b_cur = sgBucketOffset(dwest,dsouth,i,j);
                    buck_cnt++;
                    tmp.sprintf("%d of %d: ", (j*dx)+i+1, (dx+1)*(dy+1));
                    index.sprintf("%ld", b_cur.gen_index());
                    path.sprintf("%s", b_cur.gen_base_path().c_str());
                    path += "/"+index;
                    // only if the 'working' dialog is up...
                    NEWWAITMESSAGE("Moment... Doing startup of "+appname+"\nupdateCenter() - "+tmp+path+"\n");
                    hgt = srtm_bucketToHGTBase(b_cur); // get the HGTbase string
                    if ( srtm_getUrlForHgtBase( hgt, url ) ) {
                        if (list.indexOf(url) == -1)
                            list += url;    // accumulate SRTM url's
                        buck_srtm++;
                    }
                }
            }
        }
        tmp.sprintf(" - Est. %d of %d buckets.",buck_srtm, buck_cnt);
        msg += tmp;

        // build a HELPFUL SRTM list to add to the LOG
        elevList = "";
#ifdef USE_FILE_INTERFACE // use the FILE ineterface
        qmSRTM2URL.clear();
#endif
        qmSRTM2URL2.clear();
        hgturlList.clear();
        m_Hgt_Cnt = 0;
        for (i = iwest; i <= ieast; i++) {
            for (j = isouth; j <= inorth; j++) {
                hgt = srtm_latlonToHGTBase(j,i);
                if (elevList.size()) elevList += ";"; // add separator
                elevList += hgt;
#ifdef USE_FILE_INTERFACE // use the FILE ineterface
                qmSRTM2URL.insert(hgt,"");
#endif
                qmSRTM2URL2.insert(hgt,"");
                if (srtm_getUrlForHgtBase( hgt, url )) {
                    hgturlList += url+"/"+hgt; // store the HGT (base) zip name
                }
                m_Hgt_Cnt++;
            }
        }
        tmp.sprintf("Of %d HGT %d avail.",m_Hgt_Cnt,hgturlList.size());
        ui->label_hgtcnt->setText(tmp);
        ui->label_hgtcnt_3->setText(tmp);
        //tmp.sprintf(", of %d HGT %d avail.",hgt_cnt,hgturlList.size());
        //msg += tmp;
        if (hgturlList.size()) {
            outLog(tmp+"\n");
            for (i = 0; i < hgturlList.size(); i++) {
                url = hgturlList[i];
                outLog(url+"\n");
            }
        }
#ifdef TGS_DEBUG_VERS // but outputs extra info to the log file
        outLog("List of elevations\n");
        outLog(elevList+"\n");
#endif // #ifdef TGS_DEBUG_VERS // but outputs extra info to the log file
    } else {
        p.setColor(QPalette::WindowText, Qt::red);
        msg = "ERROR IN MIN/MAX RANGES";
    }
    ui->label_2->setText(msg);
    ui->label_2->setPalette(p);

}

// MENU: File -> Quit
void MainWindow::on_actionQuit_triggered()
{
    MainWindow::close();
}

// MENU: Help -> About
void MainWindow::on_actionAbout_triggered()
{
    QMessageBox::about(this,title,about);
}

/*******************************
  Setup Page
 *******************************/

// User input for current project directory
void MainWindow::on_lineEdit_pd_editingFinished()
{
    QPalette p;
    QDir d;
    QString dir;
    dir = ui->lineEdit_pd->text();
    if (dir.size() == 0)
        return; // bail if empty

    if (d.exists(dir)) {
        projectDir = dir;
        settings.setValue("paths/projectDir", projectDir);
        prevprojDir = dir;
        updateDirectories();
    } else {
        if (dir != prevprojDir) {
            QMessageBox::critical(this,"INVALID PROJECT DIRECTORY",
"The main PROJECT DIRECTORY ["+dir+"] is NOT INVALID!\n"
"Many subsequent actions can not proceed until this is FIXED!");
        }
        prevprojDir = dir;
        p.setColor(QPalette::WindowText, Qt::red);
    }
    ui->lineEdit_pd->setPalette(p);
    //ui->lineEdit_pd->repaint();
    ui->lineEdit_pd->update();
}

void MainWindow::on_lineEdit_pd_textChanged(QString dir)
{
    QPalette p;
    QDir d;
    //QString dir;
    //dir = ui->lineEdit_pd->text();
    if (dir.size() == 0)
        return; // not given a 'directory'

    if (d.exists(dir)) {
        p.setColor(QPalette::WindowText, Qt::black);
    } else {
        p.setColor(QPalette::WindowText, Qt::red);
    }
    ui->lineEdit_pd->setPalette(p); // does not seem to WORK???
    ui->lineEdit_pd->update();
}


// Browse for project directory
void MainWindow::on_pushButton_ble_clicked()
{
    projectDir = QFileDialog::getExistingDirectory(this,"Select a project folder. Everthing used and created during scenery generation will be in here.", projectDir);
    ui->lineEdit_pd->setText(projectDir);
    // on_lineEdit_pd_editingFinished();
    settings.setValue("paths/projectDir", projectDir);
    updateDirectories();
}

// Browse for terrgear tool directory
void MainWindow::on_pushButton_gtd_clicked()
{
    toolDir = QFileDialog::getExistingDirectory(this,"Select the tool directory for the TerraGear executables.", toolDir);
    ui->lineEdit_td->setText(toolDir);
    settings.setValue("paths/toolDir", toolDir);
    updateDirectories();
}

// Browse for fg root directory
void MainWindow::on_pushButton_grd_clicked()
{
    fgrootDir = QFileDialog::getExistingDirectory(this,"Select the FG ROOT directory.", fgrootDir);
    ui->lineEdit_rd->setText(fgrootDir);
    settings.setValue("paths/fgrootDir", fgrootDir);
    updateDirectories();
}

// set the tools, to use...
void MainWindow::on_pushButton_gwg_clicked()
{
    int isok = 0;
    QString fileName = QFileDialog::getOpenFileName(this,
                                                    tr("Select File"),
                                                     "",
                                                     tr("All (*.*)"));
    if (fileName.size()) {
        QString cmd = fileName+" --version";
        QString msg;
        msg = "";
        int ret = util_testApplication(cmd, msg);
        if (ret == 0) {
            isok = 1;
        } else {
            cmd.sprintf("Attempt to run cmd returned %d\n",ret);
            cmd += "Ran command: ["+fileName+" --version]\n";
            if (msg.size()) {
                cmd += "It returned an output of ["+msg+"]\n";
            }
            cmd += "Perhaps this is not the wget you need?\n";
            cmd += "Click yes to continue to use this file?";
            ret = getYesNo("CONFIRMATION ON ERROR",cmd);
            if (ret)
                isok = 1;
        }
    }
    if (isok) {
        toolWget = fileName;
        ui->lineEdit_wg->setText(toolWget);
        settings.setValue("tools/wget",toolWget);
    }
}

void MainWindow::on_pushButton_guz_clicked()
{
    int isok = 0;
    QString fileName = QFileDialog::getOpenFileName(this,
                                                    tr("Select File"),
                                                     "",
                                                     tr("All (*.*)"));
    if (fileName.size()) {
        QString cmd = fileName;
        QString msg;
        msg = "";
        int ret = util_testApplication(cmd, msg);
        if (ret == 0) {
            isok = 1;
        } else {
            cmd.sprintf("Attempt to run cmd returned %d\n",ret);
            cmd += "Ran command: ["+fileName+"]\n";
            if (msg.size()) {
                cmd += "It returned an output of ["+msg+"]\n";
            }
            cmd += "Perhaps this is not the unzip you need?\n";
            cmd += "Click yes to continue to use this file?";
            ret = getYesNo("CONFIRMATION ON ERROR",cmd);
            if (ret)
                isok = 1;
        }
    }
    if (isok) {
        toolUnzip = fileName;
        ui->lineEdit_uz->setText(toolUnzip);
        settings.setValue("tools/unzip",toolUnzip);
    }
}

// LIMITS OF THE SCENERY
// =====================
void MainWindow::on_lineEdit_maxlat_editingFinished()
{
    m_north = ui->lineEdit_maxlat->text();
    settings.setValue("limits/north",m_north);
    updateCenter();
}

void MainWindow::on_lineEdit_minlon_editingFinished()
{
    m_west = ui->lineEdit_minlon->text();
    settings.setValue("limits/west",m_west);
    updateCenter();
}

void MainWindow::on_lineEdit_maxlon_editingFinished()
{
    m_east = ui->lineEdit_maxlon->text();
    settings.setValue("limits/east",m_east);
    updateCenter();
}

void MainWindow::on_lineEdit_minlat_editingFinished()
{
    m_south = ui->lineEdit_minlat->text();
    settings.setValue("limits/south",m_south);
    updateCenter();
}

/***********************************
  Elevations Page
 ***********************************/
// download the STRM elevation files
void MainWindow::on_pushButton_srtm_clicked()
{
    int i;
    // ===============================================
    // Unable to proceed, if no valid project directory
    if (!util_verifyProjDir(projectDir))
        return; // down hgt - projectDir invlaid
    if (!util_createDirectory(m_srtmDir))
        return; // down hgt - m_srtmDir for output not valid
    QStringList allZips = findFiles(m_srtmDir, QStringList() << "*.zip", false);
    QStringList allHGT = findFiles(m_srtmDir, QStringList() << "*.hgt", false);
    bool skip_exist_zip = ui->checkBox_skipzip->isChecked();
    bool skip_cfm = ui->checkBox_skipcfm->isChecked();
    bool ign_errors = ui->checkBox_skiperr_2->isChecked();
    m_Verify_Tools = ui->checkBox_verify->isChecked();

    // ================================================
    QString url = "http://dds.cr.usgs.gov";
    bool opnd = false;
    outLog("Base URL: "+url+"\n");
#ifndef TGS_DEBUG_VERS // do not load url, for debug, just a time cost...
    QUrl qu(url);
    opnd = QDesktopServices::openUrl(qu);
#endif
    QString msg;
    QString info;
#ifdef USE_FILE_INTERFACE // use the FILE ineterface
    int srtmcnt = qmSRTM2URL.count();
#endif
    int srtmcnt2 = qmSRTM2URL2.count();
    // QString minElev and maxElev set
    outLog(elevList+"\n"); /* provide a 'helpful' list of SRTM files */
    info = "\ndownload the SRTM in the range ["+minElev+"] to ["+maxElev+"]";
    info += "\nA list of the ranges should be in the templog.txt file";
    info += "\nWhen the zips are downloaded they should be expanded into a\ndirectory of your choice before running 'decode hgt'.";
    int count2 = util_checkSRTMFiles2(qmSRTM2URL2);
#if (defined(TGS_DEBUG_VERS) && defined(USE_FILE_INTERFACE)) // but outputs extra info to the log file
    QString m_appDir = "C:/Qt/2010.05/qt/projects/TgScenery";
    int count = util_checkSRTMFiles(qmSRTM2URL,m_appDir);
    // DEBUG - a file read versus the new 'internal' data functions
    // these two functions should ABSOLUTELY yield the SAME result ;=))
    // and DEBUG is only if they are NOT '=='
    if (count2 != count) {
        {
            msg.sprintf("Cnt %d of %d", count2, qmSRTM2URL2.size());
            outLog("List of elevations found in 2 "+msg+"\n");
            QMap<QString, QString>::iterator ii = qmSRTM2URL2.begin();
            for ( ; ii != qmSRTM2URL2.end(); ii++) {
                QString val = ii.value();
                if (val.size()) {
                    QString key = ii.key();
                    outLog(key+": "+val+"\n");
                }
            }
        }
        {
            msg.sprintf("Cnt %d of %d", count, qmSRTM2URL.size());
            outLog("List of elevations found in 1 "+msg+"\n");
            QMap<QString, QString>::iterator ii = qmSRTM2URL.begin();
            for ( ; ii != qmSRTM2URL.end(); ii++) {
                QString val = ii.value();
                if (val.size()) {
                    QString key = ii.key();
                    outLog(key+": "+val+"\n");
                }
            }
        }
    }
#endif // #ifdef TGS_DEBUG_VERS // but outputs extra info to the log file
    // ok, we proceed with the DOWNLOAD of the desired HGT files
    // to the temp/hgt directory...
    //QString tm;
    QStringList argList; // list of arguments to run
    QStringList filList; // and 'information' file list
    QString file;        // file name only
    QString outfile;     // fully qualified file name
    QString argument;    // build each argument
    QString runtime = toolWget; // get 'wget' tool
    // QStringList zipList;
    QString tmp;
    QFile f;
    int ext, res;
    int skipped = 0;
    int wait_secs = ui->lineEdit_secs->text().toInt();
    if (wait_secs < 1) wait_secs = 1;
    bool do_wait;

    // check the wget tool functions
    if (runtime.contains(QChar(' ')))
        runtime = "\""+runtime+"\"";
    argument = runtime;
    argument += " --help";
    res = util_testApplication(argument, info);
    if (res || m_Verify_Tools) {
        util_trimLongMessage(m_MaxDiag, info);
        tmp.sprintf("\nTool returned value %d",res);
        info += tmp;
        info += "\nThe name (and location if not in your PATH)";
        info += "\nis set on the 'Advanced' pages.";
        outLog(info+"\n");
        info += "\n\nClick 'Yes' to continue to use this tool.";
        res = getYesNo("DOWNLOAD TOOL",info);
        if (!res)
            return; // elevation download tool NOT acceptable
    }

    // BUILD THE ARGUMENT LIST
    m_zipList.clear();
    QMap<QString, QString>::iterator i2 = qmSRTM2URL2.begin();
    for ( ; i2 != qmSRTM2URL2.end(); i2++) {
        QString val = i2.value(); // get the URL
        // if we have a URL for this SRTM item...
        if (val.size()) {
            QString key = i2.key(); // get the SRTM we want
            file = key+".hgt.zip";  // build the 'file'
            // outLog(key+": "+val+"\n");
            argument = runtime+" "; // the 'tested' agreed wget tool
            argument += url+val+"/"+file; // add what we want...
            outfile = m_srtmDir+"/"+file;
            // if (skip_exist_zip && util_isinFileList(allZips,file))
            if (skip_exist_zip && (allZips.indexOf(outfile) != -1)) {
                skipped++;
                continue;
            }
            argList += argument; // wget request argument
            filList += file;
        }
    }
    if (count2 == 0) {
        msg.sprintf("Of the desired %d HGT elevation files, found NONE!", srtmcnt2);
    } else if (count2 < srtmcnt2) {
        msg.sprintf("Of the possible %d HGT elevation files, found URL for %d ", srtmcnt2, count2);
        msg += "\nIt must be remembered there will be NO elevation files for ocean areas, so this could be the correct count of what is needed.";
    } else {
        msg.sprintf("Found the URL for ALL the desired %d HGT elevation files.", srtmcnt2);
    }
    if (opnd) {
        msg += "\nIn any case the URL should be open in your browser, so you could MANUALLY";
        msg += info;
    }
    if (skip_exist_zip && skipped) {
        argument.sprintf("\n\nDue to skip existing checked, skipping %d, leaving %d for download", skipped, argList.size());
        msg += argument;
    }
    if (skip_cfm) {
        msg += "\nOption 'Skip confirmation' checked, so this will be the only confirmation.";
        if (ign_errors)
            msg += "\nand will also not confirm on a download error.";
        else
            msg += "\nbut will confirm if a download error detected.";
    } else {
        msg += "\nOption 'Skip confirmation' is not checked, so will confirm after each download.";
    }
    if (argList.count() == 0) {
        msg += "\n\nAdvice only, since there is no work to do.";
    } else {
        msg += "\n\nDo you want to CONTINUE with this download?";
    }
    outLog("CONFIRMATION TO PROCEED\n");
    outLog(msg+"\n");
    res = getYesNo("CONFIRMATION TO PROCEED", msg);
    if (!res) {
       return; // down hgt - no confirmation to proceed
    }
    // got the set of download arguments to run...
    // and the name of the desired file
    m_User_Break = false;
    for (i = 0; i < argList.size(); i++) {
        if (m_User_Break)
            break;
        do_wait = (skip_cfm ? false : true);
        msg.sprintf("%d of %d: ", (i+1), argList.size());
        argument = argList[i];
        file = filList[i];
        outfile = m_srtmDir+"/"+file;
        outLog(msg+"Downloading ["+file+"] with arg ["+argument+"]\n");
        // YUK, 'wget' needs params to overwrite existing,
        // but that gets complicated, so...
        ext = util_renameToOLDBAK(outfile); // rename any exsiting
        outArg(argument);
        info = util_runProcess(argument,m_srtmDir);
        util_trimLongMessage(m_MaxDiag, info);
        info += "\nFile "+file+" ";
        if (f.exists(outfile)) {
            // was this an already existing
            if (allZips.indexOf(outfile) == -1)
                info += "DOWNLOADED\n"; // no, is NEW download
            else
                info += "RE-DOWNLOAD\n";
            m_zipList += outfile;
        } else {
            if (!ign_errors)
                do_wait = true;
            info += "failed (does not exist)\n";
        }
        info += "\n";
        outLog(info+"\n");
        if (do_wait) {
            if ((i + 1) < argList.size()) {
                info += msg+"Continue with next download?";
                res = getYesNo("DONE DOWNLOAD",info);
                if (!res)
                    return; // down hgt - done one, but no cfm to proceeed
            }
        } else {
            openNonModalDialog("DOWNLOAD INFORMATION",info,wait_secs);
        }
    }

    // update plainTextEdit_hgtinfo
    on_tabWidget_currentChanged(ind_Elev);
    // done the LAST, or aborted by user
    msg.sprintf("%d of %d: ", i, argList.size());
    if (m_User_Break)
        info += "\nUser abort"+msg;
    else
        info += "\nDone "+msg;
    if (i == argList.size()) {
        info += "\nThe next step would be to do the UNZIPPING?";
    } else {
        info += "\nIt seems the download is unfinished!";
        info += "\nThat needs to be completed before the UNZIPPING?";
    }
    m_User_Break = false;
    wait_secs = (wait_secs > 5) ? wait_secs : 5;
    openNonModalDialog("DONE HGT ZIP DOWNLOAD",info,wait_secs);

}

// unzipping the HGT ZIP files, if any found
void MainWindow::on_pushButton_srtmuz_clicked()
{
    QString info;
    // ===============================================
    // Unable to proceed, if no valid project directory
    if (!util_verifyProjDir(projectDir))
        return; // uz hgt - projectDir invalid
    if (!util_createDirectory(m_srtmDir))
        return; // uz hgt - m_srtmDir for output invalid
    int i;
    QStringList allZips = findFiles(m_srtmDir, QStringList() << "*.zip", false);
    QStringList allHGT = findFiles(m_srtmDir, QStringList() << "*.hgt", false);
    bool skip_exist_hgt = ui->checkBox_skiphgt->isChecked();
    int wait_secs = ui->lineEdit_secs->text().toInt();
    if (wait_secs < 1) wait_secs = 1;
    bool do_wait;
    bool skip_cfm = ui->checkBox_skipcfm->isChecked();
    bool ign_errors = ui->checkBox_skiperr_2->isChecked();
    m_Verify_Tools = ui->checkBox_verify->isChecked();

    // ================================================
    QStringList argList; // list of arguments to run
    QStringList filList; // and 'information' file list
    QString file;        // file name only
    QString outfile;     // fully qualified file name
    QString argument;    // build each argument
    QString runtime = toolUnzip; // get 'unzipt' tool
    // QStringList zipList;
    QString tmp;
    QString msg;
    QFile f;
    int ext, added, hgtskip, res;
    // potentially add previously UNZIPPED files to the arguments,
    // but ONLY if there is NO current unzipped HGT file
    added = 0;
    for (i = 0; i < allZips.size(); i++) {
        outfile = allZips[i]; // get the ZIP
        file = outfile;
        file.chop(4); // drop the '.zip', to get HGT file name
        if (m_zipList.indexOf(outfile) == -1) {
            // ok this zip is NOT in the LIST already
            if ( allHGT.indexOf(file) == -1) {
                // ok, there is no HGT for this ZIP
                m_zipList += outfile;
                added++;
            }
        }
    }

    // proceed to UNZIP the downloads, and/or others
    // not previously unzipped...
    if (m_zipList.size() == 0) {
        info.sprintf("Started with potentially %d files to unzip.", allZips.size());
        info += "\nBut it seems all have already been unzipped, so no unzipping to do.";
        info += "\n\nClick 'Yes' to continue anyway, else click 'No";
        if (!getYesNo("UNZIP ADVICE",info))
            return; // hgt unzip - appears all done, and user not continue
    }

    // check the unzip tool functions
    if (runtime.contains(QChar(' ')))
        runtime = "\""+runtime+"\"";
    argument = runtime;
    argument += " --help";
    res = util_testApplication(argument, info);
    if (res || m_Verify_Tools) {
        util_trimLongMessage(m_MaxDiag, info);
        tmp.sprintf("\nTool returned value %d",res);
        info += tmp;
        info += "\nThe name (and location, if not in your PATH)";
        info += " is set on the 'Advanced' pages.";
        outLog(info+"\n");
        info += "\n\nClick 'Yes' to continue to use this tool.";
        res = getYesNo("UNZIP TOOL",info);
        if (!res)
            return; // unzip tool NOT acceptable
    }

    // build the UNZIP set of arguments
    argList.clear();
    filList.clear();
    hgtskip = 0;
    for (i = 0; i < m_zipList.size(); i++) {
        outfile = m_zipList[i];
        argument = runtime+" "; // the 'tested' agreed wget tool
        argument += outfile;
        file = outfile;
        file.chop(4);
        if (skip_exist_hgt && (allHGT.indexOf(file) != -1)) {
            hgtskip++;
            continue;
        }
        argList += argument;
        filList += outfile;
    }

    if (argList.size() == 0) {
        info.sprintf("Started with %d files to unzip!", m_zipList.size());
        info += "\nBut due to skip exising options checked have NONE to process.";
        info += "\nRemove the check, and re-run if you want to re-do the unzipping.";
        QMessageBox::information(this,"NULL ADVICE",info);
        return; // no zip files to process
    }



    info.sprintf("Have found %d files to UNZIP.\n",argList.size());
    if (skip_exist_hgt && hgtskip) {
        tmp.sprintf("Found %d zip files, but skip existing checked, so skipping %d.\n", allZips.count(), hgtskip);
        info += tmp;
    }
    for (i = 0; i < filList.size(); i++) {
        QFileInfo fi(filList[i]);
        info += fi.fileName()+" ";
    }
    info += "\n";
    if (skip_cfm) {
        info += "\nDue to the skip confirm option, this will be the last confirmation";
        if (ign_errors)
            info += " even if there is a perceived error.";
        else
            info += " but will still ask on a perceived error.";
    } else {
        info += "\nSince skip confirm is not checked, will ask for continuation after each unzip.";
    }

    info += "\n\nProceed with the unzip?";
    res = getYesNo("UNZIP CONFIRMATION",info);
    if (!res)
        return; // hgt unzip - no cfm to proceeed

    m_User_Break = false;
    for (i = 0; i < argList.size(); i++) {
        if (m_User_Break)
            break;
        do_wait = (skip_cfm ? false : true);
        msg.sprintf("%d of %d: ", (i+1), argList.size());
        argument = argList[i];
        outfile = filList[i];
        file = outfile;
        file.chop(4);
        outLog("Unzipping ["+outfile+"] with arg ["+argument+"]\n");
        ext = util_renameToOLDBAK(file); // rename any exsiting
        outArg(argument);
        info = util_runProcess(argument,m_srtmDir);
        if ((m_MaxDiag > 2)&&(info.size() > m_MaxDiag)) {
            // we need to CHOP some info
            tmp = info.left(m_MaxDiag / 2);
            tmp += "...";
            tmp += info.right(m_MaxDiag / 2);
            outLog("Chop =["+info+"]=\n"); // to  =["+tmp+"]=\n");
            info = tmp;
        }
        info += "\nFile "+file+" ";
        if (f.exists(file)) {
            // was this an already existing
            if (allHGT.indexOf(file) == -1)
                info += "UNZIPPED\n"; // no, is NEW unzip
            else
                info += "RE-UNZIP\n";
        } else {
            if (!ign_errors)
                do_wait = true;
            info += "failed! (does not exist?)\n";
        }
        info += "\n";
        outLog(info+"\n");
        if ((i + 1) < argList.size()) {
            info += msg;
            if (do_wait) {
                info +="\n\nContinue with next unzip?";
                res = getYesNo("DONE UNZIP",info);
                if (!res)
                    return; // hgt unzip - done one, but no cfm to proceeed
            } else {
                openNonModalDialog("UNZIP INFORMATION",info,wait_secs);
            }
        }
    }
    if (m_User_Break) {
        info += "\nAborted due to user Abort";
    }
    if (i == argList.size())
        info += "\nCompleted all arguments.";
    else {
        tmp.sprintf("\nCompleted %d of %d arguments.",i,argList.size());
        info += tmp;
    }


    m_User_Break = false;
    wait_secs = (wait_secs > 5) ? wait_secs : 5;
    openNonModalDialog("UNZIP INFORMATION",info,wait_secs);
    // if we got to the end, then CLEAR the zip list
    m_zipList.clear();
    // update plainTextEdit_hgtinfo
    on_tabWidget_currentChanged(ind_Elev);
}

// decode HGT files into array (*.arr.gz) files
void MainWindow::on_pushButton_srtmdc_clicked()
{
    QString tot;
    QString cnt;
    QStringList argList;
    QStringList filList;
    QString elevationFile;
    QString arguments;
    QString info;
    QString msg;
    QString tmp;
    int i, res;
    QString elevationRes = ui->comboBox_hgt->currentText();
    int wait_secs = ui->lineEdit_secs->text().toInt();
    if (wait_secs < 1) wait_secs = 1;
    bool do_wait;
    bool skip_cfm = ui->checkBox_skipcfm->isChecked();
    bool ign_errors = ui->checkBox_skiperr_2->isChecked();
    m_Verify_Tools = ui->checkBox_verify->isChecked();

    QString outDir = m_elevDir; // = m_workDir+"/SRTM-30";
    QStringList workList;
    QStringList arrList1;
    QStringList fitList1;
    QStringList newList2;
    QStringList arrList2;
    QStringList fitList2;
    bool got_out = false;
    QDir wd;
    int work_cnt, arr_cnt, hgt_cnt, new_cnt;

    // Unable to proceed, if no valid project directory
    if (!util_verifyProjDir(projectDir))
        return; // decode hgt - projectDir invalid
    if (!util_createDirectory(m_srtmDir))
        return; // decode hgt - m_srtmDir for output invalid

    // find all the HGT files - NOT recursively
    QStringList hgt_list = findFiles(m_srtmDir, QStringList() << "*.hgt");
    hgt_cnt = hgt_list.size();
    // do we have some files?
    if (hgt_cnt == 0) {
        info += "\nThere are no elevation files in ["+m_srtmDir+"]";
        info += "\nPerhaps they are in some other directory?";
        info += "\n\nNothing to do! Aborting.";
        QMessageBox::information(this,"NOTHING TO DO",info);
        return; // nothing to PROCESS
    }

    QString runtime;
    runtime = toolDir+"/hgtchop";
    if (runtime.contains(QChar(' ')))
        runtime = "\""+runtime+"\"";
    arguments = runtime;
    arguments += " --help"; // we 'know' there is no such command!!!
    res = util_testApplication(arguments, info);
    if (res || m_Verify_Tools) {
        if ( !m_Verify_Tools && (res == 255) && util_lookslike_hgtchop(info)) {
            res = 0;
        }
        if (res || m_Verify_Tools) {
            util_trimLongMessage(m_MaxDiag, info);
            tmp.sprintf("\nTool returned value %d",res);
            info += tmp;
            info += "\nThe path to all TG tools ";
            info += " is set on the 'Setup' pages.";
            outLog(info+"\n");
            info += "\n\nClick 'Yes' to continue to use this tool.";
            res = getYesNo("DECODE TOOL",info);
            if (!res)
                return; // decode tool NOT acceptable
        }
    }

    work_cnt = 0;
    arr_cnt = 0;
    if ( wd.exists(outDir) ) {
        workList = findFiles(outDir,QStringList() << "*",true);
        arrList1 = findFiles(outDir,QStringList() << "*.arr.gz",true);
        fitList1 = findFiles(outDir,QStringList() << "*.fit.gz",true);
        work_cnt = workList.count();
        arr_cnt = arrList1.count();
        info.sprintf("Found work directory with %d files,\n%d array files, and %d fit files.",
                     work_cnt, arr_cnt, fitList1.count());
        if (work_cnt && (work_cnt == arr_cnt) &&
            (fitList1.size() == 0)) {
            info += "\nThis suggests the need to run 'Terrafit'!";
        }
        got_out = true;
    } else {
        info = "Output directory ["+outDir+"] does NOT yet exist!";
    }

    tot.sprintf("%d", hgt_list.size()); //
    if (outDir.contains(QChar(' ')))
        outDir = "\""+outDir+"\"";

    // build set of arguments
    for (i = 0; i < hgt_list.size(); ++i) {
        elevationFile = hgt_list.at(i);
        if (elevationFile.contains(QChar(' ')))
            elevationFile = "\""+elevationFile+"\"";

        arguments = runtime;
        arguments += " "+elevationRes+" "+elevationFile+" "+outDir;
        // store runtime argument, and file name
        argList += arguments;
        filList += elevationFile;
    }

    if (argList.size()) {
        msg.sprintf("Found %d *.hgt files to process (%d)", hgt_cnt, argList.count());
        msg += "\n"+info;
        if (got_out) {
            tot.sprintf("Found %d arrays already processed.", arrList1.count());
            msg += "\n"+tot;
        }
        if (skip_cfm) {
            msg += "\nThis will be the last confirmation.";
            if (ign_errors)
                msg += " Even if an appearent error is detected!";
            else
                msg += " But will stop of confirm if an error suspected.";
        } else {
            msg += "\nWill stop to confirm after each HGT decode.";
        }
        outLog(msg+"\n");
        msg += "\n\nProceed to process these HGT files?";
        res = getYesNo("DECODE CONFIRMATION",msg);
       if (!res) {
           return; // process hgt - no confirmation
        }
    } else {
        info = "There are no elevation files (*.hgt) in "+m_srtmDir+"\nNothing to do!";
        outLog(info+"\n");
        QMessageBox::information(this,"ERROR",info);
        return; // process hgt - no files found to process
    }

    tot.sprintf("%d", hgt_list.size());
    // process set of arguments
    m_User_Break = false;
    for (i = 0; i < argList.size(); i++) {
        if (m_User_Break)
            break;
        do_wait = (skip_cfm ? false : true);
        arguments = argList[i];
        elevationFile = filList[i];
        outLog(elevationFile+"\n"+arguments+"\n");
        cnt.sprintf("%d of %d:", (i + 1), argList.size());
        outArg(arguments);
        info = util_runProcess(arguments,"");
        arrList2 = util_getArrayFileList(info);
        util_trimLongMessage(m_MaxDiag, info);
        info += "\n*PROC_END*\n";
        // newList2 = findFiles(outDir,QStringList() << "*.*",true);
        // arrList2 = findFiles(outDir,QStringList() << "*.arr.gz",true);
        if (arrList2.count()) {
            tmp.sprintf("Genererated %d array files.\n", arrList2.count());
            info += tmp;
            new_cnt = 0;
            foreach (tmp, arrList2) {
                if (arrList1.indexOf(tmp) == -1) {
                    new_cnt++;
                    arrList1 += tmp;
                }
            }
            if (new_cnt)
                tmp.sprintf("%d were NEW files.",new_cnt);
            else
                tmp = "\nbut no NEW files.";
            info += tmp;
        } else {
            // this is really the ONLY indication of error
            // *** NOTHING WAS GENERATED ***
            info += "WARNING: No new array files generated!\n";
            if (!ign_errors)
                do_wait = true;

        }
        arr_cnt = arrList2.count();
        outLog(info);
        if ((i + 1) < argList.size()) {
            if (do_wait) {
                info += "\n\n"+cnt+"Continue with next DECODING?";
                res = getYesNo("DONE DECODE",info);
                if (!res)
                    return; // hgt chop - done one, but no cfm to proceed
            } else {
                openNonModalDialog("HGT CHOP INFORMATION",info,wait_secs);
            }
        }
    }

    newList2 = findFiles(outDir,QStringList() << "*.*",true);
    arrList2 = findFiles(outDir,QStringList() << "*.arr.gz",true);
    fitList2 = findFiles(outDir,QStringList() << "*.fit.gz",true);
    msg.sprintf("Started with %d chopped files, now have %d (arr=%d, fit=%d)", workList.size(), newList2.size(), arrList2.size(), fitList2.size());
    if (arrList2.size() == 0) {
        msg += "\nWARNING: Appears 'decode hgt' FAILED to generate any array files!";
        msg += "\nWITHOUT these elevation arrays, any scenery generated will be FLAT ;=((";
        msg += "\nCan only suggest check ALL parameters, folders, and running this AGAIN";
    } else if (newList2.size() > workList.size()) {
        msg += "\nCongrats you have successfully generated some elevation files.";
    } else {
        msg += "\nWARNING: No NEW array files generated!";
    }
    outLog(msg+"\n");
    m_User_Break = false;
    wait_secs = (wait_secs > 5 ? wait_secs : 5);
    openNonModalDialog("HGT CHOP INFORMATION",info,wait_secs);
    // update plainTextEdit_hgtinfo
    on_tabWidget_currentChanged(ind_Elev);
}

// run terrafit
void MainWindow::on_pushButton_srtmtf_clicked()
{
    QString cnt;
    QString info;
    QString msg;
    QString tmp;
    QString minnode      = ui->lineEdit_minnodes->text();
    QString maxnode      = ui->lineEdit_maxnodes->text();
    QString maxerror     = ui->lineEdit_maxerr->text();
    //bool skip_err        = ui->checkBox_skiperr_2->isChecked();
    QString outDir = m_workDir+"/SRTM-30";
    QStringList workList;
    QStringList arrList1;
    QStringList fitList1;
    QString arguments;
    QString runtime;
    QDir wd;
    int res;

    // Unable to proceed, if no valid project directory
    if (!util_verifyProjDir(projectDir))
        return; // decode hgt - projectDir invalid
    if (!util_createDirectory(m_srtmDir))
        return; // decode hgt - m_srtmDir for output invalid

    m_Verify_Tools = ui->checkBox_verify->isChecked();
    runtime = toolDir+"/terrafit";
    if (runtime.contains(QChar(' ')))
        runtime = "\""+runtime+"\"";
    arguments = runtime;
    arguments += " --help"; // we 'know' there is no such command!!!
    res = util_testApplication(arguments, info);
    if (res || m_Verify_Tools) {
        // any other test?
        if (!m_Verify_Tools && (res == 1) && util_lookslike_terrafit(info))
            res = 0;

        if (res || m_Verify_Tools) {
            util_trimLongMessage(m_MaxDiag, info);
            tmp.sprintf("\nTool returned value %d",res);
            info += tmp;
            info += "\nThe path to all TG tools ";
            info += " is set on the 'Setup' pages.";
            outLog(info+"\n");
            info += "\n\nClick 'Yes' to continue to use this tool.";
            res = getYesNo("TERRAFIT TOOL",info);
            if (!res)
                return; // terrfit tool NOT acceptable
        }
    }

    if ( wd.exists(outDir) ) {
        workList = findFiles(outDir,QStringList() << "*",true);
        arrList1 = findFiles(outDir,QStringList() << "*.arr.gz",true);
        fitList1 = findFiles(outDir,QStringList() << "*.fit.gz",true);
        info.sprintf("Found work directory with %d files,\n%d array files, and %d fit files.",
                     workList.count(), arrList1.count(), fitList1.count());
    } else {
        info = "Work directory ["+outDir+"] does NOT exist!";
        info += "\nNeed to run 'decode hgt' to generate these files.";
        // get list of HGT files - NOT recursively - if any
        QStringList list = findFiles(m_srtmDir, QStringList() << "*.hgt");
        if (list.size()) {
            cnt.sprintf("\n\nHave found %d hgt files, ", list.size());
            cnt += "ready to run 'decode hgt' in ["+m_srtmDir+"]";
            info += cnt;
        } else {
            info += "\nAlso have not found any hgt files to process!";
            info += "\nDo you need to download, and/or unzip these?";
        }
        info += "\n\nNothing for 'terrafit' to do here!";
        QMessageBox::information(this,"ERROR",info);
        return; // terrafit - nothing to do!?
    }

    if (arrList1.size() == 0) {
        info += "\nNote the are NO array files to process!";
        info += "\n\nNothing to do!";
        QMessageBox::information(this,"ERROR",info);
        return; // terrafit - no arrays to process
    }
    if (arrList1.size() == fitList1.size()) {
        info += "\nNote the number of array equals the number of fit files already.";
        info += "\nterrafit will skip where there is an existing 'fit' file.";
        info += "\nIf you have changed some parameter, then you need to delete the current files first to force 'terrafit' to redo them.";
    } else if ((arrList1.size() > fitList1.size())) {
        if (fitList1.size() == 0)
            info += "\nThis looks like the first run of 'terrafit'";
        else {
            cnt.sprintf("\nRun to get balance of %d fit files.", arrList1.size() - fitList1.size());
            info += cnt;
        }
    }

    // generate the terrafit command
    arguments = runtime;
    if (minnode.size() > 0){
        arguments += " --minnodes "+minnode;
    }
    if (maxnode.size() > 0){
        arguments += " --maxnodes "+maxnode;
    }
    if (maxerror.size() > 0){
        arguments += " --maxerror "+maxerror;
    }

    if (outDir.contains(QChar(' ')))
        outDir += "\""+outDir+"\"";
    arguments +=" "+outDir;

    info += "\nRun terrafit with ["+arguments+"]";

    info += "\n\nContinue with 'terrafit'?";
    if (!getYesNo("TERRAFIT CONFIRMATION",info))
        return; // terrafit - no confirmation to proceed

    outArg(arguments);
    info = "Moment... Doing terrafit... with arguments\n"+arguments;
    info += "\nDuring this period the GUI will be FROZEN\nNo paints, etc will be done\nHave patiences until this dialog closes...";
    SHOWWAITDIALOG("DOING TERRAFIT",info);
    qApp->processEvents();
    info = util_runProcess(arguments,"");
    CLOSEWAITDIALOG;
    util_trimLongMessage(m_MaxDiag, info);

    QStringList newList2 = findFiles(outDir,QStringList() << "*.*",true);
    QStringList arrList2 = findFiles(outDir,QStringList() << "*.arr.gz",true);
    QStringList fitList2 = findFiles(outDir,QStringList() << "*.fit.gz",true);
    msg.sprintf("\nStarted with %d chopped files, now have %d (arr=%d, fit=%d)", workList.size(), newList2.size(), arrList2.size(), fitList2.size());
    if (fitList2.size() == 0) {
        msg += "\nWARNING: Appears 'terrafit' FAILED to generate any 'fit' files!";
        if (arrList2.size()) {
            msg += "\nThe construction tool does have a built-in terrafit function,";
            msg += " so this is not a big problem. But why did '"+runtime+"' fail to generate fit files?";
        }
        msg += "\nWITHOUT these elevation arrays, any scenery generated will be FLAT ;=((";
        msg += "\nCan only suggest check ALL parameters, folders, and running this AGAIN";
    } else if (fitList1.size()) {
        // we started with some fit files, were any NEW ones generated
        if (fitList2.size() > fitList1.size()) {
            tmp.sprintf("%d", fitList2.size() - fitList1.size());
            msg += "\nYou have successfully generated "+tmp+" new fit files.";
        } else {
            msg += "\nWARNING: It appears no new fit files were generated!";
        }
    } else if (arrList2.size()) {
        tmp.sprintf("%d", fitList2.size());
        msg += "\nCongrats! You have generated "+tmp+" fit files.";
    } else {
        msg += "\nWARNING: No NEW fit array files generated!";
    }
    if (arrList2.size() && (arrList2.size() == fitList2.size())) {
        msg += "\nFit file count is the same as the arr file count, which is good, and suggests you can move onto the next stage, ";
        msg += "getting the Landuse arrays, or generating airports.";
    }

    // update plainTextEdit_hgtinfo
    on_tabWidget_currentChanged(ind_Elev);
    info += msg;
    outLog(info+"\n");
    QMessageBox::information(this,"TERRAFIT INFORMATION",info);
}

// automate the above steps
void MainWindow::on_pushButton_elev_clicked()
{
    on_pushButton_srtm_clicked();   // download
    on_pushButton_srtmuz_clicked(); // unzip
    on_pushButton_srtmdc_clicked(); // decode
    on_pushButton_srtmtf_clicked(); // terrafit
}


// some utility functions
void MainWindow::updateDirectories()
{
    QPalette p;
    QDir d;
    if (projectDir.size() && d.exists(projectDir))
        p.setColor(QPalette::WindowText, Qt::black);
    else
       p.setColor(QPalette::WindowText, Qt::red);
    ui->label_pd->setPalette(p);

    if (toolDir.size() && d.exists(toolDir))
        p.setColor(QPalette::WindowText, Qt::black);
    else
        p.setColor(QPalette::WindowText, Qt::red);
    ui->label_td->setPalette(p);

    if (fgrootDir.size() && d.exists(fgrootDir)) {
        p.setColor(QPalette::WindowText, Qt::black);
        // check for the airport database
        if (m_aptFile.size() == 0) {
            QString af = fgrootDir+"/Airports/apt.dat.gz";
            QFile apf(af);
            if (apf.exists()) {
                m_aptFile = af;
                settings.setValue("files/airport",m_aptFile);
                ui->lineEdit_ap->setText(m_aptFile);
            }
        }
    } else
       p.setColor(QPalette::WindowText, Qt::red);
    ui->label_rd->setPalette(p);

    // set composite directories
    if (projectDir.size()) {
        m_dataDir = projectDir+"/data"; // source (raw) data
        m_outpDir = projectDir+"/scenery"; // final construction
        m_workDir = projectDir+"/work"; // source (chopped) data
        m_tempDir = projectDir+"/temp"; // temporary work directory - for downloads
        m_elevDir = m_workDir+"/SRTM-30"; // HGT decode destination

        if (m_srtmDir.size() == 0) {
            m_srtmDir = m_tempDir+"/hgt"; // temp for HGT work files
            settings.setValue("paths/temphgt",m_srtmDir);
            ui->lineEdit_hgt->setText(m_srtmDir);
        }
        if (m_shapDir.size() == 0) {
            m_shapDir = m_tempDir+"/shape"; // temp for SHAPE work files
            settings.setValue("paths/tempshp",m_shapDir);
            ui->lineEdit_shp->setText(m_shapDir);
        }
        if (m_logFile.size() == 0) {
            m_logFile = projectDir+"/templog.txt";
            ui->lineEdit_lf->setText(m_logFile);
        }
        util_createLogFile(m_logFile); // RESTART LOG FILE
        if (m_argFile.size() == 0) {
            m_argFile = projectDir+"/temparg.txt";
            ui->lineEdit_lf_2->setText(m_logFile);
        }
    }
}

void MainWindow::outArg(QString s)
{
    if (m_argFile.size()) {
        QFile data(m_argFile);
        if (data.open(QFile::WriteOnly | QFile::Append | QFile::Text)) {
            QTextStream out(&data);
            out << s;
            out << endl; // terminate the runtime argument
        }
    }
}

// Landuse - download shape files
void MainWindow::on_pushButton_shp_clicked()
{
    bool add_landmass = ui->checkBox_lmadd->isChecked();
    double eastInt    = m_east.toDouble();
    double northInt   = m_north.toDouble();
    double southInt   = m_south.toDouble();
    double westInt    = m_west.toDouble();
    bool is_ok = false;
    QString base = "http://mapserver.flightgear.org/";
    QString url1 = "dlcs?xmin="+m_west+"&xmax="+m_east+"&ymin="+m_south+"&ymax="+m_north;
    QString url2 = "dlaction.psp?xmin="+m_west+"&xmax="+m_east+"&ymin="+m_south+"&ymax="+m_north+"&pgislayer=v0_landmass";
    QString info;
    QString tmp;
    QString argument1;
    QString argument2;
    QString msg;
    QString file1 = m_shapDir+"/"+url1;
    QString file2 = m_shapDir+"/"+url2;
    QString file1n = m_shapDir+"/landuse.zip";
    QString file2n = m_shapDir+"/landmass.zip";
    QFile f;
    int i, res;
    m_Verify_Tools = ui->checkBox_verify->isChecked();

    if ((westInt < eastInt) && (northInt > southInt) &&
         latlonInWorld(northInt,eastInt) &&
         latlonInWorld(southInt,westInt)) {
        is_ok = true;
    } else {
        QString msg = tr("Boundaries ERROR! min. not less than max, or out of world range.\nCorrect the strings on Setup, and try again.");
        outLog(msg+"\n");
        QMessageBox::about(this,tr("BOUNDARY ERROR"),msg);
    }
    if (!is_ok)
        return; // shp process - boundaries error

    if (!util_verifyProjDir(projectDir))
        return; // shp process - projectDir invalid
    // ensure download directory exist
    if (!util_createDirectory(m_shapDir))
        return; // shp process - m_shapDir output invlaid

    QString runtime = toolWget; // get 'wget' tool
    // check the wget tool functions
    if (runtime.contains(QChar(' ')))
        runtime = "\""+runtime+"\"";
    argument1 = runtime;
    argument1 += " --help";
    res = util_testApplication(argument1, info);
    if (res || m_Verify_Tools) {
        util_trimLongMessage(m_MaxDiag, info);
        tmp.sprintf("\nTool returned value %d",res);
        info += tmp;
        info += "\nThe name (and location if not in your PATH)";
        info += "\nis set on the 'Advanced' pages.";
        outLog(info+"\n");
        info += "\n\nClick 'Yes' to continue to use this tool.";
        res = getYesNo("DOWNLOAD TOOL",info);
        if (!res)
            return; // elevation download tool NOT acceptable
    }
    bool no_overwrite = ui->checkBox_noovr_2->isChecked();
    QStringList argList;
    QStringList filList;
    QStringList tarList;
    // going ahead to try the DOWNLOAD
    QStringList allList1 = findFiles(m_shapDir, QStringList() << "*");
    QStringList zipList1 = findFiles(m_shapDir, QStringList() << "*.zip");
    argument1 = runtime+" ";
    argument1 += base+url1;
    argument2 = runtime+" ";
    argument2 += base+url2;

    info = "About to attempt the download of mapserver shapefiles ";
    is_ok = true;
    if (no_overwrite && f.exists(file1n)) {
        is_ok = false;
    }
    if (is_ok) {
        argList += argument1;
        filList += file1; // url name
        tarList += file1n; // target (simple) name
        info += "landuse.zip ";
    }
    if (add_landmass) {
        is_ok = true;
        if (no_overwrite && f.exists(file2n)) {
            is_ok = false;
        }
        if (is_ok) {
            argList += argument2;
            filList += file2;   // URL NAME
            tarList += file2n;  // target (simple) name
            info += "landmass.zip ";
        }
    }
    if (argList.size() == 0) {
        info = "Zip file 'landuse.zip'' ";
        if (add_landmass)
            info += "and 'landmass.zip' ";
        info += "\nalready exist in ["+m_shapDir+"]";
        info += "\nEither remove or rename these files,";
        info += "\nor uncheck the no overwrite option.";
        info += "\n\nPresently nothing to do!";
        QMessageBox::information(this,"DOWNLOAD INFORMATION",info);
        return; // shp unzip - but no overwrite option on
    }

    //info += "\nUsing the command ["+argument1+"]";
    info += "\nThe donwload will be to ["+m_shapDir+"]";
    info += "\n\nProceed with the download?";
    if (!getYesNo("DOWNLOAD CONFIRMATION",info))
        return; // shp download - no confirmation

    QString argument;
    QString file;
    QString tarfile;
    m_User_Break = false;
    for (i = 0; i < argList.size(); i++) {
        if (m_User_Break)
            break;
        argument = argList[i];
        file = filList[i];
        tarfile = tarList[i];
        outArg(argument+"\n"+tarfile+"\n");
        info = "Moment... Doing download... with arguments\n"+argument;
        info += "\nDuring this period the GUI will be FROZEN\nNo paints, etc will be done\nHave patiences until this dialog closes...";
        SHOWWAITDIALOG("DOWNLOAD SHAPE FILES",info)
        qApp->processEvents();
        info = util_runProcess(argument,m_shapDir);
        CLOSEWAITDIALOG;
        tmp = util_extractZipFile(info);
        if (tmp.size() && f.exists(m_shapDir+"/"+tmp))
            file = m_shapDir+"/"+tmp;
        util_trimLongMessage(m_MaxDiag, info);
        if (!f.exists(file)) {
            // ok, has a NEW new zip file arrived?
            is_ok = false;
            QStringList zipList2 = findFiles(m_shapDir, QStringList() << "*.zip");
            if (zipList2.size() > zipList1.size()) {
                // well, there is a new file
                for (i = 0; i < zipList2.size(); i++) {
                    tmp = zipList2[i];
                    if (zipList1.indexOf(tmp) == -1) {
                        // HA, a NEW zip file ;=))
                        file = tmp; // assume this is it
                        is_ok = true;
                        break;
                    }
                }
            }
            if (!is_ok) {
                // ok, has a NEW new file arrived, of any name?
                QStringList allList2 = findFiles(m_shapDir, QStringList() << "*");
                if (allList2.size() > allList1.size()) {
                    // well, there is a new file, or more
                    for (i = 0; i < allList2.size(); i++) {
                        tmp = allList2[i];
                        if (allList1.indexOf(tmp) == -1) {
                            // HA, a NEW zip file ;=))
                            file = tmp; // assume this is it
                            is_ok = true;
                            break;
                        }
                    }
                }
            }
        }
        if (f.exists(file)) {
            info += "\nAppears the download succeeded";
            info += "\nreceived ["+file+"]";
            if (tarfile != file) {
                if (f.exists(tarfile))
                    util_renameToOLDBAK(tarfile);
                if (f.rename(file,tarfile)) {
                    info += "\nrenamed to ["+tarfile+"]";
                } else {
                    info += "\nrename failed! Keeping received";
                    tarfile = file;
                }
            }
        } else {
            info += "\nERROR: Appears the DONWLOAD FAILED!";
            info += "\nNo file ["+file+"]";
        }
        if ((i+1) < argList.size()) {
            if ( ! getYesNo("CONFIRMATION TO CONTINUE",info) )
                return; // doing shp download - no configration to proceeed
        }
    } // for argument loop

    // maybe enable pushButton_shpuz, pushButton_fill, and pushButton_shpdec
    // and maybe run the fill, but that should be after the 'unzip'
    // on_tabWidget_currentChanged(ind_Elev);
    on_tabWidget_currentChanged(ind_Landuse);

    msg = "End of DOWNLOADS";
    outLog(info+"\n"+msg);
    QMessageBox::information(this,"DOWNLOAD INFORMATION",info);
    // =========================
    return; // STOP HERE FOR NOW - code following is just the old style manual download
    // =========================

    // previous MANUAL download and UNZIP
    // ==================================
    info = "\nWhen the zip is downloaded it should be expanded into your\n[Project Directory]/data.";
    // save output to log

    outLog(url1+"\n");
    QUrl qu(url1);
    if ( ! QDesktopServices::openUrl(qu) ) {
        QMessageBox::critical(this,"OPEN URL FAILED","Attempted to opn the URL ["+url1+"] but FAILED.\nCopy the URL to your browser"+info);
    } else {
        QMessageBox::information(this,"DATA DOWNLOAD","The URL ["+url1+"] should be displayed in your default browser, and a zip file downloaded.\nIf this fails, copy the URL to your browser manually."+info);
    }
    if (add_landmass) {
        info = "\nWhen the zip is downloaded it should be expanded into your\n[Project Directory]/data/v0_landmass/v0_landmass.[dbf|prj|shp|shx] files.";
        outLog(url2+"\n");
        QUrl q2(url2);
        if ( ! QDesktopServices::openUrl(q2) ) {
            QMessageBox::critical(this,"OPEN URL FAILED","Attempted to opn the URL ["+url2+"] but FAILED.\nCopy the URL to your browser"+info);
        } else {
            QMessageBox::information(this,"DATA DOWNLOAD","The URL ["+url2+"] should be displayed in your default browser, and a zip file downloaded\nof the form v0_landmass-<large-index>.zip...\nIf this fails, copy the URL to your browser manually."+info);
        }
    }

}

// unzip landmass (srtm) files
void MainWindow::on_pushButton_shpuz_clicked()
{
    QString file1n = m_shapDir+"/landuse.zip";
    QString file2n = m_shapDir+"/landmass.zip";
    QFile f;
    bool add_landmass = ui->checkBox_lmadd->isChecked();
    //bool no_overwrite = ui->checkBox_noovr_2->isChecked();

    QStringList argList;
    QStringList filList;
    QString runtime = toolUnzip;
    QString info;
    QString argument;
    QString file;
    QString tmp;
    QString cnt;
    int i, res;
    m_Verify_Tools = ui->checkBox_verify->isChecked();

    if (!util_verifyProjDir(projectDir))
        return; // shp process - projectDir invalid
    // ensure target directory exist
    if (!util_createDirectory(m_dataDir))
        return;

    // check the unzip tool functions
    if (runtime.contains(QChar(' ')))
        runtime = "\""+runtime+"\"";
    argument = runtime;
    argument += " --help";
    res = util_testApplication(argument, info);
    if (res || m_Verify_Tools) {
        util_trimLongMessage(m_MaxDiag, info);
        tmp.sprintf("\nTool returned value %d",res);
        info += tmp;
        info += "\nThe name (and location, if not in your PATH)";
        info += " is set on the 'Advanced' pages.";
        outLog(info+"\n");
        info += "\n\nClick 'Yes' to continue to use this tool.";
        res = getYesNo("UNZIP TOOL",info);
        if (!res)
            return; // unzip tool NOT acceptable
    }

    info = "Unzip file ";
    if (f.exists(file1n)) {
        argList += runtime+" "+file1n;
        filList += file1n;
        info += "landuse.zip ";
    }
    if (add_landmass && f.exists(file2n)) {
        argList += runtime+" "+file2n;
        filList += file2n;
        info += "and landmass.zip ";
    }
    if (argList.size() == 0) {
        info = "No relevant zip files found in ["+m_shapDir+"]";
        info += "\n\nNothing to do!";
        QMessageBox::information(this,"UNZIP INFORMATION",info);
        return;
    }
    info += "\nOutput to be put in ["+m_dataDir+"]";
    QStringList shpList = findFiles(m_dataDir, QStringList() << "*.shp" );
    if (shpList.size()) {
        tmp.sprintf("\nBut found %d 'shape' files!", shpList.size());
        info += tmp;
        info += "\nThis suggests UNZIP has already been run,";
        info += "\nand what is required is to 'Refresh' the list'!";
    }

    info += "\n\nContinue with UNZIP?";
    if (!getYesNo("UNZIP CONFIRMATION",info))
        return;

    m_User_Break = false;
    for (i = 0; i < argList.size(); i++) {
        if (m_User_Break)
            break;
        argument = argList[i];
        file = filList[i];
        outArg(argument);
        info = util_runProcess(argument,m_dataDir);
        if ((m_MaxDiag > 2)&&(info.size() > m_MaxDiag)) {
            // we need to CHOP some info
            tmp = info.left(m_MaxDiag / 2);
            tmp += "...";
            tmp += info.right(m_MaxDiag / 2);
            outLog("Chop =["+info+"]=\n"); // to  =["+tmp+"]=\n");
            info = tmp;
        }
        cnt.sprintf("%d of %d:", i+1, argList.size());
        if ((i+1) < argList.size()) {
            info += "\n\n"+cnt+" Done UNZIP. Continue with next?";
            if ( ! getYesNo("UNZIP CONFIRMATION",info) )
                return;
        }
    }

    // POST UNZIP
    tmp.sprintf("\nDone %d of %d UNZIP arguments.",i, argList.size());
    info += tmp;
    on_tabWidget_currentChanged(ind_Landuse);
    // enable push button if unzip appears successful
    bool enabled = false;
    QDir d;
    if (m_dataDir.size() && d.exists(m_dataDir)) {
        QDir sdir(m_dataDir);
        sdir.setFilter(QDir::Files | QDir::NoDotAndDotDot);
        QFileInfoList list = sdir.entryInfoList();
        for (i = 0; i < list.size(); i++) {
            QFileInfo finfo = list.at(i);
            if (finfo.suffix() == "shp") {
                enabled = true;
                break;
            }
        }
    }
    // enable/disable shape fill list
    ui->pushButton_fill->setEnabled(enabled);
    if (enabled && (ui->tblShapesAlign->rowCount() == 0)) {
        enabled = m_Auto_Roll;
        m_Auto_Roll = true;
        on_pushButton_fill_clicked();
        m_Auto_Roll = enabled;
    }
    QMessageBox::information(this,"DONE UNZIP",info);
}

// update shapefiles list for ogr-decode/shape-decode
void MainWindow::on_pushButton_fill_clicked()
{
    QString info;
    QString tmp;
    QFileInfo fInfo;
    QString file;
    int i, cnt;
    // move shapefiles to "private" directories
    QDir dir(m_dataDir); // search 'data' folder, for 'files'
    dir.setFilter(QDir::Files | QDir::NoDotAndDotDot);
    QFileInfoList fList = dir.entryInfoList();
    // search 'data' folder, for 'directories'
    dir.setFilter(QDir::Dirs | QDir::NoDotAndDotDot);
    QFileInfoList dList = dir.entryInfoList();
    QStringList dirs;
    info = "Searching in ["+m_dataDir+"] for shape files, directories.";

    cnt = 0;
    for (i = 0; i < fList.size(); ++i) {
        fInfo = fList.at(i);
        if (fInfo.suffix() == "shp") {
            file = fInfo.fileName();
            file.chop(4);
            if (dirs.indexOf(file) == -1) {
                dirs += file;
            }
            cnt++;
        }
    }
    tmp.sprintf("Found total %d files, %d dirs, %d shapefiles, %d types.",
                fList.size(),dList.size(),cnt,dirs.size());
    info += "\n"+tmp;
    if ((fList.size() == 0) && dList.size())
        info += "\nAppeara first part of listing is done.";

    outLog(info);
    info += "\n\nContinue to process these files?";
    if (!m_Auto_Roll) {
        if (!getYesNo("CONFIRM SHAPEFILE LISTING",info))
            return;
    }

    for (i = 0; i < fList.size(); ++i) {
        fInfo = fList.at(i);
        QString fPath = fInfo.absolutePath();
        QString fFilePath = fInfo.absoluteFilePath();
        QString fFileName1 = fInfo.fileName();
        QString fFileName2 = fInfo.fileName();

        // move only shapefiles
        if ((fInfo.suffix() == "dbf")||
            (fInfo.suffix() == "prj")||
            (fInfo.suffix() == "shp")||
            (fInfo.suffix() == "shx")) {
            fFileName1.chop(4);     // remove fileformat from name
            //QFile file (fFilePath);
            QString fPath_ren = fPath+"/"+fFileName1+"/"+fFileName2;
            dir.mkpath(fPath+"/"+fFileName1);
            dir.rename(fFilePath, fPath_ren);
        }
    }

    // update list
    while (ui->tblShapesAlign->rowCount() != 0)
    {
        ui->tblShapesAlign->removeRow(0);
    }

    // list of custom scenery shapefiles
    QStringList csShape;
    csShape = util_getCustomShapes();
    // list of correpsonding materials
    QStringList csMater; // *TBD* - should compare this to default_priorities.txt
    csMater = util_getMaterialList();
    dir.setFilter(QDir::Dirs | QDir::NoDotAndDotDot);
    QFileInfoList dirList = dir.entryInfoList(); // get updated directory list

    for (int i = 0; i < dirList.size(); ++i) {
        QFileInfo dirInfo = dirList.at(i);
        file = dirInfo.fileName();
        if (srtm_isElevationDir(file)) // should NOT be in here, but
            continue;
        // add this one
        ui->tblShapesAlign->insertRow(ui->tblShapesAlign->rowCount());
        QTableWidgetItem *twiCellShape = new QTableWidgetItem(0);
        twiCellShape->setText(tr(qPrintable(QString("%1").arg(dirInfo.fileName()))));
        twiCellShape->setFlags(Qt::ItemFlags(Qt::ItemIsSelectable | Qt::ItemIsEnabled));
        ui->tblShapesAlign->setItem(ui->tblShapesAlign->rowCount()-1, 0, twiCellShape);
        QTableWidgetItem *twiCellMater = new QTableWidgetItem(0);

        // suggest a material
        QString suggestedMaterial;
        for (int j = 0; j < csShape.length(); ++j) {
            if (dirInfo.fileName() == csShape[j]){
                suggestedMaterial = csMater[j];
            }
        }
        twiCellMater->setText(suggestedMaterial);
        ui->tblShapesAlign->setItem(i, 1, twiCellMater);
    }
    ui->tblShapesAlign->resizeRowsToContents();

    // check for tool existance
    QString tgTool = toolDir+"/ogr-decode";
#ifdef Q_OS_WIN
    tgTool += ".exe";
#endif
    QFile f(tgTool);
    if ( ! f.exists() ) {
        tgTool = toolDir+"/shape-decode";
#ifdef Q_OS_WIN
        tgTool += ".exe";
#endif
        if (f.exists(tgTool)) {
            ui->checkBox_useshp->setCheckState(Qt::Checked);
        }
    }
    on_tabWidget_currentChanged(ind_Landuse);
}


// run ogr-decode (or shape-decode)
void MainWindow::on_pushButton_shpdec_clicked()
{
    QString msg;
    QStringList argList;
    QStringList shpList;
    QString arguments;
    QString info;
    QString shapefile;
    QString runtime;
    QString tmp;
    QString material;
    QString lineWidth;
    QString outDir;
    QStringList list;
    QFile f;
    QDir d;
    int i, res, no_ovr_count;
    bool shp_checked = ui->checkBox_useshp->isChecked();
    bool skip_cfm = ui->checkBox_skipcfm_3->isChecked();
    m_Verify_Tools = ui->checkBox_verify->isChecked();
    bool skip_errors = ui->checkBox_skiperr_3->isChecked();
    bool no_ovr = ui->checkBox_noovr_3->isChecked();
    bool do_wait;
    int wait_secs = ui->lineEdit_secs->text().toInt();
    if (wait_secs < 1) wait_secs = 1;

    // check if decode tool exists
    runtime = toolDir+"/ogr-decode";
    if (shp_checked) {
        runtime = toolDir+"/shape-decode";
    }
#ifdef Q_OS_WIN
    // does not seem necessary - runtime.replace(QChar('/'),QChar('\\'));
    runtime += ".exe"; // add EXE for windows
#endif
    if ( !f.exists(runtime) ) {
        msg = "Unable to locate executable ["+runtime+"]";
        if (!shp_checked) {
            runtime = toolDir+"/shape-decode";
#ifdef Q_OS_WIN
            // does not seem necessary runtime.replace(QChar('/'),QChar('\\'));
            runtime += ".exe";
#endif
            if (f.exists(runtime)) {
                msg += "\nhowever have found ["+runtime+"]";
                msg += "\nwhich will do the SAME job.";
                msg += "\nCheck the 'Use shaped-decode' option";
                msg += "\n and try again.";
            } else {
                msg += "\nand also can NOT locate ["+runtime+"]!";
                msg += "\nCheck 'Setup' for the correct TG Tool Directory!";
            }
        }
        if (!d.exists(toolDir)) {
            msg += "\nIN FACT can NOT find ["+toolDir+"]!";
            msg += "\nTHIS MUST BE FIXED IN SETUP";
        }
        QMessageBox::critical(this,"ERROR: NO TOOL FILE", msg);
        return;
    }

    // check the decode tool functions
    if (runtime.contains(QChar(' ')))
        runtime = "\""+runtime+"\"";
    arguments = runtime;
    arguments += " --help";
    res = util_testApplication(arguments, info);
    if (res || m_Verify_Tools) {
        if (!m_Verify_Tools && (res == 255) && util_lookslike_shape_decode(info)) {
            // looks good to go ;=))
        } else {
            // ask the user to continue
            util_trimLongMessage(m_MaxDiag, info);
            tmp.sprintf("\nTool returned value %d",res);
            info += tmp;
            info += "\nThe Terragear Tool directory is set on the Setup page.";
            outLog(info+"\n");
            info += "\n\nClick 'Yes' to continue to use this tool.";
            res = getYesNo("DECODE TOOL",info);
            if (!res)
                return; // decode tool NOT acceptable
        }
    }

    // check that EACH has a material assigned
    for (i = 0; i < ui->tblShapesAlign->rowCount(); i++) {
        // skip if material are not assigned
        material = ui->tblShapesAlign->item(i, 0)->text();
        if ((ui->tblShapesAlign->item(i, 1) == 0)||
            (ui->tblShapesAlign->item(i, 1)->text().length() == 0)) {
            info = "ERROR: You did not assign a material to ["+material+"] shapefile.";
            QMessageBox::critical(this,"ERROR: NO MATERIAL", info);
            return;
        }
    }

    // got executable, and have assigned materials,
    // so build up a work list
    no_ovr_count = 0;
    msg = "";
    for (i = 0; i < ui->tblShapesAlign->rowCount(); i++) {

        // first column - get the mapeserver material
        material = ui->tblShapesAlign->item(i, 0)->text();
        if (ui->tblShapesAlign->item(i, 2) != 0) {
            // cell item already created
            lineWidth = ui->tblShapesAlign->item(i, 2)->text();
        } else {
            // cell item are not created - default width
            lineWidth = "10";
        }

        // we get material assigned to this shapefile name
        shapefile = ui->tblShapesAlign->item(i, 1)->text();
        outDir = m_workDir+"/"+shapefile;
        if (outDir.contains(QChar(' ')))
            outDir = "\""+outDir+"\"";
        if (no_ovr) {
            // could go on and check there are decode array files, for these buckets, in this folder,
            // but for the moment, assume yes if the directory exists, and contains files
            if ( d.exists(outDir) ) {
                list = findFiles(outDir, QStringList() << "*", true);
                if (list.size()) {
                    tmp.sprintf("%d", list.size());
                    msg += "\nMaterial ["+shapefile+"], folder ["+outDir+"] already contains "+tmp+" files.";
                    no_ovr_count++;
                    continue; // onto the next
                }
            }
        }

        if (shp_checked) {
            /* until I can get ogr-decode built ;=() */
            arguments   = runtime+" ";
            arguments += "--line-width "+lineWidth+" ";

            if (ui->lineEdit_24->text() > 0){
                arguments += "--point-width "+ui->lineEdit_24->text()+" ";
            }
            if (ui->checkBox_2->isChecked() == 1){
                arguments += "--continue-on-errors ";
            }
            if (ui->lineEdit_26->text() > 0){
                arguments += "--max-segment "+ui->lineEdit_26->text()+" ";
            }
            // last 3 arguments
            // shape file, with no extension, in 'data' directory
            arguments += "\""+m_dataDir+"/"+material+"/"+material+"\" ";
            // work directory - where to put the decoded output
            arguments += outDir+" ";
            // area string = MATERIAL
            arguments += " "+shapefile+" ";
        } else {
            arguments   = runtime+" ";
            arguments += "--line-width "+lineWidth+" ";
            if (ui->lineEdit_24->text() > 0){
                arguments += "--point-width "+ui->lineEdit_24->text()+" ";
            }
            if (ui->checkBox_2->isChecked() == 1){
                arguments += "--continue-on-errors ";
            }
            if (ui->lineEdit_26->text() > 0){
                arguments += "--max-segment "+ui->lineEdit_26->text()+" ";
            }
            arguments += "--area-type "+shapefile+" ";
            arguments += "\""+m_workDir+"/"+shapefile+"\" ";
            arguments += "\""+m_dataDir+"/"+material+"\"";
        }
        argList += arguments;
        shpList += shapefile;
    }

    info.sprintf("Got %d arguments to process", argList.size());
    info += "\nwith ["+runtime+"]\n";
    for (i = 0; i < shpList.size(); i++) {
        shapefile = shpList[i]; // this is actually the MATERIAL name
        info += shapefile+" ";  // which I prefer to add, rather than mapserver's
        // cs_<material> strings...
    }
    if (skip_cfm) {
        info += "\nDue to the checked skip confirm option, this will the the last confirmation";
        if (skip_errors)
            info += " including a perceived error condition.";
        else
            info += " but will stop of a perceived error condtion.";
    } else {
        info += "\nWill confirm after each processing.";
    }

    if (no_ovr_count) {
        tmp.sprintf("\nSkipping %d due to 'No overwrite' options", no_ovr_count);
        info += tmp;
        info += msg;
    } else {
        info += "\nNo check made for no overwrites.";
    }
    if (argList.size() == 0) {
        // oops, nothing to do
        info += "\n\nDue to the options chosen, or for other reasons, there are NO shapefiles to decode.";
        QMessageBox::information(this,"NO SHAPE DECODE",info);
        return; // no arguments available for shape decode

    }
    info += "\n\nContinue SHAPE DECODE processing?";
    if (!getYesNo("SHAPE DECODE CONFIRMATION",info))
        return;

    // got all the arguments prepared,
    // now process then one by one...
    info = "";

    m_User_Break = false;
    for (i = 0; i < argList.size(); i++) {
        if (m_User_Break)
            break;
        do_wait = (skip_cfm ? false : true);
        arguments = argList[i];
        shapefile = shpList[i];
        // save commandline to log
        outArg(arguments);
        SHOWWAITDIALOG("DECODE SHAPE FILES",info)
        qApp->processEvents();
        info = util_runProcess(arguments,"");
        CLOSEWAITDIALOG;
        util_trimLongMessage(m_MaxDiag, info);
        msg.sprintf("%d of %d:", i+1, argList.size());
        info += "\n"+msg+" Done shape ["+shapefile+"]";
        if ((i + 1) < argList.size()) {
            outLog(info+"\n");
            if (do_wait) {
                info += "\n\nContinue to next?";
                res = getYesNo("CONTINUE SHAPE DECODE",info);
                if (!res)
                    return; // no confirmation to continue shape decode
            } else {
                openNonModalDialog("DECODE INFORMATION",info,wait_secs);
            }

        }
    }

    // POST DECODE
    msg.sprintf("\nDone %d of %d DECODE arguments", i, argList.size());
    info += msg;
    if (m_User_Break)
        info += "\nAborted due to a user break!";
    outLog(info+"\n");
    on_tabWidget_currentChanged(ind_Landuse);
    do_wait = (skip_cfm ? false : true);
    m_User_Break = false;
    if (do_wait) {
        QMessageBox::information(this,"DONE DECODE",info);
    } else {
        wait_secs = (wait_secs > 5 ? wait_secs : 5);
        openNonModalDialog("DECODE INFORMATION",info,wait_secs);
    }
    m_User_Break = false;
}


// populate material list with materials from FG's materials.xml
// or a default set if no 'materials.xml' file
void MainWindow::updateMaterials()
{
    QStringList materialList;
    QFile materialfile(fgrootDir+"/materials.xml");
    if (materialfile.exists() == true) {
        if (materialfile.open(QIODevice::ReadOnly)) {
            QXmlStreamReader materialreader(&materialfile);
            QXmlStreamReader::TokenType tokenType;
            QString material;
            while ((tokenType = materialreader.readNext()) != QXmlStreamReader::EndDocument) {
                if (materialreader.name() == "material") {
                    while ((tokenType = materialreader.readNext()) != QXmlStreamReader::EndDocument) {
                        if (materialreader.name() == "name") {
                            material = materialreader.readElementText();
                            // ignore materials already present
                            if (materialList.indexOf(material, 0) == -1)
                                materialList.append(material);
                        }
                        // ignore sign materials
                        if (materialreader.name() == "glyph") {
                            materialreader.skipCurrentElement();
                        }
                    }
                }
            }
            materialfile.close();
        }
    }
    if (materialList.size() == 0)
        materialList = util_getMaterialList();
    materialList.sort();
    ui->comboBox_mat->clear();
    ui->comboBox_mat->addItems(materialList);
}

/************************
    Advanced Page
 ************************/
// given a log NEW file
void MainWindow::on_lineEdit_lf_editingFinished()
{
    QString file;
    QFile f;
    file = ui->lineEdit_lf->text();
    if (m_logFile != file) {
        m_logFile = file;
        settings.setValue("files/log",m_logFile);
        if (!f.exists(m_logFile))
            util_createLogFile(m_logFile); // init file
    }

}

// selected, or entered a new log file to use
void MainWindow::on_pushButton_blf_clicked()
{
    QString file;
    file = QFileDialog::getSaveFileName(this, tr("Log File"),
                                        m_logFile,
                                        tr("All (*.*)"),
                                        0,
                                        QFileDialog::DontConfirmOverwrite);
    if (file.size() && (m_logFile != file)) {
        m_logFile = file;
        settings.setValue("files/log",m_logFile);
        QFile f;
        if (!f.exists(m_logFile))
            util_createLogFile(m_logFile); // init file
        ui->lineEdit_lf->setText(m_logFile);
    }

}

// argument log
void MainWindow::on_lineEdit_lf_2_editingFinished()
{
    QString file;
    file = ui->lineEdit_lf_2->text();
    if (m_argFile != file) {
        m_argFile = file;
        settings.setValue("files/args",m_argFile);
    }
}

// get new argument log
void MainWindow::on_pushButton_blf_2_clicked()
{
    QString file;
    file = QFileDialog::getSaveFileName(this, tr("Argument Log"),
                                        m_argFile,
                                        tr("All (*.*)"),
                                        0,
                                        QFileDialog::DontConfirmOverwrite);
    if (file.size() && (m_argFile != file)) {
        m_argFile = file;
        settings.setValue("files/args",m_argFile);
        ui->lineEdit_lf_2->setText(m_logFile);
    }

}

// given a NEW airport file
void MainWindow::on_pushButton_gap_clicked()
{
    QString file;
    file = QFileDialog::getOpenFileName(this,tr("Open airport file"), m_aptFile, tr("Airport files (*.dat *.dat.gz)"));
    QFile f;
    if (file.size() && (m_aptFile != file) && f.exists(file)) {
        m_aptFile = file;
        settings.setValue("files/airport",m_aptFile);
    }

}

// user has entered a new valid file, so
void MainWindow::on_lineEdit_ap_editingFinished()
{
    QString file;
    file = ui->lineEdit_ap->text(); // get user entered text
    QFile f;
    if (file.size() && (m_aptFile != file) && f.exists(file)) {
        m_aptFile = file;
        settings.setValue("files/airport",m_aptFile);
    }
}

void MainWindow::dialog_abort_clicked()
{
    m_User_Break = true;
}


void MainWindow::openNonModalDialog(QString title, QString msg, int secs)
{

    QString tm;
    tm.sprintf("\n\nThis dialog will close after %d seconds.\n",secs);
    QDialog *d = new QDialog(this);

    d->setModal(false);
    d->setWindowTitle(title);

    QVBoxLayout *l = new QVBoxLayout(d);
    QLabel * pLab = new QLabel(d);
    // pLab->setSizePolicy(QSizePolicy::ExpandFlag);
    pLab->setWordWrap(true);
    pLab->setText(msg+tm);

#ifdef ADD_PUSH_BUTTONS
  QPushButton *button = new QPushButton;
  button->setText("Yes");
  connect(button, SIGNAL(clicked()), SLOT(close()));
  l->addWidget(button);
  button = new QPushButton;
  button->setText("No");
  connect(button, SIGNAL(clicked()), SLOT(close()));
  l->addWidget(button);
#endif

    QPushButton *button = new QPushButton;
    button->setText("Abort");
    connect(button, SIGNAL(clicked()), this, SLOT(dialog_abort_clicked()));

    l->addWidget(pLab);
    l->addWidget(button);
    //d->setFocusPolicy(Qt::FocusPolicy);
    d->show();
    d->repaint();
    qApp->processEvents();

    QTime to;
    to.start();
    int i;
    int sec_cnt = 0;
    int last_sec = 0;
    int elap;
    int ms = secs * 1000;
    elap = 0; // to.elapsed();
    while (!m_User_Break && (elap < ms)) {
        i++;
        elap = to.elapsed();
        sec_cnt = elap / 1000;
        d->update();
        if (sec_cnt != last_sec) {
            last_sec = sec_cnt;
            tm.sprintf("\n\nThis dialog will close after %d seconds.",secs - last_sec);
            pLab->setText(msg+tm);
            d->repaint();
            qApp->processEvents();
        }
        if (m_User_Break)
            break;
    }

    d->close();
    delete d;
    qApp->processEvents();
}


// genapts command
void MainWindow::on_pushButton_genair_clicked()
{
    LL ll;
    QString info;
    QString tmp;

    // options
    QString maxSlope  = ui->lineEdit_21->text();
    QString nudge = ui->lineEdit_nudge->text();
    bool limit_dem = ui->checkBox_limitdem->isChecked();
    bool ignore_ranges = ui->checkBox_igrng->isChecked();

    QString airportId   = ui->lineEdit_apid->text(); // given a SPECIFIC a/p
    QString startAptId  = ui->lineEdit_apid_2->text(); // set a START

    QString tileId  = ui->lineEdit_apid_3->text(); // set min/max from this
    QString chunkId = ui->lineEdit_apid_4->text(); // or from this

    QString aptFile = ui->lineEdit_ap->text();

    // GLOBAL Scenery Limits
    QString minLat  = m_south; // = min. lat
    QString maxLat  = m_north; // = max. lat
    QString minLon  = m_west;  // = min. lon
    QString maxLon  = m_east;  // = max. lon

    QString runtime = toolDir+"/genapts";
    QString airOut = m_workDir+"/AirportObj";
    QString outDir = m_workDir;
    QString demDir = m_workDir+"/SRTM-30";

    QString arguments;
    int res, ver, air_cnt, ind_cnt;
    bool check_limits = true;
    bool range_ok = true; // assumed ok - only checked if used

    if (runtime.contains(QChar(' ')))
        runtime = "\""+runtime+"\"";
    if (aptFile.contains(QChar(' ')))
        aptFile = "\""+aptFile+"\"";
    if (outDir.contains(QChar(' ')))
        outDir = "\""+outDir+"\"";
    if (demDir.contains(QChar(' ')))
        demDir = "\""+demDir+"\"";

    m_Verify_Tools = ui->checkBox_verify->isChecked();

    double south = minLat.toDouble();
    double north = maxLat.toDouble();
    double east = maxLon.toDouble();
    double west = minLon.toDouble();

    // Unable to proceed, if no valid project directory
    if (!util_verifyProjDir(projectDir))
        return; // genapts - projectDir invlaid

    QStringList airList = findFiles(airOut, QStringList() << "*.gz", true);
    QStringList indList = findFiles(airOut, QStringList() << "*.ind", true);
    air_cnt = airList.size();
    ind_cnt = indList.size();

    arguments = runtime;
    arguments += " --help";
    res = util_testApplication(arguments, info);
    ver = util_lookslike_genapts(info);
    if (res || m_Verify_Tools) {
        if (!m_Verify_Tools && (res == 255) && ver) {
            // looks good to go ;=))
        } else {
            // ask the user to continue
            util_trimLongMessage(m_MaxDiag, info);
            tmp.sprintf("\nTool returned value %d",res);
            info += tmp;
            info += "\nThe Terragear Tool directory is set on the Setup page.";
            outLog(info+"\n");
            info += "\n\nClick 'Yes' to continue to use this tool.";
            res = getYesNo("AIRPORT TOOL",info);
            if (!res)
                return; // airport tool NOT acceptable
        }
    }

    // begin command
    arguments = runtime + " --input="+aptFile+" --work="+outDir+" ";
    info = "About to run airport generation,\n";
    // ensure only the 'known'/'generated' elevation directory is used
    if (limit_dem) {
        arguments += "--clear-dem-path --dem-path="+demDir+" ";
        info += "limiting the elevation input to ["+demDir+"]\n";
    } else {
        info += "Using built-in default DEM paths.\n";
        info += "This can generate a lot of worthless output where elevation files do not exist.\n";
    }
    if (nudge.size() > 0) {
        arguments += "--nudge="+nudge+" ";
        info += "applying a nudge factor of ["+nudge+"]\n";
    }
    if (maxSlope.size() > 0) {
        arguments += "--max-slope="+maxSlope+" ";
        info += "setting max. runway slope to ["+maxSlope+"]\n";
    }
    if (airportId.size() > 0) {
        arguments += "--airport="+airportId+" ";
        info += "ONLY processing airport ["+airportId+"]\n";
    }
    if (startAptId.size() > 0) {
        arguments += "--start-id="+startAptId+" ";
        info += "STARTING at airport ["+startAptId+"]\n";
    }

    // set limits for airport generation
    if (tileId.size() > 0) {
        if ( !util_ValidTileId(tileId, &ll)) {
            info = "Parameter error. The tile ID must be of the form\n";
            info += "[e|w]nnn[n|s]nn, length 7. Got ["+tileId+"]?\n";
            info += "Either remove this option, or correct to continue.";
            QMessageBox::information(this,"OPTION ERROR",info);
            return; // genapts aborted - not a valid tile ID
        }
        // use min 1x1 tile ID sizing
        arguments += "--tile="+tileId+" ";
        info += "Area limited to 1x1 degree tile ID ["+tileId+"]\n";
        south = ll.lat;
        north = south + 1.0;
        west = ll.lon;
        east = west + 1.0;
        check_limits = true;
    } else if (chunkId.size() > 0) {
        // use 10x10 degree chunk size
        if ( !util_ValidChunkId(tileId, &ll)) {
            info = "Parameter error. The tile ID must be of the form\n";
            info += "[e|w]nnn[n|s]nn, length 7. Got ["+chunkId+"]?\n";
            info += "Either remove this option, or correct to continue.";
            QMessageBox::information(this,"OPTION ERROR",info);
            return; // genapts aborted - not a valid chunk ID
        }
        arguments += "--chunk="+chunkId+" ";
        info += "Area limited to 10x10 degree chunk ID ["+chunkId+"]\n";
        south = ll.lat;
        north = south + 9.0;
        west = ll.lon;
        east = west + 9.0;
        check_limits = true;
    } else if (ignore_ranges) {
        info += "No area limit = Do all in airport file given.\n";
        info += "Make sure you have all the elevations needed, ";
        info += "downloaded and chopped!\n";
        info += "This can take a LONG time if using the 27K fgdata file.\n";
        south = -90.0;
        north = 90.0;
        west = -180.0;
        east = 180.0;
        check_limits = false;
    } else {
        if ((west < east) && (north > south) &&
             latlonInWorld(north,east) &&
             latlonInWorld(south,west)) {
            range_ok = true; // all in world range
        } else
            range_ok = false;
        // or use existing min/max
        arguments += "--min-lon="+minLon+" ";
        arguments += "--max-lon="+maxLon+" ";
        arguments += "--min-lat="+minLat+" ";
        arguments += "--max-lat="+maxLat+" ";
        info += "Area limited to lat/lon min "+minLat+","+minLon+", max "+maxLat+","+maxLon+"\n";
        // south, north, west and east already set
        check_limits = true;
    }

    info += "Full tools arguments are -\n";
    info += arguments;
    if (check_limits) {
        int total = 0;
        int count = srtm_verifySRTMfiles(south, north, west, east, m_workDir, &total);
        tmp.sprintf("\nFound %d of %d elevations.\n", count, total);
        info += tmp;
    } else {
        // hmmm doing all - how can I help - check what
        QStringList elevList = srtm_getSRTMIndexList(m_workDir);
        tmp.sprintf("\nFound %d elevation files.", elevList.size());
        info += tmp;
    }

    if (air_cnt) {
        tmp.sprintf("\n\nPresently have %d generated airports, %d *.ind files\n",
                    air_cnt, ind_cnt);
        info += tmp;
        info += "in the output target ["+airOut+"]\n";
    } else {
        info += "\nPresently no generated airports in target ["+airOut+"]\n";
    }
    if (range_ok) {
        info += "\nProceed with airport generations?";
        res = getYesNo("CONFIRM AIRPORT GENERATION",info);
        if (!res)
            return; // genapts - user aborted run
    } else {
        info += "\nPrimary min/max range error - aborting airport generation!";
        QMessageBox::critical(this,"BAD MIN/MAX RANGES",info);
        return; // genapts - ranges NOT correct
    }

    // ok, now to do it ;=))
    SHOWWAITDIALOG("AIRPORT GENERATION",info);
    qApp->processEvents();
    outArg(arguments);
    info = util_runProcess(arguments,"");
    util_trimLongMessage(m_MaxDiag, info);
    info += "\nDone airport generation.";
    airList = findFiles(airOut, QStringList() << "*.gz", true);
    indList = findFiles(airOut, QStringList() << "*.ind", true);
    if (air_cnt == 0) {
        // had none previous - was anything generated
        if (airList.count())
            tmp.sprintf("\nGenerated %d airports, %d ind files.",airList.count(), indList.count());
        else
            tmp = "\nLooks like an ERROR! No airports generated.";
    } else {
        // had some previous - were new items generated?
        if (airList.count() > air_cnt)
            tmp.sprintf("\nStarted with %d, now have %d airports. Created %d", air_cnt, airList.count(), airList.count()-air_cnt);
        else
            tmp.sprintf("\nStarted with %d, and have no new airports.", air_cnt);
    }
    info += tmp;
    CLOSEWAITDIALOG;
    outLog(info+"\n");
    QMessageBox::information(this,"DONE AIRPORT GENERATION",info);

}

int MainWindow::process_Landuse(QString & selectedMaterials,
                                 QString & results,
                                 bool in_detail )
{
    int maxItems = ui->listWidget_2->count();
    int folderCnt = 0; // START COUNTER
    QString test;
    QStringList srtm;
    int flag = 0;
    int i;
    // create string with selected terraintypes
    for (i = 0; i < maxItems; ++i){
        test = ui->listWidget_2->item(i)->text(); // get landuse folder
        if (ui->listWidget_2->item(i)->isSelected() == 1) {
            selectedMaterials += test+" ";
            folderCnt++;
            if (test == "AirportObj") {
                // could also check for ICAO.btg.gz and <index>.ind files
                flag |= flg_AirObj;
            } else if (test == "AirportArea") {
                // could also check for <chunk>/<tile>/<index>.NNN array files
                // and maybe even poly_counter to check genapts WAS run
                flag |= flg_AirArea;
            } else if (test == "Default") {
                // could also check for <chunk>/<tile>/<index>.NNN array files
                flag |= flg_Default;
            } else if ( srtm_isElevationDir(test) ) {
                // hmmm, of course it IS possible to use other than this
                // hard coded set of directories, but...
                flag |= flg_SRTM;
                srtm += test;
            } else if (test == "Landmass") {
                // again could check more...
                flag |= flg_Land;
            }
        }
    }
    QString info;
    QString tmp;
    QString dir;
    QStringList list;
    info.sprintf("List of %d landuse, with %d selected items.",
                 maxItems, folderCnt);

    // Elevations
    // ==========================================================
    if (flag & flg_SRTM) {
        tmp = "\nFound Elevation folder.";
        if (in_detail) {
            i = 0;
            foreach (dir, srtm) {
                dir = m_workDir + "/" + dir;
                NEWWAITMESSAGE("Finding elevation array files...");
                list = findFiles(dir, QStringList() << "*.arr.gz", true);
                i += list.size();
            }
            tmp.sprintf("\nFound %d Elevation folder with %d arrays.", srtm.size(), i);
            if (i)
                flag |= flg_SRTM_cnt;
        }
    } else {
        tmp = "\nWARNING: NO elevation folders found = A FLAT WORLD AT -9999 FEET!";
    }
    info += tmp;


    // Airports
    // ==========================================================
    // AirportObj
    dir = m_workDir + "/AirportObj";
    if (flag & flg_AirObj) {
        tmp = "\nFound AirportObj folder.";
        if (in_detail) {
            NEWWAITMESSAGE("Finding airport BTG files...");
            list = findFiles(dir, QStringList() << "*.btg.gz", true);
            tmp.sprintf("\nFound AirportObj with %d a/p files.", list.size());
            if (list.size())
                flag |= flg_AirObj_cnt;
        }
    } else {
        tmp = "\nWARNING: No AirportObj folder found = NO AIRPORTS!";
    }
    info += tmp;

    // AirportArea
    dir = m_workDir + "/AirportArea";
    if (flag & flg_AirArea) {
        if (in_detail) {
            NEWWAITMESSAGE("Finding airport cutout arrays...");
            list = findFiles(dir, QStringList() << "*", true);
            tmp.sprintf("\nFound AirportArea with %d files.", list.size());
            if (list.size())
                flag |= flg_AirArea_cnt;
        } else {
            tmp = "\nFound AirportArea folder.";
        }
    } else {
        tmp = "\nWARNING: No AirportArea folder found = NO AIRPORT CUTOUTS!";
    }
    info += tmp;

    // Default area
    // ==========================================================
    // Default
    dir = m_workDir + "/Default";
    if (flag & flg_Default) {
        tmp = "\nFound Default folder.";
        if (in_detail) {
            NEWWAITMESSAGE("Finding default area arrays...");
            list = findFiles(dir, QStringList() << "*", true);
            tmp.sprintf("\nFound Default with %d files.", list.size());
            if (list.size())
                flag |= flg_Default_cnt;
        }
    } else {
        tmp = "\nWARNING: No Default folder found = NO DEFAULT AREA!";
    }
    info += tmp;


    // Landmass Area
    // ==========================================================
    // Landmass
    dir = m_workDir + "/Landmass";
    if (flag & flg_Land) {
        tmp = "\nFound Landmass folder.";
        if (in_detail) {
            NEWWAITMESSAGE("Finding Landmass area arrays...");
            list = findFiles(dir, QStringList() << "*", true);
            tmp.sprintf("\nFound Landmass with %d files.", list.size());
            if (list.size())
                flag |= flg_Land_cnt;
        }
    } else {
        tmp = "\nWARNING: No Landmass folder found = NO LANDMASS AREA!";
    }
    info += tmp;

    // Summary
    // ==========================================================
    if ((flag & flg_ALL) == flg_ALL) {
        info += "\nConstruction looks good to go ;=))";
    }
    if ((flag & flg_ALL_cnt) == flg_ALL_cnt) {
        info += "\nwith files in each category- congrats!";
    }
    results = info;
    return folderCnt;
}

void MainWindow::on_pushButton_updt_clicked()
{
    int i, j = 0;
    QString tmp;
    QString info;
    QDir dir(m_workDir);
    dir.setFilter(QDir::Dirs | QDir::NoDotAndDotDot);

    ui->listWidget_2->clear(); // clear the list
    QFileInfoList list = dir.entryInfoList();
    j = 0;
    for (i = 0; i < list.size(); ++i) {
        QFileInfo fileInfo = list.at(i);

        // do not add the Shared folder
        if (fileInfo.fileName() != "Shared"){
            //QString test = qPrintable(QString("%1").arg(fileInfo.fileName()));
            QString test = fileInfo.fileName();
            new QListWidgetItem(tr(qPrintable(QString("%1").arg(fileInfo.fileName()))), ui->listWidget_2);
            // select all materials per default
            ui->listWidget_2->item(j)->setSelected(1);
            j++;
        }
    }
    j = process_Landuse(tmp,info);
    outLog("Landuse info\n"+info+"\n");
    tmp = info;
    util_CleanLine(tmp);
    ui->textEdit_info->setText(tmp);
}

// final scenery construction clicked
void MainWindow::on_pushButton_gen_clicked()
{
    QString lat;
    QString lon;
    QString x;
    QString y;
    int maxItems, folderCnt;
    QString selectedMaterials;
    QString info;
    QString msg;
    QString tmp;
    QFile f;
    bool brk = false;
    QString arguments;
    int exe_res, arg_res;

    SHOWWAITDIALOG("SCENERY GENERATION","Moment...\ngathering information...\n...\n");

    bool no_priorities = ui->checkBox_nopri->isChecked();
    bool ignore_lm = ui->checkBox_lm->isChecked();

    QString runtime = toolDir+"/fgfs-construct";
    brk = false;
#ifdef Q_OS_WIN
    runtime += ".exe";
#endif
    if ( ! f.exists(runtime) ) {
#ifdef Q_OS_WIN
        // UGH! Must 'fix' my terragear tool name
        runtime = toolDir+"/fgfs_construct.exe";
        if ( ! f.exists(runtime) )
            brk = true;
#else
        brk = true;
#endif
    }
    if (runtime.contains(QChar(' ')))
        runtime = "\""+runtime+"\""; // if a space, add double quotes

    if (brk) {
        // catastophic FAILURE - no find of tool
        info = "ERROR: Unable to locate the terragear construction tool!";
        info += "\nSearched in ["+toolDir+"] for ["+runtime+"]";
        outLog(info+"\n");
        info += "\n\nSuggest 'No' to abort operation, or 'Yes' to continue regardless!";
        CLOSEWAITDIALOG;
        if ( !getYesNo("TOOL ERROR",info) )
            return; // construction - no find of tool
        SHOWWAITDIALOG("SCENERY GENERATION","Moment... gathering information...");
    }
    arguments = runtime+" --help";
    exe_res = (util_testApplication(arguments, info) & 0xff); // hmmm, seems linux returns max 255, while windows an int -1!!! so limit to byte
    if (exe_res) {
        // But if it is -1, and contains lots of help lines, then accept it...
        arg_res = util_lookslike_Fgfs_Construct(info); // check options

        if ((exe_res == 255) && (arg_res & 1) && (arg_res & 2)) {
            // reluctantly accept this modern tool -
            // it has enough 'character' ;=))
        } else if ((exe_res == -1) && (arg_res & 1) &&
                   (no_priorities) && (!ignore_lm)) {
            // has the basics - probably an older tool
            // accept since we do not need to add --priorities
            // nor --ignore_landmass, only available on modern tools
        } else {
            // ask user to CONFIRM the use of this tool
            tmp.sprintf("\nGot error exit %d", exe_res);
            info += tmp;
            if ((arg_res & 1) && !(arg_res & 2)) {
                if (!no_priorities) {
                    info += "\nERROR: This look like an OLDER tool, but you have NOT checked 'No priorities' option";
                    info += "\nand the tool indicates it does NOT support this option!";
                }
                if (ignore_lm) {
                    info += "\nERROR: You have checked 'Ignore landmass', but it appears this option is NOT supported on this older tool.";
                }
            }

            info += "\nBecause not ALL versions of the terragear tools exit(0) on a --help command, maybe there is in fact NO problem using this tool.";
            outLog(info+"\n");
            info += "\n\nClick 'Yes'' to continue and use this tool, else 'No' to abort.";
            CLOSEWAITDIALOG;
            // ==========================================================
            if (!getYesNo("TOOL CONFIRATION",info))
                return; // construction - user rejects tool output
            // ==========================================================
            SHOWWAITDIALOG("SCENERY GENERATION","Moment... gathering information...");
        }
    }

    QString defTxt = "tg_priorities.txt";
    QString usgsTxt = "tg_usgsmap.txt";
    if ( !no_priorities ) {
        if ( ! util_ensure_Priorities_Text(projectDir, defTxt, usgsTxt) ) {
            // EEK failed to find/create priorities text files
            // what has happened - is the directory not writable, WHAT???
            info = "ERROR: Failed to find/create ["+defTxt+"] and/or ["+usgsTxt+"]!";
            info += "\nCan ONLY guess the directory ["+projectDir+"] is NOT writable!";
            info += "\nPerhaps you need to 'manually' create these files?";
            info += "\n\nSuggest 'No' to abort the operation, or 'Yes' to continue regardless";
            CLOSEWAITDIALOG;
            if (!getYesNo("FILE FIND/CREATE FAILED",info) )
                return; // construction - no find/create of priorities files
            SHOWWAITDIALOG("SCENERY GENERATION","Moment... gathering information...");
        }
        tmp = projectDir + "/" + defTxt;
        if (tmp.contains(QChar(' ')))
            tmp = "\""+tmp+"\"";
        runtime += " --priorities="+tmp;
        tmp = projectDir + "/" + usgsTxt;
        if (tmp.contains(QChar(' ')))
            tmp = "\""+tmp+"\"";
        runtime += " --usgs-map="+tmp;
    }

    lat = ui->label_clat_2->text();
    lon = ui->label_clon_2->text();
    x = ui->label_xdist_2->text();
    y = ui->label_ydist_2->text();
    maxItems = ui->listWidget_2->count();
    folderCnt = process_Landuse( selectedMaterials, msg, true );
    outLog("Landuse info - full\n"+msg+"\n");
    tmp = msg;
    util_CleanLine(tmp);
    ui->textEdit_info->setText(tmp);

    if (folderCnt == 0) {
        if (maxItems)
            msg += "\nThere are NO material folders SELECTED!";
        else
            msg += "\nThere are NO material folders!";
        msg += "\nMay need to download and process land use!";
        CLOSEWAITDIALOG;
        msg += "\n\nSuggest 'No' to abort the operation, or 'Yes' to continue regardless";
        if ( !getYesNo("ERROR NO LANDUSE",msg) )
            return; // construction - no landuse folders
        SHOWWAITDIALOG("SCENERY GENERATION","Moment... gathering information...");
    }

    // we have SGBucket capability
    // construct fgfs-construct commandline,
    // FOR EACH BUCKET SEPARATELY, like master/client do
    // We could concurrently run multiple constructions, but then like server.cxx
    // we should skip every other column, to avoid two clients working on adjacent tiles
    // but here fgfs-construct is just run with for each 'bucket'
    QStringList argList; // build a string list to run
    QStringList pathList; // and the PATH to each bucket
    QString index = ui->lineEdit_tileid->text();
    QString path;
    QTime rt;
    QTime pt;
    QString tm;
    QString em;
    QStringList list;

    bool no_overwrite = ui->checkBox_noovr->isChecked();
    bool skip_error = ui->checkBox_skiperr->isChecked();
    bool skip_nodata = ui->checkBox_nodata->isChecked();
    bool skip_cfm = ui->checkBox_skipcfm_2->isChecked();

    bool do_confirm;
    int d_secs = ui->lineEdit_secs->text().toInt();
    if (d_secs < 1) d_secs = 1;

    long ind;
    int dx, dy, i, j;
    int gotcnt = 0;
    bool add_it = true; // ADD all buckets, unless there is a reason not to
    int ovr_exist = 0;  // would skip if ON
    int skip_ovr = 0;   // skip of exsting data
    int skip_cnt = 0;
    int data_fnd = 0;
    int nodata_skip = 0;
    int errCode;
    int pos1, pos2;

    SGBucket b_cur;
    rt.start();

    NEWWAITMESSAGE("Building construction arguments...");
    // build the general runtime string
    runtime += " --work-dir=\""+m_workDir+"\"";
    runtime += " --output-dir=\""+m_outpDir+"/Terrain\"";

    if (ui->checkBox_UK->isChecked()) {
        runtime += " --useUKgrid";
    }
    if (ignore_lm) {
        runtime += " --ignore-landmass";
    }
    gotcnt = 0;
    if (index.size() > 0) {
        // just ONE to do - DO IT
        NEWWAITMESSAGE("Building construction arguments... just one index...");
        ind = index.toLong();
        SGBucket b(ind);
        path.sprintf("%s", b.gen_base_path().c_str());
        path += "/"+index;
        arguments = runtime; // common runtime and params
        arguments += " --tile-id="+index+" ";
        arguments += selectedMaterials;
        tmp = m_outpDir+"/Terrain/"+path+".btg.gz";
        if ( f.exists(tmp) ) {
            if (no_overwrite) {
                add_it = false;
            } else {
                ovr_exist++;
            }
        }
        if (add_it) {
            argList << arguments; // set the ONE argument
            pathList << path;
        }
        gotcnt++;
    } else {
        // break the set into buckets
        double dlon = lon.toDouble();
        double dlat = lat.toDouble();
        double xdist = x.toDouble();
        double ydist = y.toDouble();

        double min_x = dlon - xdist;
        double min_y = dlat - ydist;
        SGBucket b_min( min_x, min_y );
        SGBucket b_max( dlon + xdist, dlat + ydist );
        if (b_min == b_max) {
            // just ONE bucket
            NEWWAITMESSAGE("Building construction arguments... just one bucket...");
            index.sprintf("%ld", b_min.gen_index());
            path.sprintf("%s", b_min.gen_base_path().c_str());
            path += "/"+index;
            arguments = runtime; // common runtime and params
            arguments += " --tile-id="+index+" ";
            arguments += selectedMaterials;
            tmp = m_outpDir+"/Terrain/"+path+".btg.gz";
            if ( f.exists(tmp) ) {
                if (no_overwrite) {
                    add_it = false;
                    skip_ovr++; // skip count of items NOT being done
                } else {
                    ovr_exist++;
                }
            }
            data_fnd = countDataFound(path,selectedMaterials,m_workDir);
            if (!data_fnd) {
                if (skip_nodata) {
                    add_it = false;
                    nodata_skip++; // count skipped doe to NO DATA
                } else {
                    skip_cnt++;
                }
            }
            if (add_it) {
                argList << arguments; // set the ONE argument
                pathList << path;
            }
            gotcnt++;
        } else {
            // a range of buckets
            NEWWAITMESSAGE("Building construction arguments... for range of buckets...");
            sgBucketDiff(b_min, b_max, &dx, &dy);
            for ( j = 0; j <= dy; j++ ) {
                for ( i = 0; i <= dx; i++ ) {
                    add_it = true; // initially ADD ALL buckets, for this INDEX
                    b_cur = sgBucketOffset(min_x, min_y, i, j);
                    index.sprintf("%ld", b_cur.gen_index());
                    tmp.sprintf("for bucket %d of %d", (j*dx)+i+1, (dx+1)*(dy+1));
                    path.sprintf("%s", b_cur.gen_base_path().c_str());
                    path += "/"+index;
                    tmp += ", path="+path;
                    NEWWAITMESSAGE("Building construction arguments... "+tmp);
                    arguments = runtime;
                    arguments += " --tile-id="+index+" ";
                    arguments += selectedMaterials;
                    tmp = m_outpDir+"/Terrain/"+path+".btg.gz";
                    if ( f.exists(tmp) ) {
                        if (no_overwrite) {
                            add_it = false; // KILLED BY NO OVERWRITE RULE
                            skip_ovr++; // skip count of items NOT being done
                        } else {
                            ovr_exist++;
                        }
                    }
                    data_fnd = countDataFound(path,selectedMaterials,m_workDir);
                    if (!data_fnd) {
                        if (skip_nodata) {
                            add_it = false; // KILLED BY NO ARRAY FILES FOUND - build with what??
                            nodata_skip++; // count skipped due to NO DATA
                        } else {
                            skip_cnt++;
                        }
                    }
                    if (add_it) {
                        argList << arguments; // set the EACH argument
                        pathList << path;
                    }
                    gotcnt++; // count the max. possible total for range
                }
            }
        }
    }
    ind = argList.size();
    if (ind != pathList.size()) {
        // catastophic FAILURE
        CLOSEWAITDIALOG;
        QMessageBox::critical(this,"INTERNAL ERROR","Lists are NOT equal in length!");
        return; // construction - internal arg count not equal
    }
    if (ind == 0) {
        tmp.sprintf("\nWith the current min/max, and options, have %d buckets to process, but perhpas no overwrite, and/or skip no data are checked, so nothing to do!", gotcnt);
        msg += tmp;
        CLOSEWAITDIALOG;
        msg += "\n\nSuggest 'No' to abort the operation, or 'Yes' to continue regardless";
        if (! getYesNo("NO BUCKETS TO PROCESS",msg) )
            return; // construction - no bucket to process
        SHOWWAITDIALOG("SCENERY GENERATION","Moment... gathering information...");
    }

    tmp.sprintf("\nTo process %d buckets, of total %d, for %d folders.\n", argList.size(), gotcnt, folderCnt);

    tmp += "Options:\nOverwrite=";
    if (no_overwrite) {
        tmp += "ON";
        if (skip_ovr) {
            // show number being skipped due already DONE
            tm.sprintf(" (Skipping %d.)",skip_ovr);
            tmp += tm;
        }
    } else {
        tmp += "OFF";
        if (ovr_exist) {
            tm.sprintf(" (Would skip %d if ON!)",ovr_exist);
            tmp += tm;
        }
    }
    tmp += "\nSkip_no_data=";
    if (skip_nodata) {
        tmp += "ON";
        if (nodata_skip) {
            // show number being skipped due NO DATA
            tm.sprintf(" (Skipping %d.)",nodata_skip);
            tmp += tm;
        }
    } else {
        tmp += "OFF";
        if (skip_cnt) {
            tm.sprintf(" (would skip %d if ON!)",skip_cnt);
            tmp += tm;
        }
    }
    tmp += "\nIgnore_Err=";
    tmp += skip_error ? "ON" : "OFF";
    tmp += "\n";
    msg += tmp;

    if ( no_priorities ) {
        msg += "Not adding a --priorities=file command.\n";
    } else {
        msg += "Adding --priorities= file "+defTxt+" command.\n";
    }

    if (skip_cfm) {
        msg += "Skip confirm is ON (this will be the only)!\n";
        if (skip_error) {
            msg += "even when there may be an error\n";
        } else {
            msg += "But will stop and confirm on apparent errors.\n";
        }
    } else {
        msg += "Skip confirm is OFF (ask after each bucket).\n";
    }

    outLog(msg);
    msg += "\n\nClick Yes to CONTINUE";
    CLOSEWAITDIALOG; // OK, close WAIT

    // =============================================================
    if ( !getYesNo("CONSTRUCTION PROCESS",msg) ) {
        return; // construction - no cfm to proceed
    }

    // we are on our way. to construct <index>.btg.gz file(s),
    // plus the associated <index>.stg to load these items in FG
    // int setup_ms = rt.restart(); // restart timer

#ifdef ADD_DIAG_OUTPUT
    // add a complete set of arguments to a templog.txt
    dy = 0;
    tm = "Setup: "+getElapTimeStg(setup_ms);
    outLog(tm+"\n");
    for (i = 0; i < argList.size(); i++) {
        dy++;
        arguments = argList[i]; // get command line arguments
        path = pathList[i]; // get the destination path
        // output commandline to log.txt
        outLog(arguments+"\n");
        msg.sprintf("%d: ", dy);
        msg += m_outpDir+"/Terrain/"+path+".btg.gz\n";
        outLog(msg);
    }
#endif // #ifdef ADD_DIAG_OUTPUT

    do_confirm = skip_cfm;
    dx = argList.size();
    dy = 0;
    brk = false;
    m_User_Break = false;
    // this is the section that should be run on a thread
    // ==================================================
    for (i = 0; i < argList.size(); i++) {
        if (brk || m_User_Break)
            break; // all over - user requested a break
        dy++;
        pt.start();
        do_confirm = skip_cfm ? false : true;
        // about to RUN fgfs-construct, for each bucket argument
        arguments = argList[i]; // get command line arguments
        path = pathList[i]; // get the destination path
        msg.sprintf("%d of %d: ", dy, argList.size());
        msg += "Target: ["+m_outpDir+"/Terrain/"+path+".btg.gz]\n";
        msg += "Args: ["+arguments+"]\n";
        outArg(msg); // show what we are about to do...
        info = "Moment...\nrunnin argument\n"+arguments;
        SHOWWAITDIALOG("SCENERY GENERATION",info);
        qApp->processEvents(); // re-paint the app - but does NOT seem to do a repaint???
        NEWWAITMESSAGE(info);
        qApp->processEvents(); // re-paint the app - but does NOT seem to do a repaint???
        // OLD tools seems to want to be in the 'work' directory
        //info = util_runProcess(arguments,"", &errCode);
        info = util_runProcess(arguments, m_workDir, &errCode);
        list = util_getOutputFiles(info);
        pos1 = info.indexOf("[Finished successfully]");
        if ((pos1 < 0) || !info.contains("[Finished successfully]") ) {
            if (!skip_error)
                do_confirm = true;
            em.sprintf("%d",errCode);
            msg.sprintf("WARNING: %d of %d: fgfs-construct ", dy, dx);
            msg += "exited with error "+em+"\n";
            msg += "or failed to output [Finished successfully]\n";
            msg += "while processing "+path+"\n";
            msg += "This usually indicates some error in processing!\n";
            tmp = "unknown area = '";
            if (info.contains(tmp)) {
                pos1 = info.indexOf(tmp);
                if (pos1 > 0) {
                    pos1 += tmp.size(); // bump past the find
                    pos2 = info.indexOf("'", pos1+1); // find next "'"
                    if ((pos2 > 0)&&(pos2 > pos1)) {
                        msg += "\nIn this case it seems the 'area' ["+info.mid(pos1, pos2-pos1)+"]\n";
                        msg += "needs to be added to the ["+projectDir+"/"+defTxt+"] file.\n\n";
                    }
                }
            }
        } else {
            msg = "Got [Finished successfully]\n";
        }
        if (list.size()) {
            foreach (tmp, list)
                msg += "out="+tmp+"\n";
        }
        util_trimLongMessage(m_MaxDiag, info);
        tmp.sprintf("%d of %d:", i+1, argList.size());
        info += "\n"+msg+" Done construction "+tmp+"\n";
        outLog(info+"\n"); // ALWAYS write it to the LOG
        CLOSEWAITDIALOG;
        qApp->processEvents(); // re-paint the app - but does NOT seem to do a repaint???
        if ((i+1) < argList.size()) {
            if (do_confirm) {
                info += "\nContinue to next?";
                if (!getYesNo("CONTINUE CONSTRUCTION",info)) {
                    return; // construction - done bucket, but no cfm to proceed
                }
            } else {
                openNonModalDialog("CONSTRUCTION INFORMATION",info,d_secs);
            }
        }
        qApp->processEvents(); // re-paint the app - but does NOT seem to do a repaint???
    }
    // ==================================================================
    if (m_User_Break)
        info += "\nUser break.";
    else
        info += "\nDone last.";
    msg.sprintf("\nfgfs-construct did %d buckets (%d/%d)", i, argList.count(), gotcnt);
    msg += " in "+getElapTimeStg(rt.elapsed());
    tmp = ui->textEdit_info->document()->toPlainText();
    if (tmp.contains("fgfs-construct did")) {
        i = tmp.indexOf("fgfs-construct did");
        if (i > 0)
            tmp.chop(i-1);
    }
    tmp += " "+msg;
    util_CleanLine(tmp);
    ui->textEdit_info->setText(tmp);
    info += msg;
    outLog(info+"\n"); // ALWAYS write it to the LOG
    d_secs = (d_secs < 5 ? 5 : d_secs);
    if (m_User_Break)
        msg = "CONSTRUCTION ABORT - USER BREAK";
    else
        msg = "CONSTRUCTION END INFORMATION";
    m_User_Break = false;
    openNonModalDialog(msg,info,d_secs);
    m_User_Break = false;

}

// construction - toggle skip each confirmation dialog
void MainWindow::on_checkBox_skipcfm_2_clicked()
{
    bool chk = ui->checkBox_skipcfm_2->isChecked();
    // if checked, enable the delay value - move to advanced, and always enabled
    // ui->label_secs->setEnabled(chk);
    // ui->lineEdit_secs->setEnabled(chk);
    if (chk) {
        int secs = ui->lineEdit_secs->text().toInt();
        if (secs < 1)
            ui->lineEdit_secs->setText("1");
    }
}

void MainWindow::on_lineEdit_secs_editingFinished()
{
    m_waitSecs = ui->lineEdit_secs->text().toInt();
    if (m_waitSecs < 1) {
        QString info;
        info = "At present the MINIMUM delay between sets of actions is 1 second.";
        info += "\nThis gives at least this amount of time to click on the 'Abort' button, if things are not going correctly, or as expected.";
        // could get noisy, with dialog, or just fix it !!!
        ui->lineEdit_secs->setText("1");
        QMessageBox::information(this,"MINIMUM DELAY",info);
        m_waitSecs = 1;
    }
    settings.setValue("settings/waitsecs",m_waitSecs);

}

void MainWindow::on_lineEdit_maxtext_editingFinished()
{
    m_MaxDiag = ui->lineEdit_maxtext->text().toInt();
    if (m_MaxDiag < 10) {
        QString info;
        info.sprintf("You have entered a value of %d.",m_MaxDiag);
        info += "\nBy using a value less than 10, you have effectively disabled this feature.";
        info += "\nBe warned that very long outputs may cause the dialog to be larger than the screen height,";
        info += "and thus it may be difficult to 'see' the button choices at the bottom.";
        info += "\nNote even when the text is truncated for display, the =[...]=FULL text is always written to the ["+m_logFile+"] log file";
        info += " thus is available there for review. The default value is 512.";
        QMessageBox::information(this,"MINIMUM DIALOG TEXT",info);
    }
    settings.setValue("settings/maxtext",m_MaxDiag);

}


void MainWindow::on_checkBox_noovr_clicked(bool checked)
{
    settings.setValue("check/const_no_overwrite",checked);
}

void MainWindow::on_checkBox_lm_clicked(bool checked)
{
    settings.setValue("check/const_ignore_landmass",checked);
}

void MainWindow::on_checkBox_nodata_clicked(bool checked)
{
    settings.setValue("check/const_no_data", checked);
}

void MainWindow::on_checkBox_skiperr_clicked(bool checked)
{
    settings.setValue("check/const_ign_errors",checked);
}

void MainWindow::on_checkBox_skipcfm_2_clicked(bool checked)
{
    settings.setValue("check/const_skip_cfm_2",checked);
}

void MainWindow::on_lineEdit_hgt_editingFinished()
{
    QString tmp;
    tmp = ui->lineEdit_hgt->text();
    if (tmp.length())
        settings.setValue("paths/temphgt",tmp);

}

void MainWindow::on_lineEdit_shp_editingFinished()
{
    QString tmp;
    tmp = ui->lineEdit_shp->text();
    if (tmp.length())
        settings.setValue("paths/tempshp",tmp);

}

void MainWindow::on_pushButton_gshp_clicked()
{
    QString tmp;
    tmp = ui->lineEdit_shp->text();
    tmp = QFileDialog::getExistingDirectory(this,"Create/Select directory. Suggest <dir>/temp/shape", tmp);
    if (tmp.length()) {
        ui->lineEdit_shp->setText(tmp);
        on_lineEdit_shp_editingFinished();
    }

}

void MainWindow::on_pushButton_bhgt_clicked()
{
    QString tmp;
    tmp = ui->lineEdit_hgt->text();
    tmp = QFileDialog::getExistingDirectory(this,"Create/Select directory. Suggest <dir>/temp/hgt", tmp);
    if (tmp.length()) {
        ui->lineEdit_hgt->setText(tmp);
        on_lineEdit_hgt_editingFinished();
    }
}
