/*
    file: tggui_utils.cpp

    Written by Geoff R. Mclane, started May 2011.

    Copyright (C) 2011  Geoff R. McLane - reports@geoffair.info

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License as
    published by the Free Software Foundation; either version 2 of the
    License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful, but
    WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    $Id: 0.0.1 2001-05-07 - version 0.0.1 $
   */

#include "tgs_config.h"
#include "tggui_utils.h"
#include <QTextStream>
#include <QVBoxLayout>
#include <QLabel>


static QString m_logFile;

/*
    LOG FILE FUNCTIONS
 */

// set log file name
void util_setLogFile(QString file)
{
    m_logFile = file; // set name
}

// create a log file and set name
bool util_createLogFile(QString file)
{
    bool bret = false;
    QFile data(file);
    if (data.open(QIODevice::ReadWrite | QIODevice::Truncate | QFile::Text)) {
        QTextStream out(&data);
        QString dt = util_getDTstg();
        out << dt;
        out.flush();
        data.close();
        bret = true;    // log is ok
        util_setLogFile(file); // set name
    }
    if (bret)
        outLog(" - Commenced log\n");
    return bret;
}

// append to LOG file
void outLog(QString s)
{
    if (m_logFile.size()) {
        QFile data(m_logFile);
        if (data.open(QFile::WriteOnly | QFile::Append | QFile::Text)) {
            QTextStream out(&data);
            out << s;
        }
    }
}

/*
    Utility function that recursively searches, if desired, for files per filters.
*/
QStringList findFiles(const QString &startDir, QStringList filters, bool recurse)
{
    QStringList names;
    QDir dir(startDir);

    // 1: Get the files matching the filter
    foreach (QString file, dir.entryList(filters, QDir::Files))
        names += startDir + "/" + file;

    if (recurse) {
        // 2: If recursive, get ALL directories, and try for more files
        foreach (QString subdir, dir.entryList(QDir::AllDirs | QDir::NoDotAndDotDot))
            names += findFiles(startDir + "/" + subdir, filters, recurse);
    }
    return names;
}

/*********************************************
  Run a Yes, No dialog, given Title and Question
// return
// 1 = Yes was clicked - the default
// 0 = No or otherwise
  ********************************************/
int getYesNo( QString title, QString msg )
{
    QMessageBox msgBox;
    //msgBox.setText(title); // set TITLE text
    msgBox.setWindowTitle(title);
    msgBox.setIcon(QMessageBox::Information); // set predefined icon, icon is show on left side of text.
    msgBox.setInformativeText(msg); // set the Question
    msgBox.setStandardButtons(QMessageBox::Yes | QMessageBox::No); // Add Yes and No buttons.
    msgBox.setDefaultButton(QMessageBox::Yes);  //Set focus on Yes button
    //execute message box. method exec() return the button value of clicked button
    int ret = msgBox.exec();
    //User get input from returned value (ret). you can handle it here.
    switch (ret)
    {
    case QMessageBox::Yes:
        ret = 1;    // 1 for OK to continue
        break;
    case QMessageBox::No:
        ret = 0; // Cancel was clicked
        break;
    default:
        // should never be reached, or maybe with ESCAPE?
        ret = 0; // but assume NO
        break;
    }
    return ret;
}

void timedMessage( QString title, QString msg, int secs)
{
    QDialog *d = new QDialog(0);
    QVBoxLayout *l = new QVBoxLayout(d);
    QString tm;
    tm.sprintf("\n\nThis dialog will close after %d seconds.",secs);
    l->addWidget(new QLabel(msg+tm));
    d->setWindowTitle(title);
    d->show();
    d->repaint();   // re-paint it...
    // maybe d->raise();
    // maybe d->activateWindow();
    QTime to;
    int cnt = 0;
    int ms = secs * 1000;
    to.start();
    while (to.elapsed() < ms) {
        // would like to add an ok button, and if the user clicks this
        // before this wasted timeout, then exit the loop
        //if (d->isFinished())
        //    break;
        cnt++;
    }
    //d->result();
    d->close();
    delete d;
}


// given millisecond, return appropriate (nice) string
QString getElapTimeStg(int ms)
{
    QString fmt = "";
    if (ms < 1000) {
        fmt.sprintf("%d ms", ms);
    } else {
        int secs = ms / 1000;
        ms = ms % 1000;
        if (secs < 60) {
            if (ms)
                fmt.sprintf("%d.%03d secs", secs, ms);
            else
                fmt.sprintf("%d secs", secs);
        } else {
            int mins = secs / 60;
            secs = secs % 60;
            if (mins < 60) {
                if (ms)
                    fmt.sprintf("%d:%02d.%03d mins", mins, secs, ms);
                else if (secs)
                    fmt.sprintf("%d:%02d mins", mins, secs);
                else
                    fmt.sprintf("%d mins", mins);
            } else {
                int hrs = mins / 60;
                mins = mins % 60;
                if (ms)
                    fmt.sprintf("%d:%02d:%02d.%03d hrs", hrs, mins, secs, ms);
                else if (secs)
                    fmt.sprintf("%d:%02d:%02d hrs", hrs, mins, secs);
                else if (mins)
                    fmt.sprintf("%d:%02d hrs", hrs, mins);
                else
                    fmt.sprintf("%d hrs", hrs);
            }
        }
    }
    return fmt;
}

// given a SGBucket path = <chunk>/<1x1>/index, and a space separated list of directories to try,
int countDataFound(QString in_path, QString selectedMaterials, QString workDirectory)
{
    QStringList materList = selectedMaterials.split(QChar(' '),QString::SkipEmptyParts); // space split list
    QStringList pathList = in_path.split(QChar('/'));
    int i, totFound;
    totFound = 0;
    if ((pathList.size() == 3) && (materList.size() > 0)) {
        // we have a chance...
        QString path = pathList[0]+"/"+pathList[1];
        QString index = pathList[2]; // the bucket INDEX
        for (i = 0; i < materList.size(); i++) {
            QString landUse = materList[i];
            // got <chunk>/<1x1> path
            QString test = workDirectory+"/"+landUse+"/"+path;
            QDir dir(test);
            if (dir.exists()) {
                // ok, this <worK>/<landuse>/<chunk>/<1x1>/<index>.* exists
                QStringList files = findFiles(dir.absolutePath(), QStringList() << index+".*", true);
                totFound += files.count();
            }
        }
    }
    return totFound; // ZERO if NO DATA FOUND - not good to continue ;=))
}

bool checkTGTools( QString & dir )
{
    QStringList tgtoolList;
    tgtoolList << "hgt-chop" << "genapts" << "ogr-decode" << "shape-decode" << "fgfs-construct";
    QDir d;
    int count = 0;
    int i;
    if (d.exists(dir)) {
        QString tool;
        QString exe;
        for (i = 0; i < tgtoolList.count(); i++) {
            tool = tgtoolList[i];
            exe = dir+"/"+tool;
#ifdef Q_OS_WIN
            exe += ".exe";
#endif
            QFile f(exe);
            if (f.exists()) {
                count++;
            }
        }
        if (i == count)
            return true;
    }
    return false;
}

bool latlonInWorld( double lat, double lon )
{
    if ((lat >= -90.0)&&(lat <= 90.0)&&
        (lon >= -180.0)&&(lon <= 180.0)) {
        return true;
    }
    return false;
}

int util_testApplication(QString cmd, QString & msg)
{
    int errCode = -2;
    QTime rt;
    QString tm;
    QString std;
    QString err;

    rt.start();
    tm = "";
    msg = "Running ["+cmd+"]\n";
    QProcess proc;
    // run command
    proc.start(cmd, QIODevice::ReadWrite);
    //proc.waitForReadyRead();
    if (proc.waitForStarted()) {
        if (proc.QProcess::waitForFinished()) {
            errCode = proc.exitCode();
            std = proc.readAllStandardOutput();
            err = proc.readAllStandardError();
            std.trimmed();
            std = std.simplified();
            err = err.simplified();
            if ((std.size() == 0)&&(err.size() == 0)) {
                msg += "No 'stdout' or 'errout' text!\n";
            } else {
                if (std.size())
                    msg += "S["+std+"]\n";
                if (err.size())
                    msg += "E["+err+"]\n";
            }
            tm.sprintf(" for %d ms.", rt.elapsed());
            msg += "*PROC_ENDED*"+tm+"\n";
        } else {
            msg += "proc.QProcess::waitForFinished() returned FALSE";
        }
    } else {
        msg += "proc.waitForStarted() returned FALSE";
    }
    return errCode;
}

/* ==========================================
S[Usage: C:\FG\TG2\terragear-cs\msvc\bin\fgfs_construct.exe
[ --output-dir=<directory>
--work-dir=<directory>
--cover=<path to land-cover raster>
--tile-id=<id>
--lon=<degrees>
--lat=<degrees>
--xdist=<degrees>
--ydist=<degrees>
--nudge=<float>
--priorities=<filename>
--usgs-map=<filename>
--useUKgrid
--no-write-shared-edges
--use-own-shared-edges
--ignore-landmass
] <load directory...>
BUT older versions had -
S[Usage: C:\FG\TG2\terragear-cs\msvc\bin\fgfs-construct.exe
[ --output-dir=<directory>
  --work-dir=<directory>
  --cover=<path to land-cover raster>
  --tile-id=<id>
  --lon=<degrees>
  --lat=<degrees>
  --xdist=<degrees>
  --ydist=<degrees>
  --nudge=<float>
  --useUKgrid
 ] <load directory...>
 ======================================== */
int util_lookslike_Fgfs_Construct( QString info ) {
    int flag = 0;
    if (info.contains("Usage") &&
        info.contains("--output-dir=") &&
        info.contains("--work-dir") &&
        info.contains("--tile-id") &&
        info.contains("--useUKgrid") &&
        info.contains("--nudge") )
        flag |= 1;  // got the basics
    if (info.contains("--priorities=") &&
        info.contains("--no-write-shared-edges") &&
        info.contains("--ignore-landmass") )
        flag |= 2; // newer version
    return flag;
}

/* --------------------------------------
Usage hgtchop <resolution> <hgt_file> <work_dir>

        resolution must be either 1 or 3 for 1arcsec or 3arcsec
   -------------------------------------- */
int util_lookslike_hgtchop( QString info )
{
    if ( info.contains("<resolution>") &&
         info.contains("<hgt_file>") &&
         info.contains("<work_dir>"))
        return 1; // has enough of hgtchop's personality ;=))
    return 0; // nope, it ain't a known hgtchop ;=((
}

/* -----------------------------------------

$ terrafit --help
Usage: terrafit [options] <file | path to walk>
         -h | --help
         -m | --minnodes 50
         -x | --maxnodes 1000
         -e | --maxerror 40
         -v | --version

Algorithm will produce at least <minnodes> fitted nodes, but no
more than <maxnodes>.  Within that range, the algorithm will stop
if the maximum elevation error for any remaining point
drops below <maxerror> meters.

Increasing the maxnodes value and/or decreasing maxerror
will produce a better surface approximation.

The input file must be a .arr.gz file such as that produced
by demchop or hgtchop utils.

**** NOTE ****:
If a directory is input all .arr.gz files in directory will be
processed recursively.

The output file(s) is/are called .fit.gz and is simply a list of
from the resulting fitted surface nodes.  The user of the
.fit.gz file will need to retriangulate the surface.
Min points = 50
Max points = 1000
Max error  = 40
Insufficient arguments

  ------------------------------------------- */
int util_lookslike_terrafit( QString info )
{
    if ( info.contains(" --minnodes ") &&
         info.contains(" --maxnodes ") &&
         info.contains(" --maxerror "))
        return 1; // has enough of terrafit's personality ;=))
    return 0; // nope, it ain't a known terrafit ;=((
}

/* ********************************************************
Usage: shape-decode [--line-width width] [--point-width width] [--area-column col] [--code-col col] [--line-width-column col ]  [--continue-on-errors] [--max-segment max_segment_length] [--start-record num] <shape_file> <work_dir> [ area_string ]
Options:
--line-width width
        Width in meters for the lines
--point-width width
        Size in meters of the squares generated from points
--max-segment max_segment_length
        Maximum segment length in meters
--area-column col
        Get areatype for objects from column number col
        in associated dbf-file (only if <area_string> is not given)
--code-column col
        Get codetype for objects from column number col
        in associated dbf-file
        (currently code is only used in debug printouts)
--continue-on-errors
        Continue even if the file seems fishy

--start-record record-number
        Start processing at the specified record number (first record num=0)
--end-record record-number
        End processing at (immediately after) the specified record number (first record num=0)

<shape_file>
        Name of the shape-file to process, without .shp extension
<work_dir>
        Directory to put the polygon files in
<area_string>
        (Optional) Area type for all objects in file
        Overrides --area-column option if present

   ******************************************************** */
int util_lookslike_shape_decode( QString info )
{
    if ( info.contains("--line-width ") &&
         info.contains("--point-width ") &&
         info.contains("--area-column ") &&
         info.contains("--continue-on-errors") )
        return 1; // has enough shape-code, and hopefull ogr-decode personality
    return 0;
}

/* *************************************************************
~/projects$ genapts --help
genapts generates airports for use in generating scenery for the FlightGear flight simulator.  Airport, runway, and taxiway vector data and attributes are input, and generated 3D airports are output for further processing by the TerraGear scenery creation tools.

The standard input file is runways.dat.gz which is found in $FG_ROOT/Airports.  This file is periodically generated for the FlightGear project by Robin Peel, who maintains an airport database for both the X-Plane and FlightGear simulators.  The format of this file is documented on the FlightGear web site.  Any other input file corresponding to this format may be used as input to genapts.  Input files may be gzipped or left as plain text as required.

Processing all the world's airports takes a *long* time.  To cut down processing time when only some airports are required, you may refine the input selection either by airport or by area.  By airport, either one airport can be specified using --airport=abcd, where abcd is a valid airport code eg. --airport-id=KORD, or a starting airport can be specified using --start-id=abcd where once again abcd is a valid airport code.  In this case, all airports in the file subsequent to the start-id are done.  This is convienient when re-starting after a previous error.
An input area may be specified by lat and lon extent using min and max lat and lon.  Alternatively, you may specify a chunk (10 x 10 degrees) or tile (1 x 1 degree) using a string such as eg. w080n40, e000s27.
An input file containing only a subset of the world's airports may of course be used.

It is necessary to generate the elevation data for the area of interest PRIOR TO GENERATING THE AIRPORTS.  Failure to do this will result in airports being generated with an elevation of zero.  The following subdirectories of the work-dir will be searched for elevation files:

SRTM2-Africa-3
SRTM2-Australia-3
SRTM2-Eurasia-3
SRTM2-Islands-3
SRTM2-North_America-3
SRTM2-South_America-3
DEM-USGS-3
SRTM-1
SRTM-3
SRTM-30

Usage genapts --input=<apt_file> --work=<work_dir> [ --start-id=abcd ] [ --nudge=n ]
[--min-lon=<deg>] [--max-lon=<deg>] [--min-lat=<deg>] [--max-lat=<deg>] [--clear-dem-path]
[--dem-path=<path>] [--max-slope=<decimal>] [ --airport=abcd ]  [--tile=<tile>]
[--chunk=<chunk>] [--verbose] [--help]

   ************************************************************* */
int util_lookslike_genapts( QString info )
{
    int flag = 0;
    if ( info.contains("--input=") &&
         info.contains("--work=") &&
         info.contains("--start-id=") &&
         info.contains("--min-lon=") &&
         info.contains("--max-lon=") &&
         info.contains("--min-lat=") &&
         info.contains("--airport=") &&
         info.contains("--tile=") &&
         info.contains("--chunk=") &&
         info.contains("--max-lat=") &&
         info.contains("--nudge=") )
        flag |= 1; // has enough genapts personality
    if ( info.contains("--clear-dem-path") &&
         info.contains("--dem-path="))
        flag |= 2; // this is a later version
    return flag;
}

bool util_getWgetVersion(QString & vers)
{
    QString out;
    out = "";
    int ret = util_testApplication("wget --version",out);
    if ((out.size() > 0)&&(ret == 0)) {
        QString stg = "GNU Wget ";
        if (out.contains(stg,Qt::CaseInsensitive)) {
            int off = out.indexOf(stg,0,Qt::CaseInsensitive);
            if (off >= 0) {
                vers = out.mid(off,stg.size()+5);
                vers.trimmed();
            }
            return true;
        }
    }
    return false;
}

QString util_getDTstg(void)
{
    QDateTime datetime = QDateTime::currentDateTime();
    QString dt = datetime.toString("yyyy/MM/dd HH:mm:ss");
    return dt;
}


// list of correpsonding materials
QStringList util_getMaterialList(void)
{
    QStringList csMater;
    // *TBD* - should compare this to default_priorities.txt
    csMater << "AgroForest" << "Airport" << "Asphalt" << "BarrenCover" << "Bog" <<
            "Burnt" << "Canal" << "Cemetery" << "ComplexCrop" << "Construction" <<
            "CropGrass" << "DeciduousForest" << "Default" << "Dirt" << "DryCrop" <<
            "Dump" << "Estuary" << "EvergreenForest" << "FloodLand" << "Freeway" <<
            "Glacier" << "GolfCourse" << "GrassLand" << "GreenSpace" << "Heath" <<
            "HerbTundra" << "Industrial" << "IntermittentLake" << "IntermittentStream" <<
            "IrrCrop" << "Lagoon" << "Lake" << "Lava" << "Littoral" << "Marsh" <<
            "MixedCrop" << "MixedForest" << "NaturalCrop" << "Olives" << "OpenMining" <<
            "Orchard" << "PackIce" << "PolarIce" << "Port" << "Railroad" << "Railroad" <<
            "RainForest" << "Rice" << "Road" << "Rock" << "Saline" << "SaltMarsh" <<
            "Sand" << "Sclerophyllous" << "ScrubCover" << "Stream" << "SubUrban" << "Town" <<
            "Transport" << "Urban" << "Vineyard" << "Watercourse" << "Landmass";
    return csMater;
}

// list of custom scenery shapefiles
QStringList util_getCustomShapes(void)
{
    QStringList csShape;
    csShape << "cs_agroforest" << "cs_airport" << "cs_asphalt" << "cs_barrencover" << "cs_bog" <<
            "cs_burnt" << "cs_canal" << "cs_cemetery" << "cs_complexcrop" << "cs_construction" <<
            "cs_cropgrass" << "cs_deciduousforest" << "cs_default" << "cs_dirt" << "cs_drycrop" <<
            "cs_dump" << "cs_estuary" << "cs_evergreenforest" << "cs_floodland" << "cs_freeway" <<
            "cs_glacier" << "cs_golfcourse" << "cs_grassland" << "cs_greenspace" << "cs_heath" <<
            "cs_hebtundra" << "cs_industrial" << "cs_intermittentlake" << "cs_intermittentstream" <<
            "cs_irrcrop" << "cs_lagoon" << "cs_lake" << "cs_lava" << "cs_littoral" << "cs_marsh" <<
            "cs_mixedcrop" << "cs_mixedforest" << "cs_naturalcrop" << "cs_olives" << "cs_openmining" <<
            "cs_orchard" << "cs_packice" << "cs_polarice" << "cs_port" << "cs_railroad1" << "cs_railroad2" <<
            "cs_rainforest" << "cs_rice" << "cs_road" << "cs_rock" << "cs_saline" << "cs_saltmarsh" <<
            "cs_sand" << "cs_sclerophyllous" << "cs_scrub" << "cs_stream" << "cs_suburban" << "cs_town" <<
            "cs_transport" << "cs_urban" << "cs_vineyard" << "cs_watercourse" << "v0_landmass";
    return csShape;
}

// need QApplication::processEvents(); somewhere
// run a process with 'arguments', in 'dir', if any
QString util_runProcess(QString arguments, QString dir, int * pErr)
{
    QTime rt;
    QString info;
    QString tm;
    rt.start();
    QProcess proc;
    int errCode = -2;
    QString std;
    QString err;
    if (dir.size())
        proc.setWorkingDirectory(dir);
    proc.start(arguments, QIODevice::ReadWrite);
    // proc.waitForReadyRead();
    if (proc.waitForStarted()) {
        // proc.QProcess::waitForFinished(-1);
        proc.waitForFinished(-1);
        errCode = proc.exitCode();
        std = proc.readAllStandardOutput();
        err = proc.readAllStandardError();
    } else {
        info = "waitForStarted() FAILED or timed out!";
        errCode = -2;
    }
    tm = " in "+getElapTimeStg(rt.elapsed());
    if (errCode) {
        info.sprintf("*PROC_ENDED* with error %d",errCode);
    } else {
        info = "*PROC_ENDED*";
    }
    info += tm+"\n";
    info += "arg="+arguments;
    if (dir.size())
        info += ", in "+dir;
    info += "\n";
    // std.trimmed();
    std.simplified();
    err.simplified();
    if (std.size())
        info += "stdout=["+std+"]";
    if (err.size())
        info += "errout=["+err+"]";
    if (pErr)
        *pErr = errCode;
    return info;
}

// is this string in this list - there is probably a function for this
// hmmm, I suppose it is list.indexOf(item) != -1
bool util_isinFileList(QStringList list, QString item)
{
    for (int i = 0; i < list.size(); i++) {
        if (list[i] == item)
            return true;
    }
    return false;
}

// rename 'file' to file.OLD, file.BAK, file.BAK1, file.BAK2, etc
// never deletes anything
// returns -1 if error
// returns 0 if nothing done
// returns 1 if renamed to OLD
// returns 2 if renamed to BAK
// returns 3++ if renamed to BAK+number
int util_renameToOLDBAK(QString file)
{
    QFile f(file);
    if (f.exists()) {
        QString filNew = file + ".old";
        if (f.exists(filNew)) {
            filNew = file + ".bak";
            if (f.exists(filNew)) {
                int i = 1;
                QString ext;
                ext.sprintf(".%d",i);
                filNew = ".bak"+ext;
                while (f.exists(filNew)) {
                    i++;
                    ext.sprintf(".%d",i);
                    filNew = ".bak"+ext;
                }
                if (f.rename(file,filNew))
                    return 2+i;
                else
                    return -1;
            } else {
                if (f.rename(file,filNew))
                    return 2;
                else
                    return -1;
            }
        } else {
            if (f.rename(file,filNew))
                return 1;
            else
                return -1;
        }
    }
    return 0;
}

QString util_getRenameExt(int n)
{
    QString ext = "";
    if (n > 0) {
        switch (n)
        {
        case 1:
            ext = ".old";
            break;
        case 2:
            ext = ".bak";
            break;
        default:
            ext.sprintf(".bak%d", (n - 2));
            break;
        }
    }
    return ext;
}

bool util_verifyProjDir(QString path)
{
    QDir d;
    if ((path.size() == 0)|| // MAIN project directory
        ( !d.exists(path) )) {
        QMessageBox::critical(0,"NO PROJECT DIECTORY",
"The main PROJECT DIRECTORY, has NOT been set, OR is INVALID!\n"
"Can NOT proceed to process anything until this is fixed!\n\n"
"Return to the SetUp page, to correct this problem.");
        return false;
    }
    return true;
}

bool util_createDirectory(QString path)
{
    QDir d;
    if ( ! d.exists(path) ) {
        //download.mkdir(path); // hmmmm this FAILS
        d.mkpath(path); // but this seems ok???
        if ( ! d.exists(path) ) {
            QString info;
            info = "Failed to create ["+path+"]!\n";
            info += "Can only abort the action.\n";
            info += "Perhaps change the 'Project Directory'\n";
            info += "and/or check permissions on this path.\n";
            QMessageBox::critical(0,"MAKE PATH FAILED", info);
            return false;
        }
    }
    return true;
}

// in windows wget, expect a file name of  the form
//                      012345678911234567892123456789312345678941
// 16:09:23 (122.06 KB/s) - `754cd722-7370-47e3-8bed-f548a954c34b.zip' saved [193363/193363]
QString util_extractZipFile(QString info)
{
    QString file = "";
    QString find1 = ") - `";
    QString find2 = ".zip' saved [";
    int from, len;
    int pos1 = info.indexOf(find1);
    int pos2 = info.indexOf(find2);
    if ((pos1 > 0) &&
        (pos2 > 0) &&
        ((pos2 - pos1) > 30) &&
        ((pos2 - pos1) < 50)){
        from = pos1+find1.size();
        len = pos2 - from;
        file = info.mid(from,len)+".zip";
    }
    return file;
}

// say e151s34 - 1x1 tile id of YSSY, YCOB, ...
bool util_ValidTileId(QString tileId, PLL pLL)
{
    bool ok;
    QChar ch;
    if (tileId.size() != 7)
        return false;

    ch = tileId.at(0);
    if ( !((ch == QChar('e'))||(ch == QChar('w'))) )
        return false;
    pLL->lon = tileId.mid(1,3).toDouble(&ok);
    if (!ok)
        return false;
    if (ch == QChar('w'))
        pLL->lon *= -1.0;

    ch = tileId.at(4);
    if ( !((ch == QChar('n'))||(ch == QChar('s'))) )
        return false;
    pLL->lat = tileId.mid(5,2).toDouble(&ok);
    if (!ok)
        return false;
    if (ch == QChar('s'))
        pLL->lat *= -1.0;
    if ( !latlonInWorld( pLL->lat, pLL->lon ) )
        return false;

    return true;
}

// say e150s40 - 10x10 chunk id of YSSY, YCOB, ...
bool util_ValidChunkId(QString chunkId, PLL pLL)
{
    if (!util_ValidTileId(chunkId, pLL))
        return false;
    int ill = (int)pLL->lon;
    if (ill % 10)
        return false;
    ill = (int)pLL->lat;
    if (ill % 10)
        return false;
    return true;
}

void util_CleanLine( QString & line )
{
    line.replace(QChar('\n'),QChar(' '));
}

// this is REALY only for diagnostics -
// to waste some time only
void util_uselessWait(int secs)
{
    QTime to;
    to.start();
    int ms = secs * 1000;
    while (to.elapsed() < ms) {
        // what to do???
        /* DO NOTHING */;
    }
}

QString util_get_UsgsMap_text(void)
{
    QString text = "# 1: Urban and Built-Up Land\n"
    "BuiltUpCover\n"
    "# 2: Dryland Cropland and Pasture\n"
    "DryCropPastureCover\n"
    "# 3: Irrigated Cropland and Pasture\n"
    "IrrCropPastureCover\n"
    "# 4: Mixed Dryland/Irrigated Cropland and Pasture\n"
    "MixedCropPastureCover\n"
    "# 5: Cropland/Grassland Mosaic\n"
    "CropGrassCover\n"
    "# 6: Cropland/Woodland Mosaic\n"
    "CropWoodCover\n"
    "# 7: Grassland\n"
    "GrassCover\n"
    "# 8: Shrubland\n"
    "ShrubCover\n"
    "# 9: Mixed Shrubland/Grassland\n"
    "ShrubGrassCover\n"
    "# 10: Savanna\n"
    "SavannaCover\n"
    "# 11: Deciduous Broadleaf Forest\n"
    "DeciduousBroadCover\n"
    "# 12: Deciduous Needleleaf Forest\n"
    "DeciduousNeedleCover\n"
    "# 13: Evergreen Broadleaf Forest\n"
    "EvergreenBroadCover\n"
    "# 14: Evergreen Needleleaf Forest\n"
    "EvergreenNeedleCover\n"
    "# 15: Mixed Forest\n"
    "MixedForestCover\n"
    "# 16: Water Bodies\n"
    "Default\n"
    "# 17: Herbaceous Wetland\n"
    "HerbWetlandCover\n"
    "# 18: Wooded Wetland\n"
    "WoodedWetlandCover\n"
    "# 19: Barren or Sparsely Vegetated\n"
    "BarrenCover\n"
    "# 20: Herbaceous Tundra\n"
    "HerbTundraCover\n"
    "# 21: Wooded Tundra\n"
    "WoodedTundraCover\n"
    "# 22: Mixed Tundra\n"
    "MixedTundraCover\n"
    "# 23: Bare Ground Tundra\n"
    "BareTundraCover\n"
    "# 24: Snow or Ice\n"
    "SnowCover\n";

    return text;
}

QString util_get_Default_Priorities_text(void)
{
    QString text = "Default\t\t\t\t# Default area which can be overridden by\n"
            "# raster landcover data (e.g. USGS)\n"
            "Ocean\t\t\t\t# collect slivers as ocean\n"
            "# Area types in order of descending priority\n"
            "SomeSort\t\tother\n"
            "Hole\t\t\thole\t# Leave area completely empty\n"
            "Airport\t\t\tother\n"
            "Freeway\t\t\troad\n"
            "Road\t\t\troad\n"
            "Railroad\t\troad\n"
            "Pond\t\t\tlake\n"
            "Lake\t\t\tlake\n"
            "DryLake\t\t\tlake\n"
            "IntLake\t\t\tlake\n"
            "Reservoir\t\tlake\n"
            "IntReservoir\t\tlake\n"
            "Stream\t\t\tstream\n"
            "IntStream\t\tstream\n"
            "Canal\t\t\tstream\n"
            "Glacier\t\t\tother\t# Solid ice/snow\n"
            "PackIce\t\t\tother\t# Water with ice packs\n"
            "PolarIce\t\tother\n"
            "Ocean\t\t\tocean\n"
            "Urban\t\t\tother\t# Densely-populated city or large town\n"
            "Town\t\t\tother\t# Small town or village\n"
            "FloodLand\t\tother\t# Land subject to flooding\n"
            "Bog\t\t\tother\t# Bog\n"
            "Marsh\t\t\tother\t# Marshland or swamp\n"
            "Sand\t\t\tother\t# Sand-covered area\n"
            "Littoral\t\tother\t# Tidal, Sand-covered area\n"
            "Lava\t\t\tother\t# Lava-covered area\n"
            "# USGS Land Covers\n"
            "# These are low-priority, since known polygons should always win.\n"
            "BuiltUpCover\t\tother\t# Urban and Built-Up Land\n"
            "DryCropPastureCover\tother\t# Dryland Cropland and Pasture\n"
            "IrrCropPastureCover\tother\t# Irrigated Cropland and Pasture\n"
            "MixedCropPastureCover\tother\t# Mixed Dryland/Irrigated Cropland and Pasture\n"
            "CropGrassCover\t\tother\t# Cropland/Grassland Mosaic\n"
            "CropWoodCover\t\tother\t# Cropland/Woodland Mosaic\n"
            "GrassCover\t\tother\t# Grassland\n"
            "GrassLand       other\n"
            "ShrubCover\t\tother\t# Shrubland\n"
            "ShrubGrassCover\t\tother\t# Mixed Shrubland/Grassland\n"
            "SavannaCover\t\tother\t# Savanna\n"
            "DeciduousBroadCover\tother\t# Deciduous Broadleaf Forest\n"
            "DeciduousNeedleCover\tother\t# Deciduous Needleleaf Forest\n"
            "EvergreenBroadCover\tother\t# Evergreen Broadleaf Forest\n"
            "EvergreenNeedleCover\tother\t# Evergreen Needleleaf Forest\n"
            "MixedForestCover\tother\t# Mixed Forest\n"
            "MixedForest         other\n"
            "EvergreenForest     other\n"
            "WaterBodyCover\t\tlake\t# Water Bodies\n"
            "HerbWetlandCover\tother\t# Herbaceous Wetland\n"
            "WoodedWetlandCover\tother\t# Wooded Wetland\n"
            "BarrenCover\t\tother\t# Barren or Sparsely Vegetated\n"
            "HerbTundraCover\t\tother\t# Herbaceous Tundra\n"
            "WoodedTundraCover\tother\t# Wooded Tundra\n"
            "MixedTundraCover\tother\t# Mixed Tundra\n"
            "BareTundraCover\t\tother\t# Bare Ground Tundra\n"
            "SnowCover\t\tother\t# Snow or Ice\n"
            "AgroForest\t\tother\n"
            "Asphalt\t\tother\n"
            "Burnt\t\tother\n"
            "Cemetery\t\tother\n"
            "ComplexCrop\t\tother\n"
            "Construction\t\tother\n"
            "CropGrass\t\tother\n"
            "DeciduousForest\t\tother\n"
            "Dirt\t\tother\n"
            "DryCrop\t\tother\n"
            "Dump\t\tother\n"
            "Estuary\t\tother\n"
            "GolfCourse\t\tother\n"
            "GreenSpace\t\tother\n"
            "Heath\t\tother\n"
            "HerbTundra\t\tother\n"
            "Industrial\t\tother\n"
            "IntermittentLake\t\tother\n"
            "IntermittentStream\t\tother\n"
            "IrrCrop\t\tother\n"
            "Lagoon\t\tother\n"
            "MixedCrop\t\tother\n"
            "NaturalCrop\t\tother\n"
            "Olives\t\tother\n"
            "OpenMining\t\tother\n"
            "Orchard\t\tother\n"
            "Port\t\tother\n"
            "RainForest\t\tother\n"
            "Rice\t\tother\n"
            "Rock\t\tother\n"
            "Saline\t\tother\n"
            "SaltMarsh\t\tother\n"
            "Sclerophyllous\t\tother\n"
            "ScrubCover\t\tother\n"
            "SubUrban\t\tother\n"
            "Transport\t\tother\n"
            "Vineyard\t\tother\n"
            "Watercourse\t\tother\n"
            "Landmass\t\t\tlandmass\n"
            "Island\t\t\tisland\t# any island area not covered otherwise\n"
            "Default\t\t\tlandmass # any land area not covered otherwise\n"
            "Void\t\t\tother\n"
            "Null\t\t\tother\n"
            "Unknown\t\t\tother\n";
    return text;
}

bool util_ensure_Priorities_Text(QString workDir,
                                 QString defIn,
                                 QString usgsIn)
{
    QString text;
    QString defTxt = workDir+"/"+defIn;
    QString usgsTxt = workDir+"/"+usgsIn;
    QFile def(defTxt);
    int flag = 0;
    if ( def.exists() ) {
        flag |= 1;
    } else {
        if (def.open(QIODevice::ReadWrite | QIODevice::Truncate | QFile::Text)) {
            QTextStream out(&def);
            text = util_get_Default_Priorities_text();
            out << text;
            out.flush();
            def.close();
            flag |= 1;
        }
    }
    QFile usgs(usgsTxt);
    if (usgs.exists()) {
        flag |= 2;
    } else {
        if (usgs.open(QIODevice::ReadWrite | QIODevice::Truncate | QFile::Text)) {
            QTextStream out(&usgs);
            text = util_get_UsgsMap_text();
            out << text;
            out.flush();
            usgs.close();
            flag |= 2;
        }
    }
    return ((flag == 3) ? true : false);
}

// Output file = C:/FG/29/Scene2/scenery/Terrain/e150s40/e150s37/5410128.btg.gz
QStringList util_getOutputFiles( QString info )
{
    int pos0, pos1, pos2;
    QString find = "Output file = ";
    QString file;
    QStringList fileList;
    pos0 = 0;
    while (1) {
        pos1 = info.indexOf(find,pos0);
        if (pos1 > 0) {
            pos1 += find.size();
            pos2 = info.indexOf(".btg.gz");
            if (pos2 > 0) {
                file = info.mid(pos1, pos2 - pos1 + 7);
                // ignore if it involves a line wrap
                if (! file.contains(QChar('\n')))
                    fileList += file; // add an OUT file
            } else
                break;
            pos0 = pos1; // search forward from last find
        } else
            break;
    }
    return fileList;
}

bool util_trimLongMessage( int max, QString & info )
{
    bool iret = false;
    if ((max > 10) && (info.size() > max)) {
        // we need to CHOP some info
        // 1: write the FULL string to the LOG
        outLog("Chop =["+info+"]=FULL\n"); // to  =["+tmp+"]=\n");
        // 2: try just 'simplify'
        info = info.simplified();
        if (info.size() > max) {
            // 3: ok, really TRUCATE
            QString tmp = info.left(max / 2);
            tmp += " ... ";
            tmp += info.right(max / 2);
            info = tmp;
            iret = true;
        }
    }
    return iret;
}

// list like from multiple strings like
// array_file = /home/geoff/fg/fg14/scenery/work/SRTM-30/e150s40/e150s31/5410553.arr.gz
QStringList util_getArrayFileList(QString info)
{
    QStringList list;
    QString find = "array_file = ";
    QString file;
    int pos1,pos2,off;
    off = 0;
    while (1) {
        pos1 = info.indexOf(find,off);
        if (pos1 > 0) {
            pos1 += find.size();
            pos2 = info.indexOf(".arr.gz",pos1);
            if (pos2 > 0) {
                file = info.mid(pos1, pos2 - pos1 + 7);
                if (list.indexOf(file) == -1)
                    list += file;
                off = pos2 + 7; // move onto next position
            } else
                break;
        } else
            break;
    }
    return list;
}

// eof - tggui_utils.cpp


