/*-------------------------------------------------------------------------
  win_utils.cxx

  Written by Geoff R. McLane, started January 2008.

  Copyright (C) 2008 Geoff R. McLane

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  ---------------------------------------------------------------------------*/

// win_utils.cxx
// Various minor utility functions for Atlas/Map suite of applications

#pragma warning(disable:4996)
#include <string>
#include <sstream>
#include <iostream>
#include <math.h>

#define ATLAS_STRINGIZE(X) ATLAS_DO_STRINGIZE(X)
#define ATLAS_DO_STRINGIZE(X) #X
#define ATLAS_COMPILER_STR "Microsoft Visual C++ version " ATLAS_STRINGIZE(_MSC_VER)

using namespace std;

/* -----------------------
DESCRIPTION
The remainder () function computes the remainder of dividing x by y.
The return value is x " - " n " * " y, where n is the value x " / " y,
rounded to the nearest integer. If this quotient is 1/2 (mod 1), it is
rounded to the nearest even number (independent of the current rounding 
mode). If the return value is 0, it has the sign of x. The drem () function
does precisely the same thing. 
or
The remainder() function returns the remainder r: 
x - n * y;
where n is the integer nearest the exact value of x/y; if |n - x/y| = 1/2,
then n is even. 
   ----------------------- */

float remainderf( float x, float y )
{
   int n = (int)(( x / y ) + 0.5);
   float result = x - ( n * y );
   return result;
}

char * basename( char * name )
{
   size_t len = strlen(name);
   size_t i, j;
   int c;
   j = 0;
   for( i = 0; i < len; i++ )
   {
      c = name[i];
      if(( c == '/' )||
         ( c == '\\'))
      {
         j = i + 1;
      }
   }
   return &name[j];
}

void set_win_path_sep( std::string & s )
{
   size_t fnd;
   while( (fnd = s.find('/')) != -1 )
      s.replace(fnd, 1, "\\");
}

double er_equ = 6378137.0;   // earth radius, equator (6,378.1370 km?)
double er_pol = 6356752.314; // earth radius, polar   (6,356.7523 km?)
double get_erad( double lat )
{
	// case SANSON_FLAMSTEED:
	double a = cos(lat) / er_equ;
	double b = sin(lat) / er_pol;
	return (1.0 / sqrt( a * a + b * b ));
}

void ll2pt( double obj_latr, double obj_lonr,
           double cent_latr, double cent_lonr,
           double map_size,  double map_zoom,
           double * px_pixel, double * py_pixel )
{
	double x,y,r;
	r = get_erad(cent_latr); // case SANSON_FLAMSTEED:
   x = r * cos(obj_latr)*(obj_lonr - cent_lonr);
   y = r * (obj_latr - cent_latr);
	*px_pixel = (map_size / 2.0) + (x * map_zoom);
	*py_pixel = (map_size / 2.0) + (y * map_zoom);
}

void pt2ll( double * pobj_latr, double * pobj_lonr,
           double cent_latr, double cent_lonr,
           double map_size,  double map_zoom,
           double x_pixel, double y_pixel )
{
	double x,y,r;
	r = get_erad(cent_latr); // case SANSON_FLAMSTEED:
	x = (x_pixel - (map_size / 2.0)) / map_zoom;
	y = (y_pixel - (map_size / 2.0)) / map_zoom;
      *pobj_latr = (y / r) + cent_latr;
      *pobj_lonr = (x / (r * cos(*pobj_latr))) + cent_lonr; 
}

bool  check_map_executable( std::string exe )
{
   bool failed = true;  // assume it will
   FILE *f;
   std::ostringstream cmd;
   std::string str = "";
   char c;
   size_t n;

   set_win_path_sep(exe);
   cmd << exe << " --version";
   if ((f = _popen(cmd.str().c_str(), "r")) == NULL) {
   } else {
      while (true) {
         n = fread(&c, 1, 1, f);
         if (n == 0) {
            if (feof(f)) {
               fclose(f);
               // check for 'version' is output by map
               if(( str.find(": version: ") != -1 )||   // found new version
                  ( str.find("FlightGear mapping utility") != -1 )) // or even OLD version
                  failed = false;
               break;
            }
         }
         str.append(1, c);
      }
   }
   if(failed) {
      if( str.length() > 1 )
         printf("WARNING: Got output '%s'\n", str.c_str() );
      printf("WARNING: The exe '%s' doesn't exist.\nAny missing maps can not be generated!\n",
         exe.c_str() );
      printf("Continue regardless? Enter y to continue, else will exit: ");
      std::cin >> c;
      if(( c == 'y' )||( c == 'Y' )) {
         printf( " continuing ...\n" );
         failed = false;
      } else {
         printf( " aborting ...\n" );
         exit(1);
      }
   }
   return failed;
}

std::string get_compiler_version_string( void )
{
   string s = ATLAS_COMPILER_STR;
   s += "\n";
   s += "Compiled on ";
   s += __DATE__;
   s += " at ";
   s += __TIME__;
   char * env = getenv("COMPUTERNAME");
   if(env) {
      s += "\n";
      s += "In ";
      s += env;
      env = getenv("USERNAME");
      if(env) {
         s += " by ";
         s += env;
      }
   }
   return s;
}


// eof - win_utils.cxx
