/*-------------------------------------------------------------------------
  Program for getting landsat image and preparing them for Atlas

  Written by Frederic Bouvier, started September 2004.

  Copyright (C) 2005 Frederic Bouvier, fredb@users.sourceforge.net

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
---------------------------------------------------------------------------

Try with --base-url='http://wms.jpl.nasa.gov/cgi-bin/wms.cgi?LAYERS=modis,global_mosaic&styles=default,visual&'

---------------------------------------------------------------------------
  CHANGES
  2009-01-11        Minor changes for WIN32
  2005-09-28        Initial version
---------------------------------------------------------------------------*/
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif
#define  OUT_HTML_FILE  "tempmaps.htm"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>

#include <iostream>
#include <string>
#include <iomanip>
#include <sstream>
#include <fstream>
#include <curl/curl.h>

bool verbose = false;
int size = 256;
int min_lat = 1000;
int min_lon = 1000;
int max_lat = -1000;
int max_lon = -1000;
std::string outp;
std::string base_url;
size_t file_size;
std::ostringstream memstream;

/*****************************************************************************/

void print_help() {
  printf("GetMap - FlightGear mapping retrieval utility\n\n");
  printf("Usage:\n");
  printf("  --size=pixels           Create map of size pixels*pixels (default 256)\n");
  printf("  --base-url=url          Beginning of the url of the WMS server\n");
  printf("  --atlas=path            Create maps of all scenery, and store them in path\n");
  printf("  --min-lat               Minimum latitude\n");
  printf("  --max-lat               Maximum latitude\n");
  printf("  --min-lon               Minimum longitude\n");
  printf("  --max-lon               Maximum longitude\n");
  printf("  --verbose               Display information during processing\n");
}

bool parse_arg(char* arg) {

  if ( sscanf(arg, "--size=%d", &size) == 1 ) {
    // Nothing
  } else if ( sscanf(arg, "--min-lat=%d", &min_lat) == 1 ) {
    // Nothing
  } else if ( sscanf(arg, "--min-lon=%d", &min_lon) == 1 ) {
    // Nothing
  } else if ( sscanf(arg, "--max-lat=%d", &max_lat) == 1 ) {
    // Nothing
  } else if ( sscanf(arg, "--max-lon=%d", &max_lon) == 1 ) {
    // Nothing
  } else if ( strncmp(arg, "--atlas=", 8) == 0 ) {
    outp = arg+8;
  } else if ( strncmp(arg, "--base-url=", 11) == 0 ) {
    base_url = arg+11;
  } else if ( strcmp(arg, "--verbose") == 0 ) {
    verbose = true;
  } else if ( strcmp(arg, "--help") == 0 ) {
    print_help();
    exit(0);
  } else {
    return false;
  }

  return true;
}

size_t writeData_cb( void *buffer, size_t size, size_t nmemb, void *userp ) {
  memstream.write( (char *)buffer, size * nmemb );
  file_size += size * nmemb;
  return nmemb;
}

int progress_cb( void *clientp, double dltotal, double dlnow, double ultotal, double ulnow ) {
  printf( "Gotten %lu bytes                  \r", (unsigned long)dlnow );
  return 0;
}

// helper to condition the URL give
void ensure_trailing_at( std::string & s)
{
   std::string::size_type len, ind;
   len = s.length();
   if(len) {
      len--;
      ind = s.find("&");
      if( ind != len )
         s += "&";
   }
}

size_t _instri( const char * str, char * fnd )
{
   size_t iret = -1;
   if( stricmp( str, fnd ) == 0 )
      iret = 0;
   else {
      size_t len = strlen(str);
      size_t fl = strlen(fnd);
      if(len && (len > fl)) {
         if(fl == 0)
            iret = 0;   // everything matches nothing!!!
         else {
            size_t i, max;
            max = len - fl;
            for( i = 0; i <= max; i++ ) {
               if( strnicmp( &str[i], fnd, fl ) == 0 ) {
                  iret = i;
                  break;
               }
            }
         }
      }
   }
   return iret;
}


int main( int argc, char **argv ) {
  if (argc < 5)   // need, base url, plus min, max, lat, lon
    print_help();

  // process command line arguments
  for (int arg = 1; arg < argc; arg++) {
    if (!parse_arg(argv[arg])) {
      fprintf(stderr, "%s: unknown argument '%s'.\n", argv[0], argv[arg]);
      print_help();
      exit(1);
    }
  }

  if ( size & ( size-1 ) ) {
    fprintf(stderr, "%s: --size should be a power of 2.\n", argv[0]);
    exit(1);
  }

  if ( outp.empty() ) {
    fprintf(stderr, "%s: --atlas option missing.\n", argv[0]);
    exit(1);
  } else {
     if( access(outp.c_str(), F_OK) == -1 ) {
       fprintf(stderr, "%s: --atlas=%s directory is NOT accessable.\n", argv[0], outp.c_str());
       exit(1);
     }
  }

  if ( base_url.empty() ) {
    fprintf(stderr, "%s: --base-url option missing.\n", argv[0]);
    exit(1);
  }
  if ( min_lat == 1000 ) {
    fprintf(stderr, "%s: --min-lat option missing.\n", argv[0]);
    exit(1);
  }
  if ( min_lat < -90 || min_lat >= 90 ) {
    fprintf(stderr, "%s: --min-lat out of range. Should be between -90 and 90.\n", argv[0]);
    exit(1);
  }
  if ( min_lon == 1000 ) {
    fprintf(stderr, "%s: --min-lon option missing.\n", argv[0]);
    exit(1);
  }
  if ( min_lon < -180 || min_lon >= 180 ) {
    fprintf(stderr, "%s: --min-lon out of range. Should be between -180 and 180.\n", argv[0]);
    exit(1);
  }
  if ( max_lat == -1000 ) {
    fprintf(stderr, "%s: --max-lat option missing.\n", argv[0]);
    exit(1);
  }
  if ( max_lat < -90 || max_lat >= 90 ) {
    fprintf(stderr, "%s: --max-lat out of range. Should be between -90 and 90.\n", argv[0]);
    exit(1);
  }
  if ( max_lon == -1000 ) {
    fprintf(stderr, "%s: --max-lon option missing.\n", argv[0]);
    exit(1);
  }
  if ( max_lon < -180 || max_lon >= 180 ) {
    fprintf(stderr, "%s: --max-lon out of range. Should be between -180 and 180.\n", argv[0]);
    exit(1);
  }

  if ( min_lon > max_lon ) {
    max_lon += 360;
  }

  std::string::size_type indexCh = base_url.find("?");   // is there a '?'
  if( indexCh == std::string.npos ) // no, add full command
     base_url += "?layers=modis,global_mosaic&styles=default,visual";
  else {
     std::string ss = base_url.substr(indexCh+1); // get the SUB-string, after '?'
     if( _instri( ss.c_str(), "layers=" ) == -1 ) {
        ensure_trailing_at(base_url);
        base_url += "layers=modis,global_mosaic";
     }
     if( _instri( ss.c_str(), "styles=" ) == -1 ) {
        ensure_trailing_at(base_url);
        base_url += "styles=default,visual";
     }
  }
  ensure_trailing_at(base_url);

  std::cout << "Getting landsat images from " << base_url.c_str() << std::endl;

  CURL *ceh = curl_easy_init();
  std::ostringstream html;    // for simple HTML output
  html << "<html>" << std::endl;
  html << "<head>" << std::endl;
  html << "<title>Images from landsat</title>" << std::endl;
  html << "</head>" << std::endl;
  html << "<body>" << std::endl;
  html << "<h1 align=\"center\">Images from " << max_lat << "," << min_lon << " to " <<
     min_lon << "," << max_lon << "</h1>" << std::endl;
  html << "<table border=\"0\" cellspacing=\"0\" cellpadding=\"0\" summary=\"Fetched images\">" << std::endl;

  if ( ceh != 0 ) {
    for ( int y = max_lat; y > min_lat; y -= 1 ) { // top, down to bottom
       html << "<tr>" << std::endl;
      for ( int x = min_lon; x < max_lon; x += 1 ) { // left, to right
        int rx = x;
        if ( rx >= 180 ) {
          rx -= 360;
        }
        std::ostringstream fname;
        std::ostringstream fn;   // file name ONLY, for HTML output
        fname << outp << "/" << ( rx < 0 ? "w" : "e" ) << std::setw( 3 ) << std::setfill( '0' ) << abs(rx)
                            << ( y < 0 ? "s" : "n" ) << std::setw( 2 ) << std::setfill( '0' ) << abs(y)
                            << ".jpg";
        fn << ( rx < 0 ? "w" : "e" ) << std::setw( 3 ) << std::setfill( '0' ) << abs(rx)
                            << ( y < 0 ? "s" : "n" ) << std::setw( 2 ) << std::setfill( '0' ) << abs(y)
                            << ".jpg";

        struct stat stat_buf;
        if ( stat( fname.str().c_str(), &stat_buf ) == 0 ) {
          html << "<td>";
          html << "<img src=\"" << fn.str() << "\" alt=\"Existing " << fn.str() << "\">";
          html << "</td>" << std::endl;
          if(verbose)
             std::cout << "Skipping existing '" << fname.str().c_str() << "'" << std::endl;
          continue;
        }

        std::ostringstream surl;
        surl << base_url << "REQUEST=GetMap&VERSION=1.1.1&WIDTH=" << size << "&HEIGHT=" << size << "&BBOX="
            << rx << "," << y << "," << rx+1 << "," << y+1 << "&FORMAT=image/jpeg&SRS=EPSG:4326";

        std::string url = surl.str();
        curl_easy_setopt( ceh, CURLOPT_URL, url.c_str() );
        curl_easy_setopt( ceh, CURLOPT_WRITEFUNCTION, writeData_cb );
        curl_easy_setopt( ceh, CURLOPT_WRITEDATA, 0 );
	curl_easy_setopt( ceh, CURLOPT_NOPROGRESS, 0 );
	curl_easy_setopt( ceh, CURLOPT_PROGRESSFUNCTION, progress_cb );
	curl_easy_setopt( ceh, CURLOPT_PROGRESSDATA, 0 );
	//curl_easy_setopt( ceh, CURLOPT_INFILESIZE, -1 );

        if( verbose )
           std::cout << "Fetching '" << fname.str().c_str() << "'" << std::endl;

        file_size = 0;
        memstream.str("");
        int success = curl_easy_perform( ceh );
        html << "<td>";
        if ( success == 0 ) {
          char *cType;
          curl_easy_getinfo( ceh, CURLINFO_CONTENT_TYPE, &cType );
          if ( strcmp( cType, "image/jpeg" ) == 0 ) {
            std::ofstream fstr( fname.str().c_str(), std::ios::binary );
            if( fstr.good() ) {
               fstr.write( memstream.str().data(), file_size );
               fstr.close();
               std::cout << "Written '" << fname.str().c_str() << "', size " <<
                  file_size << " bytes." << std::endl;
               html << "<img src=\"" << fn.str() << "\" alt=\"" << fn.str() << "\">";
            } else {
               std::cout << "FAILED Write '" << fname.str().c_str() << "'" << std::endl;
               html << "no image";
            }
          } else {
            std::cout << "FAILED to fetch image/jpeg! Got '" << cType << "'?" << std::endl;
            html << "no image";
          }
        } else {
          std::cout << "FAILED to in curl fetch!" << std::endl;
          html << "no image";
        }
        html << "</td>" << std::endl;
      }
      html << "</tr>" << std::endl;
    }
    // finish off HTML
    html << "</table>" << std::endl;
    html << "</body>" << std::endl;
    html << "</html>" << std::endl;
    std::ostringstream htmname;
    htmname << outp << "/" << OUT_HTML_FILE;
    std::ofstream fil( htmname.str().c_str(), std::ios::out );
    fil.write( html.str().c_str(), html.str().length() );
    fil.close();
    std::cout << "Written '" << htmname.str().c_str() << "'" << std::endl;
  }
}
