/*-------------------------------------------------------------------------
  Preferences.cxx

  Written by Brian Schack, started August 2007.

  Copyright (C) 2007 Brian Schack

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  ---------------------------------------------------------------------------*/
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif
#include <stdlib.h>
#include <getopt.h>
#ifndef _MSC_VER
#include "libgen.h"
#endif   // !_MSC_VER
#include <fstream>

// This is a space-saving macro used when parsing command-line
// arguments.  The function 'f' is assumed to be sscanf().
#define OPTION_CHECK(f,n,t) if ((f) != (n)) {\
	fprintf(stderr, "%s: bad option argument\n", basename(argv[0]));\
	print_help_for(t);						\
	return false;	  \
    }

#include "Preferences.hxx"
#include "MapBrowser.hxx"

// Preferences file.
const char *atlasrc = ".atlasrc";

// These are used to tag options.  They *cannot* begin at 0, they
// *must* be consecutive, FIRST_OPTION *must* come first, LAST_OPTION
// *must* come last, and the option *cannot* have the value 63 (ASCII
// '?').  Other than that, you can do anything you want. :-)
enum {FIRST_OPTION, 
      LAT_OPTION, LON_OPTION, AIRPORT_OPTION, PATH_OPTION, 
      FG_ROOT_OPTION, GLUTFONTS_OPTION, GEOMETRY_OPTION, 
      SOFTCURSOR_OPTION, UDP_OPTION, SERIAL_OPTION, BAUD_OPTION, 
      SQUARE_OPTION, FG_SCENERY_OPTION,
#ifndef NO_RSYNC_APP
      SERVER_OPTION,
#endif // #ifndef NO_RSYNC_APP
      MAP_PATH_OPTION, 
      SIZE_OPTION, LOWRES_SIZE_OPTION, MAX_TRACK_OPTION, 
#ifndef NO_RSYNC_APP
      TERRASYNC_MODE_OPTION,
#endif // #ifndef NO_RSYNC_APP
      CONCURRENCY_OPTION, UPDATE_OPTION,
      AUTO_CENTER_MODE_OPTION, VERSION_OPTION, HELP_OPTION,
      LAST_OPTION
};

// Used by getopt_long()
static struct option long_options[] = {
    {"lat", required_argument, 0, LAT_OPTION},
    {"lon", required_argument, 0, LON_OPTION},
    {"airport", required_argument, 0, AIRPORT_OPTION},
    {"path", required_argument, 0, PATH_OPTION},
    {"fg-root", required_argument, 0, FG_ROOT_OPTION},
    {"glutfonts", no_argument, 0, GLUTFONTS_OPTION},
    {"geometry", required_argument, 0, GEOMETRY_OPTION},
    {"softcursor", no_argument, 0, SOFTCURSOR_OPTION},
    {"udp", optional_argument, 0, UDP_OPTION},
    {"serial", optional_argument, 0, SERIAL_OPTION},
    {"baud", required_argument, 0, BAUD_OPTION},
    {"update", required_argument, 0, UPDATE_OPTION},
    {"square", no_argument, 0, SQUARE_OPTION},
    {"fg-scenery", required_argument, 0, FG_SCENERY_OPTION},
#ifndef NO_RSYNC_APP
    {"server", required_argument, 0, SERVER_OPTION},
#endif // #ifndef NO_RSYNC_APP
    {"map-executable", required_argument, 0, MAP_PATH_OPTION},
    {"size", required_argument, 0, SIZE_OPTION},
    {"lowres-size", required_argument, 0, LOWRES_SIZE_OPTION},
    {"max-track", required_argument, 0, MAX_TRACK_OPTION},
#ifndef NO_RSYNC_APP
    {"terrasync-mode", no_argument, 0, TERRASYNC_MODE_OPTION},
#endif // #ifndef NO_RSYNC_APP
    {"concurrency", required_argument, 0, CONCURRENCY_OPTION},
    {"autocenter-mode", no_argument, 0, AUTO_CENTER_MODE_OPTION},
    {"version", no_argument, 0, VERSION_OPTION},
    {"help", no_argument, 0, HELP_OPTION},
    {0, 0, 0, 0}
};

static void print_short_help(char *name) {
    printf("usage: %s [--lat=<x>] [--lon=<x>] [--airport=<icao>] [--path=<path>]\n", name);
    printf("\t[--fg-root=<path>] [--glutfonts] [--geometry=<w>x<h>]\n");
    printf("\t[--softcursor] [--udp[=<port>]] [--serial=[<dev>]] [--baud=<rate>]\n");
    printf("\t[--square] [--fg-scenery=<path>] ");
#ifndef NO_RSYNC_APP
    printf("[--server=<addr>]");
#endif // #ifndef NO_RSYNC_APP
    printf("\n");
    printf("\t[--map-executable=<path>] [--size=<pixels>] [--lowres-size=<pixels>]\n");
    printf("\t[--max-track=<x>] ");
#ifndef NO_RSYNC_APP
    printf("[--terrasync-mode] ");
#endif // #ifndef NO_RSYNC_APP
    printf("[--concurrency=<n>]\n");
    printf("\t[--update=<s>] [--autocenter-mode] [--version]\n");
    printf("\t[--help] [<flight file>] ...\n");
}

// Prints a long entry for the give option.
static void print_help_for(int option) 
{
    switch(option) {
    case LAT_OPTION:
	printf("--lat=<x>\tStart browsing at latitude x (deg. south i neg.)\n");
	break;
    case LON_OPTION:
	printf("--lon=<x>\tStart browsing at longitude x (deg. west i neg.)\n");
	break;
    case AIRPORT_OPTION:
	printf("--airport=<icao>\tStart browsing at an airport specified by ICAO code icao\n");
	break;
    case PATH_OPTION:
	printf("--path=<path>\tSet path for map images\n");
	break;
    case FG_ROOT_OPTION:
	printf("--fg-root=<path>\tOverrides FG_ROOT environment variable\n");
	break;
    case GLUTFONTS_OPTION:
	printf("--glutfonts\tUse GLUT bitmap fonts (fast for software rendering)\n");
	break;
    case GEOMETRY_OPTION:
	printf("--geometry=<w>x<h>\tSet initial window size\n");
	break;
    case SOFTCURSOR_OPTION:
	printf("--softcursor\tDraw mouse cursor using OpenGL (for fullscreen Voodoo cards)\n");
	break;
    case UDP_OPTION:
	printf("--udp[=<port>]\tInput read from UDP socket at specified port\n");
	printf("\t(defaults to 5500)\n");
	break;
    case SERIAL_OPTION:
	printf("--serial[=<dev>]\tInput read from serial port with specified device\n");
	printf("\t(defaults to /dev/ttyS0)\n");
	break;
    case BAUD_OPTION:
	printf("--baud=<rate>\tSet serial port baud rate (defaults to 4800)\n");
	break;
    case UPDATE_OPTION:
	printf("--update=<s>\tCheck for position updates every x seconds (defaults to 1.0)\n");
	break;
    case SQUARE_OPTION:
	printf("--square\tSet square mode (map 1x1 degree area on the whole image)\n");
	printf("\tto be compatible with images retrieved by GetMap\n");
	break;
    case FG_SCENERY_OPTION:
	printf("--fg-scenery=<path>\tLocation of FlightGear scenery (defaults to\n");
	printf("\t<fg-root>/Scenery-Terrasync)\n");
	break;
#ifndef NO_RSYNC_APP
    case SERVER_OPTION:
	printf("--server=<addr>\tRsync scenery server (defaults to\n");
	printf("\t'scenery.flightgear.org')\n");
	break;
#endif // #ifndef NO_RSYNC_APP
    case MAP_PATH_OPTION:
	printf("--map-executable=<path>\tLocation of Map executable (defaults to 'Map')\n");
	break;
    case SIZE_OPTION:
	printf("--size=<pixels>\tCreate maps of size pixels*pixels (default 256)\n");
	break;
    case LOWRES_SIZE_OPTION:
	printf("--lowres-size=<pixels>\tCreate lowres maps of size pixels*pixels\n");
	printf("\t(defaults to 0, meaning don't generate lowres maps)\n");
	break;
    case MAX_TRACK_OPTION:
	printf("--max-track=<x>\tMaximum number of points to record while tracking a\n");
	printf("\tflight (defaults to 2000, 0 = unlimited)\n");
	break;
#ifndef NO_RSYNC_APP
    case TERRASYNC_MODE_OPTION:
	printf("--terrasync-mode\tDownload scenery while tracking a flight (default is\n");
	printf("\tto not download)\n");
	break;
#endif // #ifndef NO_RSYNC_APP
    case CONCURRENCY_OPTION:
	printf("--concurrency=<n>\tNumber of tiles to simultaneously update (defaults to\n");
	printf("\t1, 0 = unlimited)\n");
	break;
    case AUTO_CENTER_MODE_OPTION:
	printf("--autocenter-mode\tAutomatically center map on aircraft (default is\n");
	printf("\tto not auto-center)\n");
	break;
    case VERSION_OPTION:
	printf("--version\tPrint version number\n");
	break;
    case HELP_OPTION:
	printf("--help\tPrint this help\n");
	break;
    }
}

static char * atlas_keyboard_help( void )
{
   static char keyhelp[] =
      "Atlas keyboard, when running ...\n"
      " +/-    Zoom In/Out.\n"
      " A/a    Toggle Airport show.\n"
      " C      Toggle Auto Centering.\n"
      " c      Center Map on Aircraft.\n"
      " D/d    Toggle Hide/Show the info interface and the graphs window.\n"
      " F/f    Select the next ('f') or previous ('F') flight track.\n"
      " J/j    Toggle the search interface.\n"
#ifndef NO_RSYNC_APP
      " L      Show the next downloading tile.\n"
      " l      Schedule or de-schedule the 1x1 tile at our current lat/lon for updating.\n"
#endif // #ifndef NO_RSYNC_APP
      " N/n    Toggle Show Navigation Aids.\n"
      " O/o    Open a Flight Track File.\n"
      " q      Quit, prompting for save of any active track.\n"
      " S/s    Save Flight Track File.\n"
      " T/t    Toggle Texturing.\n"
      " W/w   Clear/Close the current track. No warning if the track is unsaved.\n"
      " U/u     Un-attach (detach) Network, and restart tracking.\n"
      " V/v     Toggle Show Names.\n";
   return keyhelp;
}

// This prints a long help message.
static void print_help() {
  // EYE - use executable name here?
  printf("ATLAS - A map browsing utility for FlightGear\n\nUsage:\n\n");
  // EYE - use executable name here?
  printf("Atlas <options> [<flight file>] ...\n\n");
  for (int i = FIRST_OPTION + 1; i < LAST_OPTION; i++) {
      printf("   ");
      print_help_for(i);
  }
  printf( atlas_keyboard_help() );
}

// All preferences should be given default values here.
Preferences::Preferences()
{
    latitude = 37.5;
    longitude = -122.25;
    icao = strdup("");

    char *env = getenv("FG_ROOT");
    if (env == NULL) {
	fg_root.set(FGBASE_DIR);
    } else {
	fg_root.set(env);
    }

    if (fg_root.str().length() != 0) {
	path.set(fg_root.str());
    } else {
	path.set(FGBASE_DIR);
    }
    path.append("Atlas");

    textureFonts = true;
    width = 800;
    height = 600;
    softcursor = false;
//     port = strdup("5500");
//     device = strdup("/dev/ttyS0");
//     baud = strdup("4800");
    _port = 5500;
#ifdef _MSC_VER
    _serial.device = strdup("COM1");
#else // !_MSC_VER
    _serial.device = strdup("/dev/ttyS0");
#endif // _MSC_VER y/n
    _serial.baud = 4800;
    update = 1.0;
    mode = MapBrowser::ATLAS;

    scenery_root.set(fg_root.str());
    scenery_root.append("Scenery-Terrasync");

#ifndef NO_RSYNC_APP
    rsync_server = strdup("scenery.flightgear.org");
#endif // #ifndef NO_RSYNC_APP
    map_executable.set("Map");
    map_size = 256;
    lowres_map_size = 0;
    max_track = 2000;
#ifndef NO_RSYNC_APP
    terrasync_mode = false;
#endif // #ifndef NO_RSYNC_APP
    concurrency = 1;
    autocenter_mode = false;
}

// First loads preferences from ~/.atlasrc (if it exists), then checks
// the command line options passed in via argc and argv.
bool Preferences::loadPreferences(int argc, char *argv[])
{
    // Check for a preferences file.
    char* homedir = getenv("HOME");
    SGPath rcpath;

#ifdef   _MSC_VER
    std::ifstream locrc("tempatlasrc");
    if (locrc.is_open()) {
      char *lines[2];
      string line;
      lines[0] = argv[0];
      while (!locrc.eof()) {
   	   getline(locrc, line);
   	   if (line.length() == 0)
	         continue;
   	   if (line[0] == '#')
            continue;
   	   lines[1] = (char *)line.c_str();
         // Try to make sense of it.
         if (!_loadPreferences(2, lines)) {
            fprintf(stderr, "%s: Error in %s: '%s'.\n",
                basename(argv[0]), atlasrc, lines[1]);
            locrc.close();
            return false;
	      }
       }
       locrc.close();
    }
#endif   // _MSC_VER

    if (homedir != NULL) {
	rcpath.set(homedir);
	rcpath.append(atlasrc);
    } else {
	rcpath.set(atlasrc);
    }

    std::ifstream rc(rcpath.c_str());
    if (rc.is_open()) {
	char *lines[2];
	string line;

	// By default, getopt_long() (called in _loadPreferences())
	// skips past the first argument, which is usually the
	// executable name.  Theoretically, we should be able to tell
	// it to start processing from the first argument by setting
	// optind to 0, but I can't seem to get it to work, and
	// anyway, our error messages depend on argv[0] being set to
	// the executable.
	lines[0] = argv[0];
	while (!rc.eof()) {
	    getline(rc, line);

	    // Skip comments and emtpy lines.  Our version is given in
	    // a special comment of the format "#ATLASRC Version x".
	    int version;
	    if (line.length() == 0) {
		continue;
	    }
	    if (sscanf(line.c_str(), "#ATLASRC Version %d", &version) == 1) {
		if (version > 1) {
		    fprintf(stderr, "%s: Unknown %s version: %d\n",
			    basename(argv[0]), atlasrc, version);
		    return false;
		}
		continue;
	    }
	    if (line[0] == '#') {
		continue;
	    }

	    // EYE - should we remove leading and trailing whitespace?
	    // I guess it's a real option.
	    lines[1] = (char *)line.c_str();

	    // Try to make sense of it.
	    if (!_loadPreferences(2, lines)) {
		fprintf(stderr, "%s: Error in %s: '%s'.\n",
			basename(argv[0]), atlasrc, lines[1]);
		return false;
	    }
	}

	rc.close();
    }

    // Now parse the real command-line arguments.
    return _loadPreferences(argc, argv);
}

/* =============================================
   Output current settings, per 'rc' file format
  ============================================== */
void Preferences::savePreferences()
{
#ifdef _MSC_VER
   int i;
   char * tmprc = "tempatlasrc";
   FILE * fp = fopen( tmprc, "w" );
   if ( fp )
   {
      fprintf( fp, "#ATLASRC Version %s\n", VERSION );
      fprintf( fp, "# List of current preferences - \n" );
      // Start browsing at latitude x (deg. south i neg.)
      fprintf( fp, "--lat=%.2f\n", latitude );
      // Start browsing at longitude x (deg. west i neg.)
      fprintf( fp, "--lon=%.2f\n", longitude );
      // Start browsing at an airport specified by ICAO code icao
      fprintf( fp, "--airport=%s\n", icao );
      // Set path for map images
      fprintf( fp, "--path=%s\n", path.c_str() );
      // Overrides FG_ROOT environment variable
      fprintf( fp, "--fg-root=%s\n", fg_root.c_str() );
      if( !textureFonts ) // Use GLUT bitmap fonts (fast for software rendering)
         fprintf( fp, "# " );
      fprintf( fp, "--glutfonts\n" );
      // Set initial window size (geometry)
      fprintf( fp, "--geometry=%dx%d\n", width, height );
      if( !softcursor ) // Draw mouse cursor using OpenGL (for fullscreen Voodoo cards)
         fprintf( fp, "# " );
      fprintf( fp, "--softcursor\n" );

      fprintf( fp, "# Networks connections %d\n", networkConnections.size() );
      for (i = 0; i < networkConnections.size(); i++) {
	      fprintf( fp, "--udp=%u\n", networkConnections[i]) ; // Input read from UDP socket at specified port
      }

      fprintf( fp, "# Serial connections %d\n", serialConnections.size() );
      for (i = 0; i < serialConnections.size(); i++) {
	      fprintf( fp, "--serial=%s\n", serialConnections[i].device ); // Input read from serial port with specified device
         fprintf( fp, "--baud=%u\n", serialConnections[i].baud ); // Set serial port baud rate (defaults to 4800)
      }
      if( !mode )
         fprintf( fp, "# " );
      fprintf( fp, "--square\n" );   //  Set square mode (map 1x1 degree area on the whole image)
      // to be compatible with images retrieved by GetMap
      fprintf( fp, "--fg-scenery=%s\n", scenery_root.c_str() ); // Location of FlightGear scenery
#ifndef NO_RSYNC_APP
      fprintf( fp, "--server=%s\n", rsync_server );   // Rsync scenery server
#endif // #ifndef NO_RSYNC_APP
      fprintf( fp, "--map-executable=%s\n", map_executable.c_str() ); // Location of Map exe
      fprintf( fp, "--size=%d\n", map_size ); // Create maps of size pixels*pixels
      fprintf( fp, "--lowres-size=%d\n", lowres_map_size ); // Create lowres maps of size pixels*pixels
        // (defaults to 0, meaning don't generate lowres maps)
      fprintf( fp, "--max-track=%d\n", max_track ); // Maximum number of points to record
         // while tracking a flight (defaults to 2000, 0 = unlimited)
#ifndef NO_RSYNC_APP
      if( !terrasync_mode )   // Download scenery while tracking a flight
         fprintf( fp, "# " );  // kill it
      fprintf( fp, "--terrasync-mode\n" ); //  (default is to not download)
#endif // #ifndef NO_RSYNC_APP
      fprintf( fp, "--concurrency=%d\n", concurrency ); // Number of tiles to simultaneously update 
         // (defaults to 1, 0 = unlimited)
      if( !autocenter_mode ) // Automatically center map on aircraft
         fprintf( fp, "# " );
      fprintf( fp, "--autocenter-mode\n" ); // (default is to not auto-center)
      fprintf( fp, "--update=%.1f\n", update ); // Check for position updates every x seconds
         // (defaults to 1.0)
      fprintf( fp, "# Flight Track file(s) %d\n", flightFiles.size() );
      for (i = 0; i < flightFiles.size(); i++) {
      	fprintf( fp, "%s\n", flightFiles[i].c_str() );
      }
      fprintf( fp, "# eof - end of preferences\n" );
      printf("Preferences written to [%s] file.\n", tmprc );
   }
   else
   {
      printf("WARNING: Failed to write preferences to %s file.\n", tmprc );
   }
#endif // _MSC_VER
}

// Checks the given set of preferences.  Returns true (and sets the
// appropriate variables in Preferences) if there are no problems.
// Returns false (and prints an error message as appropriate) if
// there's a problem, or if the user asked for --version or --help.
bool Preferences::_loadPreferences(int argc, char *argv[])
{
    int c;
    int option_index = 0;
    SGPath p;

    // The use of optind (or optind and optreset, depending on your
    // system) is necessary because we may call getopt_long() many
    // times.
#ifdef HAVE_OPTRESET
    optreset = 1;
    optind = 1;
#else
    optind = 0;
#endif
    while ((c = getopt_long(argc, argv, "", long_options, &option_index)) 
	   != -1) {
	switch (c) {
	case LAT_OPTION:
	    OPTION_CHECK(sscanf(optarg, "%f", &latitude), 1, LAT_OPTION);
	    break;
	case LON_OPTION:
	    OPTION_CHECK(sscanf(optarg, "%f", &longitude), 1, LON_OPTION);
	    break;
	case AIRPORT_OPTION:
	    free(icao);
	    icao = strdup(optarg);
	    // Make sure it's in uppercase only.
	    for (unsigned int i = 0; i < strlen(icao); ++i) {
		icao[i] = toupper(icao[i]);
	    }
	    break;
	case PATH_OPTION:
	    path.set(optarg);
	    break;
	case FG_ROOT_OPTION:
	    fg_root.set(optarg);
	    break;
	case GLUTFONTS_OPTION:
	    textureFonts = false;
	    break;
	case GEOMETRY_OPTION:
	    OPTION_CHECK(sscanf(optarg, "%dx%d", &width, &height), 2, GEOMETRY_OPTION);
	    break;
	case SOFTCURSOR_OPTION:
	    softcursor = true;
	    break;
	case UDP_OPTION: {
	    // EYE - we need better documentation about how the UDP,
	    // SERIAL, and BAUD options interact.

	    // Whenever a unique --udp appears on the command line, we
	    // create an entry for it in networkConnections.  Whenever
	    // a unique --serial appears, we create an entry for it
	    // (using the current baud rate).  Whenever --baud
	    // appears, we just change the baud variable.  It does not
	    // affect --serial's that appear before it.
	    unsigned int thisPort = _port;
	    if (optarg) {
		OPTION_CHECK(sscanf(optarg, "%u", &thisPort), 1, UDP_OPTION);
	    }
       int fnd = 0; // do not generate another UDP connection,
       // which then also generates another track for same PORT
       for( size_t st = 0; st < networkConnections.size(); st++ ) {
          if( networkConnections[st] == thisPort ) {
             fnd = 1;
             break;
          }
       }
       if( !fnd )
	       networkConnections.push_back(thisPort);
	    break;
	}
	case SERIAL_OPTION: {
	    SerialConnection thisConnection;
	    thisConnection.baud = _serial.baud;
	    if (optarg) {
		thisConnection.device = strdup(optarg);
	    } else {
		thisConnection.device = strdup(_serial.device);
	    }
	    serialConnections.push_back(thisConnection);
	    break;
	}
	case BAUD_OPTION:
	    OPTION_CHECK(sscanf(optarg, "%u", &_serial.baud), 1, BAUD_OPTION);
	    break;
	case UPDATE_OPTION:
	    OPTION_CHECK(sscanf(optarg, "%f", &update), 1, UPDATE_OPTION);
	    break;
	case SQUARE_OPTION:
	    mode = MapBrowser::SQUARE;
	    break;
	case FG_SCENERY_OPTION:
	    scenery_root.set(optarg);
	    break;
#ifndef NO_RSYNC_APP
	case SERVER_OPTION:
	    free(rsync_server);
	    rsync_server = strdup(optarg);
	    break;
#endif // #ifndef NO_RSYNC_APP
	case MAP_PATH_OPTION:
	    map_executable.set(optarg);
	    break;
	case SIZE_OPTION:
	    OPTION_CHECK(sscanf(optarg, "%d", &map_size), 1, SIZE_OPTION);
	    break;
	case LOWRES_SIZE_OPTION:
	    OPTION_CHECK(sscanf(optarg, "%d", &lowres_map_size), 1, LOWRES_SIZE_OPTION);
	    break;
	case MAX_TRACK_OPTION:
	    OPTION_CHECK(sscanf(optarg, "%d", &max_track), 1, MAX_TRACK_OPTION);
	    break;
#ifndef NO_RSYNC_APP
	case TERRASYNC_MODE_OPTION:
	    terrasync_mode = true;
	    break;
#endif // #ifndef NO_RSYNC_APP
	case CONCURRENCY_OPTION:
	    OPTION_CHECK(sscanf(optarg, "%d", &concurrency), 1, CONCURRENCY_OPTION);
	    break;
	case AUTO_CENTER_MODE_OPTION:
	    autocenter_mode = true;
	    break;
	case HELP_OPTION:
	    print_help();
	    return false;
	    break;
	case VERSION_OPTION:
	    printf("%s version %s\n", basename(argv[0]), VERSION);
	    return false;
	    break;
	case '?':
	    // Note: We must make sure our _OPTION variables don't
	    // equal '?' (63).
	    print_short_help(basename(argv[0]));
	    return false;
	    break;
	default:
	    // We should never get here.
	    assert(false);   // but in case we do
       fprintf(stderr, "%s: unknown option!\n", basename(argv[0]) );
       return false;
	}
    }
    while (optind < argc) {
	p.set(argv[optind++]);
	flightFiles.push_back(p);
    }

    return true;
}
