/*
 * error.c - error handling functions
 *
 *   Author: Gabor Toth <tgbp@freemail.hu>
 *   License: GPL
 *
 *   $Log: error.c,v $
 *   Revision 1.2  2006/05/10 21:22:34  koversrac
 *   Comment with author and license has been added.
 *
 */

#include    "common.h"
#include    "error.h"
#include    <stdarg.h>          /* ANSI C header file */ 
#include    <syslog.h>          /* for syslog() */ 

int     daemon_proc;            /* set nonzero by daemon_init() */ 

/* Nonfatal error related to system call 
 * Print message and return */ 
void 
err_ret(const char *fmt, ...) 
{ 
    va_list ap; 
    va_start(ap, fmt); 
    err_doit(1, LOG_INFO, fmt, ap); 
    va_end(ap); 
    return; 
} 


/* Fatal error related to system call 
 * Print message and terminate */ 
void 
err_sys(const char *fmt, ...) 
{ 
    va_list ap; 
    va_start(ap, fmt); 
    err_doit(1, LOG_ERR, fmt, ap); 
    va_end(ap); 
    exit(1); 
} 


/* Fatal error related to system call 
 * Print message, dump core, and terminate */ 
void 
err_dump(const char *fmt, ...) 
{ 
    va_list ap; 
    va_start(ap, fmt); 
    err_doit(1, LOG_ERR, fmt, ap); 
    va_end(ap); 
    abort();                    /* dump core and terminate */ 
    exit(1);                    /* shouldn't get here */ 
} 


/* Nonfatal error unrelated to system call 
 * Print message and return */ 
void 
err_msg(const char *fmt, ...) 
{ 
    va_list ap; 
    va_start(ap, fmt); 
    err_doit(0, LOG_INFO, fmt, ap); 
    va_end(ap); 
    return; 
} 


/* Fatal error unrelated to system call 
 * Print message and terminate */ 
void 
err_quit(const char *fmt, ...) 
{ 
    va_list ap; 
    va_start(ap, fmt); 
    err_doit(0, LOG_ERR, fmt, ap); 
    va_end(ap); 
    exit(1); 
} 


/* Print message and return to caller 
 * Caller specifies "errnoflag" and "level" */ 
static void 
err_doit(int errnoflag, int level, const char *fmt, va_list ap) 
{ 
    int     errno_save, n; 
    char    buf[MAXLINE + 1]; 
    errno_save = errno;         /* value caller might want printed */ 
#ifdef HAVE_VSNPRINTF 
    vsnprintf(buf, MAXLINE, fmt, ap);   * safe */ 
#else 
    vsprintf(buf, fmt, ap);     /* not safe */ 
#endif 
    n = strlen(buf); 
    if (errnoflag) 
        snprintf(buf + n, MAXLINE - n, ": %s", strerror(errno_save)); 
    strcat(buf, "\n"); 
    if (daemon_proc) { 
        syslog(level, buf); 
    } else { 
        fflush(stdout);         /* in case stdout and stderr are the same */ 
        fputs(buf, stderr); 
        fflush(stderr); 
    } 
    return; 
} 

void
debug(int level,char *str)
{
	char buf[MAXLINE+1];
	if (level<=DEBUG_LEVEL)
	{
		//printf("%s\n",str);
		snprintf(buf,MAXLINE,"%s",str);
	       	syslog(level, buf); 
	}
}
