#!/perl -w
# NAME: fg_telnet.pl
# AIM: To RUN FlightGear, and get/send information to it using TELNET
# With much thanks to Franz Melchior for the 'signs' perl script,
# on which this is based.
#
# Note, although Term::ReadKey is used to CHECK for any keyboard input,
# and the main_loop() is terminated on any keyboard input, the process
# will NOT exit, due to the nature of fork() and exec() as implemented in WIN32
# The secondary process of fork() will WAIT until exec(FG) exits, and at present it appears
# sending the command "quit" is ignored by FG - maybe something is wrong here???
#
# 13/12/2008 geoff mclane http://geoffair.net/mperl
use strict;
use warnings;
use IO::Socket;
use Cwd;
use Win32::Console::ANSI;   # for WIN32
use Term::ReadKey;
require 'logfile.pl' or die "Unable to load logfile.pl ...\n";
# log file stuff
my ($LF);
my $pgmname = $0;
if ($pgmname =~ /\w{1}:\\.*/) {
    my @tmpsp = split(/\\/,$pgmname);
    $pgmname = $tmpsp[-1];
}
my $outfile = "temp.$pgmname.txt";
open_log($outfile);

my $debug_on = 1;   # force DEBUG on

my $def_fg_root = "C:\\FG\\27\\data";
my $def_fg_rt = 'C:\FG\27\bin';
my $def_fg_binary = 'flightgear';

my $cwd = cwd();
my $HOME = $ENV{HOME} || ".";
my $FG_HOME = $ENV{FG_HOME} || $HOME . "/.fgfs";
my $FG_ROOT = $ENV{FG_ROOT} || $def_fg_root;
my $BASEDIR = "$FG_ROOT/Local/signs";

my $FGFS = $def_fg_binary;
my $HOST = "localhost";
my $PORT = 5500;

my $INTERVAL = 1;       # get postion EACH second
my $USECOLOR = 1;
my $MIN_CHANGE = 0.00001;

my $CLR_ERR = "41m\033[33;1";
my $CLR_WARN = "47m\033[31;1";
my $CLR_INFO = "32";
my $CLR_BULK = "";
my $CLR_DEBUG = "36;1";
#             $ERR      $WARN      $INFO      $BULK      $DEBUG
my @COLOR = ( $CLR_ERR, $CLR_WARN, $CLR_INFO, $CLR_BULK, $CLR_DEBUG );

my $PI = 3.1415926535897932384626433832795029;
my $D2R = $PI / 180;
my $R2D = 180 / $PI;
my $ERAD = 6378138.12;
my $FGFS_IO;

my $ERR = 0;
my $WARN = 1;
my $INFO = 2;
my $BULK = 3;
my $DEBUG = 4;
my $VERBOSITY = $INFO;

my $help = <<EOF;
Usage: $pgmname
   -h or -?          This brief help.
   -v[vvv...]        Add to verbosity level.
   -q                Quiet (verbosity=0).
   -rt=path          Runtime folder. (Def=$def_fg_rt).
                     NOTE: Will change to this directory to run FG!
   -binary=name      FG binary EXE. (Def=$def_fg_binary).
   -root=path        Set FG root. (Def=$def_fg_root).
   --add-fg-params   Ends internal commands, and this, plus any following, will be passed to FG.
EOF

# command arguments to FG
my @fgfsargs = ( "--fg-root=$FG_ROOT",
    "--aircraft=ufo",
    "--fdm=ufo",
    "--prop:/sim/rendering/fps-display=true",
    "--timeofday=noon",
    "--disable-random-objects",
    "--disable-ai-models",
    "--fog-disable",
    "--disable-real-weather-fetch",
    "--altitude=1000",
    "--lon=-122.33276046",
    "--lat=37.60364931",
    "--heading=297"
    );

sub set_verbosity {
    my (@av) = @_;
    while (@av) {
        my $arg = $av[0];
        if ($arg =~ /^-(v+)$/) {
            $VERBOSITY += length($1);
        } elsif ($arg =~ /^-q$/) {
            $VERBOSITY = 0;
        }
        shift @av;
    }
    mylog( $DEBUG, "Verbosity set to $VERBOSITY\n" );
}

sub chk_arg {
    my ($arg, @av) = @_;
    fatal( "Invalid $arg - needs value ... -? for help ... aborting!\n" ) if !(@av);
}

sub parse_args {
    my (@av) = @_;
    set_verbosity(@av);     # parse only for verbosity
    my $cnt = 0;
    while (@av) {
        my $arg = $av[0];
        my $len = length($arg);
        $cnt++;
        mylog( $DEBUG, "$cnt: $arg($len)\n" );
        last if (($len > 2)&&(substr($arg,0,2) eq '--'));   # assume this is a FG argument
        if (($arg =~ /^-h$/)||
            ($arg =~ /^-\?$/)) {
            print $help;
            if ($VERBOSITY > $WARN) {
                print "Arguments to FG are :\n";
                print join("\n", @fgfsargs);
            }
            print "\n";
            exit(0);
        } elsif ($arg =~ /^-(v+)$/) {
            # done - $VERBOSITY += length($1);
        } elsif ($arg =~ /^-q$/) {
            # done $VERBOSITY = 0;
        } elsif ($arg =~ /^-rt=(.+)$/) {
            $def_fg_rt = $1;
            mylog( $DEBUG, "Runtime folder to [$def_fg_rt].\n" );
        } elsif ($arg =~ /^-rt$/) {
            chk_arg(@av);
            shift @av;
            $arg = $av[0];
            $def_fg_rt = $arg;
            mylog( $DEBUG, "Runtime folder to [$def_fg_rt].\n" );
        } elsif ($arg =~ /^-binary=(.+)$/) {
            $def_fg_binary = $1;
            mylog( $DEBUG, "FG binary EXE to [$def_fg_binary].\n" );
        } elsif ($arg =~ /^-binary$/) {
            chk_arg(@av);
            shift @av;
            $arg = $av[0];
            $def_fg_binary = $arg;
            mylog( $DEBUG, "FG binary EXE to [$def_fg_binary].\n" );
        } elsif ($arg =~ /^-root=(.+)$/) {
            $def_fg_root = $1;
            mylog( $DEBUG, "FG root to [$def_fg_root].\n" );
        } elsif ($arg =~ /^-root$/) {
            chk_arg(@av);
            shift @av;
            $arg = $av[0];
            $def_fg_root = $arg;
            mylog( $DEBUG, "FG root to [$def_fg_root].\n" );
        } else {
            mylog( $ERR, "Unknown argument! [$arg] ... -? for help ... aborting ...\n" );
            exit(-1);
        }
        shift @av;
    }
    if (@av) {
        mylog( $DEBUG, "Adding ".join(" ",@av)." to exec commands\n" );

        push(@fgfsargs, @av);
    }

    ##print "aborting ...\n";
    ##exit(0);
}

sub main() {

    $VERBOSITY = $DEBUG if ($debug_on);

    parse_args(@ARGV);

    if (! chdir($def_fg_rt) ) {
        fatal( "ERROR: Unable to change to $def_fg_rt ...\n" );
    }

    my $dir = cwd();
    mylog( $BULK, "Current work directory = $dir\n" );

	if (my $pid = fork) {
		main_loop();    # main processing loop
        mylog( $DEBUG, "Returned from main_loop();\n" );
	} else {
		defined $pid or fatal("cannot fork: $!");
		#exec("$FGFS --telnet=$PORT --config=$BASEDIR/signs.xml @fgfsargs");
		exec("$FGFS --telnet=$PORT @fgfsargs") or
        mylog( $WARN, "Exec FAILED ...\n" );
	}

    mylog( $DEBUG, "Closing output LOG ...\n" );
    close_log($outfile,0);
    mylog( $DEBUG, "Exit, returning zero ...\n" );
    exit(0);

} main;

sub got_keyboard {
    my ($rc) = shift;
    if (defined (my $char = ReadKey(-1)) ) {
		# input was waiting and it was $char
        $$rc = $char;
        return 1;
	}
    return 0;
}

sub main_loop() {
    my ($x, $y, $z);
    my ($px, $py, $pz);
	my ($lon, $lat, $i, $i2);
	my ($oldlon, $oldlat);
    my ($dist, $dtot);
    my ($alt, $agl);          # alititude
    my ($dmsg, $pmsg);
    my ($char);

    # If it takes a WHILE for FG to start, use greater than 2 minutes (120 seconds)
	$FGFS_IO = fgfs_connect($HOST, $PORT, 120) || die " can't open socket\n";

    ReadMode('cbreak'); # not sure this is required, or what it does exactly

	fgfs_send("data");

	fgfs_get_coord(\$oldlon, \$oldlat) or return 0;

    ($x, $y, $z) = ll2xyz($oldlon, $oldlat);

    mylog( $INFO, "Initial: Lat=$oldlat, Lon=$oldlon, xyz=($x,$y,$z)\n" );

    $px = $x;
    $py = $y;
    $pz = $z;
    $dtot = 0;

	for ($i = 0;; $i++) {
        # to exit, just EXIT FG should work
		sleep $INTERVAL;    # sampling interval
		fgfs_get_coord(\$lon, \$lat) or last;
        $i2 = $i + 1;
        if ((abs($oldlat - $lat) > $MIN_CHANGE)||
            (abs($oldlon - $lon) > $MIN_CHANGE)) {
            fgfs_get_altitude( \$alt );
            fgfs_get_agl( \$agl );
    		($x, $y, $z) = ll2xyz($lon, $lat);
            $dist = sqrt( coord_dist_sq( $px, $py, $pz, $x, $y, $z ) ) * 1000;  # Km??? maybe???
            $dtot += $dist;
            #mylog( $BULK, "$i2: Lat=$lat, Lon=$lon, xyz=($x,$y,$z) d=$dist, t=$dtot\n" );
            $dmsg = sprintf( "d=%0.6f, t=%0.6f", $dist, $dtot );
            $pmsg = sprintf( "lat=%0.8f, lon=%0.8f", $lat, $lon );
            mylog( $INFO, "$i2: $pmsg, $dmsg, alt=".int($alt + 0.5).", agl=".int($agl + 0.5)." ft\n" );
            $oldlat = $lat;
            $oldlon = $lon;
            $px = $x;
            $py = $y;
            $pz = $z;
        }
        if ( got_keyboard(\$char) ) {
            $pmsg = sprintf( "%02X", ord($char) );
            mylog( $WARN, "Got keyboard input hex[$pmsg]...\n" );
            last;
        }
	}

    mylog( $DEBUG, "Sending 'quit' to FG ...\n" );
	#fgfs_send("quit");  # this ONLY closes the interface
	#fgfs_send("\033");  # try an ESC key, did nothing
	fgfs_send("run exit"); # YAHOO! THAT WORKED!!! PHEW!!!
    sleep(5);
    mylog( $DEBUG, "Closing telnet IO ...\n" );
	close $FGFS_IO;
	undef $FGFS_IO;
    ReadMode('normal'); # not sure this is required, or what it does exactly
}

sub fgfs_connect() {
	my $host = shift;
	my $port = shift;
	my $timeout = (shift || 120);
	my $socket;
	STDOUT->autoflush(1);
	print "connect $host, $port, timeout $timeout secs ";
	while ($timeout--) {
		if ($socket = IO::Socket::INET->new(
				Proto => 'tcp',
				PeerAddr => $host,
				PeerPort => $port)) {
			print ".. done.\n";
			$socket->autoflush(1);
			sleep 1;
			return $socket;
		}	
		print ".";
		sleep(1);
	}
	return 0;
}

sub fgfs_send {
	print $FGFS_IO shift, "\015\012";
}

sub fgfs_get_coord($$) {
	my $lon = shift;
	my $lat = shift;
	fgfs_get("/position/longitude-deg", $lon) or exit -2;
	fgfs_get("/position/latitude-deg", $lat) or exit -2;
	return 1;
}

sub fgfs_get_altitude($) {
	my $ref_alt = shift;
	fgfs_get("/position/altitude-ft", $ref_alt) or exit -2;
	return 1;
}

sub fgfs_get_agl($) {
	my $ref_alt = shift;
	fgfs_get("/position/altitude-agl-ft", $ref_alt) or exit -2;
	return 1;
}

sub fgfs_get() {
	fgfs_send("get " . shift);
	eof $FGFS_IO and return 0;
	my $val = shift;
	$$val = <$FGFS_IO>;
	$$val =~ s/\015?\012$//;
	$$val =~ /^-ERR (.*)/ and (mylog($WARN, "$1") and return 0);
	return 1;
}

END {
	if (defined $FGFS_IO) {
        mylog( $WARN, "$pgmname: Ending ...\n\n" );
		fgfs_send("run exit");
		close $FGFS_IO;
        undef $FGFS_IO;
	}
    print "\n";
}

sub ll2xyz($$) {
	my $lon = (shift) * $D2R;
	my $lat = (shift) * $D2R;
	my $cosphi = cos $lat;
	my $di = $cosphi * cos $lon;
	my $dj = $cosphi * sin $lon;
	my $dk = sin $lat;
	return ($di, $dj, $dk);
}


sub xyz2ll($$$) {
	my ($di, $dj, $dk) = @_;
	my $aux = $di * $di + $dj * $dj;
	my $lat = atan2($dk, sqrt $aux) * $R2D;
	my $lon = atan2($dj, $di) * $R2D;
	return ($lon, $lat);
}


sub coord_dist_sq($$$$$$) {
	my ($xa, $ya, $za, $xb, $yb, $zb) = @_;
	my $x = $xb - $xa;
	my $y = $yb - $ya;
	my $z = $zb - $za;
	return $x * $x + $y * $y + $z * $z;
}

sub fatal {
	mylog($ERR, "$pgmname: @_");
	exit -1;
}

sub mylog {
    my ($v) = shift;
	return if $v > $VERBOSITY;
	$v = 4 if $v > 4;
    my $msg = '';
	$msg .= "\033[$COLOR[$v]m" if $USECOLOR;
	$msg .= "@_";
	$msg .= "\033[m" if $USECOLOR;
    prt( "$msg" );
}

# eof - fg_telnet.pl
