#!/usr/bin/perl -w
# NAME: srtm-list.pl
# AIM: Given a lat, lon, get the STRM3 list
use strict;
use warnings;
use File::Basename;  # split path ($name,$dir,$ext) = fileparse($file [, qr/\.[^.]*/] )
use Cwd;
my $perl_dir = 'C:\GTools\perl';
unshift(@INC, $perl_dir);
require 'lib_utils.pl' or die "Unable to load 'lib_utils.pl' Check paths in \@INC...\n";
require 'Bucket2.pm' or die "Unable to load 'Bucket2.pm' Check paths in \@INC...\n";
require 'srtm_data.pl' or die "Unable to load 'srtm_data.pl' Check paths in \@INC...\n";
# log file stuff
our ($LF);
my $pgmname = $0;
if ($pgmname =~ /(\\|\/)/) {
    my @tmpsp = split(/(\\|\/)/,$pgmname);
    $pgmname = $tmpsp[-1];
}
my $outfile = $perl_dir."\\temp.$pgmname.txt";
open_log($outfile);

# user variables
my $VERS = "0.0.1 2011-11-03";
my $load_log = 0;
my $in_file = '';
my $verbosity = 0;
my $debug_on = 1;
my $def_file = 'def_file';
my $out_xml = '';
my $badlatlon = 400;
my $user_lat = $badlatlon;
my $user_lon = $badlatlon;

my $def_lat = -31.697287500;
my $def_lon = 148.636942500;

### program variables
my @warnings = ();
my $cwd = cwd();
my $os = $^O;

our $Region_01;
our $Region_02;
our $Region_03;
our $Region_04;
our $Region_05;
our $Region_06;
our $Region_07;
our $Africa;
our $Australia;
our $Eurasia;
our $Islands;
our $North_America;
our $South_America;

# from - http://dds.cr.usgs.gov/srtm/version2_1/SRTM3/Australia/
# Index of /srtm/version2_1/SRTM3/Australia
my @OZ_SRTM_List = qw( S11E119 S11E120 S11E121 S11E122 S11E123 S11E124 S11E132 S11E133 S11E141 S11E142 
S11E143 S11E147 S11E148 S11E149 S11E150 S11E151 S11E152 S11E153 S11E161 S11E162 S11E165 S11E166 
S11E179 S11W140 S11W151 S11W161 S11W162 S11W166 S12E122 S12E130 S12E131 S12E132 S12E133 S12E134 
S12E135 S12E136 S12E141 S12E142 S12E143 S12E144 S12E151 S12E152 S12E153 S12E154 S12E159 S12E160 
S12E166 S12E169 S12E170 S12W152 S12W166 S12W172 S13E122 S13E123 S13E130 S13E131 S13E132 S13E133 
S13E134 S13E135 S13E136 S13E141 S13E142 S13E143 S13E168 S13E176 S13E177 S14E125 S14E126 S14E127 
S14E129 S14E130 S14E131 S14E132 S14E133 S14E134 S14E135 S14E136 S14E141 S14E142 S14E143 S14E144 
S14E166 S14E167 S14W164 S14W172 S14W173 S14W177 S15E121 S15E123 S15E124 S15E125 S15E126 S15E127 
S15E128 S15E129 S15E130 S15E131 S15E132 S15E133 S15E134 S15E135 S15E136 S15E141 S15E142 S15E143 
S15E144 S15E145 S15E166 S15E167 S15E168 S15W139 S15W142 S15W145 S15W146 S15W147 S15W148 S15W149 
S15W169 S15W170 S15W171 S15W172 S15W178 S15W179 S16E123 S16E124 S16E125 S16E126 S16E127 S16E128 
S16E129 S16E130 S16E131 S16E132 S16E133 S16E134 S16E135 S16E136 S16E137 S16E141 S16E142 S16E143 
S16E144 S16E145 S16E166 S16E167 S16E168 S16W141 S16W143 S16W145 S16W146 S16W147 S16W148 S16W149 
S16W155 S16W174 S16W176 S16W180 S17E122 S17E123 S17E124 S17E125 S17E126 S17E127 S17E128 S17E129 
S17E130 S17E131 S17E132 S17E133 S17E134 S17E135 S17E136 S17E137 S17E138 S17E139 S17E140 S17E141 
S17E142 S17E143 S17E144 S17E145 S17E146 S17E149 S17E150 S17E167 S17E168 S17E177 S17E178 S17E179 
S17W141 S17W142 S17W143 S17W144 S17W145 S17W146 S17W147 S17W150 S17W151 S17W152 S17W153 S17W154 
S17W155 S17W180 S18E118 S18E119 S18E122 S18E123 S18E124 S18E125 S18E126 S18E127 S18E128 S18E129 
S18E130 S18E131 S18E132 S18E133 S18E134 S18E135 S18E136 S18E137 S18E138 S18E139 S18E140 S18E141 
S18E142 S18E143 S18E144 S18E145 S18E146 S18E148 S18E155 S18E168 S18E176 S18E177 S18E178 S18E179 
S18W139 S18W141 S18W142 S18W143 S18W144 S18W145 S18W146 S18W149 S18W150 S18W151 S18W179 S18W180 
S19E121 S19E122 S19E123 S19E124 S19E125 S19E126 S19E127 S19E128 S19E129 S19E130 S19E131 S19E132 
S19E133 S19E134 S19E135 S19E136 S19E137 S19E138 S19E139 S19E140 S19E141 S19E142 S19E143 S19E144 
S19E145 S19E146 S19E162 S19E163 S19E168 S19E169 S19E177 S19E178 S19E179 S19W137 S19W138 S19W139 
S19W140 S19W141 S19W142 S19W143 S19W160 S19W164 S19W170 S19W174 S19W175 S19W179 S19W180 S20E118 
S20E119 S20E120 S20E121 S20E122 S20E123 S20E124 S20E125 S20E126 S20E127 S20E128 S20E129 S20E130 
S20E131 S20E132 S20E133 S20E134 S20E135 S20E136 S20E137 S20E138 S20E139 S20E140 S20E141 S20E142 
S20E143 S20E144 S20E145 S20E146 S20E147 S20E148 S20E158 S20E163 S20E169 S20E170 S20E177 S20E178 
S20E179 S20W139 S20W140 S20W141 S20W142 S20W145 S20W146 S20W158 S20W159 S20W170 S20W175 S20W176 
S20W179 S20W180 S21E115 S21E116 S21E117 S21E118 S21E119 S21E120 S21E121 S21E122 S21E123 S21E124 
S21E125 S21E126 S21E127 S21E128 S21E129 S21E130 S21E131 S21E132 S21E133 S21E134 S21E135 S21E136 
S21E137 S21E138 S21E139 S21E140 S21E141 S21E142 S21E143 S21E144 S21E145 S21E146 S21E147 S21E148 
S21E149 S21E150 S21E154 S21E163 S21E164 S21E165 S21E166 S21E167 S21E169 S21W139 S21W140 S21W144 
S21W158 S21W159 S21W175 S21W176 S21W179 S22E113 S22E114 S22E115 S22E116 S22E117 S22E118 S22E119 
S22E120 S22E121 S22E122 S22E123 S22E124 S22E125 S22E126 S22E127 S22E128 S22E129 S22E130 S22E131 
S22E132 S22E133 S22E134 S22E135 S22E136 S22E137 S22E138 S22E139 S22E140 S22E141 S22E142 S22E143 
S22E144 S22E145 S22E146 S22E147 S22E148 S22E149 S22E150 S22E151 S22E152 S22E153 S22E154 S22E155 
S22E158 S22E164 S22E165 S22E166 S22E167 S22E168 S22W136 S22W137 S22W139 S22W140 S22W141 S22W155 
S22W158 S22W160 S22W175 S22W176 S22W179 S23E113 S23E114 S23E115 S23E116 S23E117 S23E118 S23E119 
S23E120 S23E121 S23E122 S23E123 S23E124 S23E125 S23E126 S23E127 S23E128 S23E129 S23E130 S23E131 
S23E132 S23E133 S23E134 S23E135 S23E136 S23E137 S23E138 S23E139 S23E140 S23E141 S23E142 S23E143 
S23E144 S23E145 S23E146 S23E147 S23E148 S23E149 S23E150 S23E152 S23E155 S23E165 S23E166 S23E167 
S23E168 S23E171 S23E172 S23W135 S23W137 S23W139 S23W152 S23W153 S23W177 S24E113 S24E114 S24E115 
S24E116 S24E117 S24E118 S24E119 S24E120 S24E121 S24E122 S24E123 S24E124 S24E125 S24E126 S24E127 
S24E128 S24E129 S24E130 S24E131 S24E132 S24E133 S24E134 S24E135 S24E136 S24E137 S24E138 S24E139 
S24E140 S24E141 S24E142 S24E143 S24E144 S24E145 S24E146 S24E147 S24E148 S24E149 S24E150 S24E151 
S24E152 S24E155 S24W131 S24W135 S24W136 S24W138 S24W148 S24W150 S25E113 S25E114 S25E115 S25E116 
S25E117 S25E118 S25E119 S25E120 S25E121 S25E122 S25E123 S25E124 S25E125 S25E126 S25E127 S25E128 
S25E129 S25E130 S25E131 S25E132 S25E133 S25E134 S25E135 S25E136 S25E137 S25E138 S25E139 S25E140 
S25E141 S25E142 S25E143 S25E144 S25E145 S25E146 S25E147 S25E148 S25E149 S25E150 S25E151 S25E152 
S25E153 S25W125 S25W129 S26E112 S26E113 S26E114 S26E115 S26E116 S26E117 S26E118 S26E119 S26E120 
S26E121 S26E122 S26E123 S26E124 S26E125 S26E126 S26E127 S26E128 S26E129 S26E130 S26E131 S26E132 
S26E133 S26E134 S26E135 S26E136 S26E137 S26E138 S26E139 S26E140 S26E141 S26E142 S26E143 S26E144 
S26E145 S26E146 S26E147 S26E148 S26E149 S26E150 S26E151 S26E152 S26E153 S26W131 S27E113 S27E114 
S27E115 S27E116 S27E117 S27E118 S27E119 S27E120 S27E121 S27E122 S27E123 S27E124 S27E125 S27E126 
S27E127 S27E128 S27E129 S27E130 S27E131 S27E132 S27E133 S27E134 S27E135 S27E136 S27E137 S27E138 
S27E139 S27E140 S27E141 S27E142 S27E143 S27E144 S27E145 S27E146 S27E147 S27E148 S27E149 S27E150 
S27E151 S27E152 S27E153 S27W106 S28E113 S28E114 S28E115 S28E116 S28E117 S28E118 S28E119 S28E120 
S28E121 S28E122 S28E123 S28E124 S28E125 S28E126 S28E127 S28E128 S28E129 S28E130 S28E131 S28E132 
S28E133 S28E134 S28E135 S28E136 S28E137 S28E138 S28E139 S28E140 S28E141 S28E142 S28E143 S28E144 
S28E145 S28E146 S28E147 S28E148 S28E149 S28E150 S28E151 S28E152 S28E153 S28W110 S28W144 S28W145 
S29E113 S29E114 S29E115 S29E116 S29E117 S29E118 S29E119 S29E120 S29E121 S29E122 S29E123 S29E124 
S29E125 S29E126 S29E127 S29E128 S29E129 S29E130 S29E131 S29E132 S29E133 S29E134 S29E135 S29E136 
S29E137 S29E138 S29E139 S29E140 S29E141 S29E142 S29E143 S29E144 S29E145 S29E146 S29E147 S29E148 
S29E149 S29E150 S29E151 S29E152 S29E153 S30E114 S30E115 S30E116 S30E117 S30E118 S30E119 S30E120 
S30E121 S30E122 S30E123 S30E124 S30E125 S30E126 S30E127 S30E128 S30E129 S30E130 S30E131 S30E132 
S30E133 S30E134 S30E135 S30E136 S30E137 S30E138 S30E139 S30E140 S30E141 S30E142 S30E143 S30E144 
S30E145 S30E146 S30E147 S30E148 S30E149 S30E150 S30E151 S30E152 S30E153 S31E114 S31E115 S31E116 
S31E117 S31E118 S31E119 S31E120 S31E121 S31E122 S31E123 S31E124 S31E125 S31E126 S31E127 S31E128 
S31E129 S31E130 S31E131 S31E132 S31E133 S31E134 S31E135 S31E136 S31E137 S31E138 S31E139 S31E140 
S31E141 S31E142 S31E143 S31E144 S31E145 S31E146 S31E147 S31E148 S31E149 S31E150 S31E151 S31E152 
S31E153 S32E115 S32E116 S32E117 S32E118 S32E119 S32E120 S32E121 S32E122 S32E123 S32E124 S32E125 
S32E126 S32E127 S32E128 S32E129 S32E130 S32E131 S32E132 S32E133 S32E134 S32E135 S32E136 S32E137 
S32E138 S32E139 S32E140 S32E141 S32E142 S32E143 S32E144 S32E145 S32E146 S32E147 S32E148 S32E149 
S32E150 S32E151 S32E152 S32E153 S32E159 S33E115 S33E116 S33E117 S33E118 S33E119 S33E120 S33E121 
S33E122 S33E123 S33E124 S33E125 S33E126 S33E127 S33E128 S33E132 S33E133 S33E134 S33E135 S33E136 
S33E137 S33E138 S33E139 S33E140 S33E141 S33E142 S33E143 S33E144 S33E145 S33E146 S33E147 S33E148 
S33E149 S33E150 S33E151 S33E152 S34E114 S34E115 S34E116 S34E117 S34E118 S34E119 S34E120 S34E121 
S34E122 S34E123 S34E124 S34E134 S34E135 S34E136 S34E137 S34E138 S34E139 S34E140 S34E141 S34E142 
S34E143 S34E144 S34E145 S34E146 S34E147 S34E148 S34E149 S34E150 S34E151 S35E114 S35E115 S35E116 
S35E117 S35E118 S35E119 S35E120 S35E121 S35E122 S35E123 S35E134 S35E135 S35E136 S35E137 S35E138 
S35E139 S35E140 S35E141 S35E142 S35E143 S35E144 S35E145 S35E146 S35E147 S35E148 S35E149 S35E150 
S35E151 S36E116 S36E117 S36E118 S36E135 S36E136 S36E137 S36E138 S36E139 S36E140 S36E141 S36E142 
S36E143 S36E144 S36E145 S36E146 S36E147 S36E148 S36E149 S36E150 S37E136 S37E137 S37E139 S37E140 
S37E141 S37E142 S37E143 S37E144 S37E145 S37E146 S37E147 S37E148 S37E149 S37E150 S38E139 S38E140 
S38E141 S38E142 S38E143 S38E144 S38E145 S38E146 S38E147 S38E148 S38E149 S38E150 S39E140 S39E141 
S39E142 S39E143 S39E144 S39E145 S39E146 S39E147 S40E143 S40E144 S40E146 S40E147 S40E148 S41E143 
S41E144 S41E145 S41E146 S41E147 S41E148 S42E144 S42E145 S42E146 S42E147 S42E148 S43E145 S43E146 
S43E147 S43E148 S44E145 S44E146 S44E147 S44E148 
);

sub VERB1() { return $verbosity >= 1; }
sub VERB2() { return $verbosity >= 2; }
sub VERB5() { return $verbosity >= 5; }
sub VERB9() { return $verbosity >= 9; }

sub show_warnings($) {
    my ($val) = @_;
    if (@warnings) {
        prt( "\nGot ".scalar @warnings." WARNINGS...\n" );
        foreach my $itm (@warnings) {
           prt("$itm\n");
        }
        prt("\n");
    } else {
        prt( "\nNo warnings issued.\n\n" ) if (VERB9());
    }
}

sub pgm_exit($$) {
    my ($val,$msg) = @_;
    if (length($msg)) {
        $msg .= "\n" if (!($msg =~ /\n$/));
        prt($msg);
    }
    show_warnings($val);
    close_log($outfile,$load_log);
    exit($val);
}


sub prtw($) {
   my ($tx) = shift;
   $tx =~ s/\n$//;
   prt("$tx\n");
   push(@warnings,$tx);
}

sub in_world_range($$) {
    my ($lat,$lon) = @_;
    return 0 if (($lat < -90)||($lat > 90)||($lon < -180)||($lon > 180));
    return 1;
}

sub process_latlon($$) {
    my ($lat,$lon) = @_;
    my $b = Bucket2->new();
    $b->set_bucket($lon,$lat);
    my $path = $b->gen_base_path();
    my @arr = split(/\//,$path);
    my $chunk = $arr[0];
    my $tile = $arr[1];
    my $index = $b->gen_index();
    my $clat = $b->get_center_lat();
    my $clon = $b->get_center_lon();
    my $width = $b->get_width();
    my $height = $b->get_height();
    prt("Bucket path [$path/$index] cent $clat,$clon, wid $width, hgt $height chunk $chunk\n");
    # chunk e140s40
    my ($ns,$ew,$chlat,$chlon,$mxlat,$mxlon);
    if ($chunk =~ /^(e|w){1}(\d{3})(n|s){1}(\d{2})$/) {
        $ew = $1;
        $chlon = $2;
        $ns = $3;
        $chlat = $4;
        $chlon *= -1 if ($ew eq 'w');
        $chlat *= -1 if ($ns eq 's');
        $mxlat = $chlat + 9;
        $mxlon = $chlon + 9;
        prt("Range $chlat,$chlon to $mxlat,$mxlon\n");
        my ($srtm,$len);
        my @newarr = split(/\s+/,$Australia);
        my $ra = \@newarr;
        my $cnt = scalar @{$ra};
        prt("Got $cnt from \@newarr...\n");
        my $minlat = 400;
        my $minlon = 400;
        my $maxlat = -400;
        my $maxlon = -400;
        $mxlat = $clat + 2;
        $mxlon = $clon + 2;
        $chlat = $clat - 2;
        $chlon = $clon - 2;
        prt("SRTM Range $chlat,$chlon to $mxlat,$mxlon\n");
        foreach $srtm (@{$ra}) {
            $len = length($srtm);
            if (($len == 7)&&($srtm =~ /^(N|S){1}(\d{2})(E|W){1}(\d{3})$/)) {
                $ns = $1;
                $lat = $2;
                $ew = $3;
                $lon = $4;
                $lat *= -1 if ($ns eq 'S');
                $lon *= -1 if ($ew eq 'W');
                $minlat = $lat if ($lat < $minlat);
                $minlon = $lon if ($lon < $minlon);
                $maxlat = $lat if ($lat > $maxlat);
                $maxlon = $lon if ($lon > $maxlon);
                if (($lat >= $chlat)&&($lat <= $mxlat)&&($lon >= $chlon)&&($lon <= $mxlon)) {
                    prt("$srtm ");
                }
            } else {
                prt("WARNING: [$srtm] does not conform!\n");
            }
        }
        prt("\n");
        prt("Bounding box: lat/lon min : $minlat,$minlon max : $maxlat,$maxlon\n");
        prt("\@REM RAW2ASCII LIMITS\n");
        prt("\@set TEMPCTLAT=$clat\n");
        prt("\@set TEMPCTLON=$clon\n");
        prt("\@set TEMPMILON=$chlon\n");
        prt("\@set TEMPMXLON=$mxlon\n");
        prt("\@set TEMPMILAT=$chlat\n");
        prt("\@set TEMPMXLAT=$mxlat\n");
        prt("\@REM Constuct requires a width\n");
        prt("\@set TEMPX=2\n");
        prt("\@set TEMPY=2\n");
        prt("\@set TEMPRANGE=--min-lon=\%TEMPMILON\% --max-lon=\%TEMPMXLON\% --min-lat=\%TEMPMILAT\% --max-lat=\%TEMPMXLAT\%\n");
        prt("\@set TEMPCRNG=--lon=\%TEMPCTLON\% --lat=\%TEMPCTLAT\% --xdist=\%TEMPX\% --ydist=\%TEMPY\%\n");
    } else {
        prt("Bad chunk!!!!!\n");
    }

}


sub process_in_file($) {
    my ($inf) = @_;
    if (! open INF, "<$inf") {
        pgm_exit(1,"ERROR: Unable to open file [$inf]\n"); 
    }
    my @lines = <INF>;
    close INF;
    my $lncnt = scalar @lines;
    prt("Processing $lncnt lines, from [$inf]...\n");
    my ($line,$inc,$lnn);
    $lnn = 0;
    foreach $line (@lines) {
        chomp $line;
        $lnn++;
        if ($line =~ /\s*#\s*include\s+(.+)$/) {
            $inc = $1;
            prt("$lnn: $inc\n");
        }
    }
}

sub show_bounding_box() {
    my ($cnt,$srtm,$ns,$ew,$lat,$lon,$len);
    my $minlat = 400;
    my $minlon = 400;
    my $maxlat = -400;
    my $maxlon = -400;
#    my $ra1 = \@OZ_SRTM_List;
#    $cnt = scalar @{$ra1};
#    prt("Got $cnt from \@OS_SRTM_List...\n");
#    foreach $srtm (@{$ra1}) {
#        $len = length($srtm);
#        if (($len == 7)&&($srtm =~ /^(N|S){1}(\d{2})(E|W){1}(\d{3})$/)) {
#            $ns = $1;
#            $lat = $2;
#            $ew = $3;
#            $lon = $4;
#            $lat *= -1 if ($ns eq 'S');
#            $lon *= -1 if ($ew eq 'W');
#            $minlat = $lat if ($lat < $minlat);
#            $minlon = $lon if ($lon < $minlon);
#            $maxlat = $lat if ($lat > $maxlat);
#            $maxlon = $lon if ($lon > $maxlon);
#        } else {
#            prt("WARNING: [$srtm] does not conform!\n");
#        }
#    }
#    prt("Bounding box: lat/lon min : $minlat,$minlon max : $maxlat,$maxlon\n");
    my @newarr = split(/\s+/,$Australia);
    my $ra = \@newarr;
    $cnt = scalar @{$ra};
    prt("Got $cnt from \@newarr...\n");
    $minlat = 400;
    $minlon = 400;
    $maxlat = -400;
    $maxlon = -400;
    foreach $srtm (@{$ra}) {
        $len = length($srtm);
        if (($len == 7)&&($srtm =~ /^(N|S){1}(\d{2})(E|W){1}(\d{3})$/)) {
            $ns = $1;
            $lat = $2;
            $ew = $3;
            $lon = $4;
            $lat *= -1 if ($ns eq 'S');
            $lon *= -1 if ($ew eq 'W');
            $minlat = $lat if ($lat < $minlat);
            $minlon = $lon if ($lon < $minlon);
            $maxlat = $lat if ($lat > $maxlat);
            $maxlon = $lon if ($lon > $maxlon);
        } else {
            prt("WARNING: [$srtm] does not conform!\n");
        }
    }
    prt("Bounding box: lat/lon min : $minlat,$minlon max : $maxlat,$maxlon\n");
}

#########################################
### MAIN ###
parse_args(@ARGV);
show_bounding_box();
### prt( "$pgmname: in [$cwd]: Hello, World...\n" );
#process_in_file($in_file);
process_latlon($user_lat,$user_lon);
pgm_exit(0,"");
########################################
sub give_help {
    prt("$pgmname: version $VERS\n");
    prt("Usage: $pgmname [options] in-file\n");
    prt("Options:\n");
    prt(" --help  (-h or -?) = This help, and exit 0.\n");
    prt(" --verb[n]     (-v) = Bump [or set] verbosity. def=$verbosity\n");
    prt(" --load        (-l) = Load LOG at end. ($outfile)\n");
    prt(" --out <file>  (-o) = Write output to this file.\n");
}

sub need_arg {
    my ($arg,@av) = @_;
    pgm_exit(1,"ERROR: [$arg] must have a following argument!\n") if (!@av);
}

sub parse_args {
    my (@av) = @_;
    my ($arg,$sarg,$cnt,@arr);
    $cnt = 0;
    while (@av) {
        $arg = $av[0];
        if ($arg =~ /^-/) {
            $sarg = substr($arg,1);
            $sarg = substr($sarg,1) while ($sarg =~ /^-/);
            if (($sarg =~ /^h/i)||($sarg eq '?')) {
                give_help();
                pgm_exit(0,"Help exit(0)");
            } elsif ($sarg =~ /^v/) {
                if ($sarg =~ /^v.*(\d+)$/) {
                    $verbosity = $1;
                } else {
                    while ($sarg =~ /^v/) {
                        $verbosity++;
                        $sarg = substr($sarg,1);
                    }
                }
                prt("Verbosity = $verbosity\n") if (VERB1());
            } elsif ($sarg =~ /^l/) {
                $load_log = 1;
                prt("Set to load log at end.\n") if (VERB1());
            } elsif ($sarg =~ /^o/) {
                need_arg(@av);
                shift @av;
                $sarg = $av[0];
                $out_xml = $sarg;
                prt("Set out file to [$out_xml].\n") if (VERB1());
            } else {
                pgm_exit(1,"ERROR: Invalid argument [$arg]! Try -?\n");
            }
        } else {
            if ($arg =~ /,/) {
                @arr = split(/,/,$arg);
                if (scalar @arr == 2) {
                    $user_lat = $arr[0];
                    $user_lon = $arr[1];
                    $cnt = 2;
                    prt("Set input lat [$user_lat], lon [$user_lon]\n");
                } else {
                    pgm_exit(1,"ERROR: Unknown argument! $arg - Did not split to 2 on comma!\n");
                }
            } else {
                if ($cnt == 0) {
                    $user_lat = $arg;
                    prt("Set input lat [$user_lat]\n");
                } elsif ($cnt == 1) {
                    $user_lon = $arg;
                    prt("Set input lon [$user_lon]\n");
                } else {
                    pgm_exit(1,"ERROR: Unknown argument! $arg - Already have lat $user_lat, lon $user_lon!\n");
                }
                $cnt++;
            }
        }
        shift @av;
    }
    if (($user_lat == $badlatlon)&&$debug_on) {
        $user_lat = $def_lat;
        prt("Set default lat [$user_lat]\n");
    }
    if (($user_lon == $badlatlon)&&$debug_on) {
        $user_lon = $def_lon;
        prt("Set default lon [$user_lon]\n");
    }
    if ($user_lat == $badlatlon) {
        pgm_exit(1,"ERROR: No user latitude found in input!\n");
    }
    if ($user_lon == $badlatlon) {
        pgm_exit(1,"ERROR: No user longitude found in input!\n");
    }
    if (!in_world_range($user_lat,$user_lon)) {
        pgm_exit(1,"ERROR: Bad lat $user_lat or lon $user_lon!\n");
    }
}

# eof - template.pl
