#!/usr/bin/perl -w
# NAME: genindex04.pl
# AIM: Complete re-write - Given an input folder, generate a tempsitemap.htm of 
# a complete directory scan
# 06/01/2012 geoff mclane http://geoffair.net/mperl
use strict;
use warnings;
use File::Basename;  # split path ($name,$dir,$ext) = fileparse($file [, qr/\.[^.]*/] )
use Cwd;
my $os = $^O;
my $perl_dir = '/home/geoff/bin';
my $PATH_SEP = '/';
my $temp_dir = '/tmp';
if ($os =~ /win/i) {
    $perl_dir = 'C:\GTools\perl';
    $temp_dir = $perl_dir;
    $PATH_SEP = "\\";
}
unshift(@INC, $perl_dir);
require 'lib_utils.pl' or die "Unable to load 'lib_utils.pl' Check paths in \@INC...\n";
# log file stuff
our ($LF);
my $pgmname = $0;
if ($pgmname =~ /(\\|\/)/) {
    my @tmpsp = split(/(\\|\/)/,$pgmname);
    $pgmname = $tmpsp[-1];
}
my $outfile = $temp_dir.$PATH_SEP."temp.$pgmname.txt";
open_log($outfile);

# user variables
my $VERS = "0.0.1 2012-01-06";
my $load_log = 0;
my $in_dir = '';
my $verbosity = 0;
my $out_xml = 'tempsitemap.htm';
my $xclude_repo_dirs = 1;
my @repo_dirs = qw( CVS .svn .git );

# debug
my $debug_on = 1;
my $def_file = 'C:\GTools\perl';

### program variables
my @warnings = ();
my $cwd = cwd();
# forward
sub scan_directory($$);

sub VERB1() { return $verbosity >= 1; }
sub VERB2() { return $verbosity >= 2; }
sub VERB5() { return $verbosity >= 5; }
sub VERB9() { return $verbosity >= 9; }

sub is_repo_directory($) {
    my $dir = shift;
    my ($test);
    foreach $test (@repo_dirs) {
        return 1 if ($dir eq $test);
    }
    return 0;
}

sub show_warnings($) {
    my ($val) = @_;
    if (@warnings) {
        prt( "\nGot ".scalar @warnings." WARNINGS...\n" );
        foreach my $itm (@warnings) {
           prt("$itm\n");
        }
        prt("\n");
    } else {
        prt( "\nNo warnings issued.\n\n" ) if (VERB9());
    }
}

sub pgm_exit($$) {
    my ($val,$msg) = @_;
    if (length($msg)) {
        $msg .= "\n" if (!($msg =~ /\n$/));
        prt($msg);
    }
    show_warnings($val);
    close_log($outfile,$load_log);
    exit($val);
}


sub prtw($) {
   my ($tx) = shift;
   $tx =~ s/\n$//;
   prt("$tx\n");
   push(@warnings,$tx);
}

sub process_in_file($) {
    my ($inf) = @_;
    if (! open INF, "<$inf") {
        pgm_exit(1,"ERROR: Unable to open file [$inf]\n"); 
    }
    my @lines = <INF>;
    close INF;
    my $lncnt = scalar @lines;
    prt("Processing $lncnt lines, from [$inf]...\n");
    my ($line,$inc,$lnn);
    $lnn = 0;
    foreach $line (@lines) {
        chomp $line;
        $lnn++;
        if ($line =~ /\s*#\s*include\s+(.+)$/) {
            $inc = $1;
            prt("$lnn: $inc\n");
        }
    }
}

sub scan_directory($$) {
    my ($dir,$rlist) = @_;
    if (!opendir(DIR,$dir)) {
        prtw("WARNING: Failed to open directory [$dir]\n");
    }
    my @files = readdir(DIR);
    closedir(DIR);
    my ($item,$ff);
    $dir .= $PATH_SEP if (!($dir =~ /(\\|\/)$/));
    my @dirs = ();
    foreach $item (@files) {
        next if (($item eq '.')||($item eq '..'));
        $ff = $dir.$item;
        if (-f $ff) {
            my ($n,$d,$e) = fileparse($item, qr/\.[^.]*/);
            push(@{$rlist},[$ff,$e]);    # got a FILE
        } elsif (-d $ff) {
            push(@dirs,$ff); # got a directory
        } else {
            prtw("WARNING: item [$ff] skipped!\n");
        }
    }
    foreach $dir (@dirs) {
        scan_directory($dir,$rlist);
    }
}

sub write_html($$) {
    my ($dir,$rlist) = @_;
    my $len = length($dir);
    my ($file,$cnt,$i,$html,$wrap,$cols,$rows,$mrow,$flen,$maxlen,$ind,$href);
    my ($line);
    $cnt = scalar @{$rlist};
    $html = "<html><head><title>Site Map</title></head>\n";
    $html .= "<body><a name=\"top\"></a>\n";
    $html .= "<h1 align=\"center\">Site Map - $cnt Files</h1>\n";
    $html .= "<p align=\"center\"><a href=\"index.htm\">index</a>\n";
    $html .= "<a href=\"#end\">end</a></p>\n";
    $html .= "<table border=\"1\" cellpadding=\"0\" cellspacing=\"0\" summary=\"Site List\">\n";
    $line = '';
    $wrap = 3;
    $mrow = 24;
    $maxlen = 50;
    $rows = 0;
    $cols = 0;
    for ($i = 0; $i < $cnt; $i++) {
        $file = path_d2u(substr(${$rlist}[$i][0],$len));
        $href = $file;
        $flen = length($file);
        $ind = index($file,'/');
        if (($len > $maxlen)&&($ind > 0)) {
            my ($nm,$dr) = fileparse($file);
            $dr = substr($file,0,$ind);
            $file = "$dr...$nm";
        }
        $line = "<td><a href=\"$href\">$file</a></td>\n";
        $html .= "<tr>\n" if ($cols == 0);
        $html .= $line;
        $cols++;
        if ($cols == $wrap) {
            $html .= "</tr>\n";
            $cols = 0;
            $rows++;
            if ($rows == $mrow) {
                $html .= "<tr><td align=\"center\" colspan=\"$wrap\">\n";
                $html .= "<a href=\"#top\">top</a> <a href=\"#end\">end</a>\n";
                $html .= "</td></tr>\n";
                $rows = 0;
            }
        }
    }
    if ($cols) {
        while ($cols < $wrap) {
            $html .= "<td>&nbsp;</td>";
            $cols++;
        }
        $html .= "</tr>\n";
    }
    $html .= "</table>\n";
    # maybe now another table by the extension, or whatevr
    $html .= "<p>Done Site Map of $cnt files</p>\n";
    $html .= "<a name=\"end\"></a>";
    $html .= "</body>\n";
    $html .= "</html>\n";
    write2file($html,$out_xml);
    prt("Site list written to $out_xml\n");
    if ($os =~ /win/i) {
        system($out_xml);
    } else {
        system("firefox $out_xml");
    }
}

sub process_in_directory($) {
    my $dir = shift;
    opendir(DIR,$dir) || pgm_exit(1,"ERROR: Unable to open directory [$dir]!\n");
    my @files = readdir(DIR);
    closedir(DIR);
    my $itemcnt = scalar @files;
    prt("Got $itemcnt items, from directory $dir...\n");
    my ($item,$ff);
    $dir .= $PATH_SEP if (!($dir =~ /(\\|\/)$/));
    my @file_list = ();
    my @dirs = ();
    foreach $item (@files) {
        next if (($item eq '.')||($item eq '..'));
        $ff = $dir.$item;
        if (-f $ff) {
            my ($n,$d,$e) = fileparse($item, qr/\.[^.]*/);
            push(@file_list,[$ff,$e]); # got a FILE
        } elsif (-d $ff) {
            push(@dirs,$ff); # got a directory
        } else {
            prtw("WARNING: item [$ff] skipped!\n");
        }
    }
    foreach $dir (@dirs) {
        scan_directory($dir,\@file_list);
    }
    $itemcnt = scalar @file_list;
    #my @arr = sort mysort @file_list;
    prt("Got TOTAL $itemcnt files, from directory $dir...\n");
    #write_html($dir,\@arr);
    write_html($dir,\@file_list);
}

#########################################
### MAIN ###
parse_args(@ARGV);
process_in_directory($in_dir);
pgm_exit(0,"");
########################################
sub give_help {
    prt("$pgmname: version $VERS\n");
    prt("Usage: $pgmname [options] in-file\n");
    prt("Options:\n");
    prt(" --help  (-h or -?) = This help, and exit 0.\n");
    prt(" --verb[n]     (-v) = Bump [or set] verbosity. def=$verbosity\n");
    prt(" --load        (-l) = Load LOG at end. ($outfile)\n");
    prt(" --out <file>  (-o) = Write output to this file.\n");
}

sub need_arg {
    my ($arg,@av) = @_;
    pgm_exit(1,"ERROR: [$arg] must have a following argument!\n") if (!@av);
}

sub parse_args {
    my (@av) = @_;
    my ($arg,$sarg);
    while (@av) {
        $arg = $av[0];
        if ($arg =~ /^-/) {
            $sarg = substr($arg,1);
            $sarg = substr($sarg,1) while ($sarg =~ /^-/);
            if (($sarg =~ /^h/i)||($sarg eq '?')) {
                give_help();
                pgm_exit(0,"Help exit(0)");
            } elsif ($sarg =~ /^v/) {
                if ($sarg =~ /^v.*(\d+)$/) {
                    $verbosity = $1;
                } else {
                    while ($sarg =~ /^v/) {
                        $verbosity++;
                        $sarg = substr($sarg,1);
                    }
                }
                prt("Verbosity = $verbosity\n") if (VERB1());
            } elsif ($sarg =~ /^l/) {
                $load_log = 1;
                prt("Set to load log at end.\n") if (VERB1());
            } elsif ($sarg =~ /^o/) {
                need_arg(@av);
                shift @av;
                $sarg = $av[0];
                $out_xml = $sarg;
                prt("Set out file to [$out_xml].\n") if (VERB1());
            } else {
                pgm_exit(1,"ERROR: Invalid argument [$arg]! Try -?\n");
            }
        } else {
            $in_dir = $arg;
            prt("Set input to [$in_dir]\n") if (VERB1());
        }
        shift @av;
    }

    if ((length($in_dir) ==  0) && $debug_on) {
        $in_dir = $def_file;
        prt("Set DEFAULT input to [$in_dir]\n");
    }
    if (length($in_dir) ==  0) {
        pgm_exit(1,"ERROR: No input files found in command!\n");
    }
    if (! -d $in_dir) {
        pgm_exit(1,"ERROR: Unable to find in directory [$in_dir]! Check name, location...\n");
    }
}

# eof - genindex04.pl
