#!/usr/bin/perl -w
# NAME: difflist.pl
# AIM: Read a *.diff -u output and pepae a list of files that differ
# 15/08/2013 geoff mclane http://geoffair.net/mperl
use strict;
use warnings;
use File::Basename;  # split path ($name,$dir,$ext) = fileparse($file [, qr/\.[^.]*/] )
use File::Spec; # File::Spec->rel2abs($rel); # we are IN the SLN directory, get ABSOLUTE form
use Cwd;
my $os = $^O;
my $perl_dir = '/home/geoff/bin';
my $PATH_SEP = '/';
my $temp_dir = '/tmp';
if ($os =~ /win/i) {
    $perl_dir = 'C:\GTools\perl';
    $temp_dir = $perl_dir;
    $PATH_SEP = "\\";
}
unshift(@INC, $perl_dir);
require 'lib_utils.pl' or die "Unable to load 'lib_utils.pl' Check paths in \@INC...\n";
# log file stuff
our ($LF);
my $pgmname = $0;
if ($pgmname =~ /(\\|\/)/) {
    my @tmpsp = split(/(\\|\/)/,$pgmname);
    $pgmname = $tmpsp[-1];
}
my $outfile = $temp_dir.$PATH_SEP."temp.$pgmname.txt";
open_log($outfile);

# user variables
my $VERS = "0.0.1 2013-03-17";
my $load_log = 0;
my $in_file = '';
my $verbosity = 0;
my $out_file = $temp_dir.$PATH_SEP."templist.txt";

# ### DEBUG ###
my $debug_on = 1;
my $def_file = 'C:\FG\18\temp2.diff';

### program variables
my @warnings = ();
my $cwd = cwd();

sub VERB1() { return $verbosity >= 1; }
sub VERB2() { return $verbosity >= 2; }
sub VERB5() { return $verbosity >= 5; }
sub VERB9() { return $verbosity >= 9; }

sub show_warnings($) {
    my ($val) = @_;
    if (@warnings) {
        prt( "\nGot ".scalar @warnings." WARNINGS...\n" );
        foreach my $itm (@warnings) {
           prt("$itm\n");
        }
        prt("\n");
    } else {
        prt( "\nNo warnings issued.\n\n" ) if (VERB9());
    }
}

sub pgm_exit($$) {
    my ($val,$msg) = @_;
    if (length($msg)) {
        $msg .= "\n" if (!($msg =~ /\n$/));
        prt($msg);
    }
    show_warnings($val);
    close_log($outfile,$load_log);
    exit($val);
}


sub prtw($) {
   my ($tx) = shift;
   $tx =~ s/\n$//;
   prt("$tx\n");
   push(@warnings,$tx);
}

# process a diff -u file, extracting a list of file
# also list 'Only in fgcom\iaxclient\simpleclient\iaxcomm: iaxcomm.exe.manifest'
sub process_in_file($) {
    my ($inf) = @_;
    if (! open INF, "<$inf") {
        pgm_exit(1,"ERROR: Unable to open file [$inf]\n"); 
    }
    my @lines = <INF>;
    close INF;
    my $lncnt = scalar @lines;
    prt("Processing $lncnt lines, from [$inf]...\n");
    my ($line,$diff,$lnn,@arr,$cnt,$i,$fil1,$fil2,$ok,$ff1,$ff2,$ok1,$ok2,$msg);
    $lnn = 0;
    my $absfile = File::Spec->rel2abs($inf); # we are IN the SLN directory, get ABSOLUTE form
    my ($name,$dir) = fileparse($absfile);
    ut_fix_directory(\$dir);
    my @file_list = ();
    my @missed_list = ();
    $msg = '';
    foreach $line (@lines) {
        chomp $line;
        $lnn++;
        @arr = space_split($line);
        $cnt = scalar @arr;
        $diff = $arr[0];
        if ($diff && ($diff eq 'diff')) {
            $fil1 = '';
            $fil2 = '';
            $ok = 0;
            for ($i = 1; $i < $cnt; $i++) {
                $diff = $arr[$i];
                next if ($diff =~ /^-/);
                if (length($fil1) == 0) {
                    $fil1 = $diff;
                    $ok++;
                } elsif (length($fil2) == 0) {
                    $fil2 = $diff;
                    $ok++;
                } else {
                    prtw("WARNING: $lnn: Bad diff line [$line]\n");
                    $ok++;
                    last;
                }
            }
            if ($ok == 2) {
                $ff1 = $dir.$fil1;
                $ff2 = $dir.$fil2;
                $ok1 = (-f $ff1) ? "ok" : "NF";
                $ok2 = (-f $ff2) ? "ok" : "NF";
                prt("$lnn: $ff1 $ok1 $ff2 $ok2\n");
                if (($ok1 eq 'ok')&&($ok2 eq 'ok')) {
                    push(@file_list, [$ff1,$ff2]);
                    $msg .= "$ff1 $ff2\n";
                } else {
                    push(@missed_list, [$ff1,$ff2]);
                }
            }
        }
    }
    $cnt = scalar @file_list;
    prt("Got $cnt diff files...\n");
    if (length($out_file) && length($msg)) {
        write2file($msg,$out_file);
        $msg = "Written list to [$out_file]";
        prt("$msg\n");
    }
    my ($len);
    my $min = 0;
    for ($i = 0; $i < $cnt; $i++) {
        $ff1 = $file_list[$i][0];
        $ff2 = $file_list[$i][1];
        $len = length($ff1);
        $min = $len if ($len > $min);
    }
    for ($i = 0; $i < $cnt; $i++) {
        $ff1 = $file_list[$i][0];
        $ff2 = $file_list[$i][1];
        $ff1 .= ' ' while (length($ff1) < $min);
        prt("$ff1 $ff2\n");
    }
    prt("$msg\n") if (length($msg));

}

#########################################
### MAIN ###
parse_args(@ARGV);
process_in_file($in_file);
pgm_exit(0,"");
########################################

sub need_arg {
    my ($arg,@av) = @_;
    pgm_exit(1,"ERROR: [$arg] must have a following argument!\n") if (!@av);
}

sub parse_args {
    my (@av) = @_;
    my ($arg,$sarg);
    while (@av) {
        $arg = $av[0];
        if ($arg =~ /^-/) {
            $sarg = substr($arg,1);
            $sarg = substr($sarg,1) while ($sarg =~ /^-/);
            if (($sarg =~ /^h/i)||($sarg eq '?')) {
                give_help();
                pgm_exit(0,"Help exit(0)");
            } elsif ($sarg =~ /^v/) {
                if ($sarg =~ /^v.*(\d+)$/) {
                    $verbosity = $1;
                } else {
                    while ($sarg =~ /^v/) {
                        $verbosity++;
                        $sarg = substr($sarg,1);
                    }
                }
                prt("Verbosity = $verbosity\n") if (VERB1());
            } elsif ($sarg =~ /^l/) {
                if ($sarg =~ /^ll/) {
                    $load_log = 2;
                } else {
                    $load_log = 1;
                }
                prt("Set to load log at end. ($load_log)\n") if (VERB1());
            } elsif ($sarg =~ /^o/) {
                need_arg(@av);
                shift @av;
                $sarg = $av[0];
                $out_file = $sarg;
                prt("Set out file to [$out_file].\n") if (VERB1());
            } else {
                pgm_exit(1,"ERROR: Invalid argument [$arg]! Try -?\n");
            }
        } else {
            $in_file = $arg;
            prt("Set input to [$in_file]\n") if (VERB1());
        }
        shift @av;
    }

    if ($debug_on) {
        prtw("WARNING: DEBUG is ON!\n");
        if ((length($in_file) ==  0) && $debug_on) {
            $in_file = $def_file;
            prt("Set DEFAULT input to [$in_file]\n");
        }
    }
    if (length($in_file) ==  0) {
        pgm_exit(1,"ERROR: No input files found in command!\n");
    }
    if (! -f $in_file) {
        pgm_exit(1,"ERROR: Unable to find in file [$in_file]! Check name, location...\n");
    }
}

sub give_help {
    prt("$pgmname: version $VERS\n");
    prt("Usage: $pgmname [options] in-file\n");
    prt("Options:\n");
    prt(" --help  (-h or -?) = This help, and exit 0.\n");
    prt(" --verb[n]     (-v) = Bump [or set] verbosity. def=$verbosity\n");
    prt(" --load        (-l) = Load LOG at end. ($outfile)\n");
    prt(" --out <file>  (-o) = Write output to this file.\n");
}

# eof - template.pl
