#!/usr/bin/perl -w
# NAME: fg_pattern.pl
# AIM: Fly a pattern supplied in a file...
use strict;
use warnings;
use File::Basename;  # split path ($name,$dir,$ext) = fileparse($file [, qr/\.[^.]*/] )
use Cwd;
my $os = $^O;
my $perl_dir = '/home/geoff/bin';
my $PATH_SEP = '/';
my $temp_dir = '/tmp';
if ($os =~ /win/i) {
    $perl_dir = 'C:\GTools\perl';
    $temp_dir = $perl_dir;
    $PATH_SEP = "\\";
}
unshift(@INC, $perl_dir);
require 'lib_utils.pl' or die "Unable to load 'lib_utils.pl' Check paths in \@INC...\n";
require 'lib_fgio.pl' or die "Unable to load 'lib_fgio.pl' Check paths in \@INC...\n";
require 'fg_wsg84.pl' or die "Unable to load fg_wsg84.pl ...\n";
require "Bucket.pm" or die "Unable to load Bucket.pm ...\n";
# log file stuff
our ($LF);
my $pgmname = $0;
if ($pgmname =~ /(\\|\/)/) {
    my @tmpsp = split(/(\\|\/)/,$pgmname);
    $pgmname = $tmpsp[-1];
}
my $outfile = $temp_dir.$PATH_SEP."temp.$pgmname.txt";
open_log($outfile);

# user variables
my $VERS = "0.0.1 2013-03-17";
my $load_log = 0;
my $in_file = '';
my $verbosity = 0;
my $out_file = '';

my $HOST = "localhost";
my $PORT = 5556;
my $TIMEOUT = 3;
my $DELAY = 5;

# ### DEBUG ###
my $debug_on = 1;
my $def_file = 'C:\GTools\perl\tempcircuit.txt';

### program variables
my @warnings = ();
my $cwd = cwd();

my %circuits = ();

sub VERB1() { return $verbosity >= 1; }
sub VERB2() { return $verbosity >= 2; }
sub VERB5() { return $verbosity >= 5; }
sub VERB9() { return $verbosity >= 9; }

sub show_warnings($) {
    my ($val) = @_;
    if (@warnings) {
        prt( "\nGot ".scalar @warnings." WARNINGS...\n" );
        foreach my $itm (@warnings) {
           prt("$itm\n");
        }
        prt("\n");
    } else {
        prt( "\nNo warnings issued.\n\n" ) if (VERB9());
    }
}

sub pgm_exit($$) {
    my ($val,$msg) = @_;
    if (length($msg)) {
        $msg .= "\n" if (!($msg =~ /\n$/));
        prt($msg);
    }
    show_warnings($val);
    close_log($outfile,$load_log);
    exit($val);
}


sub prtw($) {
   my ($tx) = shift;
   $tx =~ s/\n$//;
   prt("$tx\n");
   push(@warnings,$tx);
}

sub prtt($) {
    my $txt = shift;
    prt(lu_get_hhmmss_UTC(time()).": $txt");
}

sub hash_ref_valid($) {
    my $rh = shift;
    my ($key,$val,$k2,$v2);
    my $cnt = 0;
    foreach $key (keys %{$rh}) {
        $val = ${$rh}{$key};
        if ($key eq 'direction') {
            prt("Direction: $val\n");
        } else {
            foreach $k2 (keys %{$val}) {
                $v2 = ${$val}{$k2};
                prt("$key: $k2 = $v2\n");
                $cnt++;
            }
        }
    }
    return $cnt;
}

sub show_ref_hash($) {
    my $rh = shift;
    my ($key,$val);
    foreach $key (keys %{$rh}) {
        $val = ${$rh}{$key};
        prt("[$key]\n");
        hash_ref_valid($val);
    }
}

sub process_in_file($) {
    my ($inf) = @_;
    if (! open INF, "<$inf") {
        pgm_exit(1,"ERROR: Unable to open file [$inf]\n"); 
    }
    my @lines = <INF>;
    close INF;
    my $lncnt = scalar @lines;
    prt("Processing $lncnt lines, from [$inf]...\n");
    my ($line,$inc,$lnn,$num,$val,$dir,@arr,$sect,@a,$as,$lat,$lon,$alt);
    $lnn = 0;
    $inc = '';
    $sect = '';
    my %hash = ();
    foreach $line (@lines) {
        chomp $line;
        $lnn++;
        next if ($line =~ /^\#/);
        if ($line =~ /^\[(\w+)\]\s*$/) {
            $inc = $1;
            prt("$lnn: $inc\n");
            if (length($sect)) {
                my %h = %hash;
                if (hash_ref_valid(\%h)) {
                    $dir = 'anticlockwise' if (length($dir) == 0);
                    $h{'direction'} = $dir;
                    $circuits{$sect} = \%h;
                } else {
                    prtw("WARNING: Section $sect skipped!\n");
                }
            }
            $sect = $inc;
            @arr = ();
            $dir = '';
            %hash = ();
        } elsif ($line =~ /^P(\d+)=(.+)$/) {
            $num = $1;
            $val = $2;
            @a = split(",",$val);
            $as = scalar @a;
            if ($as >= 2) {
                $lat = $a[0];
                $lon = $a[1];
                $hash{$num}{'lat'} = $lat;
                $hash{$num}{'lon'} = $lon;
                $alt = 0;
                if ($as > 2) {
                    $alt = $a[2];
                }
                $hash{$num}{'alt'} = $alt;
            } else {
                pgm_exit(1,"Point $num - $val does not have lat,lon minimum!\nLine [$line]\n");
            }
            prt("Point $num - $val\n");
        } elsif ($line =~ /^direction=(\w+)$/) {
            $dir = $1;
            if (($dir eq 'anitclockwise')||($dir = 'clockwise')) {
                prt("Direction = $dir\n");
            } else {
                pgm_exit(1,"Directions can only be 'anitclockwise' (def), or 'clockwise'!\n$lnn: [$line]\n");
            }
        } else {
            prtw("WARNING: $lnn: [$line] NOT PARSED! FIX ME!\n");
        }
    }
    if (length($sect)) {
        my %h = %hash;
        if (hash_ref_valid(\%h)) {
            $dir = 'anticlockwise' if (length($dir) == 0);
            $h{'direction'} = $dir;
            $circuits{$sect} = \%h;
        } else {
            prtw("WARNING: Section $sect skipped!\n");
        }
    }
}

sub show_klocks() {
    show_K_locks();
}

sub show_klocks2($) {
    my $show = shift;
    my $iret = 0;
    my $rk = fgfs_get_K_locks();
    my ($ah,$pa,$pm,$ra,$rm,$hh,$msg);
    $ah = ${$rk}{'ah'};
    $pa = ${$rk}{'pa'};
    $pm = ${$rk}{'pm'};
    $ra = ${$rk}{'ra'};
    $rm = ${$rk}{'rm'};
    $hh = ${$rk}{'hh'};
    $msg = '';
    if ((defined $ah && ($ah eq 'true'))&&
        (defined $pa && ($pa eq 'true'))&&
        (defined $ra && ($ra eq 'true'))&&
        (defined $hh && ($hh eq 'true'))) {
        $msg = "Full ON";
        $iret = 1;
    }
    prt(" ah=$ah, pa=$pa, pm=$pm, ra=$ra, rm=$rm, hh=$hh $msg\n") if ($show);
    return $iret;
}

sub check_keys() {
    my ($char,$pmsg,$val);
    if (got_keyboard(\$char)) {
        $val = ord($char);
        $pmsg = sprintf( "%02X", $val );
        if ($val == 27) {
            prt("ESC key... Exiting...\n");
            return 1;
        } elsif ($char eq '+') {
            $DELAY++;
            prt("Increase delay to $DELAY seconds...\n");
        } elsif ($char eq '-') {
            $DELAY-- if ($DELAY);
            prt("Decrease delay to $DELAY seconds...\n");
        } else {
            prt("Got keyboard input hex[$pmsg]...\n");
        }
    }
    return 0;
}


sub connect_to_fg() {
    my ($ok,$ct,$lt);
    if (fgfs_connect($HOST,$PORT,$TIMEOUT)) {
        prt("Connection established...\n");
       	fgfs_send("data");  # switch exchange to data mode
        $ok = 1;
        $lt = time();
        show_klocks2($ok);
        while ($ok) {
            last if (check_keys());
            $ct = time();
            if ($ct > ($lt + $DELAY)) {
                show_klocks2($ok);
                $lt = $ct;
            }
        }
        fgfs_disconnect();
    } else {
        prt("Connection FAILED!\n");
    }
    pgm_exit(1,"");
}

#########################################
### MAIN ###
connect_to_fg();
parse_args(@ARGV);
process_in_file($in_file);
show_ref_hash(\%circuits);
pgm_exit(0,"");
########################################

sub need_arg {
    my ($arg,@av) = @_;
    pgm_exit(1,"ERROR: [$arg] must have a following argument!\n") if (!@av);
}

sub parse_args {
    my (@av) = @_;
    my ($arg,$sarg);
    while (@av) {
        $arg = $av[0];
        if ($arg =~ /^-/) {
            $sarg = substr($arg,1);
            $sarg = substr($sarg,1) while ($sarg =~ /^-/);
            if (($sarg =~ /^h/i)||($sarg eq '?')) {
                give_help();
                pgm_exit(0,"Help exit(0)");
            } elsif ($sarg =~ /^v/) {
                if ($sarg =~ /^v.*(\d+)$/) {
                    $verbosity = $1;
                } else {
                    while ($sarg =~ /^v/) {
                        $verbosity++;
                        $sarg = substr($sarg,1);
                    }
                }
                prt("Verbosity = $verbosity\n") if (VERB1());
            } elsif ($sarg =~ /^l/) {
                if ($sarg =~ /^ll/) {
                    $load_log = 2;
                } else {
                    $load_log = 1;
                }
                prt("Set to load log at end. ($load_log)\n") if (VERB1());
            } elsif ($sarg =~ /^o/) {
                need_arg(@av);
                shift @av;
                $sarg = $av[0];
                $out_file = $sarg;
                prt("Set out file to [$out_file].\n") if (VERB1());
            } else {
                pgm_exit(1,"ERROR: Invalid argument [$arg]! Try -?\n");
            }
        } else {
            $in_file = $arg;
            prt("Set input to [$in_file]\n") if (VERB1());
        }
        shift @av;
    }

    if ($debug_on) {
        prtw("WARNING: DEBUG is ON!\n");
        if ((length($in_file) ==  0) && $debug_on) {
            $in_file = $def_file;
            prt("Set DEFAULT input to [$in_file]\n");
        }
    }
    if (length($in_file) ==  0) {
        pgm_exit(1,"ERROR: No input files found in command!\n");
    }
    if (! -f $in_file) {
        pgm_exit(1,"ERROR: Unable to find in file [$in_file]! Check name, location...\n");
    }
}

sub give_help {
    prt("$pgmname: version $VERS\n");
    prt("Usage: $pgmname [options] in-file\n");
    prt("Options:\n");
    prt(" --help  (-h or -?) = This help, and exit 0.\n");
    prt(" --verb[n]     (-v) = Bump [or set] verbosity. def=$verbosity\n");
    prt(" --load        (-l) = Load LOG at end. ($outfile)\n");
    prt(" --out <file>  (-o) = Write output to this file.\n");
}

# eof - template.pl
