#!/usr/bin/perl -w
# NAME: genhtmlindex.pl
# AIM: Given a folder name, seek ALL .htm[l] docs and generate an index.html to jump to each
# Maybe extract the <title> from each to fill out the index table
# 05/11/2013 geoff mclane http://geoffair.net/mperl

use strict;
use warnings;
use File::Basename;  # split path ($name,$dir,$ext) = fileparse($file [, qr/\.[^.]*/] )
use Cwd;
my $os = $^O;
my $perl_dir = '/home/geoff/bin';
my $PATH_SEP = '/';
my $temp_dir = '/tmp';
if ($os =~ /win/i) {
    $perl_dir = 'C:\GTools\perl';
    $temp_dir = $perl_dir;
    $PATH_SEP = "\\";
}
unshift(@INC, $perl_dir);
require 'lib_utils.pl' or die "Unable to load 'lib_utils.pl' Check paths in \@INC...\n";
require 'lib_html.pl' or die "Unable to load 'lib_html.pl' Check paths in \@INC...\n";
# log file stuff
our ($LF);
my $pgmname = $0;
if ($pgmname =~ /(\\|\/)/) {
    my @tmpsp = split(/(\\|\/)/,$pgmname);
    $pgmname = $tmpsp[-1];
}
my $outfile = $temp_dir.$PATH_SEP."temp.$pgmname.txt";
open_log($outfile);

# user variables
my $VERS = "0.0.1 2013-03-17";
my $load_log = 0;
my $in_dir = '';
my $verbosity = 0;
my $out_file = '';
my $tmp_out = $temp_dir.$PATH_SEP."tempindex.html";

# ### DEBUG ###
my $debug_on = 1;
my $def_file = 'C:\OSGeo4W\apache\htdocs\openlayers\examples';

### program variables
my @warnings = ();
my $cwd = cwd();

sub VERB1() { return $verbosity >= 1; }
sub VERB2() { return $verbosity >= 2; }
sub VERB5() { return $verbosity >= 5; }
sub VERB9() { return $verbosity >= 9; }

sub show_warnings($) {
    my ($val) = @_;
    if (@warnings) {
        prt( "\nGot ".scalar @warnings." WARNINGS...\n" );
        foreach my $itm (@warnings) {
           prt("$itm\n");
        }
        prt("\n");
    } else {
        prt( "\nNo warnings issued.\n\n" ) if (VERB9());
    }
}

sub pgm_exit($$) {
    my ($val,$msg) = @_;
    if (length($msg)) {
        $msg .= "\n" if (!($msg =~ /\n$/));
        prt($msg);
    }
    show_warnings($val);
    close_log($outfile,$load_log);
    exit($val);
}


sub prtw($) {
   my ($tx) = shift;
   $tx =~ s/\n$//;
   prt("$tx\n");
   push(@warnings,$tx);
}

sub process_in_file($) {
    my ($inf) = @_;
    if (! open INF, "<$inf") {
        pgm_exit(1,"ERROR: Unable to open file [$inf]\n"); 
    }
    my @lines = <INF>;
    close INF;
    my $lncnt = scalar @lines;
    prt("Processing $lncnt lines, from [$inf]...\n");
    my ($line,$inc,$lnn);
    $lnn = 0;
    foreach $line (@lines) {
        chomp $line;
        $lnn++;
        if ($line =~ /\s*#\s*include\s+(.+)$/) {
            $inc = $1;
            prt("$lnn: $inc\n");
        }
    }
}

sub get_html_head() {
    my $txt = <<EOF;
<!DOCTYPE html>
<html>
 <head>
  <title>HTML index</title>
 </head>
 <body>
  <h1 align="center">Index of HTML files</h1>
  <table summary="Index of HTML files">
EOF
    return $txt;
}

sub process_html($) {
    my $ra = shift;
    my ($file,$title,$name,$dir,$min,$len,$i);
    set_show_warnings(0);
    my @arr = ();
    $min = 0;
    foreach $file (@{$ra}) {
        ($name,$dir) = fileparse($file);
        $title = get_html_title_from_file($file);
        $title = $name if (length($title) == 0);
        prt("$name [$title]\n") if (VERB9());
        push(@arr,[$name,$title]);
        $len = length($name);
        $min = $len if ($len > $min);
    }
    $len = scalar @arr;
    prt("Collected $len titles... using file name if none...\n");
    my $html = get_html_head();
    my $col = 0;
    my $wrap = 4;
    for ($i = 0; $i < $len; $i++) {
        $name = $arr[$i][0];
        $title = $arr[$i][1];
        $html .= "   <tr>\n" if ($col == 0);
        $html .= "    <td><a target=\"_blank\" href=\"$name\">$name</a></td>\n";
        $col++;
        $html .= "    <td>$title</td>\n"; 
        $col++;
        if ($col == $wrap) {
            $col = 0;
            $html .= "   </tr>\n";
        }
        # for display only
        $name .= ' ' while (length($name) < $min);
        prt("$name [$title]\n") if (VERB5());
    }
    if ($col) {
        while ($col < $wrap) {
            $col++;
            $html .= "    <td>&nbsp;</td>\n"; 
            $col++;
            $html .= "    <td>&nbsp;</td>\n"; 
        }
        $html .= "   </tr>\n";
    }
    $html .= "   </table>\n";
    $html .= " <!-- generated by $pgmname on ".lu_get_YYYYMMDD_hhmmss(time())."\n";
    $html .= "      from directory $in_dir -->\n";
    $html .= "</html>\n";
    $name = '';
    if (length($out_file) == 0) {
        $out_file = $tmp_out;
        $name = 'since no -o file given...';
    }
    write2file($html,$out_file);
    prt("HTML written to $out_file... $name\n");
}


sub my_type($) {
    my $fil = shift;
    return 1 if ($fil =~ /\.html$/i);
    return 1 if ($fil =~ /\.htm$/i);
    return 0;
}

sub process_in_dir($) {
    my $dir = shift;
    if (!opendir(DIR,$dir)) {
        pgm_exit(1,"ERROR: Unable to open directory $dir!\n");
    }
    my @files = readdir(DIR);
    closedir(DIR);
    my ($file,$ff,$cnt);
    ut_fix_directory(\$dir);
    my @dirs = ();
    my @hfiles = ();
    foreach $file (@files) {
        next if ($file eq '.');
        next if ($file eq '..');
        $ff = $dir.$file;
        if (-d $ff) {
            push(@dirs,$ff);
        } elsif (-f $ff) {
            if (my_type($file)) {
                push(@hfiles,$ff);
            }
        } else {
            prtw("WARNING: WHAT IS THIS [$ff]\n");
        }
    }
    $cnt = scalar @hfiles;
    prt("Found $cnt HTML files in $dir...\n");
    process_html(\@hfiles) if ($cnt);
}

#########################################
### MAIN ###
parse_args(@ARGV);
process_in_dir($in_dir);
pgm_exit(0,"");
########################################

sub need_arg {
    my ($arg,@av) = @_;
    pgm_exit(1,"ERROR: [$arg] must have a following argument!\n") if (!@av);
}

sub parse_args {
    my (@av) = @_;
    my ($arg,$sarg);
    while (@av) {
        $arg = $av[0];
        if ($arg =~ /^-/) {
            $sarg = substr($arg,1);
            $sarg = substr($sarg,1) while ($sarg =~ /^-/);
            if (($sarg =~ /^h/i)||($sarg eq '?')) {
                give_help();
                pgm_exit(0,"Help exit(0)");
            } elsif ($sarg =~ /^v/) {
                if ($sarg =~ /^v.*(\d+)$/) {
                    $verbosity = $1;
                } else {
                    while ($sarg =~ /^v/) {
                        $verbosity++;
                        $sarg = substr($sarg,1);
                    }
                }
                prt("Verbosity = $verbosity\n") if (VERB1());
            } elsif ($sarg =~ /^l/) {
                if ($sarg =~ /^ll/) {
                    $load_log = 2;
                } else {
                    $load_log = 1;
                }
                prt("Set to load log at end. ($load_log)\n") if (VERB1());
            } elsif ($sarg =~ /^o/) {
                need_arg(@av);
                shift @av;
                $sarg = $av[0];
                $out_file = $sarg;
                prt("Set out file to [$out_file].\n") if (VERB1());
            } else {
                pgm_exit(1,"ERROR: Invalid argument [$arg]! Try -?\n");
            }
        } else {
            $in_dir = $arg;
            prt("Set input to [$in_dir]\n") if (VERB1());
        }
        shift @av;
    }

    if ($debug_on) {
        prtw("WARNING: DEBUG is ON!\n");
        if (length($in_dir) ==  0) {
            $in_dir = $def_file;
            prt("Set DEFAULT input to [$in_dir]\n");
        }
    }
    if (length($in_dir) ==  0) {
        pgm_exit(1,"ERROR: No input directory found in command!\n");
    }
    if (! -d $in_dir) {
        pgm_exit(1,"ERROR: Unable to find in directory [$in_dir]! Check name, location...\n");
    }
}

sub give_help {
    prt("$pgmname: version $VERS\n");
    prt("Usage: $pgmname [options] in-dir\n");
    prt("Options:\n");
    prt(" --help  (-h or -?) = This help, and exit 0.\n");
    prt(" --verb[n]     (-v) = Bump [or set] verbosity. def=$verbosity\n");
    prt(" --load        (-l) = Load LOG at end. ($outfile)\n");
    prt(" --out <file>  (-o) = Write output to this file.\n");
    prt(" Given an input directory, search for all htm[l] files, and generate an index.\n");
}

# eof - genhtmlindex.pl
