#!/usr/bin/perl -w
# NAME: msvclog.pl
# AIM: Read a MSVC build log output, and report success and failed project
# 20/07/2013 geoff mclane http://geoffair.net/mperl
use strict;
use warnings;
use File::Basename;  # split path ($name,$dir,$ext) = fileparse($file [, qr/\.[^.]*/] )
use Cwd;
my $os = $^O;
my $perl_dir = '/home/geoff/bin';
my $PATH_SEP = '/';
my $temp_dir = '/tmp';
if ($os =~ /win/i) {
    $perl_dir = 'C:\GTools\perl';
    $temp_dir = $perl_dir;
    $PATH_SEP = "\\";
}
unshift(@INC, $perl_dir);
require 'lib_utils.pl' or die "Unable to load 'lib_utils.pl' Check paths in \@INC...\n";
# log file stuff
our ($LF);
my $pgmname = $0;
if ($pgmname =~ /(\\|\/)/) {
    my @tmpsp = split(/(\\|\/)/,$pgmname);
    $pgmname = $tmpsp[-1];
}
my $outfile = $temp_dir.$PATH_SEP."temp.$pgmname.txt";
open_log($outfile);

# user variables
my $VERS = "0.0.1 2013-07-20";
my $load_log = 0;
my $in_file = '';
my $verbosity = 0;
my $out_file = '';

# ### DEBUG ###
my $debug_on = 0;
my $def_file = 'C:\FG\18\build-sdl2\bldlog-1.txt';

### program variables
my @warnings = ();
my $cwd = cwd();

sub VERB1() { return $verbosity >= 1; }
sub VERB2() { return $verbosity >= 2; }
sub VERB5() { return $verbosity >= 5; }
sub VERB9() { return $verbosity >= 9; }

sub show_warnings($) {
    my ($val) = @_;
    if (@warnings) {
        prt( "\nGot ".scalar @warnings." WARNINGS...\n" );
        foreach my $itm (@warnings) {
           prt("$itm\n");
        }
        prt("\n");
    } else {
        prt( "\nNo warnings issued.\n\n" ) if (VERB9());
    }
}

sub pgm_exit($$) {
    my ($val,$msg) = @_;
    if (length($msg)) {
        $msg .= "\n" if (!($msg =~ /\n$/));
        prt($msg);
    }
    show_warnings($val);
    close_log($outfile,$load_log);
    exit($val);
}


sub prtw($) {
   my ($tx) = shift;
   $tx =~ s/\n$//;
   prt("$tx\n");
   push(@warnings,$tx);
}

# file lines - cmake header

#-- Configuring done
#-- Generating done
#-- Build files have been written to: C:/FG/18/build-sdl2
#
#Microsoft (R) Visual Studio Version 10.0.40219.1.
#Copyright (C) Microsoft Corp. All rights reserved.
#------ Build started: Project: ZERO_CHECK, Configuration: Debug Win32 ------
#Build started 20/07/2013 16:00:33.
#InitializeBuildStatus:
#  Creating "Win32\Debug\ZERO_CHECK\ZERO_CHECK.unsuccessfulbuild" because "AlwaysCreate" was specified.
#FinalizeBuildStatus:
#  Deleting file "Win32\Debug\ZERO_CHECK\ZERO_CHECK.unsuccessfulbuild".
#  Touching "Win32\Debug\ZERO_CHECK\ZERO_CHECK.lastbuildstate".
#
#Build succeeded.
#
#Time Elapsed 00:00:00.06

#Time Elapsed 00:00:00.39
#------ Build started: Project: ALL_BUILD, Configuration: Debug Win32 ------
#Build started 20/07/2013 16:01:02.
#InitializeBuildStatus:
#  Creating "Win32\Debug\ALL_BUILD\ALL_BUILD.unsuccessfulbuild" because "AlwaysCreate" was specified.
#CustomBuild:
#  Build all projects
#FinalizeBuildStatus:
#  Deleting file "Win32\Debug\ALL_BUILD\ALL_BUILD.unsuccessfulbuild".
#  Touching "Win32\Debug\ALL_BUILD\ALL_BUILD.lastbuildstate".
#
#Build succeeded.
#
#Time Elapsed 00:00:00.26
#========== Build: 47 succeeded, 21 failed, 0 up-to-date, 0 skipped ==========


sub process_in_file($) {
    my ($inf) = @_;
    if (! open INF, "<$inf") {
        pgm_exit(1,"ERROR: Unable to open file [$inf]\n"); 
    }
    my @lines = <INF>;
    close INF;
    my $lncnt = scalar @lines;
    prt("Processing $lncnt lines, from [$inf]...\n");
    my ($line,$inc,$lnn,$tline,$len,$dncmake,$proj,$conf,$had_blank);
    my ($hour,$mins,$secs,$tsecs,$time,$csecs,$ok,@arr);
    my ($hadprj,$hadtm,$hadbld,$nproj,$nconf);
    $lnn = 0;
    $dncmake = 0;
    $had_blank = 0;
    $csecs = 0;
    $ok = '';
    @arr = ();
    my %h = ();
    $hadprj = 0;
    $hadtm = 0;
    $hadbld = 0;
    foreach $line (@lines) {
        chomp $line;
        $lnn++;
        $tline = trim_all($line);
        $len = length($tline);
        if ($len == 0) {
            $had_blank = 1;
            next;
        }
        #------ Build started: Project: ZERO_CHECK, Configuration: Debug Win32 ------
        if ($line =~ /------\s+Build\s+started:\s+Project:\s+(.+),\s+Configuration:\s+(.+)\s+------/) {
            $nproj = $1;
            $nconf = $2;
            if ($hadprj && $hadbld && $hadtm) {
                # time to store the RESULTS
                my %ph = ();
                $ph{'name'} = $proj;
                $ph{'conf'} = $conf;
                $ph{'time'} = $time;
                $ph{'res'}  = $ok;
                $ph{'lines'} = [ @arr ];
                $h{$proj.':'.$conf} = \%ph;
            }
            prt("Proj: $nproj Conf: $nconf\n") if (VERB9());
            $proj = $nproj;
            $conf = $nconf;
            $dncmake = 1;
            @arr = ();
            if (($proj eq 'ZERO_CHECK')||($proj eq 'ALL_BUILD')) {
                $hadprj = 0;
            } else {
                $hadprj = 1;
            }
        } elsif ($line =~ /Time\s+Elapsed\s+(\d{2}):(\d{2}):(\d{2})\.(\d{2})/) {
            $hour  = $1;
            $mins  = $2;
            $secs  = $3;
            $tsecs = $4;
            $time = sprintf("Elapsed %02d:%02d:%02d.%02d", $hour, $mins, $secs, $tsecs);
            $csecs += ($hour * 60 * 60);
            $csecs += ($mins * 60);
            $csecs += $secs;
            $csecs += $tsecs / 100;
            # prt("$time Cum $csecs secs\n");
            prt("Proj: $proj Conf: $conf $time Cum $csecs secs $ok\n");
            $hadtm = 1;
        } elsif ($line =~ /==========\s+Build:\s+(\d+)\s+succeeded,\s+(\d+)\s+failed,\s+(\d+)\s+up-to-date,\s+(\d+)\s+skipped\s+==========/) {
            prt("$line\n");
        }
        if ($had_blank) {
            if ($line =~ /^Build\s+(.+)\./ ) {
                $ok = $1;
                $hadbld = 1;
            }
        }
        if ($hadprj) {
           push(@arr,$line);
        }
        $had_blank = 0;
    }
}

#########################################
### MAIN ###
parse_args(@ARGV);
process_in_file($in_file);
pgm_exit(0,"");
########################################

sub need_arg {
    my ($arg,@av) = @_;
    pgm_exit(1,"ERROR: [$arg] must have a following argument!\n") if (!@av);
}

sub parse_args {
    my (@av) = @_;
    my ($arg,$sarg);
    while (@av) {
        $arg = $av[0];
        if ($arg =~ /^-/) {
            $sarg = substr($arg,1);
            $sarg = substr($sarg,1) while ($sarg =~ /^-/);
            if (($sarg =~ /^h/i)||($sarg eq '?')) {
                give_help();
                pgm_exit(0,"Help exit(0)");
            } elsif ($sarg =~ /^v/) {
                if ($sarg =~ /^v.*(\d+)$/) {
                    $verbosity = $1;
                } else {
                    while ($sarg =~ /^v/) {
                        $verbosity++;
                        $sarg = substr($sarg,1);
                    }
                }
                prt("Verbosity = $verbosity\n") if (VERB1());
            } elsif ($sarg =~ /^l/) {
                if ($sarg =~ /^ll/) {
                    $load_log = 2;
                } else {
                    $load_log = 1;
                }
                prt("Set to load log at end. ($load_log)\n") if (VERB1());
            } elsif ($sarg =~ /^o/) {
                need_arg(@av);
                shift @av;
                $sarg = $av[0];
                $out_file = $sarg;
                prt("Set out file to [$out_file].\n") if (VERB1());
            } else {
                pgm_exit(1,"ERROR: Invalid argument [$arg]! Try -?\n");
            }
        } else {
            $in_file = $arg;
            prt("Set input to [$in_file]\n") if (VERB1());
        }
        shift @av;
    }

    if ($debug_on) {
        prtw("WARNING: DEBUG is ON!\n");
        if (length($in_file) ==  0) {
            $in_file = $def_file;
            prt("Set DEFAULT input to [$in_file]\n");
        }
    }
    if (length($in_file) ==  0) {
        pgm_exit(1,"ERROR: No input files found in command!\n");
    }
    if (! -f $in_file) {
        pgm_exit(1,"ERROR: Unable to find in file [$in_file]! Check name, location...\n");
    }
}

sub give_help {
    prt("$pgmname: version $VERS\n");
    prt("Usage: $pgmname [options] in-file\n");
    prt("Options:\n");
    prt(" --help  (-h or -?) = This help, and exit 0.\n");
    prt(" --verb[n]     (-v) = Bump [or set] verbosity. def=$verbosity\n");
    prt(" --load        (-l) = Load LOG at end. ($outfile)\n");
    prt(" --out <file>  (-o) = Write output to this file.\n");
}

# eof - template.pl
