#!/usr/bin/perl -w
# NAME: navcsv.pl
# AIM: Parse apt1000-nav.csv and show items within a bounding box
# 06/11/2013 geoff mclane http://geoffair.net/mperl
use strict;
use warnings;
use File::Basename;  # split path ($name,$dir,$ext) = fileparse($file [, qr/\.[^.]*/] )
use Cwd;
my $os = $^O;
my $perl_dir = '/home/geoff/bin';
my $PATH_SEP = '/';
my $temp_dir = '/tmp';
if ($os =~ /win/i) {
    $perl_dir = 'C:\GTools\perl';
    $temp_dir = $perl_dir;
    $PATH_SEP = "\\";
}
unshift(@INC, $perl_dir);
require 'lib_utils.pl' or die "Unable to load 'lib_utils.pl' Check paths in \@INC...\n";
# log file stuff
our ($LF);
my $pgmname = $0;
if ($pgmname =~ /(\\|\/)/) {
    my @tmpsp = split(/(\\|\/)/,$pgmname);
    $pgmname = $tmpsp[-1];
}
my $outfile = $temp_dir.$PATH_SEP."temp.$pgmname.txt";
open_log($outfile);

# user variables
my $VERS = "0.0.1 2013-03-17";
my $load_log = 0;
my $in_file = '';
my $verbosity = 0;
my $out_file = '';

# ### DEBUG ###
my $debug_on = 1;
my $def_file = 'C:\OSGeo4W\apache\htdocs\test\data\apt1000-nav.csv';

### program variables
my @warnings = ();
my $cwd = cwd();

sub VERB1() { return $verbosity >= 1; }
sub VERB2() { return $verbosity >= 2; }
sub VERB5() { return $verbosity >= 5; }
sub VERB9() { return $verbosity >= 9; }

sub show_warnings($) {
    my ($val) = @_;
    if (@warnings) {
        prt( "\nGot ".scalar @warnings." WARNINGS...\n" );
        foreach my $itm (@warnings) {
           prt("$itm\n");
        }
        prt("\n");
    } else {
        prt( "\nNo warnings issued.\n\n" ) if (VERB9());
    }
}

sub pgm_exit($$) {
    my ($val,$msg) = @_;
    if (length($msg)) {
        $msg .= "\n" if (!($msg =~ /\n$/));
        prt($msg);
    }
    show_warnings($val);
    close_log($outfile,$load_log);
    exit($val);
}


sub prtw($) {
   my ($tx) = shift;
   $tx =~ s/\n$//;
   prt("$tx\n");
   push(@warnings,$tx);
}

# VOR   48.77797200,  002.37872200,   583,   302,    22,        0.0,   TH, VILLACOUBLAY NDB (ap=1 5.71Km on 6.5)
# 0    1           2            3    4    5   6    7  8    9   10
# type,lat,        lon,         feet,freq,rng,bear,id,icao,rwy,"name"
# 2,   48.77158400,002.09744500,583, 286, 22, 0.0, TA,    ,   ,"VILLACOUBLAY NDB"
#// 2 NDB (Non-Directional Beacon) Includes NDB component of Locator Outer Markers (LOM)
#// 3 VOR (including VOR-DME and VORTACs) Includes VORs, VOR-DMEs and VORTACs
#// 4 Localiser component of an ILS (Instrument Landing System)
#// 5 Localiser component of a localiser-only approach Includes for LDAs and SDFs
#// 6 Glideslope component of an ILS Frequency shown is paired frequency, not the DME channel
#// 7 Outer markers (OM) for an ILS Includes outer maker component of LOMs
#// 8 Middle markers (MM) for an ILS
#// 9 Inner markers (IM) for an ILS
#// 12 DME, including the DME component of an ILS, VORTAC or VOR-DME Frequency display suppressed on X-Planes charts
#// 13 Stand-alone DME, or the DME component of an NDB-DME Frequency will displayed on X-Planes charts

my $lat1 = 48.77870225311;
my $lon1 = 2.04085815625;
my $lat2 = 48.675204494487;
my $lon2 = 2.70003784375;
my $show_vors = 1;

sub process_in_file($) {
    my ($inf) = @_;
    if (! open INF, "<$inf") {
        pgm_exit(1,"ERROR: Unable to open file [$inf]\n"); 
    }
    my @lines = <INF>;
    close INF;
    my $lncnt = scalar @lines;
    prt("Processing $lncnt lines, from [$inf]...\n");
    my ($line,$inc,$lnn,$nlat,$nlon);
    $lnn = 0;
    my @found = ();
    foreach $line (@lines) {
        chomp $line;
        $lnn++;
        my @arr = split(",",$line);
        if ($lnn > 1) {
            $nlat = $arr[1];
            $nlon = $arr[2];
            if (($nlat <= $lat1)&&($nlat >= $lat2)&&
                ($nlon >= $lon1)&&($nlon <= $lon2)) {
                push(@found,\@arr);
            }
        }
    }
    $inc = scalar @found;
    prt("Found $inc navaids with box $lat1,$lon1,$lat2,$lon2\n");
    if ($show_vors) {
        my ($typ,$name,$freq,$i,$ra,$len,$j,$tmp); 
        # somehow to decide if VOR, VOR-DME, or VORTAC
        for ($i = 0; $i < $inc; $i++) {
            $ra = $found[$i];
            $typ = ${$ra}[0];
            $freq = ${$ra}[4];
            $name = ${$ra}[10];
            $len = scalar @{$ra};
            for ($j = 11; $j < $len; $j++) {
                $name .= ' ';
                $name .= ${$ra}[$j];
            }
            if ($typ == 3) {
                if ($name =~ /VORTAC/) {
                    $tmp = 'VORTAC ';
                } elsif ($name =~ /DME/) {
                    $tmp = 'VOR-DME'
                } else {
                    $tmp = 'VOR    ';
                }
                prt("$tmp $freq $name\n");
            }
        }

    }
}

#########################################
### MAIN ###
parse_args(@ARGV);
process_in_file($in_file);
pgm_exit(0,"");
########################################

sub need_arg {
    my ($arg,@av) = @_;
    pgm_exit(1,"ERROR: [$arg] must have a following argument!\n") if (!@av);
}

sub parse_args {
    my (@av) = @_;
    my ($arg,$sarg);
    while (@av) {
        $arg = $av[0];
        if ($arg =~ /^-/) {
            $sarg = substr($arg,1);
            $sarg = substr($sarg,1) while ($sarg =~ /^-/);
            if (($sarg =~ /^h/i)||($sarg eq '?')) {
                give_help();
                pgm_exit(0,"Help exit(0)");
            } elsif ($sarg =~ /^v/) {
                if ($sarg =~ /^v.*(\d+)$/) {
                    $verbosity = $1;
                } else {
                    while ($sarg =~ /^v/) {
                        $verbosity++;
                        $sarg = substr($sarg,1);
                    }
                }
                prt("Verbosity = $verbosity\n") if (VERB1());
            } elsif ($sarg =~ /^l/) {
                if ($sarg =~ /^ll/) {
                    $load_log = 2;
                } else {
                    $load_log = 1;
                }
                prt("Set to load log at end. ($load_log)\n") if (VERB1());
            } elsif ($sarg =~ /^o/) {
                need_arg(@av);
                shift @av;
                $sarg = $av[0];
                $out_file = $sarg;
                prt("Set out file to [$out_file].\n") if (VERB1());
            } else {
                pgm_exit(1,"ERROR: Invalid argument [$arg]! Try -?\n");
            }
        } else {
            $in_file = $arg;
            prt("Set input to [$in_file]\n") if (VERB1());
        }
        shift @av;
    }

    if ($debug_on) {
        prtw("WARNING: DEBUG is ON!\n");
        if (length($in_file) ==  0) {
            $in_file = $def_file;
            prt("Set DEFAULT input to [$in_file]\n");
        }
    }
    if (length($in_file) ==  0) {
        pgm_exit(1,"ERROR: No input files found in command!\n");
    }
    if (! -f $in_file) {
        pgm_exit(1,"ERROR: Unable to find in file [$in_file]! Check name, location...\n");
    }
}

sub give_help {
    prt("$pgmname: version $VERS\n");
    prt("Usage: $pgmname [options] in-file\n");
    prt("Options:\n");
    prt(" --help  (-h or -?) = This help, and exit 0.\n");
    prt(" --verb[n]     (-v) = Bump [or set] verbosity. def=$verbosity\n");
    prt(" --load        (-l) = Load LOG at end. ($outfile)\n");
    prt(" --out <file>  (-o) = Write output to this file.\n");
}

# eof - template.pl
