#!/usr/bin/perl -w
# NAME: 3js2ini.pl
# AIM: Given a folder, or a single js file, uses showthreejs to generate an 
# ini file with the dimension...
use strict;
use warnings;
use File::Basename;  # split path ($name,$dir,$ext) = fileparse($file [, qr/\.[^.]*/] )
use Cwd;
my $os = $^O;
my $perl_dir = '/home/geoff/bin';
my $PATH_SEP = '/';
my $temp_dir = '/tmp';
if ($os =~ /win/i) {
    $perl_dir = 'C:\GTools\perl';
    $temp_dir = $perl_dir;
    $PATH_SEP = "\\";
}
unshift(@INC, $perl_dir);
require 'lib_utils.pl' or die "Unable to load 'lib_utils.pl' Check paths in \@INC...\n";
# log file stuff
our ($LF);
my $pgmname = $0;
if ($pgmname =~ /(\\|\/)/) {
    my @tmpsp = split(/(\\|\/)/,$pgmname);
    $pgmname = $tmpsp[-1];
}
my $outfile = $temp_dir.$PATH_SEP."temp.$pgmname.txt";
open_log($outfile);

# user variables
my $VERS = "0.0.5 2015-01-09";
my $load_log = 0;
my $in_file = '';
my $verbosity = 0;
my $out_file = '';
my $recursive = 0;
my @usr_inputs = ();

# ### DEBUG ###
my $debug_on = 0;
my $def_file = 'F:\FGx\fgx-aircraft\data\zzz-operations';
##my $def_file = 'F:\FGx\fgx-aircraft\data\zzz-operations\bus-ba.js';

### program variables
my @warnings = ();
my $cwd = cwd();

sub VERB1() { return $verbosity >= 1; }
sub VERB2() { return $verbosity >= 2; }
sub VERB5() { return $verbosity >= 5; }
sub VERB9() { return $verbosity >= 9; }

sub show_warnings($) {
    my ($val) = @_;
    if (@warnings) {
        prt( "\nGot ".scalar @warnings." WARNINGS...\n" );
        foreach my $itm (@warnings) {
           prt("$itm\n");
        }
        prt("\n");
    } else {
        prt( "\nNo warnings issued.\n\n" ) if (VERB9());
    }
}

sub pgm_exit($$) {
    my ($val,$msg) = @_;
    if (length($msg)) {
        $msg .= "\n" if (!($msg =~ /\n$/));
        prt($msg);
    }
    show_warnings($val);
    close_log($outfile,$load_log);
    exit($val);
}


sub prtw($) {
   my ($tx) = shift;
   $tx =~ s/\n$//;
   prt("$tx\n");
   push(@warnings,$tx);
}

sub process_in_file($) {
    my ($json) = @_;
    if (! -f $json) {
        pgm_exit(1,"ERROR: Unable to 'stat' file [$json]\n"); 
    }
    my $ofile = $temp_dir.$PATH_SEP."tempxmlset.txt";
    unlink $ofile if (-f $ofile);
    my ($specs,@lines,$spec);
    my ($name,$dir,$ext) = fileparse($json, qr/\.[^.]*/);
    $spec = $name.$ext;
    my $sfile = $dir.$name.".ini";
    $specs = '';
    system("showthreejs $json -o $ofile");
    if (-f $ofile) {
        if (open INF, "<$ofile") {
            @lines = <INF>;
            close INF;
            $specs .= "# dimensions from '$spec' ".lu_get_YYYYMMDD_hhmmss_UTC(time())." UTC\n";
            $specs .= join("",@lines)."\n";
            $specs .= "\n";
            rename_2_old_bak($sfile);
            ##prt($specs);
            write2file($specs,$sfile);
            prt("Specs written to $sfile\n");
        } else {
            prtw("WARNING: showthreejs $json open $ofile FAILED!\n");
        }
    } else {
        prtw("WARNING: showthreejs $json no out $ofile\n");
    }
}

sub process_dir($);

sub process_dir($) {
    my $dir = shift;
    if (! opendir(DIR,$dir)) {
        pgm_exit(1,"ERROR: Can NOT open $dir\n");
    }
    my @files = readdir(DIR);
    closedir(DIR);
    my ($file,$ff,$cnt,$json,$ra,$ra2,$ra3,$set);
    my @dirs = ();
    prt("Processing directory $dir...\n") if (VERB5());
    ut_fix_directory(\$dir);
    foreach $file (@files) {
        next if ($file eq '.');
        next if ($file eq '..');
        $ff = $dir.$file;
        if (-f $ff) {
            next if ( !( $file =~ /\.js/ ) );
            process_in_file($ff);
        } elsif (-d $ff) {
            push(@dirs,$ff);
            prt("Added directory $ff...\n") if (VERB9());
        } else {
            prtw("WARNING: What is this $ff! Skipping...\n");
        }
    }
    if ($recursive) {
        foreach $dir (@dirs) {
            process_dir($dir);
        }
    }
}


sub process_inputs() {
    my ($file);
    foreach $file (@usr_inputs) {
        if (-d $file) {
            process_dir($file);
        } elsif (-f $file) {
            process_in_file($file);
        } else {
            prtw("WARNING: $file is not file or director! Skipping\n");
        }
    }
}

#########################################
### MAIN ###
parse_args(@ARGV);
process_inputs();
pgm_exit(0,"");
########################################

sub need_arg {
    my ($arg,@av) = @_;
    pgm_exit(1,"ERROR: [$arg] must have a following argument!\n") if (!@av);
}

sub parse_args {
    my (@av) = @_;
    my ($arg,$sarg);
    my $verb = VERB2();
    while (@av) {
        $arg = $av[0];
        if ($arg =~ /^-/) {
            $sarg = substr($arg,1);
            $sarg = substr($sarg,1) while ($sarg =~ /^-/);
            if (($sarg =~ /^h/i)||($sarg eq '?')) {
                give_help();
                pgm_exit(0,"Help exit(0)");
            } elsif ($sarg =~ /^v/) {
                if ($sarg =~ /^v.*(\d+)$/) {
                    $verbosity = $1;
                } else {
                    while ($sarg =~ /^v/) {
                        $verbosity++;
                        $sarg = substr($sarg,1);
                    }
                }
                $verb = VERB2();
                prt("Verbosity = $verbosity\n") if ($verb);
            } elsif ($sarg =~ /^l/) {
                if ($sarg =~ /^ll/) {
                    $load_log = 2;
                } else {
                    $load_log = 1;
                }
                prt("Set to load log at end. ($load_log)\n") if ($verb);
            } elsif ($sarg =~ /^o/) {
                need_arg(@av);
                shift @av;
                $sarg = $av[0];
                $out_file = $sarg;
                prt("Set out file to [$out_file].\n") if ($verb);
            } elsif ($sarg =~ /^r/) {
                $recursive = 1;
                prt("Set directory reursive.\n") if ($verb);
            } else {
                pgm_exit(1,"ERROR: Invalid argument [$arg]! Try -?\n");
            }
        } else {
            $in_file = $arg;
            prt("Set input to [$in_file]\n") if ($verb);
            if ( !((-f $in_file)||(-d $in_file))) {
                pgm_exit(1,"ERROR: Unable to find in file or directory [$in_file]! Check name, location...\n");
            }
            push(@usr_inputs,$in_file);
        }
        shift @av;
    }

    if ($debug_on) {
        prtw("WARNING: DEBUG is ON!\n");
        if (length($in_file) ==  0) {
            $in_file = $def_file;
            push(@usr_inputs,$in_file);
            prt("Set DEFAULT input to [$in_file]\n");
            $recursive = 1;
            $load_log = 1;
        }
    }
    if (length($in_file) ==  0) {
        pgm_exit(1,"ERROR: No input files found in command!\n");
    }
    if ( !((-f $in_file)||(-d $in_file))) {
        pgm_exit(1,"ERROR: Unable to find in file or directory [$in_file]! Check name, location...\n");
    }
}

sub give_help {
    prt("$pgmname: version $VERS\n");
    prt("Usage: $pgmname [options] in-file\n");
    prt("Options:\n");
    prt(" --help  (-h or -?) = This help, and exit 0.\n");
    prt(" --verb[n]     (-v) = Bump [or set] verbosity. def=$verbosity\n");
    prt(" --load        (-l) = Load LOG at end. ($outfile)\n");
    prt(" --resursive   (-r) = Process directories recursively. ($recursive)\n");
    ###prt(" --out <file>  (-o) = Write output to this file.\n");
    prt(" Given a three.js file, or a directory, process the *.js files,\n");
    prt(" and write an ini file of the same name.\n");
}

# eof - template.pl
