#!/usr/bin/perl -w
# NAME: astro.pl
# AIM: SPECIFIC - Load xplane earth_astro.dat and output to csv file
use strict;
use warnings;
use File::Basename;  # split path ($name,$dir,$ext) = fileparse($file [, qr/\.[^.]*/] )
use Cwd;
my $os = $^O;
my $perl_dir = '/home/geoff/bin';
my $PATH_SEP = '/';
my $temp_dir = '/tmp';
if ($os =~ /win/i) {
    $perl_dir = 'C:\GTools\perl';
    $temp_dir = $perl_dir;
    $PATH_SEP = "\\";
}
unshift(@INC, $perl_dir);
require 'lib_utils.pl' or die "Unable to load 'lib_utils.pl' Check paths in \@INC...\n";
# log file stuff
our ($LF);
my $pgmname = $0;
if ($pgmname =~ /(\\|\/)/) {
    my @tmpsp = split(/(\\|\/)/,$pgmname);
    $pgmname = $tmpsp[-1];
}
my $outfile = $temp_dir.$PATH_SEP."temp.$pgmname.txt";
open_log($outfile);

# user variables
my $VERS = "0.0.1 2013-03-02";
my $load_log = 0;
my $in_file = '';
my $verbosity = 0;
my $out_file = $temp_dir.$PATH_SEP.'temp-astro.csv';
my $out_file2 = $temp_dir.$PATH_SEP.'temp-visible.csv';

# ### DEBUG ###
my $debug_on = 1;
my $def_file = 'D:\FG\xplane\1000\earth_astro.dat';

### program variables
my @warnings = ();
my $cwd = cwd();

sub VERB1() { return $verbosity >= 1; }
sub VERB2() { return $verbosity >= 2; }
sub VERB5() { return $verbosity >= 5; }
sub VERB9() { return $verbosity >= 9; }

sub show_warnings($) {
    my ($val) = @_;
    if (@warnings) {
        prt( "\nGot ".scalar @warnings." WARNINGS...\n" );
        foreach my $itm (@warnings) {
           prt("$itm\n");
        }
        prt("\n");
    } else {
        prt( "\nNo warnings issued.\n\n" ) if (VERB9());
    }
}

sub pgm_exit($$) {
    my ($val,$msg) = @_;
    if (length($msg)) {
        $msg .= "\n" if (!($msg =~ /\n$/));
        prt($msg);
    }
    show_warnings($val);
    close_log($outfile,$load_log);
    exit($val);
}


sub prtw($) {
   my ($tx) = shift;
   $tx =~ s/\n$//;
   prt("$tx\n");
   push(@warnings,$tx);
}

sub process_in_file($) {
    my ($inf) = @_;
    if (! open INF, "<$inf") {
        pgm_exit(1,"ERROR: Unable to open file [$inf]\n"); 
    }
    my @lines = <INF>;
    close INF;
    my $lncnt = scalar @lines;
    prt("Processing $lncnt lines, from [$inf]...\n");
    my ($line,$i,$lnn,$indata,$len,@arr,$acnt,$type);
    my ($rasc,$dec,$lum,$name,$total);
    $lnn = 0;
    $indata = 0;
    my @astro = ();
    my @all = ();
    $total = 0;
    for ($i = 0; $i < $lncnt; $i++) {
        $line = $lines[$i];
        chomp $line;
        $line = trim_all($line);
        $lnn++;
        $len = length($line);
        next if ($len == 0);
        if ($indata) {
            @arr = split(/\s+/,$line);
            $acnt = scalar @arr;
            $type = $arr[0];
            if ($type == 99) {
                prt("$lnn: End of file 99\n");
                last;
            }
            if ($acnt < 3) {
                prt("$lnn: Unknown! [$line]\n");
                next;
            }
            $rasc = $arr[0];
            $dec  = $arr[1];
            $lum  = $arr[2];
            $name = '';
            $name = join(' ',splice(@arr,3)) if ($acnt > 3);
            #$name = ($acnt > 3) ? $arr[3] : '';
            #prt("$rasc,$dec,$lum,$name\n");
            if ($lum <= 6.5 ) {
                push(@astro,[$rasc,$dec,$lum,$name]);
            }
            push(@all,[$rasc,$dec,$lum,$name]);
            $total++;
        } else {
            # wait for version line
            # 740 Version - build 1795
            if ($line =~ /\d+\s+Version\s+-\s+build/) {
                prt(substr($line,0,24)."\n");
                $indata = 1;
            }
        }
    }
    $acnt = scalar @astro;
    prt("Loaded $acnt of $total BRIGHT stars from [$inf]\n");
    my ($ofile,$ra);
    $line = '"Right Ascension",Declination,"Visible Magnitude",name'."\n";
    $ofile = $out_file2;
    for ($i = 0; $i < $acnt; $i++) {
        $ra = $astro[$i];
        $rasc = ${$ra}[0];
        $dec  = ${$ra}[1];
        $lum  = ${$ra}[2];
        $name = ${$ra}[3];
        $line .= "$rasc,$dec,$lum,$name\n";
    }
    write2file($line,$ofile);
    prt("$acnt written to $ofile\n");

    $acnt = scalar @all;
    $line = '"Right Ascension",Declination,"Visible Magnitude",name'."\n";
    $ofile = $out_file;
    for ($i = 0; $i < $acnt; $i++) {
        $ra = $all[$i];
        $rasc = ${$ra}[0];
        $dec  = ${$ra}[1];
        $lum  = ${$ra}[2];
        $name = ${$ra}[3];
        $line .= "$rasc,$dec,$lum,$name\n";
    }
    write2file($line,$ofile);
    prt("$acnt written to $ofile\n");

}

#########################################
### MAIN ###
parse_args(@ARGV);
process_in_file($in_file);
pgm_exit(0,"");
########################################

sub need_arg {
    my ($arg,@av) = @_;
    pgm_exit(1,"ERROR: [$arg] must have a following argument!\n") if (!@av);
}

sub parse_args {
    my (@av) = @_;
    my ($arg,$sarg);
    while (@av) {
        $arg = $av[0];
        if ($arg =~ /^-/) {
            $sarg = substr($arg,1);
            $sarg = substr($sarg,1) while ($sarg =~ /^-/);
            if (($sarg =~ /^h/i)||($sarg eq '?')) {
                give_help();
                pgm_exit(0,"Help exit(0)");
            } elsif ($sarg =~ /^v/) {
                if ($sarg =~ /^v.*(\d+)$/) {
                    $verbosity = $1;
                } else {
                    while ($sarg =~ /^v/) {
                        $verbosity++;
                        $sarg = substr($sarg,1);
                    }
                }
                prt("Verbosity = $verbosity\n") if (VERB1());
            } elsif ($sarg =~ /^l/) {
                if ($sarg =~ /^ll/) {
                    $load_log = 2;
                } else {
                    $load_log = 1;
                }
                prt("Set to load log at end. ($load_log)\n") if (VERB1());
            } elsif ($sarg =~ /^o/) {
                need_arg(@av);
                shift @av;
                $sarg = $av[0];
                $out_file = $sarg;
                prt("Set out file to [$out_file].\n") if (VERB1());
            } else {
                pgm_exit(1,"ERROR: Invalid argument [$arg]! Try -?\n");
            }
        } else {
            $in_file = $arg;
            prt("Set input to [$in_file]\n") if (VERB1());
        }
        shift @av;
    }

    if ((length($in_file) ==  0) && $debug_on) {
        $in_file = $def_file;
        prt("Set DEFAULT input to [$in_file]\n");
    }
    if (length($in_file) ==  0) {
        pgm_exit(1,"ERROR: No input files found in command!\n");
    }
    if (! -f $in_file) {
        pgm_exit(1,"ERROR: Unable to find in file [$in_file]! Check name, location...\n");
    }
}

sub give_help {
    prt("$pgmname: version $VERS\n");
    prt("Usage: $pgmname [options] in-file\n");
    prt("Options:\n");
    prt(" --help  (-h or -?) = This help, and exit 0.\n");
    prt(" --verb[n]     (-v) = Bump [or set] verbosity. def=$verbosity\n");
    prt(" --load        (-l) = Load LOG at end. ($outfile)\n");
    prt(" --out <file>  (-o) = Write output to this file.\n");
}

# X-Plane apt.dat codes
my $x_code = <<EOF

FROM : http://data.x-plane.com/designers.html#Formats
Astronomical:  740 Version (latest)
===================================
from : http://data.x-plane.com/file_specs/Astro740.htm
sample
 6.752569 -16.713143 -1.43 Sirius
19.846301   8.867385  0.77 Altair
 2.529743  89.264138  1.97 Polaris 

6.752569   Right Ascension in decimal hours.  Always a positive number. 
-16.713143 Declination in decimal degrees.  Positive declinations are north of the celestial equator (eg. the pole star, Polaris, is at a declination of 89.264138 degrees). 
-1.43      Visible magnitude of the star.  This is a weird logarithmic scale (low numbers are brightest), 
           and stars to a magnitude of +6.5 are considered visible to the naked eye 
           (though this will vary hugely with your local seeing conditions, light pollution, altitude, etc.).  
           Sirius (the brightest star in the night sky) has a negative magnitude (-1.43) because it is very, 
           very bright.  
Sirius     Star name (optional - not used by X-Plane).  

Any units of angular measure can be used for right ascension, but it is customarily 
measured in hours ( h ), minutes ( m ), and seconds ( s ), with 24h being equivalent 
to a full circle.

EOF
# ================================


# eof - astro.pl
