#!/usr/bin/perl -w
# NAME: chkactri.pl
# AIM: Read an AC3D file, and check if ALL surfaces are tri - three vectors
# 11/05/2013 geoff mclane http://geoffair.net/mperl
use strict;
use warnings;
use File::Basename;  # split path ($name,$dir,$ext) = fileparse($file [, qr/\.[^.]*/] )
use Cwd;
my $os = $^O;
my $perl_dir = '/home/geoff/bin';
my $PATH_SEP = '/';
my $temp_dir = '/tmp';
if ($os =~ /win/i) {
    $perl_dir = 'C:\GTools\perl';
    $temp_dir = $perl_dir;
    $PATH_SEP = "\\";
}
unshift(@INC, $perl_dir);
require 'lib_utils.pl' or die "Unable to load 'lib_utils.pl' Check paths in \@INC...\n";
# log file stuff
our ($LF);
my $pgmname = $0;
if ($pgmname =~ /(\\|\/)/) {
    my @tmpsp = split(/(\\|\/)/,$pgmname);
    $pgmname = $tmpsp[-1];
}
my $outfile = $temp_dir.$PATH_SEP."temp.$pgmname.txt";
open_log($outfile);

# user variables
my $VERS = "0.0.1 2013-05-11";
my $load_log = 0;
my $in_file = '';
my $verbosity = 0;
my $out_file = '';

# ### DEBUG ###
my $debug_on = 0;
my $def_file = 'C:\FG\18\blendac3d\f16.ac';
#my $def_file = 'C:\FG\18\blendac3d\cube.ac';

### program variables
my @warnings = ();
my $cwd = cwd();

sub VERB1() { return $verbosity >= 1; }
sub VERB2() { return $verbosity >= 2; }
sub VERB5() { return $verbosity >= 5; }
sub VERB9() { return $verbosity >= 9; }

sub show_warnings($) {
    my ($val) = @_;
    if (@warnings) {
        prt( "\nGot ".scalar @warnings." WARNINGS...\n" );
        foreach my $itm (@warnings) {
           prt("$itm\n");
        }
        prt("\n");
    } else {
        prt( "\nNo warnings issued.\n\n" ) if (VERB9());
    }
}

sub pgm_exit($$) {
    my ($val,$msg) = @_;
    if (length($msg)) {
        $msg .= "\n" if (!($msg =~ /\n$/));
        prt($msg);
    }
    show_warnings($val);
    close_log($outfile,$load_log);
    exit($val);
}


sub prtw($) {
   my ($tx) = shift;
   $tx =~ s/\n$//;
   prt("$tx\n");
   push(@warnings,$tx);
}

#SURF 0x20
#mat 0
#refs 4
#3 0 0
#2 1 0
#1 1 1
#0 0 1
#kids 0

my @object_verts = ();

sub process_in_file($) {
    my ($inf) = @_;
    if (! open INF, "<$inf") {
        pgm_exit(1,"ERROR: Unable to open file [$inf]\n"); 
    }
    my @lines = <INF>;
    close INF;
    my $lncnt = scalar @lines;
    prt("Processing $lncnt lines, from [$inf]...\n") if (VERB2());
    my ($line,$i,$lnn,$inc,$mat,$refs,$rav);
    my (@arr);
    $lnn = 0;
    my @refcnts = ();
    my %refn = ();
    $line = $lines[0];
    chomp $line;
    if ($line =~ /^AC3D/) {
        $line = substr($line,4);
        prt("AC3D version = $line\n") if (VERB2());
    } else {
        prt("This is NOT an AC3d file!\n");
        return;
    }
    my ($name,$dir) = fileparse($inf);
    my @vertices = ();
    my ($verts,$cnt);
    my $total_verts = 0;
    $rav = \@vertices;
    for ($i = 1; $i < $lncnt; $i++) {
        $line = $lines[$i];
        chomp $line;
        $lnn = $i + 1;
        if ($line =~ /^\s*numvert\s+(\d+)$/) {
            $verts = $1;
            $total_verts += $verts;
            while ($verts && (($i+1) < $lncnt)) {
                $i++;
                $lnn = $i + 1;
                $line = $lines[$i];
                chomp $line;
                $line = trim_all($line);
                @arr = split(/\s+/,$line);
                $cnt = scalar @arr;
                if ($cnt == 3) {
                    push(@vertices, [$arr[0],$arr[1],$arr[2]]);
                }
                $verts--;
            }
        } elsif ($line =~ /^\s+OBJECT\s+/) {
            if (@vertices) {
                push(@object_verts, [@vertices]);
                $rav = $object_verts[-1];
            }
            @vertices = ();
        } elsif ($line =~ /^\s*SURF\s+(.+)$/) {
            $inc = $1;
            prt("$lnn: SURF $inc\n") if (VERB2());
            $i++;
            $lnn = $i + 1;
            # expect definite sequence mat refs ... kids
            if ($i < $lncnt) {
                $line = $lines[$i];
                chomp $line;
                if ($line =~ /^\s*mat\s+(\d+)$/) {
                    $mat = $1;
                    $i++;
                    $lnn = $i + 1;
                    if ($i < $lncnt) {
                        $line = $lines[$i];
                        chomp $line;
                        if ($line =~ /^\s*refs\s+(\d+)$/) {
                            $refs = $1;
                            push(@refcnts,$refs);
                            $refn{$refs}++;
                            while ($refs && (($i + 1) < $lncnt)) {
                                $i++;
                                $lnn = $i + 1;
                                $line = $lines[$i];
                                chomp $line;
                                $refs--;
                                last if ($line =~ /^\s*kids\s+/);
                                prt("$line\n") if (VERB9());
                            }
                            if ($refs) {
                                prt("$lnn: Reached end of SURF with $refs refs remaining...\n");
                            }
                        }
                    }
                }
            }
        }
    }

    $cnt = scalar @refcnts;
    @arr = sort keys(%refn);
    $i = scalar @arr;
    my $acnt = sprintf("%4d",$cnt);
    my $msg = '';
    if ($i == 1) {
        $msg = sprintf("%4d",$arr[0]);
        prt("ALL $acnt refs have count  of $msg. verts $total_verts $name $lncnt\n");
    } else {
        $msg = '';
        foreach $i (@arr) {
            $msg .= ", " if (length($msg));
            $msg .= "$i = ".sprintf("%4d",$refn{$i});
        }
        prt("The $acnt refs have counts of $msg. verts $total_verts $name $lncnt\n");
    }
}

#########################################
### MAIN ###
parse_args(@ARGV);
process_in_file($in_file);
pgm_exit(0,"");
########################################

sub need_arg {
    my ($arg,@av) = @_;
    pgm_exit(1,"ERROR: [$arg] must have a following argument!\n") if (!@av);
}

sub parse_args {
    my (@av) = @_;
    my ($arg,$sarg);
    while (@av) {
        $arg = $av[0];
        if ($arg =~ /^-/) {
            $sarg = substr($arg,1);
            $sarg = substr($sarg,1) while ($sarg =~ /^-/);
            if (($sarg =~ /^h/i)||($sarg eq '?')) {
                give_help();
                pgm_exit(0,"Help exit(0)");
            } elsif ($sarg =~ /^v/) {
                if ($sarg =~ /^v.*(\d+)$/) {
                    $verbosity = $1;
                } else {
                    while ($sarg =~ /^v/) {
                        $verbosity++;
                        $sarg = substr($sarg,1);
                    }
                }
                prt("Verbosity = $verbosity\n") if (VERB1());
            } elsif ($sarg =~ /^l/) {
                if ($sarg =~ /^ll/) {
                    $load_log = 2;
                } else {
                    $load_log = 1;
                }
                prt("Set to load log at end. ($load_log)\n") if (VERB1());
            } elsif ($sarg =~ /^o/) {
                need_arg(@av);
                shift @av;
                $sarg = $av[0];
                $out_file = $sarg;
                prt("Set out file to [$out_file].\n") if (VERB1());
            } else {
                pgm_exit(1,"ERROR: Invalid argument [$arg]! Try -?\n");
            }
        } else {
            $in_file = $arg;
            prt("Set input to [$in_file]\n") if (VERB1());
        }
        shift @av;
    }

    if ($debug_on) {
        prtw("WARNING: DEBUG is ON!\n");
        if ((length($in_file) ==  0) && $debug_on) {
            $in_file = $def_file;
            prt("Set DEFAULT input to [$in_file]\n");
        }
    }
    if (length($in_file) ==  0) {
        pgm_exit(1,"ERROR: No input files found in command!\n");
    }
    if (! -f $in_file) {
        pgm_exit(1,"ERROR: Unable to find in file [$in_file]! Check name, location...\n");
    }
}

sub give_help {
    prt("$pgmname: version $VERS\n");
    prt("Usage: $pgmname [options] in-file\n");
    prt("Options:\n");
    prt(" --help  (-h or -?) = This help, and exit 0.\n");
    prt(" --verb[n]     (-v) = Bump [or set] verbosity. def=$verbosity\n");
    prt(" --load        (-l) = Load LOG at end. ($outfile)\n");
    prt(" --out <file>  (-o) = Write output to this file.\n");
}

# eof - template.pl
