#!/usr/bin/perl -w
# NAME: chkcss.pl
# AIM: Given a CSS file, checks some syntax, and reports missing url items
# 10/11/2013 geoff mclane http://geoffair.net/mperl
use strict;
use warnings;
use File::Basename;  # split path ($name,$dir,$ext) = fileparse($file [, qr/\.[^.]*/] )
use Cwd;
my $os = $^O;
my $perl_dir = '/home/geoff/bin';
my $PATH_SEP = '/';
my $temp_dir = '/tmp';
if ($os =~ /win/i) {
    $perl_dir = 'C:\GTools\perl';
    $temp_dir = $perl_dir;
    $PATH_SEP = "\\";
}
unshift(@INC, $perl_dir);
require 'lib_utils.pl' or die "Unable to load 'lib_utils.pl' Check paths in \@INC...\n";
# log file stuff
our ($LF);
my $pgmname = $0;
if ($pgmname =~ /(\\|\/)/) {
    my @tmpsp = split(/(\\|\/)/,$pgmname);
    $pgmname = $tmpsp[-1];
}
my $outfile = $temp_dir.$PATH_SEP."temp.$pgmname.txt";
open_log($outfile);

# user variables
my $VERS = "0.0.1 2013-03-17";
my $load_log = 0;
my $in_file = '';
my $verbosity = 0;
my $out_file = '';

# ### DEBUG ###
my $debug_on = 0;
my $def_file = 'C:\OSGeo4W\apache\htdocs\test\css\ext-all.css';

### program variables
my @warnings = ();
my $cwd = cwd();

sub VERB1() { return $verbosity >= 1; }
sub VERB2() { return $verbosity >= 2; }
sub VERB5() { return $verbosity >= 5; }
sub VERB9() { return $verbosity >= 9; }

sub show_warnings($) {
    my ($val) = @_;
    if (@warnings) {
        prt( "\nGot ".scalar @warnings." WARNINGS...\n" );
        foreach my $itm (@warnings) {
           prt("$itm\n");
        }
        prt("\n");
    } else {
        prt( "\nNo warnings issued.\n\n" ) if (VERB9());
    }
}

sub pgm_exit($$) {
    my ($val,$msg) = @_;
    if (length($msg)) {
        $msg .= "\n" if (!($msg =~ /\n$/));
        prt($msg);
    }
    show_warnings($val);
    close_log($outfile,$load_log);
    exit($val);
}


sub prtw($) {
   my ($tx) = shift;
   $tx =~ s/\n$//;
   prt("$tx\n");
   push(@warnings,$tx);
}

sub get_token_keys($) {
    my $ktxt = shift;
    my @arr = ();
    my ($i,$len,$ch);
    my $token = '';
    $len = length($ktxt);
    for ($i = 0; $i < $len; $i++) {
        $ch = substr($ktxt,$i,1);
        if (($ch =~ /\s/)||($ch eq ',')) {
            # end of token
            push(@arr,$token) if (length($token));
            $token = '';    # restart token
        } else {
            $token .= $ch;
        }
    }
    push(@arr,$token) if (length($token));
    return \@arr;
}

sub get_brace_props($) {
    my $btxt = shift;
    my @arr = ();
    my ($i,$len,$ch);
    my $token = '';
    $len = length($btxt);
    my @brackets = ();
    for ($i = 0; $i < $len; $i++) {
        $ch = substr($btxt,$i,1);
        if ($ch eq ';') {
            # end of property
            push(@arr,$token) if (length($token));
            $token = '';    # restart token
        } else {
            $token .= $ch;
        }
    }
    push(@arr,$token) if (length($token));
    return \@arr;
}

sub expand_properties($) {
    my $btxt = shift;
    my ($i,$len,$ch);
    my $token = '';
    $len = length($btxt);
    my @arr = ();
    for ($i = 0; $i < $len; $i++) {
        $ch = substr($btxt,$i,1);
        if ($ch eq ':') {
            push(@arr,$token);
            $i++;
            $token = '';
            for (; $i < $len; $i++) {
                $ch = substr($btxt,$i,1);
                last if ($ch =~ /\S/);
            }
            $token .= $ch if ($ch =~ /\S/);
            $i++;
            for (; $i < $len; $i++) {
                $ch = substr($btxt,$i,1);
                $token .= $ch;
            }
            push(@arr,$token);
        } elsif ($ch =~ /\S/) {
            $token .= $ch;
        }
    }
    return \@arr;
}

sub process_in_file($) {
    my ($inf) = @_;
    if (! open INF, "<$inf") {
        pgm_exit(1,"ERROR: Unable to open file [$inf]\n"); 
    }
    my @lines = <INF>;
    close INF;
    my $lncnt = scalar @lines;
    my ($name,$dir) = fileparse($inf);
    prt("Processing $lncnt lines, from [$inf]...\n");
    my ($line,$ch,$pc,$len,$i,$nc,$lnn,$i2,$rba,$rta,$rxa,$tmp,$fil);
    $lnn = 0;
    my $incomment = 0;
    my $inbrace = 0;
    $ch = '';
    my $btxt = '';
    my $ktxt = '';
    my $brbgn = '';
    my $cmbgn = '';
    my %urls_found = ();
    foreach $line (@lines) {
        chomp $line;
        $lnn++;
        $len = length($line);
        for ($i = 0; $i < $len; $i++) {
            $i2 = $i + 1;
            $pc = $ch;
            $ch = substr($line,$i,1);
            $nc = ($i2 < $len) ? substr($line,$i2,1) : '';
            if ($incomment) {
                if (($ch eq '/')&&($pc eq '*')) {
                    $incomment = 0;
                }
                next;
            }
            if (($ch eq '/') && ($nc eq '*')) {
                $incomment = 1;
                $cmbgn = "$lnn:$i2:";
                next;
            }
            if ($inbrace) {
                if ($ch eq '}') {
                    $inbrace = 0;
                    $rta = get_token_keys($ktxt);
                    $rba = get_brace_props($btxt);
                    prt("$ktxt { $btxt }\n") if (VERB1());
                    $ktxt = join(", ",@{$rta})."\n{\n";
                    foreach $btxt (@{$rba}) {
                        $rxa = expand_properties($btxt);
                        $btxt = '';
                        foreach $tmp (@{$rxa}) {
                            $btxt .= ' : ' if (length($btxt));
                            $btxt .= $tmp;
                            if ($tmp =~ /^url\s*\(\s*('|")(.+)('|")\s*\)\s*$/) {
                                $fil = $2;
                                $fil = path_u2d($fil) if ($os =~ /win/i);
                                $urls_found{$fil} = 1;
                            }
                        }
                        $ktxt .= " $btxt;\n";
                    }
                    $ktxt .= "}\n";
                    prt($ktxt) if (VERB9());
                    $ktxt = '';
                    $btxt = '';
                } else {
                    $btxt .= $ch;
                }
            } elsif ($ch eq '{') {
                $inbrace = 1;
                $brbgn = "$lnn:$i2:";
            } else {
                $ktxt .= $ch;
            }
        }
        if ($inbrace) {
            $btxt .= ' ' if (length($btxt));
        } else {
            $ktxt .= ' ' if (length($ktxt));
        }
    }
    if ($incomment) {
        prtw("WARNING: Reached EOF in a comment! Began $cmbgn\n");
    }
    if ($inbrace) {
        prtw("WARNING: Reached EOF in a brace! Began $brbgn\n");
    }
    my @arr = sort keys(%urls_found);
    $tmp = scalar @arr;
    if ($tmp) {
        my %dirs = ();
        my ($msg,$fil);
        prt("Found $tmp url() items...\n");
        foreach $tmp (@arr) {
            $fil = $dir.$tmp;
            if (-f $fil) {
                $msg = 'ok';
                prt("$tmp $msg\n") if (VERB5());
            } else {
                $msg = 'NF $fil';
                prt("$tmp $msg\n");
            }
            my ($n,$d) = fileparse($tmp);
            $dirs{$d} = 1;
        }
        @arr = sort keys(%dirs);
        $tmp = scalar @arr;
        prt("They are in $tmp directories...\n");
        prt(join("\n",@arr)."\n");
    }
    ### $load_log = 1;
}

#########################################
### MAIN ###
parse_args(@ARGV);
process_in_file($in_file);
pgm_exit(0,"");
########################################

sub need_arg {
    my ($arg,@av) = @_;
    pgm_exit(1,"ERROR: [$arg] must have a following argument!\n") if (!@av);
}

sub parse_args {
    my (@av) = @_;
    my ($arg,$sarg);
    while (@av) {
        $arg = $av[0];
        if ($arg =~ /^-/) {
            $sarg = substr($arg,1);
            $sarg = substr($sarg,1) while ($sarg =~ /^-/);
            if (($sarg =~ /^h/i)||($sarg eq '?')) {
                give_help();
                pgm_exit(0,"Help exit(0)");
            } elsif ($sarg =~ /^v/) {
                if ($sarg =~ /^v.*(\d+)$/) {
                    $verbosity = $1;
                } else {
                    while ($sarg =~ /^v/) {
                        $verbosity++;
                        $sarg = substr($sarg,1);
                    }
                }
                prt("Verbosity = $verbosity\n") if (VERB1());
            } elsif ($sarg =~ /^l/) {
                if ($sarg =~ /^ll/) {
                    $load_log = 2;
                } else {
                    $load_log = 1;
                }
                prt("Set to load log at end. ($load_log)\n") if (VERB1());
            } elsif ($sarg =~ /^o/) {
                need_arg(@av);
                shift @av;
                $sarg = $av[0];
                $out_file = $sarg;
                prt("Set out file to [$out_file].\n") if (VERB1());
            } else {
                pgm_exit(1,"ERROR: Invalid argument [$arg]! Try -?\n");
            }
        } else {
            $in_file = $arg;
            prt("Set input to [$in_file]\n") if (VERB1());
        }
        shift @av;
    }

    if ($debug_on) {
        prtw("WARNING: DEBUG is ON!\n");
        if ((length($in_file) ==  0) && $debug_on) {
            $in_file = $def_file;
            prt("Set DEFAULT input to [$in_file]\n");
        }
    }
    if (length($in_file) ==  0) {
        pgm_exit(1,"ERROR: No input files found in command!\n");
    }
    if (! -f $in_file) {
        pgm_exit(1,"ERROR: Unable to find in file [$in_file]! Check name, location...\n");
    }
}

sub give_help {
    prt("$pgmname: version $VERS\n");
    prt("Usage: $pgmname [options] in-file\n");
    prt("Options:\n");
    prt(" --help  (-h or -?) = This help, and exit 0.\n");
    prt(" --verb[n]     (-v) = Bump [or set] verbosity. def=$verbosity\n");
    prt(" --load        (-l) = Load LOG at end. ($outfile)\n");
    prt(" --out <file>  (-o) = Write output to this file.\n");
}

# eof - template.pl
