#!/usr/bin/perl -w
# NAME: chkinstall.pl
# AIM: Read a cmake_install.cmake file, and list what will be 'installed' if run...
use strict;
use warnings;
use File::Basename;  # split path ($name,$dir,$ext) = fileparse($file [, qr/\.[^.]*/] )
use File::Spec; # File::Spec->rel2abs($rel); # we are IN the SLN directory, get ABSOLUTE from RELATIVE
use Cwd;
my $os = $^O;
my $perl_dir = '/home/geoff/bin';
my $PATH_SEP = '/';
my $temp_dir = '/tmp';
if ($os =~ /win/i) {
    $perl_dir = 'C:\GTools\perl';
    $temp_dir = $perl_dir;
    $PATH_SEP = "\\";
}
unshift(@INC, $perl_dir);
require 'lib_utils.pl' or die "Unable to load 'lib_utils.pl' Check paths in \@INC...\n";
# log file stuff
our ($LF);
my $pgmname = $0;
if ($pgmname =~ /(\\|\/)/) {
    my @tmpsp = split(/(\\|\/)/,$pgmname);
    $pgmname = $tmpsp[-1];
}
my $outfile = $temp_dir.$PATH_SEP."temp.$pgmname.txt";
open_log($outfile);

# user variables
my $VERS = "0.0.2 2014-01-13";
my $load_log = 0;
my $in_file = '';
my $verbosity = 0;
my $out_file = '';

# ### DEBUG ###
my $debug_on = 1;
my $def_file = 'Z:\build-gmt\cmake_install.cmake';

### program variables
my @warnings = ();
my $cwd = cwd();

sub VERB1() { return $verbosity >= 1; }
sub VERB2() { return $verbosity >= 2; }
sub VERB5() { return $verbosity >= 5; }
sub VERB9() { return $verbosity >= 9; }

sub show_warnings($) {
    my ($val) = @_;
    if (@warnings) {
        prt( "\nGot ".scalar @warnings." WARNINGS...\n" );
        foreach my $itm (@warnings) {
           prt("$itm\n");
        }
        prt("\n");
    } else {
        prt( "\nNo warnings issued.\n\n" ) if (VERB9());
    }
}

sub pgm_exit($$) {
    my ($val,$msg) = @_;
    if (length($msg)) {
        $msg .= "\n" if (!($msg =~ /\n$/));
        prt($msg);
    }
    show_warnings($val);
    close_log($outfile,$load_log);
    exit($val);
}


sub prtw($) {
   my ($tx) = shift;
   $tx =~ s/\n$//;
   prt("$tx\n");
   push(@warnings,$tx);
}

my %done = ();
sub process_in_file($);

sub not_complete_cmake_line($) {
    my $line = shift;
    my $len = length($line);
    my @brackets = ();
    my ($i,$ch,$inquot,$pc);
    $inquot = 0;
    $ch = '';
    for ($i = 0; $i < $len; $i++) {
        $pc = $ch;
        $ch = substr($line,$i,1);
        if ($inquot) {
            $inquot = 0 if (($ch eq '"') && ($pc ne "\\"));
        } else {
            if ($ch eq '"') {
                $inquot = 1;
            } elsif ($ch eq '(') {
                push(@brackets,$i);
            } elsif ($ch eq ')') {
                if (@brackets) {
                    pop @brackets;
                }
            } elsif ($ch eq '#') {
                if (!@brackets) {
                    last;
                }
            }
        }
    }
    $len = scalar @brackets;
    return $len;
}

sub split_cmake_line($) {
    my $oline = shift;
    my $line = trim_all($oline);
    my $len = length($line);
    my @brackets = ();
    my @tags = ();
    my ($i,$ch,$inquot,$pc,$tag);
    $inquot = 0;
    $ch = '';
    $tag = '';
    for ($i = 0; $i < $len; $i++) {
        $pc = $ch;
        $ch = substr($line,$i,1);
        if ($inquot) {
            $tag .= $ch;
            if (($ch eq '"') && ($pc ne "\\")) {
                $inquot = 0;
                push(@tags,$tag);
                $tag = '';
            }
            next;
        } else {
            if ($ch eq '"') {
                $inquot = 1;
                push(@tags,$tag) if (length($tag));
                $tag = $ch;
                next;
            } elsif ($ch eq '(') {
                if (!@brackets) {
                    push(@tags,$tag) if (length($tag));
                    $tag = '';
                }
                push(@brackets,$i);
                next;
            } elsif ($ch eq ')') {
                if (@brackets) {
                    pop @brackets;
                }
                if (!@brackets) {
                    push(@tags,$tag) if (length($tag));
                    $tag = '';
                }
                next;
            } elsif ($ch eq '#') {
                if (!@brackets) {
                    last;
                }
            }
        }
        if ($ch =~ /\s/) {
            push(@tags,$tag) if (length($tag));
            $tag = '';
        } else {
            $tag .= $ch;
        }
    }
    $len = scalar @brackets;
    return \@tags;
}

sub process_in_file($) {
    my ($inf) = @_;
    return if (defined $done{$inf});
    if (! open INF, "<$inf") {
        pgm_exit(1,"ERROR: Unable to open file [$inf]\n"); 
    }
    $done{$inf} = 1;
    my @lines = <INF>;
    close INF;
    my $lncnt = scalar @lines;
    prt("Processing $lncnt lines, from [$inf]...\n"); # if (VERB5());
    my ($i,$line,$inc,$lnn,$bal,$len,$ind,$tmp,$skip);
    my ($dest,$type,$inst);
    $lnn = 0;
    my @includes = ();
    for ($i = 0; $i < $lncnt; $i++) {
        $line = $lines[$i];
        chomp $line;
        $lnn = $i + 1;;
        $line = trim_all($line);
        $len = length($line);
        next if ($len == 0);
        next if ($line =~ /^\#/); # skip comments
        while (not_complete_cmake_line($line) && ($lnn < $lncnt)) {
            $i++;
            $lnn = $i + 1;
            $tmp = $lines[$i];
            chomp $tmp;
            $tmp = trim_all($tmp);
            $len = length($tmp);
            next if ($len == 0);
            $line .= ' '.$tmp;
        }

        $ind = index($line,'DESTINATION');
        $skip = index($line,'MinSizeRel');
        $skip = index($line,'RelWithDebInfo') if ($skip == -1);
        if ($line =~ /^\s*INCLUDE\s*\("(.+)"\)/i) {
            $inc = $1;
            prt("$lnn: $inc\n") if (VERB9());
            if (-f $inc) {
                push(@includes,$inc);
            } else {
                prtw("WARNING: Can NOT locate INCLUDE file $inc\n");
            }
        } elsif ($line =~ /^\s*FILE\s*\(\s*INSTALL\s+DESTINATION\s+(\S+)\s+TYPE\s+(\S+)\s+FILES\s+(.+)$/i) {
            $dest = $1;
            $type = $2;
            $inst = $3;
            $dest =~ s/\$\{CMAKE_INSTALL_PREFIX\}\///;
            if ($skip == -1) {
                prt("$lnn: $dest $type $inst\n");
            }
        } elsif ($line =~ /^\s*FILE\s*\(\s*INSTALL\s+DESTINATION\s+(\S+)\s+TYPE\s+(\S+)\s+OPTIONAL\s+FILES\s+(.+)$/i) {
            $dest = $1;
            $type = $2;
            $inst = $3;
            $dest =~ s/\$\{CMAKE_INSTALL_PREFIX\}\///;
            if ($skip == -1) {
                prt("$lnn: $dest $type OPT $inst\n");
            }
        } elsif ($line =~ /^\s*FILE\s*\(\s*INSTALL\s+DESTINATION\s+/i) {
            if ($ind > 0) {
                $bal = substr($line,($ind+12));
            } else {
                $bal = substr($line,25);
            }
            if ($skip == -1) {
                prt("$lnn: $bal\n");
            }
        }
    }
    foreach $inc (@includes) {
        process_in_file($inc);
    }
    $load_log = 1;
}

#########################################
### MAIN ###
parse_args(@ARGV);
process_in_file($in_file);
pgm_exit(0,"");
########################################

sub need_arg {
    my ($arg,@av) = @_;
    pgm_exit(1,"ERROR: [$arg] must have a following argument!\n") if (!@av);
}

sub parse_args {
    my (@av) = @_;
    my ($arg,$sarg);
    my $verb = VERB2();
    while (@av) {
        $arg = $av[0];
        if ($arg =~ /^-/) {
            $sarg = substr($arg,1);
            $sarg = substr($sarg,1) while ($sarg =~ /^-/);
            if (($sarg =~ /^h/i)||($sarg eq '?')) {
                give_help();
                pgm_exit(0,"Help exit(0)");
            } elsif ($sarg =~ /^v/) {
                if ($sarg =~ /^v.*(\d+)$/) {
                    $verbosity = $1;
                } else {
                    while ($sarg =~ /^v/) {
                        $verbosity++;
                        $sarg = substr($sarg,1);
                    }
                }
                $verb = VERB2();
                prt("Verbosity = $verbosity\n") if ($verb);
            } elsif ($sarg =~ /^l/) {
                if ($sarg =~ /^ll/) {
                    $load_log = 2;
                } else {
                    $load_log = 1;
                }
                prt("Set to load log at end. ($load_log)\n") if ($verb);
            } elsif ($sarg =~ /^o/) {
                need_arg(@av);
                shift @av;
                $sarg = $av[0];
                $out_file = $sarg;
                prt("Set out file to [$out_file].\n") if ($verb);
            } else {
                pgm_exit(1,"ERROR: Invalid argument [$arg]! Try -?\n");
            }
        } else {
            $in_file = $arg;
            prt("Set input to [$in_file]\n") if ($verb);
        }
        shift @av;
    }

    if ($debug_on) {
        prtw("WARNING: DEBUG is ON!\n");
        if (length($in_file) ==  0) {
            $in_file = $def_file;
            prt("Set DEFAULT input to [$in_file]\n");
        }
    }
    if (length($in_file) ==  0) {
        pgm_exit(1,"ERROR: No input files found in command!\n");
    }
    $in_file = File::Spec->rel2abs($in_file);
    if (! -f $in_file) {
        pgm_exit(1,"ERROR: Unable to find in file [$in_file]! Check name, location...\n");
    }
}

sub give_help {
    prt("$pgmname: version $VERS\n");
    prt("Usage: $pgmname [options] in-file\n");
    prt("Options:\n");
    prt(" --help  (-h or -?) = This help, and exit 0.\n");
    prt(" --verb[n]     (-v) = Bump [or set] verbosity. def=$verbosity\n");
    prt(" --load        (-l) = Load LOG at end. ($outfile)\n");
    prt(" --out <file>  (-o) = Write output to this file.\n");
}

# eof - template.pl
