#!/usr/bin/perl -w
# NAME: chktabs.pl
# AIM: Read a file and report number of lines containing a tab character
# 15/03/2015 - Allow multiple files, and wild cards...

use strict;
use warnings;
use File::Basename;  # split path ($name,$dir,$ext) = fileparse($file [, qr/\.[^.]*/] )
use Cwd;
my $os = $^O;
my $perl_dir = '/home/geoff/bin';
my $PATH_SEP = '/';
my $temp_dir = '/tmp';
if ($os =~ /win/i) {
    $perl_dir = 'C:\GTools\perl';
    $temp_dir = $perl_dir;
    $PATH_SEP = "\\";
}
unshift(@INC, $perl_dir);
require 'lib_utils.pl' or die "Unable to load 'lib_utils.pl' Check paths in \@INC...\n";
# log file stuff
our ($LF);
my $pgmname = $0;
if ($pgmname =~ /(\\|\/)/) {
    my @tmpsp = split(/(\\|\/)/,$pgmname);
    $pgmname = $tmpsp[-1];
}
my $outfile = $temp_dir.$PATH_SEP."temp.$pgmname.txt";
open_log($outfile);

# user variables
my $VERS = "0.0.6 2015-03-15";
#my $VERS = "0.0.5 2015-01-09";
my $load_log = 0;
my $in_file = '';
my @in_files = ();
my $verbosity = 0;
my $out_file = '';

my %files_with_tab = ();

# ### DEBUG ###
my $debug_on = 0;
my $def_file = 'F:\Projects\tidy-html5\src\parser.c';

### program variables
my @warnings = ();
my $cwd = cwd();

sub VERB1() { return $verbosity >= 1; }
sub VERB2() { return $verbosity >= 2; }
sub VERB5() { return $verbosity >= 5; }
sub VERB9() { return $verbosity >= 9; }

sub show_warnings($) {
    my ($val) = @_;
    if (@warnings) {
        prt( "\nGot ".scalar @warnings." WARNINGS...\n" );
        foreach my $itm (@warnings) {
           prt("$itm\n");
        }
        prt("\n");
    } else {
        prt( "\nNo warnings issued.\n\n" ) if (VERB9());
    }
}

sub pgm_exit($$) {
    my ($val,$msg) = @_;
    if (length($msg)) {
        $msg .= "\n" if (!($msg =~ /\n$/));
        prt($msg);
    }
    show_warnings($val);
    close_log($outfile,$load_log);
    exit($val);
}


sub prtw($) {
   my ($tx) = shift;
   $tx =~ s/\n$//;
   prt("$tx\n");
   push(@warnings,$tx);
}

my %done_files = ();
my $total_files = 0;
my $total_lines = 0;
my $total_bytes = 0;
sub process_in_file($) {
    my ($inf) = @_;
    if (defined $done_files{$inf}) {
        return;
    }
    $done_files{$inf} = 1;
    if (! open INF, "<$inf") {
        pgm_exit(1,"ERROR: Unable to open file [$inf]\n"); 
    }
    my @lines = <INF>;
    close INF;
    my $lncnt = scalar @lines;
    $total_files++;
    $total_lines += $lncnt;
    prt("Processing $lncnt lines, from [$inf]...\n");
    my ($line,$ch,$lnn,$i,$len);
    my $lnswtab = 0;
    my $tabcount = 0;
    $lnn = 0;
    my $gottab = 0;
    my $lnswtsp = 0;
    foreach $line (@lines) {
        chomp $line;
        $lnn++;
        $len = length($line);
        $total_bytes += $len + 1;
        $gottab = 0;
        for ($i = 0; $i < $len; $i++) {
            $ch = substr($line,$i,1);
            if ($ch eq "\t") {
                $gottab++;
            }
        }
        if ($gottab) {
            $tabcount += $gottab;
            $lnswtab++;
        }
        if ($line =~ /\s+$/) {
            $lnswtsp++;
        }
    }
    if ($lnswtab || $lnswtsp) {
        $files_with_tab{$inf} = [$lnn,$lnswtab,$tabcount,$lnswtsp];
        prt("Have $lnswtab lines contain a 'tab'. Total of $tabcount tabs. $lnswtsp end w/space\n");
    }
}

sub process_in_files() {
    foreach $in_file (@in_files) {
        process_in_file($in_file);
    }
    my @arr = keys %files_with_tab;
    prt("\nProcessed $total_files files, $total_lines lines, appx $total_bytes bytes...\n");
    my ($ra,$lns,$wtab,$tcnt,$tscnt,$endsp,$pct);
    my $cnt = scalar @arr;
    $tcnt = 0;
    $endsp = 0;
    my $total_tabs = 0;
    my $total_lwtsp = 0;
    foreach $in_file (@arr) {
        $ra = $files_with_tab{$in_file};
        $lns = ${$ra}[0];
        $wtab = ${$ra}[1];
        $cnt = ${$ra}[2];
        $tscnt = ${$ra}[3];
        if ($cnt) {
            $tcnt++;
            $total_tabs += $cnt;
        }
        if ($tscnt) {
            $endsp++;
            $total_lwtsp += $tscnt;
        }
    }
    $pct = ($total_tabs / $total_bytes) * 100;
    $pct = int($pct * 100) / 100;
    prt("\nFound $tcnt files with tabs... $total_tabs total tabs, $pct \%...\n");
    foreach $in_file (@arr) {
        $ra = $files_with_tab{$in_file};
        $lns = ${$ra}[0];
        $wtab = ${$ra}[1];
        $cnt = ${$ra}[2];
        if ($cnt) {
            prt("$in_file - $lns lines, $wtab w/tab, $cnt tabs\n");
        }
    }
    $pct = ($total_lwtsp / $total_lines) * 100;
    $pct = int($pct * 10) / 10;
    prt("\nFound $endsp files with trailing spaces... $total_lwtsp total lines end space, $pct \%\n");
    foreach $in_file (@arr) {
        $ra = $files_with_tab{$in_file};
        $lns = ${$ra}[0];
        $wtab = ${$ra}[1];
        $tcnt = ${$ra}[2];
        $cnt = ${$ra}[3];
        if ($cnt) {
            prt("$in_file - $lns lines, $wtab w/tab, $tcnt tabs, $cnt lines with trailing space\n");
        }
    }

}

#########################################
### MAIN ###
parse_args(@ARGV);
process_in_files();
pgm_exit(0,"");
########################################

sub got_wild($) {
    my $fil = shift;
    return 1 if ($fil =~ /\*/);
    return 1 if ($fil =~ /\?/);
    return 0;
}
sub glob_wild($) {
    my $fil = shift;
    my @files = glob($fil);
    my $cnt = scalar @files;
    if ($cnt) {
        prt("Adding $cnt files, from [$fil] input.\n");
        push(@in_files,@files);
        $in_file = $files[0];
    } else {
        pgm_exit(1,"ERROR: Got no files, from [$fil] input.\n");
    }
}

sub need_arg {
    my ($arg,@av) = @_;
    pgm_exit(1,"ERROR: [$arg] must have a following argument!\n") if (!@av);
}

sub parse_args {
    my (@av) = @_;
    my ($arg,$sarg);
    my $verb = VERB2();
    while (@av) {
        $arg = $av[0];
        if ($arg =~ /^-/) {
            $sarg = substr($arg,1);
            $sarg = substr($sarg,1) while ($sarg =~ /^-/);
            if (($sarg =~ /^h/i)||($sarg eq '?')) {
                give_help();
                pgm_exit(0,"Help exit(0)");
            } elsif ($sarg =~ /^v/) {
                if ($sarg =~ /^v.*(\d+)$/) {
                    $verbosity = $1;
                } else {
                    while ($sarg =~ /^v/) {
                        $verbosity++;
                        $sarg = substr($sarg,1);
                    }
                }
                $verb = VERB2();
                prt("Verbosity = $verbosity\n") if ($verb);
            } elsif ($sarg =~ /^l/) {
                if ($sarg =~ /^ll/) {
                    $load_log = 2;
                } else {
                    $load_log = 1;
                }
                prt("Set to load log at end. ($load_log)\n") if ($verb);
            } elsif ($sarg =~ /^o/) {
                need_arg(@av);
                shift @av;
                $sarg = $av[0];
                $out_file = $sarg;
                prt("Set out file to [$out_file].\n") if ($verb);
            } else {
                pgm_exit(1,"ERROR: Invalid argument [$arg]! Try -?\n");
            }
        } else {
            if (got_wild($arg)) {
                glob_wild($arg);
            } else {
                $in_file = $arg;
                push(@in_files,$in_file);
                prt("Set input to [$in_file]\n") if ($verb);
            }
        }
        shift @av;
    }

    if ($debug_on) {
        prtw("WARNING: DEBUG is ON!\n");
        if (length($in_file) ==  0) {
            $in_file = $def_file;
            push(@in_files,$in_file);
            prt("Set DEFAULT input to [$in_file]\n");
        }
    }
    if (length($in_file) ==  0) {
        pgm_exit(1,"ERROR: No input file found in command!\n");
    }
    if (! -f $in_file) {
        pgm_exit(1,"ERROR: Unable to find in file [$in_file]! Check name, location...\n");
    }
}

sub give_help {
    prt("$pgmname: version $VERS\n");
    prt("Usage: $pgmname [options] in-file\n");
    prt("Options:\n");
    prt(" --help  (-h or -?) = This help, and exit 0.\n");
    prt(" --verb[n]     (-v) = Bump [or set] verbosity. def=$verbosity\n");
    prt(" --load        (-l) = Load LOG at end. ($outfile)\n");
    prt(" --out <file>  (-o) = Write output to this file.\n");
}

# eof - template.pl
