#!/usr/bin/perl -w
# NAME: conf2cmake.pl
# AIM: Given a config.h.in file, convert #define #undef to #cmakedefine, and try to 
# read the ususal preceeding comment for the include file or function name and 
# write the equivalent CMakeLists.txt text.
# 29/10/2013 geoff mclane http://geoffair.net/mperl
use strict;
use warnings;
use File::Basename;  # split path ($name,$dir,$ext) = fileparse($file [, qr/\.[^.]*/] )
use Cwd;
my $os = $^O;
my $perl_dir = '/home/geoff/bin';
my $PATH_SEP = '/';
my $temp_dir = '/tmp';
if ($os =~ /win/i) {
    $perl_dir = 'C:\GTools\perl';
    $temp_dir = $perl_dir;
    $PATH_SEP = "\\";
}
unshift(@INC, $perl_dir);
require 'lib_utils.pl' or die "Unable to load 'lib_utils.pl' Check paths in \@INC...\n";
# log file stuff
our ($LF);
my $pgmname = $0;
if ($pgmname =~ /(\\|\/)/) {
    my @tmpsp = split(/(\\|\/)/,$pgmname);
    $pgmname = $tmpsp[-1];
}
my $outfile = $temp_dir.$PATH_SEP."temp.$pgmname.txt";
open_log($outfile);

# user variables
my $VERS = "0.0.1 2013-10-29";
my $load_log = 0;
my $in_file = '';
my $verbosity = 0;
my $out_file = '';

# ### DEBUG ###
my $debug_on = 1;
my $def_file = 'F:\FG\18\PDCurses\config.h.in';
my $def_out = $temp_dir.$PATH_SEP."tempconf.h";

### program variables
my @warnings = ();
my $cwd = cwd();

sub VERB1() { return $verbosity >= 1; }
sub VERB2() { return $verbosity >= 2; }
sub VERB5() { return $verbosity >= 5; }
sub VERB9() { return $verbosity >= 9; }

sub show_warnings($) {
    my ($val) = @_;
    if (@warnings) {
        prt( "\nGot ".scalar @warnings." WARNINGS...\n" );
        foreach my $itm (@warnings) {
           prt("$itm\n");
        }
        prt("\n");
    } else {
        prt( "\nNo warnings issued.\n\n" ) if (VERB9());
    }
}

sub pgm_exit($$) {
    my ($val,$msg) = @_;
    if (length($msg)) {
        $msg .= "\n" if (!($msg =~ /\n$/));
        prt($msg);
    }
    show_warnings($val);
    close_log($outfile,$load_log);
    exit($val);
}


sub prtw($) {
   my ($tx) = shift;
   $tx =~ s/\n$//;
   prt("$tx\n");
   push(@warnings,$tx);
}

sub process_in_file_org($) {
    my ($inf) = @_;
    if (! open INF, "<$inf") {
        pgm_exit(1,"ERROR: Unable to open file [$inf]\n"); 
    }
    my @lines = <INF>;
    close INF;
    my $lncnt = scalar @lines;
    prt("Processing $lncnt lines, from [$inf]...\n");
    my ($line,$inc,$lnn);
    $lnn = 0;
    foreach $line (@lines) {
        chomp $line;
        $lnn++;
        if ($line =~ /\s*#\s*include\s+(.+)$/) {
            $inc = $1;
            prt("$lnn: $inc\n");
        }
    }
}

sub process_in_file($);

# some SPECIAL variables, that should be defined differently, perhap
my %special_vars = (
    'PACKAGE' => 1,
    'PACKAGE_BUGREPORT' => 1,
    'PACKAGE_NAME' => 1,
    'PACKAGE_URL' => 1,
    'PACKAGE_VERSION' => 1,
    'PACKAGE_STRING' => 1,
    'PACKAGE_TARNAME' => 1
    ); # and maybe others
# want the end result like
#set(PACKAGE "dbus-python")
#set(PACKAGE_BUGREPORT "http://bugs.freedesktop.org/enter_bug.cgi?product=dbus&component=python")
#set(PACKAGE_NAME "dbus-python")
#set(PACKAGE_URL "")
#set(PACKAGE_VERSION "1.1.0")
#set(PACKAGE_STRING "${PACKAGE_NAME} ${PACKAGE_VERSION}")
#set(PACKAGE_TARNAME "${PACKAGE_NAME}")


sub get_header($$) {
    my ($txt,$rh) = @_;
    my $len = length($txt);
    my ($i,$ch);
    my $hdr = '';
    for ($i = 0; $i < $len; $i++) {
        $ch = substr($txt,$i,1);
        if ($ch eq '<') {
            $i++;
            last;
        }
    }
    if ($ch eq '<') {
        for ( ; $i < $len; $i++) {
            $ch = substr($txt,$i,1);
            last if ($ch eq '>');
            $hdr .= $ch;
        }
    }
    if (($ch eq '>')&&(length($hdr))) {
        ${$rh} = $hdr;
        return 1;
    }
    return 0;
}

sub get_function($$) {
    my ($txt,$rh) = @_;
    my $len = length($txt);
    my ($i,$ch);
    my $hdr = '';
    for ($i = 0; $i < $len; $i++) {
        $ch = substr($txt,$i,1);
        if ($ch eq '`') {
            $i++;
            last;
        }
    }
    if ($ch eq '`') {
        for ( ; $i < $len; $i++) {
            $ch = substr($txt,$i,1);
            last if ($ch eq "'");
            $hdr .= $ch;
        }
    }
    if (($ch eq "'")&&(length($hdr))) {
        ${$rh} = $hdr;
        return 1;
    }
    return 0;
}


sub process_in_file($) {
    my ($inf) = @_;
    if (! open INF, "<$inf") {
        pgm_exit(1,"ERROR: Unable to open file [$inf]\n"); 
    }
    my @lines = <INF>;
    close INF;
    my $lncnt = scalar @lines;
    my $hdr = "Processing $lncnt lines, from [$inf]...";
    my ($line,$var,$lnn,$incomm,$tline,$len,$i,$ch,$pc,$nc,$i2,$nline,$rda,$typ,$sys,$ff,$dc);
    my ($ok,$tmp,@arr,$comment,$header,$function,$ok2);
    $lnn = 0;
    $incomm = 0;
    $ch = '';
    my @incs = ();
    my @nlines = ();
    my %headers = ();
    my %functions = ();
    my %specials = ();
    $ok = 0;
    $ok2 = 0;
    foreach $line (@lines) {
        chomp $line;
        $tline = $line;
        ###$tline = trim_all($line);
        $lnn++;
        $len = length($tline);
        next if ($len == 0);
        $nline = '';
        for ($i = 0; $i < $len; $i++) {
            $pc = $ch;
            $i2 = $i + 1;
            $ch = substr($tline,$i,1);
            $nc = ($i2 < $len) ? substr($tline,$i2,1) : '';
            if ($incomm) {
                if (($ch eq '/')&&($pc eq '*')) {
                    $incomm = 0;
                }
                $comment .= $ch;
                next;
            }
            # not in a comment
            if (($ch eq '/') && ($nc eq '*')) {
                # start comment
                $incomm = 1;
                $comment = '/';
                next;
            }
            if (($ch eq '/') && ($nc eq '/')) {
                $comment .= "\n" if (length($comment));
                $comment .= substr($tline,$i);
                last;
            }
            $nline .= $ch;
        }
        # done line
        if ($incomm) {
            $comment .= "\n" if (length($comment));
        } elsif (length($nline)) {
            if (length($comment)) {
                prt("$comment\n");
                push(@nlines,$comment);
                $comment =~ s/\n/ /g;
                if ($comment =~ /\s+header\s+/) {
                    $ok = get_header($comment,\$header);
                    if ($ok) {
                        prt("Found header [$header] in comment\n") if (VERB9());
                    } else {
                        prtw("WARNING: Failed to find header in comment [$comment]\n");
                    }
                }
                if (!$ok && ($comment =~ /\s+function\b/)) {
                    $ok2 = get_function($comment,\$function);
                    if ($ok2) {
                        prt("Found function [$function] in comment\n");
                    } else {
                        prt("Failed to find function in comment [$comment]\n");
                    }
                }
            }
            if ($nline =~ /^\s*\#\s*undef\s+(.+)$/) {
                $var = $1;
                if (defined $special_vars{$var}) {
                    if (! defined $specials{$var}) {
                        $nline = "#define $var \"\@$var\@\"";
                        $specials{$var} = 1;
                    }

                } else {
                    $tmp = $nline;
                    $tmp =~ s/^\s*\#\s*undef\s+/\#cmakedefine /;
                    $tmp = trim_all($tmp);
                    $tmp .= ' 1' if ($ok || $ok2);
                    $nline = $tmp;
                }
                if ($ok) {
                    $headers{$var} = $header;
                } elsif ($ok2) {
                    $functions{$var} = $function;
                }
            }
            prt("$nline\n");
            push(@nlines,$nline);
            $nline = '';
            $comment = '';
            $ok = 0;
            $ok2 = 0;
        } else {
            push(@nlines,"");
        }
    }


    $nline = "/*\n";
    $nline .= "   config.h.cmake to produce config.h\n";
    $nline .= "   *** DO NOT EDIT config.h ***\n";
    $nline .= "   Edit config.h.cmake, and run cmake configure again\n";
    $nline .= " */\n";
    $nline .= "#ifndef _CONFIG_H_CMAKE_\n";
    $nline .= "#define _CONFIG_H_CMAKE_\n";
    $nline .= join("\n",@nlines)."\n";
    $nline .= "#endif /* _CONFIG_H_CMAKE_ */\n";
    $nline .= "/* eof */\n";

    @arr = sort keys %headers;
    $tmp = scalar @arr;
    if ($tmp) {
        prt("Got $tmp suggested cmake header searches\n");
        $nline .= "\n";
        $nline .= "/* # $tmp suggested cmake header searches\n";
        $nline .= "include (CheckIncludeFile) # add to top\n";
        $tmp = 0;
        foreach $var (@arr) {
            $header = $headers{$var};
            $len = length($header);
            $tmp = $len if ($len > $tmp);
        }

        foreach $var (@arr) {
            $header = $headers{$var};
            $header .= ' ' while (length($header) < $tmp);
            $nline .= "check_include_file($header $var)\n";
        }
        $nline .= "# */\n";
    }
    @arr = sort keys %functions;
    $tmp = scalar @arr;
    if ($tmp) {
        prt("Got $tmp suggested cmake function searches\n");
        $nline .= "\n";
        $nline .= "/* # $tmp suggested cmake function searches\n";
        $nline .= "include (CheckFunctionExists) # add near top\n";
        $tmp = 0;
        foreach $var (@arr) {
            $function = $functions{$var};
            $len = length($function);
            $tmp = $len if ($len > $tmp);
        }

        foreach $var (@arr) {
            $function = $functions{$var};
            $function .= ' ' while (length($function) < $tmp);
            $nline .= "check_function_exists($function $var)\n";
        }
        $nline .= "# */\n";
    }
    @arr = sort keys %specials;
    $tmp = scalar @arr;
    if ($tmp) {
        prt("Got $tmp suggested cmake variables to be set in CMakeLists.txt\n");
        $nline .= "\n";
        $nline .= "/* # $tmp suggested cmake variables to be set\n";
        foreach $var (@arr) {
            $nline .= "set($var \"TODO: fix this!\")\n";
        }
        $nline .= "# */\n";
    }

    if (length($out_file)) {
        write2file($nline,$out_file);
        prt("Results written to $out_file\n");
    } else {
        prt("$nline");
        prt("No -o out_file command given.\n");
    }
}


#########################################
### MAIN ###
parse_args(@ARGV);
process_in_file($in_file);
pgm_exit(0,"");
########################################

sub need_arg {
    my ($arg,@av) = @_;
    pgm_exit(1,"ERROR: [$arg] must have a following argument!\n") if (!@av);
}

sub parse_args {
    my (@av) = @_;
    my ($arg,$sarg);
    while (@av) {
        $arg = $av[0];
        if ($arg =~ /^-/) {
            $sarg = substr($arg,1);
            $sarg = substr($sarg,1) while ($sarg =~ /^-/);
            if (($sarg =~ /^h/i)||($sarg eq '?')) {
                give_help();
                pgm_exit(0,"Help exit(0)");
            } elsif ($sarg =~ /^v/) {
                if ($sarg =~ /^v.*(\d+)$/) {
                    $verbosity = $1;
                } else {
                    while ($sarg =~ /^v/) {
                        $verbosity++;
                        $sarg = substr($sarg,1);
                    }
                }
                prt("Verbosity = $verbosity\n") if (VERB1());
            } elsif ($sarg =~ /^l/) {
                if ($sarg =~ /^ll/) {
                    $load_log = 2;
                } else {
                    $load_log = 1;
                }
                prt("Set to load log at end. ($load_log)\n") if (VERB1());
            } elsif ($sarg =~ /^o/) {
                need_arg(@av);
                shift @av;
                $sarg = $av[0];
                $out_file = $sarg;
                prt("Set out file to [$out_file].\n") if (VERB1());
            } else {
                pgm_exit(1,"ERROR: Invalid argument [$arg]! Try -?\n");
            }
        } else {
            $in_file = $arg;
            prt("Set input to [$in_file]\n") if (VERB1());
        }
        shift @av;
    }

    if ($debug_on) {
        prtw("WARNING: DEBUG is ON!\n");
        if (length($in_file) ==  0) {
            $in_file = $def_file;
            prt("Set DEFAULT input to [$in_file]\n");
        }
        if (length($out_file) == 0) {
            $out_file = $def_out;
            prt("Set DEFAULT output to [$out_file]\n");
        }
    }
    if (length($in_file) ==  0) {
        pgm_exit(1,"ERROR: No input files found in command!\n");
    }
    if (! -f $in_file) {
        pgm_exit(1,"ERROR: Unable to find in file [$in_file]! Check name, location...\n");
    }
}

sub give_help {
    prt("$pgmname: version $VERS\n");
    prt("Usage: $pgmname [options] in-file\n");
    prt("Options:\n");
    prt(" --help  (-h or -?) = This help, and exit 0.\n");
    prt(" --verb[n]     (-v) = Bump [or set] verbosity. def=$verbosity\n");
    prt(" --load        (-l) = Load LOG at end. ($outfile)\n");
    prt(" --out <file>  (-o) = Write output to this file.\n");
}

# eof - template.pl
