#!/usr/bin/perl -w
# NAME: dir2nsis.pl
# AIM: Given an input directory, scan it, and write out the results
# to a NSIS nsi include file.
# 27/07/2012 - Initial cut
use strict;
use warnings;
use File::Basename;  # split path ($name,$dir,$ext) = fileparse($file [, qr/\.[^.]*/] )
use File::stat;
use Cwd;
my $os = $^O;
my $perl_dir = '/home/geoff/bin';
my $PATH_SEP = '/';
my $temp_dir = '/tmp';
if ($os =~ /win/i) {
    $perl_dir = 'C:\GTools\perl';
    $temp_dir = $perl_dir;
    $PATH_SEP = "\\";
}
unshift(@INC, $perl_dir);
require 'lib_utils.pl' or die "Unable to load 'lib_utils.pl' Check paths in \@INC...\n";
# log file stuff
our ($LF);
my $pgmname = $0;
if ($pgmname =~ /(\\|\/)/) {
    my @tmpsp = split(/(\\|\/)/,$pgmname);
    $pgmname = $tmpsp[-1];
}
my $outfile = $temp_dir.$PATH_SEP."temp.$pgmname.txt";
open_log($outfile);

# user variables
my $VERS = "0.0.1 2012-07-27";
my $load_log = 0;
my $in_file = '';
my $verbosity = 0;
my $out_file = '';

# ### DEBUG ###
my $debug_on = 0;
my $def_file = 'C:\FG\data';
my $def_out = $temp_dir.$PATH_SEP."temp.$pgmname.nsi";

### program variables
my @warnings = ();
my $cwd = cwd();
my %file_list = ();
sub process_in_dir($$);
my $total_files = 0;
my $total_dirs = 0;
my $total_bytes = 0;

sub VERB1() { return $verbosity >= 1; }
sub VERB2() { return $verbosity >= 2; }
sub VERB5() { return $verbosity >= 5; }
sub VERB9() { return $verbosity >= 9; }

sub show_warnings($) {
    my ($val) = @_;
    if (@warnings) {
        prt( "\nGot ".scalar @warnings." WARNINGS...\n" );
        foreach my $itm (@warnings) {
           prt("$itm\n");
        }
        prt("\n");
    } else {
        prt( "\nNo warnings issued.\n\n" ) if (VERB9());
    }
}

sub pgm_exit($$) {
    my ($val,$msg) = @_;
    if (length($msg)) {
        $msg .= "\n" if (!($msg =~ /\n$/));
        prt($msg);
    }
    show_warnings($val);
    close_log($outfile,$load_log);
    exit($val);
}


sub prtw($) {
   my ($tx) = shift;
   $tx =~ s/\n$//;
   prt("$tx\n");
   push(@warnings,$tx);
}

sub process_in_file($) {
    my ($inf) = @_;
    if (! open INF, "<$inf") {
        pgm_exit(1,"ERROR: Unable to open file [$inf]\n"); 
    }
    my @lines = <INF>;
    close INF;
    my $lncnt = scalar @lines;
    prt("Processing $lncnt lines, from [$inf]...\n");
    my ($line,$inc,$lnn);
    $lnn = 0;
    foreach $line (@lines) {
        chomp $line;
        $lnn++;
        if ($line =~ /\s*#\s*include\s+(.+)$/) {
            $inc = $1;
            prt("$lnn: $inc\n");
        }
    }
}

sub process_in_dir($$) {
    my ($dir,$lev) = @_;
    if (!opendir(DIR,$dir)) {
        prt("ERROR: Unable to open directory [$dir]\n");
        return;
    }
    my @files = readdir(DIR);
    closedir(DIR);
    my ($file,$ff,$ra,$sb);
    ut_fix_dir_string(\$dir);
    my @dirs = ();
    my $cnt = scalar @files;
    prt("Processing directory [$dir], with $cnt items...\n") if ($lev == 0);
    $total_dirs++;
    foreach $file (@files) {
        next if ($file eq '.');
        next if ($file eq '..');
        $ff = $dir.$file;
        if (-d $ff) {
            push(@dirs,$ff);
        } else {
            $file_list{$dir} = [] if (!defined $file_list{$dir});
            $total_files++;
            $ra = $file_list{$dir};
            push(@{$ra},$file);
            if ($sb = stat($ff)) {
                $total_bytes += $sb->size;
            }
        }
    }
    foreach $dir (@dirs) {
        process_in_dir($dir,($lev+1));
    }
    prt("Processed $total_dirs directories, for $total_files files...\n") if ($lev == 0);
}

sub sub_root_dir($) {
    my $dir = shift;
    my $len = length($in_file);
    $dir = substr($dir,($len+1));
    return $dir;
}

sub process_file_list() {
    my $cnt = 0;
    my $script = '';
    my $rh = \%file_list;
    prt("\nNow as NSIS script...\n");
    #  SetOutPath $INSTDIR\fgdata
    #  File ${FGData}\*
    my ($dir,$ra,$line);
    foreach $dir (sort keys %{$rh}) {
        $ra = ${$rh}{$dir};
        $dir = sub_root_dir($dir);
        $dir =~ s/\\$//;
        $dir = '' if ($dir eq '.');
        $script .= "  SetOutPath \$INSTDIR\\fgdata";
        $script .= "\\$dir" if (length($dir));
        $script .= "\n";
        foreach $line (sort @{$ra}) {
            $cnt++;
            prt("$cnt: $dir\\$line\n") if (VERB9());
            $script .= "  File \${FGData}\\";
            $script .= "$dir\\" if (length($dir));
            $script .= "$line\n";
        }
    }
    my $out = "# NSIS Script generated by $pgmname, on ";
    $out .= lu_get_YYYYMMDD_hhmmss(time());
    $out .= "\n";
    $out .= "# from a scan of directory [$in_file]\n";
    $out .= "# total ".get_nn($total_dirs)." directories, ";
    $out .= get_nn($cnt)." files - ";
    $out .= get_nn($total_bytes)." bytes.\n\n";
    $out .= $script;
    $out .= "\n# eof\n";
    if (length($out_file)) {
        write2file($out,$out_file);
        prt("Script written to [$out_file]\n");
    } else {
        prt($out);
        prt("No OUPUT file given. Use -o Out-File\n");
    }

}

#########################################
### MAIN ###
parse_args(@ARGV);
process_in_dir($in_file,0);
process_file_list();
pgm_exit(0,"");
########################################

sub need_arg {
    my ($arg,@av) = @_;
    pgm_exit(1,"ERROR: [$arg] must have a following argument!\n") if (!@av);
}

sub parse_args {
    my (@av) = @_;
    my ($arg,$sarg);
    while (@av) {
        $arg = $av[0];
        if ($arg =~ /^-/) {
            $sarg = substr($arg,1);
            $sarg = substr($sarg,1) while ($sarg =~ /^-/);
            if (($sarg =~ /^h/i)||($sarg eq '?')) {
                give_help();
                pgm_exit(0,"Help exit(0)");
            } elsif ($sarg =~ /^v/) {
                if ($sarg =~ /^v.*(\d+)$/) {
                    $verbosity = $1;
                } else {
                    while ($sarg =~ /^v/) {
                        $verbosity++;
                        $sarg = substr($sarg,1);
                    }
                }
                prt("Verbosity = $verbosity\n") if (VERB1());
            } elsif ($sarg =~ /^l/) {
                if ($sarg =~ /^ll/) {
                    $load_log = 2;
                } else {
                    $load_log = 1;
                }
                prt("Set to load log at end. ($load_log)\n") if (VERB1());
            } elsif ($sarg =~ /^o/) {
                need_arg(@av);
                shift @av;
                $sarg = $av[0];
                $out_file = $sarg;
                prt("Set out file to [$out_file].\n") if (VERB1());
            } else {
                pgm_exit(1,"ERROR: Invalid argument [$arg]! Try -?\n");
            }
        } else {
            $in_file = $arg;
            prt("Set input to [$in_file]\n") if (VERB1());
        }
        shift @av;
    }

    if ((length($in_file) ==  0) && $debug_on) {
        $in_file = $def_file;
        prt("Set DEFAULT input to [$in_file]\n");
    }
    if ((length($out_file) ==  0) && $debug_on) {
        $out_file = $def_out;
        prt("Set DEFAULT output to [$out_file]\n");
    }
    if (length($in_file) ==  0) {
        pgm_exit(1,"ERROR: No input directory found in command!\n");
    }
    if (! -d $in_file) {
        pgm_exit(1,"ERROR: Unable to find in directory [$in_file]! Check name, location...\n");
    }
}

sub give_help {
    prt("$pgmname: version $VERS\n");
    prt("Usage: $pgmname [options] in-directory\n");
    prt("Options:\n");
    prt(" --help  (-h or -?) = This help, and exit 0.\n");
    prt(" --verb[n]     (-v) = Bump [or set] verbosity. def=$verbosity\n");
    prt(" --load        (-l) = Load LOG at end. ($outfile)\n");
    prt(" --out <file>  (-o) = Write output to this file.\n");
}

# eof - template.pl
