#!/usr/bin/perl -w
# NAME: direct.pl
# AIM: Given a lat lon position in degrees, an azimuth (degs), and distance (Km), give end point
# 08/09/2011 geoff mclane http://geoffair.net/mperl
use strict;
use warnings;
use File::Basename;  # split path ($name,$dir,$ext) = fileparse($file [, qr/\.[^.]*/] )
use Cwd;
my $perl_dir = 'C:\GTools\perl';
unshift(@INC, $perl_dir);
require 'lib_utils.pl' or die "Unable to load 'lib_utils.pl' ...\n";
require 'fg_wsg84.pl' or die "Unable to load fg_wsg84.pl ...\n";

my $VERS = "0.0.1 2011-09-08"; # first version
# log file stuff
our ($LF);
my $pgmname = $0;
if ($pgmname =~ /(\\|\/)/) {
    my @tmpsp = split(/(\\|\/)/,$pgmname);
    $pgmname = $tmpsp[-1];
}
my $outfile = $perl_dir."\\temp.$pgmname.txt";
open_log($outfile);

# user variables
my $load_log = 0;

my $bad_latlon = -400;
my $m_lat = $bad_latlon;
my $m_lon = $bad_latlon;
my $m_az = $bad_latlon;
my $m_dist = $bad_latlon;
my $m_inter = $bad_latlon;

my $debug_on = 0;
my $def_lat = 40.6;
my $def_lon = -73.8;
my $def_az = 45;
my $def_dist = 10000;
my $def_int = 20;

### program variables
my @warnings = ();
my $cwd = cwd();
my $os = $^O;

sub show_warnings($) {
    my ($val) = @_;
    if (@warnings) {
        prt( "\nGot ".scalar @warnings." WARNINGS...\n" );
        foreach my $itm (@warnings) {
           prt("$itm\n");
        }
        prt("\n");
    } else {
        ###prt( "\nNo warnings issued.\n\n" );
    }
}

sub pgm_exit($$) {
    my ($val,$msg) = @_;
    if (length($msg)) {
        $msg .= "\n" if (!($msg =~ /\n$/));
        prt($msg);
    }
    show_warnings($val);
    close_log($outfile,$load_log);
    exit($val);
}


sub prtw($) {
   my ($tx) = shift;
   $tx =~ s/\n$//;
   prt("$tx\n");
   push(@warnings,$tx);
}

sub process_in_file($) {
    my ($inf) = @_;
    if (! open INF, "<$inf") {
        pgm_exit(1,"ERROR: Unable to open file [$inf]\n"); 
    }
    my @lines = <INF>;
    close INF;
    my $lncnt = scalar @lines;
    prt("Processing $lncnt lines, from [$inf]...\n");
    my ($line,$inc,$lnn);
    $lnn = 0;
    foreach $line (@lines) {
        chomp $line;
        $lnn++;
        if ($line =~ /\s*#\s*include\s+(.+)$/) {
            $inc = $1;
            prt("$lnn: $inc\n");
        }
    }
}

sub do_direct() {
    my ($lat2,$lon2,$az2);
    fg_geo_direct_wgs_84( $m_lat, $m_lon, $m_az, $m_dist * 1000, \$lat2, \$lon2, \$az2 );

    prt("From $m_lat,$m_lon, on azimuth $m_az, for $m_dist kilometers,\n");
    prt("you arrive at $lat2,$lon2, with azimuth $az2\n");

    if ($m_inter !=  $bad_latlon) {
        prt("Showing $m_inter intermediate points...\n");
        my $dist = $m_dist / $m_inter;
        my ($i,$i2);
        for ($i = 0; $i < $m_inter; $i++) {
            $i2 = $i + 1;
            fg_geo_direct_wgs_84( $m_lat, $m_lon, $m_az, $dist * 1000 * $i2, \$lat2, \$lon2, \$az2 );
            prt("$lat2,$lon2 $az2\n");
        }
    }

}

#########################################
### MAIN ###
parse_args(@ARGV);
# prt( "$pgmname: in [$cwd]: Hello, World...\n" );
# process_in_file($in_file);
do_direct();
pgm_exit(0,"");
########################################
sub give_help {
    prt("$pgmname: version $VERS\n");
    prt("Usage: $pgmname [options] lat lon azimuth dist_km [interval]\n");
    prt("Options:\n");
    prt(" --help (-h or -?) = This help, and exit 0.\n");
    prt("Given a position, lat lon, an azimuth, and a distance in kilometers,\n");
    prt("display the end point.\n");
    prt("If an interval is given, also output the list of intermediate points.\n");
}
sub need_arg {
    my ($arg,@av) = @_;
    pgm_exit(1,"ERROR: [$arg] must have following argument!\n") if (!@av);
}

sub parse_args {
    my (@av) = @_;
    my ($arg,$sarg);
    my $cnt = 0;
    while (@av) {
        $arg = $av[0];
        if (($arg =~ /^-/) && !($arg =~ /^-\d+/)) {
            $sarg = substr($arg,1);
            $sarg = substr($sarg,1) while ($sarg =~ /^-/);
            if (($sarg =~ /^h/i)||($sarg eq '?')) {
                give_help();
                pgm_exit(0,"Help exit(0)");
            } else {
                pgm_exit(1,"ERROR: Invalid argument [$arg]! Try -?\n");
            }
        } else {
            if ($cnt == 0) {
                $m_lat = $arg;
            } elsif ($cnt == 1) {
                $m_lon = $arg;
            } elsif ($cnt == 2) {
                $m_az = $arg;
            } elsif ($cnt == 3) {
                $m_dist = $arg;
            } elsif ($cnt == 4) {
                $m_inter = $arg;
            } else {
                pgm_exit(1,"ERROR: Invalid argument [$arg]! Try -?\n");
            }
            $cnt++;
        }
        shift @av;
    }

    if ($debug_on) {
        prtw("WARNING: DEBUG is ON\n");
        if ($m_lat ==  $bad_latlon) {
            $m_lat = $def_lat;
            $m_lon = $def_lon;
            $m_az = $def_az;
            $m_dist = $def_dist;
            $m_inter = $def_int;
            prt("Set DEFAULT lat=$m_lat lon=$m_lon az=$m_az dist=$m_dist inter=$m_inter\n");
        }
    }
    if ($m_dist ==  $bad_latlon) {
        give_help();
        pgm_exit(1,"ERROR: No input of lat, lon, az, dist...!\n");
    }
}

# eof - direct.pl
