#!/usr/bin/perl -w
# NAME: findac.pl
# AIM: Given an --fg-aicraft=/FG_ROOT/Aircraft enumerate the available model ac files
use strict;
use warnings;
use File::Basename;  # split path ($name,$dir,$ext) = fileparse($file [, qr/\.[^.]*/] )
use File::Spec; # File::Spec->rel2abs($rel);
use Cwd;
my $os = $^O;
my $perl_dir = '/home/geoff/bin';
my $PATH_SEP = '/';
my $temp_dir = '/tmp';
if ($os =~ /win/i) {
    $perl_dir = 'C:\GTools\perl';
    $temp_dir = $perl_dir;
    $PATH_SEP = "\\";
}
unshift(@INC, $perl_dir);
require 'lib_utils.pl' or die "Unable to load 'lib_utils.pl' Check paths in \@INC...\n";
# log file stuff
our ($LF);
my $pgmname = $0;
if ($pgmname =~ /(\\|\/)/) {
    my @tmpsp = split(/(\\|\/)/,$pgmname);
    $pgmname = $tmpsp[-1];
}
my $outfile = $temp_dir.$PATH_SEP."temp.$pgmname.txt";
open_log($outfile);

my $FGROOT = (exists $ENV{'FG_ROOT'})? $ENV{'FG_ROOT'} : "X:/fgdata";

# user variables
my $VERS = "0.0.3 2015-06-22";
#my $VERS = "0.0.2 2014-11-11";
my $load_log = 0;
my $in_dir = '';
my $verbosity = 0;
my $out_file = $temp_dir.$PATH_SEP."tempmodels.csv";

# ### DEBUG ###
my $debug_on = 0;
my $def_file = '$FGROOT.$PATH_SEP."Aircraft"';

### program variables
my @warnings = ();
my $cwd = cwd();

sub VERB1() { return $verbosity >= 1; }
sub VERB2() { return $verbosity >= 2; }
sub VERB5() { return $verbosity >= 5; }
sub VERB9() { return $verbosity >= 9; }

sub show_warnings($) {
    my ($val) = @_;
    if (@warnings) {
        prt( "\nGot ".scalar @warnings." WARNINGS...\n" );
        foreach my $itm (@warnings) {
           prt("$itm\n");
        }
        prt("\n");
    } else {
        prt( "\nNo warnings issued.\n\n" ) if (VERB9());
    }
}

sub pgm_exit($$) {
    my ($val,$msg) = @_;
    if (length($msg)) {
        $msg .= "\n" if (!($msg =~ /\n$/));
        prt($msg);
    }
    show_warnings($val);
    close_log($outfile,$load_log);
    exit($val);
}


sub prtw($) {
   my ($tx) = shift;
   $tx =~ s/\n$//;
   prt("$tx\n");
   push(@warnings,$tx);
}

sub process_in_file($) {
    my ($inf) = @_;
    if (! open INF, "<$inf") {
        pgm_exit(1,"ERROR: Unable to open file [$inf]\n"); 
    }
    my @lines = <INF>;
    close INF;
    my $lncnt = scalar @lines;
    prt("Processing $lncnt lines, from [$inf]...\n");
    my ($line,$inc,$lnn);
    $lnn = 0;
    foreach $line (@lines) {
        chomp $line;
        $lnn++;
        if ($line =~ /\s*#\s*include\s+(.+)$/) {
            $inc = $1;
            prt("$lnn: $inc\n");
        }
    }
}

###my $root_dir = '';
my %aircraft_ac = ();

sub show_ac_list() {
    my @arr = sort keys %aircraft_ac;
    my ($key,$ra,$cnt);
    $cnt = scalar @arr;
    my @failed = ();
    prt("Processing $cnt aircraft directories...\n");
    my $total = 0;
    my $msg = '';
    foreach $key (@arr) {
        $ra = $aircraft_ac{$key};
        $cnt = scalar @{$ra};
        if ($cnt) {
            $msg .= join("\n",@{$ra})."\n";
            $total += $cnt;
        } else {
            push(@failed,$key);
        }
    }
    if (length($out_file)) {
        write2file($msg,$out_file);
        prt("List of ac files written to $out_file\n");
    } else {
        prt($msg);
    }
    prt("Successfully found $total ac files matching dir name...\n");

    $cnt = scalar @failed;
    if ($cnt) {
        if (VERB5()) {
            prt("Did NOT find an 'ac' file in $cnt directories...\n");
            prt(join(" ",@failed)."\n");
        }
    }

}

sub process_in_dir($$);

my $min_match = 3;
my %exclude_dirs = ();


sub process_in_dir($$) {
    my ($ind,$lev) = @_;
    if (! opendir(IND, "$ind")) {
        pgm_exit(1,"ERROR: Unable to open directtory [$ind]\n"); 
    }
    my @files = readdir(IND);
    closedir(IND);
    my $lncnt = scalar @files;
    prt("Processing $lncnt file items from [$ind]...\n") if (VERB9());
    my ($file,$ff,$name,$ra,$ra2);
    ut_fix_directory(\$ind);
    $aircraft_ac{$ind} = [] if ( ! defined $aircraft_ac{$ind});
    foreach $file (@files) {
        next if ($file eq '.');
        next if ($file eq '..');
        $ff = $ind.$file;
        if ($os =~ /win/i) {
            $ff = path_u2d($ff);
        } else {
            $ff = path_d2u($ff);
        }
        if (-d $ff) {
            if (! defined $exclude_dirs{$file} ) {
                process_in_dir($ff,($lev+1));
            }
        } elsif (-f $ff) {
            if ($file =~ /^(.+)\.ac$/) {
                $name = $1;
                prt("$name,$ff\n") if (VERB5());
                $ra = $aircraft_ac{$ind};
                push(@{$ra},$ff);
            }
        } else {
            pgm_exit(1,"What is this '$ff' ($file)! *** FIX ME ***\n");
        }
    }
}


#########################################
### MAIN ###
parse_args(@ARGV);
prt("Processing directory $in_dir for model ac files...\n");
process_in_dir($in_dir,0);
show_ac_list();
pgm_exit(0,"");
########################################

sub need_arg {
    my ($arg,@av) = @_;
    pgm_exit(1,"ERROR: [$arg] must have a following argument!\n") if (!@av);
}

sub parse_args {
    my (@av) = @_;
    my ($arg,$sarg);
    my $verb = VERB2();
    while (@av) {
        $arg = $av[0];
        if ($arg =~ /^-/) {
            $sarg = substr($arg,1);
            $sarg = substr($sarg,1) while ($sarg =~ /^-/);
            if (($sarg =~ /^h/i)||($sarg eq '?')) {
                give_help();
                pgm_exit(0,"Help exit(0)");
            } elsif ($sarg =~ /^v/) {
                if ($sarg =~ /^v.*(\d+)$/) {
                    $verbosity = $1;
                } else {
                    while ($sarg =~ /^v/) {
                        $verbosity++;
                        $sarg = substr($sarg,1);
                    }
                }
                $verb = VERB2();
                prt("Verbosity = $verbosity\n") if ($verb);
            } elsif ($sarg =~ /^l/) {
                if ($sarg =~ /^ll/) {
                    $load_log = 2;
                } else {
                    $load_log = 1;
                }
                prt("Set to load log at end. ($load_log)\n") if ($verb);
            } elsif ($sarg =~ /^o/) {
                need_arg(@av);
                shift @av;
                $sarg = $av[0];
                $out_file = $sarg;
                prt("Set out file to [$out_file].\n") if ($verb);
            } else {
                pgm_exit(1,"ERROR: Invalid argument [$arg]! Try -?\n");
            }
        } else {
            if (length($in_dir)) {
                pgm_exit(1,"Error: Already have '$in_dir'. What is this '$arg'?\n");
            }
            $in_dir = File::Spec->rel2abs($arg);
            prt("Set input to [$in_dir]\n") if ($verb);
        }
        shift @av;
    }

    if ($debug_on) {
        prtw("WARNING: DEBUG is ON!\n");
        if (length($in_dir) ==  0) {
            $in_dir = $def_file;
            prt("Set DEFAULT input to [$in_dir]\n");
        }
    }
    if (length($in_dir) ==  0) {
        give_help();
        pgm_exit(1,"ERROR: No input directory found in command!\n");
    }
    if (! -d $in_dir) {
        pgm_exit(1,"ERROR: Unable to 'stat' directory [$in_dir]! Check name, location...\n");
    }
}

sub give_help {
    prt("$pgmname: version $VERS\n");
    prt("Usage: $pgmname [options] in-file\n");
    prt("Options:\n");
    prt(" --help  (-h or -?) = This help, and exit 0.\n");
    prt(" --verb[n]     (-v) = Bump [or set] verbosity. def=$verbosity\n");
    prt(" --load        (-l) = Load LOG at end. ($outfile)\n");
    prt(" --out <file>  (-o) = Write output to this file.\n");
    prt("\n");
    prt(" Simple scan the input directory for 'aero'-set.xml files, and output the list found.\n");
    prt("\n");
    prt(" See fgsetfile.pl to process a FG *-set.xml file and try to find the model ac file...\n"); 
    prt(" See findset.pl, scan the input directory for 'aero'-set.xml files, and output the list found.\n");
    prt(" See fgsetlist.pl, which perversely does the same as the above file!!!\n");
    prt(" Can use fgxmlset.exe, hfgxmlsetmf.bat, to view contents of the 'set' files.\n");
    prt(" See fgaclist.pl to output the full list of *.ac files found in a directory.\n");
    prt(" See findmodel.pl to scan dir looking for model ac files.\n");
    #prt(" See findac.pl, scan dir for model .ac file and output a list, like fgaclist.pl.\n");
    prt("\n");

}

# eof - findac.pl
