#!/usr/bin/perl -w
# NAME: findfiles.pl
# AIM: Given a search directory, and a file name, or set of files, find then 
# and report their locations in the directory structure.
use strict;
use warnings;
use File::Basename;  # split path ($name,$dir,$ext) = fileparse($file [, qr/\.[^.]*/] )
use Cwd;
my $os = $^O;
my $perl_dir = '/home/geoff/bin';
my $PATH_SEP = '/';
my $temp_dir = '/tmp';
if ($os =~ /win/i) {
    $perl_dir = 'C:\GTools\perl';
    $temp_dir = $perl_dir;
    $PATH_SEP = "\\";
}
unshift(@INC, $perl_dir);
require 'lib_utils.pl' or die "Unable to load 'lib_utils.pl' Check paths in \@INC...\n";
# log file stuff
our ($LF);
my $pgmname = $0;
if ($pgmname =~ /(\\|\/)/) {
    my @tmpsp = split(/(\\|\/)/,$pgmname);
    $pgmname = $tmpsp[-1];
}
my $outfile = $temp_dir.$PATH_SEP."temp.$pgmname.txt";
open_log($outfile);

# user variables
my $VERS = "0.0.1 2013-03-17";
my $load_log = 0;
my $in_file = '';
my $in_dir = '';
my $verbosity = 0;
my $out_file = '';
my $show_relative = 1;
my $show_linux = 1;
my @in_list = ();

# ### DEBUG ###
my $debug_on = 0;
my $def_file = 'cplkeywordparser.h';
my $def_dir = 'c:\FG\17\gdal-1.9.1';
### program variables
my @warnings = ();
my $cwd = cwd();

sub VERB1() { return $verbosity >= 1; }
sub VERB2() { return $verbosity >= 2; }
sub VERB5() { return $verbosity >= 5; }
sub VERB9() { return $verbosity >= 9; }

sub show_warnings($) {
    my ($val) = @_;
    if (@warnings) {
        prt( "\nGot ".scalar @warnings." WARNINGS...\n" );
        foreach my $itm (@warnings) {
           prt("$itm\n");
        }
        prt("\n");
    } else {
        prt( "\nNo warnings issued.\n\n" ) if (VERB9());
    }
}

sub pgm_exit($$) {
    my ($val,$msg) = @_;
    if (length($msg)) {
        $msg .= "\n" if (!($msg =~ /\n$/));
        prt($msg);
    }
    show_warnings($val);
    close_log($outfile,$load_log);
    exit($val);
}


sub prtw($) {
   my ($tx) = shift;
   $tx =~ s/\n$//;
   prt("$tx\n");
   push(@warnings,$tx);
}

my @finds = ();
sub process_in_dir($);

sub sub_in_directory($) {
    my $ff = shift;
    my $l1 = length($ff);
    my $l2 = length($in_dir);
    if ($l1 > $l2) {
        return substr($ff,$l2+1);
    }
    return $ff;
}

my %find_hash = ();

sub is_in_list($) {
    my $file = shift;
    my ($tst,$fnd);
    foreach $tst (@in_list) {
        $fnd = ($os =~ /win/i)  ? lc($tst) : $tst;
        if ($file eq $fnd) {
            $find_hash{$tst} = 1;
            return 1;
        }
    }
    return 0;
}

my $files_found = '';

sub process_in_dir($) {
    my $dir = shift;
    if (!opendir(DIR,$dir)) {
        prtw("WARNING: Failed to open directory [$dir]\n");
        return;
    }
    my @files = readdir(DIR);
    closedir(DIR);
    my ($file,$ff);
    my @dirs = ();
    ut_fix_directory(\$dir);
    foreach $file (@files) {
        next if ($file eq '.');
        next if ($file eq '..');
        $ff = $dir.$file;
        if (-d $ff) {
            push(@dirs,$ff);
        } elsif (-f $ff) {
            $file = lc($file) if ($os =~ /win/i);
            if (is_in_list($file)) {
                push(@finds,$ff);
                $ff = sub_in_directory($ff) if ($show_relative);
                $ff = path_d2u($ff) if ($show_linux);
                prt("Found: $ff\n");
                $files_found .= "Found: $ff\n";
            }
        } else {
            prtw("WARNING: What is THIS [$ff]!\n");
        }
    }
    foreach $dir (@dirs) {
        process_in_dir($dir);
    }
}

sub show_not_found() {
    my ($file);
    foreach $file (@in_list) {
        if (!defined $find_hash{$file}) {
            prt("NOT FOUND: $file\n");
            $files_found .= "NOT FOUND: $file\n";
        }
    }
    if (length($out_file) && length($files_found)) {
        write2file($files_found,$out_file);
        prt("List of found, and NOT found, written to [$out_file]\n");
    }
}

sub process_in_file($) {
    my ($inf) = @_;
    if (! open INF, "<$inf") {
        pgm_exit(1,"ERROR: Unable to open file [$inf]\n"); 
    }
    my @lines = <INF>;
    close INF;
    my $lncnt = scalar @lines;
    prt("Processing $lncnt lines, from [$inf]...\n");
    my ($line,$inc,$lnn);
    $lnn = 0;
    foreach $line (@lines) {
        chomp $line;
        $lnn++;
        if ($line =~ /\s*#\s*include\s+(.+)$/) {
            $inc = $1;
            prt("$lnn: $inc\n");
        }
    }
}

#########################################
### MAIN ###
parse_args(@ARGV);
process_in_dir($in_dir);
show_not_found();
pgm_exit(0,"");
########################################

sub need_arg {
    my ($arg,@av) = @_;
    pgm_exit(1,"ERROR: [$arg] must have a following argument!\n") if (!@av);
}

sub load_file_list($) {
    my $file = shift;
    pgm_exit(1,"ERROR: Unable to open [$file]\n") if (!open FIL,"<$file");
    my @lines = <FIL>;
    close FIL;
    my ($line,$cnt);
    $cnt = 0;
    foreach $line (@lines) {
        chomp $line;
        $line = trim_all($line);
        next if (length($line) == 0);
        push(@in_list,$line);
        $cnt++;
        $in_file = $line;
    }
    prt("Added $cnt files to find list.\n") if (VERB1());
}


sub parse_args {
    my (@av) = @_;
    my ($arg,$sarg);
    my $cnt = 0;
    while (@av) {
        $arg = $av[0];
        if ($arg =~ /^-/) {
            $sarg = substr($arg,1);
            $sarg = substr($sarg,1) while ($sarg =~ /^-/);
            if (($sarg =~ /^h/i)||($sarg eq '?')) {
                give_help();
                pgm_exit(0,"Help exit(0)");
            } elsif ($sarg =~ /^v/) {
                if ($sarg =~ /^v.*(\d+)$/) {
                    $verbosity = $1;
                } else {
                    while ($sarg =~ /^v/) {
                        $verbosity++;
                        $sarg = substr($sarg,1);
                    }
                }
                prt("Verbosity = $verbosity\n") if (VERB1());
            } elsif ($sarg =~ /^l/) {
                if ($sarg =~ /^ll/) {
                    $load_log = 2;
                } else {
                    $load_log = 1;
                }
                prt("Set to load log at end. ($load_log)\n") if (VERB1());
            } elsif ($sarg =~ /^f/) {
                need_arg(@av);
                shift @av;
                $sarg = $av[0];
                $in_file = $sarg;
                $cnt++;
                push(@in_list,$in_file);
                prt("Set file to [$in_file].\n") if (VERB1());
            } elsif ($sarg =~ /^d/) {
                need_arg(@av);
                shift @av;
                $sarg = $av[0];
                $in_dir = $sarg;
                $cnt++;
                prt("Set directory to [$in_dir].\n") if (VERB1());
            } elsif ($sarg =~ /^i/) {
                need_arg(@av);
                shift @av;
                $sarg = $av[0];
                load_file_list($sarg);
            } elsif ($sarg =~ /^o/) {
                need_arg(@av);
                shift @av;
                $sarg = $av[0];
                $out_file = $sarg;
                prt("Set out file to [$out_file].\n") if (VERB1());
            } else {
                pgm_exit(1,"ERROR: Invalid argument [$arg]! Try -?\n");
            }
        } else {
            if ($cnt == 0) {
                $in_file = $arg;
                push(@in_list,$in_file);
                prt("Set input to [$in_file]\n") if (VERB1());
            } elsif ($cnt == 1) {
                $in_dir = $arg;
                prt("Set directory to [$in_dir]\n") if (VERB1());
            } else {
                pgm_exit(1,"ERROR: Already have file [$in_file], and directory [$in_dir]!\n".
                    "What is this [$arg]?\n");
            }
            $cnt++;
        }
        shift @av;
    }

    if ($debug_on) {
        prtw("WARNING: DEBUG is ON!\n");
        if (length($in_file) ==  0) {
            $in_file = $def_file;
            push(@in_list,$in_file);
            prt("Set DEFAULT input to [$in_file]\n");
            push(@in_list,"rubbish");
        }
        if (length($in_dir) ==  0) {
            $in_dir = $def_dir;
            prt("Set DEFAULT directory to [$in_dir]\n");
        }
    }
    if (length($in_file) ==  0) {
        pgm_exit(1,"ERROR: No input files found in command!\n");
    }
    if (length($in_dir) == 0) {
        pgm_exit(1,"ERROR: No input directory found in command!\n");
    }
    if (! -d $in_dir) {
        pgm_exit(1,"ERROR: Unable to find directory [$in_dir]! Check name, location...\n");
    }
}

sub give_help {
    prt("$pgmname: version $VERS\n");
    prt("Usage: $pgmname [options] in-file in-dir\n");
    prt("Options:\n");
    prt(" --help  (-h or -?) = This help, and exit 0.\n");
    prt(" --verb[n]     (-v) = Bump [or set] verbosity. def=$verbosity\n");
    prt(" --load        (-l) = Load LOG at end. ($outfile)\n");
    prt(" --file <file> (-f) = Give the file name to search for.\n");
    prt(" --dir <dir>   (-d) = Give the directory to search in, recursively.\n");
    prt(" --inp <file>  (-i) = Find all files listed in this input.\n");
    prt(" --out <file>  (-o) = Write output to this file.\n");
    prt("AIM: Given a file name and a search directory, find and report location.\n");
}

# eof - template.pl
