#!/usr/bin/perl -w
# NAME: findset.pl
# AIM: Given an --fg-aicraft=/FG_ROOT/Aircraft enumerate the available aircraft,
# by scanning for 'aero'-set.xml files...
# See fgaclist.pl to output the full list of *.ac files found in a directory.
# See fgsetfile.pl to process a FG *-set.xml file and try to find the model ac file, 
# using XML::Simple and load it into ac3dview.exe
# Also see: F:\FG\18\fgxmlset - hfgxmlsetf.bat and hfgxmlsetmf.bat - A C/C++ aero-set.xml parser
use strict;
use warnings;
use File::Basename;  # split path ($name,$dir,$ext) = fileparse($file [, qr/\.[^.]*/] )
use Cwd;
my $os = $^O;
my $perl_dir = '/home/geoff/bin';
my $PATH_SEP = '/';
my $temp_dir = '/tmp';
if ($os =~ /win/i) {
    $perl_dir = 'C:\GTools\perl';
    $temp_dir = $perl_dir;
    $PATH_SEP = "\\";
}
unshift(@INC, $perl_dir);
require 'lib_utils.pl' or die "Unable to load 'lib_utils.pl' Check paths in \@INC...\n";
# log file stuff
our ($LF);
my $pgmname = $0;
if ($pgmname =~ /(\\|\/)/) {
    my @tmpsp = split(/(\\|\/)/,$pgmname);
    $pgmname = $tmpsp[-1];
}
my $outfile = $temp_dir.$PATH_SEP."temp.$pgmname.txt";
open_log($outfile);

my $FGROOT = (exists $ENV{'FG_ROOT'})? $ENV{'FG_ROOT'} : "X:/fgdata";

# user variables
my $VERS = "0.0.2 2014-11-11";
my $load_log = 0;
#my $in_dir = $FGROOT.$PATH_SEP."Aircraft";
my $in_dir = '';
### hmmm, many of these AI/Aircraft do NOT have a '*-set.xml' file...
###my $in_dir = $FGROOT.$PATH_SEP."AI".$PATH_SEP."Aircraft";
my $verbosity = 0;
my $out_file = $temp_dir.$PATH_SEP."tempaero.csv";

# ### DEBUG ###
my $debug_on = 0;
my $def_file = $FGROOT.$PATH_SEP."Aircraft";

### program variables
my @warnings = ();
my $cwd = cwd();

sub VERB1() { return $verbosity >= 1; }
sub VERB2() { return $verbosity >= 2; }
sub VERB5() { return $verbosity >= 5; }
sub VERB9() { return $verbosity >= 9; }

sub show_warnings($) {
    my ($val) = @_;
    if (@warnings) {
        prt( "\nGot ".scalar @warnings." WARNINGS...\n" );
        foreach my $itm (@warnings) {
           prt("$itm\n");
        }
        prt("\n");
    } else {
        prt( "\nNo warnings issued.\n\n" ) if (VERB9());
    }
}

sub pgm_exit($$) {
    my ($val,$msg) = @_;
    if (length($msg)) {
        $msg .= "\n" if (!($msg =~ /\n$/));
        prt($msg);
    }
    show_warnings($val);
    close_log($outfile,$load_log);
    exit($val);
}


sub prtw($) {
   my ($tx) = shift;
   $tx =~ s/\n$//;
   prt("$tx\n");
   push(@warnings,$tx);
}

sub process_in_file($) {
    my ($inf) = @_;
    if (! open INF, "<$inf") {
        pgm_exit(1,"ERROR: Unable to open file [$inf]\n"); 
    }
    my @lines = <INF>;
    close INF;
    my $lncnt = scalar @lines;
    prt("Processing $lncnt lines, from [$inf]...\n");
    my ($line,$inc,$lnn);
    $lnn = 0;
    foreach $line (@lines) {
        chomp $line;
        $lnn++;
        if ($line =~ /\s*#\s*include\s+(.+)$/) {
            $inc = $1;
            prt("$lnn: $inc\n");
        }
    }
}

my $root_dir = '';
my %aircraft_set = ();
my %aircraft_aero = ();

sub process_sets() {
    my @arr = sort keys %aircraft_set;
    my ($key,$ra,$cnt);
    $cnt = scalar @arr;
    my @failed = ();
    prt("Processing $cnt aircraft directories...\n");
    my $total = 0;
    foreach $key (@arr) {
        $ra = $aircraft_set{$key};
        $cnt = scalar @{$ra};
        if ($cnt) {
            prt(join("\n",@{$ra})."\n");
            $total += $cnt;
        } else {
            push(@failed,$key);
        }
    }
    prt("Successfully found $total set files matching dir name...\n");
    $cnt = scalar @failed;
    if ($cnt) {
        prt("Did NOT find a 'set' file in $cnt directories...\n");
        prt(join(" ",@failed)."\n");
    }

}
sub process_aero() {
    my @arr = sort keys %aircraft_aero;
    my ($key,$ra,$cnt,$ra2,$line);
    $cnt = scalar @arr;
    my @failed = ();
    prt("Processing $cnt aircraft directories...\n");
    my $total = 0;
    my $aero_csv = "aero,set\n";
    foreach $key (@arr) {
        $ra = $aircraft_aero{$key};
        $cnt = scalar @{$ra};
        if ($cnt) {
            foreach $ra2 (@{$ra}) {
                $line = ${$ra2}[0].",".${$ra2}[1];
                prt("$line\n") if (VERB1());
                $aero_csv .= "$line\n";
            }
            $total += $cnt;
        } else {
            push(@failed,$key);
        }
    }
    prt("Successfully found $total set files matching dir name...\n");
    if (length($out_file)) {
        write2file($aero_csv,$out_file);
        prt("Aero list written to $out_file\n");
    }
    $cnt = scalar @failed;
    if ($cnt) {
        prt("Did NOT find a 'set' file in $cnt directories...\n");
        prt(join(" ",@failed)."\n");
    }

}

sub process_in_dir($$);

my $min_match = 3;
my %exclude_dirs = (
    "Generic" => 1,
    "Instruments" => 1,
    "Instruments-3d" => 1,
    "777-200" => 1,
    "UIUC" => 1,
    "c172" => 1,
    "tu134" => 1
    );

sub match_first($$$);
sub match_first($$$) {
    my ($name,$dir,$lev) = @_;
    my $len1 = length($name);
    my $len2 = length($dir);
    return 0 if ($len1 < $min_match);
    return 0 if ($len2 < $min_match);
    $name = lc($name);
    $dir = lc($dir);
    my ($i,$ch1,$ch2,$cnt);
    $cnt = 0;
    for ($i = 0; $i < $min_match; $i++) {
        $ch1 = substr($name,$i,1);
        $ch2 = substr($dir,$i,1);
        last if ($ch1 ne $ch2);
        $cnt++;
    }
    return 1 if ($cnt >= $min_match);
    if ($lev == 0) {
        if ($dir =~ /\-/) {
            $dir =~ s/\-//g;
            return match_first($name,$dir,($lev+1));
        }
    }
    return 0;
}


sub process_in_dir($$) {
    my ($ind,$lev) = @_;
    if (! opendir(IND, "$ind")) {
        pgm_exit(1,"ERROR: Unable to open directtory [$ind]\n"); 
    }
    my @files = readdir(IND);
    closedir(IND);
    my $lncnt = scalar @files;
    prt("Processing $lncnt file items from [$ind]...\n") if (VERB9());
    my ($file,$ff,$name,$ra,$ra2);
    ut_fix_directory(\$ind);
    foreach $file (@files) {
        next if ($file eq '.');
        next if ($file eq '..');
        $ff = $ind.$file;
        if ($os =~ /win/i) {
            $ff = path_u2d($ff);
        } else {
            $ff = path_d2u($ff);
        }
        if (-d $ff) {
            if (($lev == 0) && ! defined $exclude_dirs{$file} ) {
                $root_dir = $file;
                $aircraft_set{$root_dir} = [] if ( ! defined $aircraft_set{$root_dir});
                $aircraft_aero{$root_dir} = [] if ( ! defined $aircraft_aero{$root_dir});
                process_in_dir($ff,($lev+1));
            }
        } elsif (-f $ff) {
            if ($lev == 1) {
                if ($file =~ /^([\w\-]+)-set\.xml$/) {
                    $name = $1;
                    ###if ($name =~ /$root_dir/) {
                        prt("$name,$ff\n") if (VERB5());
                        $ra = $aircraft_set{$root_dir};
                        push(@{$ra},$ff);
                        $ra2 = $aircraft_aero{$root_dir};
                        push(@{$ra2},[$name,$ff]);
                    ###} elsif (match_first($name,$root_dir,0)) {
                    ###    prt("$ff\n");
                    ###    $ra = $aircraft_set{$root_dir};
                    ###    push(@{$ra},$ff);
                    ##}
                }
            }
        } else {
            pgm_exit(1,"What is this '$ff' ($file)! *** FIX ME ***\n");
        }
    }

}


#########################################
### MAIN ###
parse_args(@ARGV);
prt("Processing directory $in_dir for aero -set.xml files...\n");
process_in_dir($in_dir,0);
###process_sets();
process_aero();
pgm_exit(0,"");
########################################

sub need_arg {
    my ($arg,@av) = @_;
    pgm_exit(1,"ERROR: [$arg] must have a following argument!\n") if (!@av);
}

sub parse_args {
    my (@av) = @_;
    my ($arg,$sarg);
    my $verb = VERB2();
    while (@av) {
        $arg = $av[0];
        if ($arg =~ /^-/) {
            $sarg = substr($arg,1);
            $sarg = substr($sarg,1) while ($sarg =~ /^-/);
            if (($sarg =~ /^h/i)||($sarg eq '?')) {
                give_help();
                pgm_exit(0,"Help exit(0)");
            } elsif ($sarg =~ /^v/) {
                if ($sarg =~ /^v.*(\d+)$/) {
                    $verbosity = $1;
                } else {
                    while ($sarg =~ /^v/) {
                        $verbosity++;
                        $sarg = substr($sarg,1);
                    }
                }
                $verb = VERB2();
                prt("Verbosity = $verbosity\n") if ($verb);
            } elsif ($sarg =~ /^l/) {
                if ($sarg =~ /^ll/) {
                    $load_log = 2;
                } else {
                    $load_log = 1;
                }
                prt("Set to load log at end. ($load_log)\n") if ($verb);
            } elsif ($sarg =~ /^o/) {
                need_arg(@av);
                shift @av;
                $sarg = $av[0];
                $out_file = $sarg;
                prt("Set out file to [$out_file].\n") if ($verb);
            } else {
                pgm_exit(1,"ERROR: Invalid argument [$arg]! Try -?\n");
            }
        } else {
            $in_dir = $arg;
            prt("Set input to [$in_dir]\n") if ($verb);
        }
        shift @av;
    }

    if ($debug_on) {
        prtw("WARNING: DEBUG is ON!\n");
        if (length($in_dir) ==  0) {
            $in_dir = $def_file;
            prt("Set DEFAULT input to [$in_dir]\n");
        }
    }
    if (length($in_dir) ==  0) {
        give_help();
        pgm_exit(1,"ERROR: No input directory found in command!\n");
    }
    if (! -d $in_dir) {
        pgm_exit(1,"ERROR: Unable to 'stat' directory [$in_dir]! Check name, location...\n");
    }
}

sub give_help {
    prt("$pgmname: version $VERS\n");
    prt("Usage: $pgmname [options] [in-dir]\n");
    prt("Options:\n");
    prt(" --help  (-h or -?) = This help, and exit 0.\n");
    prt(" --verb[n]     (-v) = Bump [or set] verbosity. def=$verbosity\n");
    prt(" --load        (-l) = Load LOG at end. ($outfile)\n");
    prt(" --out <file>  (-o) = Write csv aero,set output to this file.\n");
    prt(" Default=$out_file\n");
    prt("\n");
    prt(" Simple scan the input directory for 'aero'-set.xml files, and output the list found.\n");
    prt("\n");
    prt(" See fgsetfile.pl to process a FG *-set.xml file and try to find the model ac file...\n"); 
    #prt(" See findset.pl, scan the input directory for 'aero'-set.xml files, and output the list found.\n");
    prt(" See fgsetlist.pl, which perversely does the same as the above file!!!\n");
    prt(" Can use fgxmlset.exe, hfgxmlsetmf.bat, to view contents of the 'set' files.\n");
    prt(" See fgaclist.pl to output the full list of *.ac files found in a directory.\n");
    prt(" See findmodel.pl to scan dir looking for model ac files.\n");
    prt(" See findac.pl, scan dir for model .ac file and output a list, like fgaclist.pl.\n");
    prt("\n");

}

# eof - findset.pl
