#!/usr/bin/perl -w
# NAME: gensrclist2.pl
# AIM: Given a FOLDER input, generate a source list of files in the folder,
# EXCLUDING those with extension in the @excluded_exts list.
# Full re-write of gensrclist.pl
use strict;
use warnings;
use File::Basename;  # split path ($name,$dir,$ext) = fileparse($file [, qr/\.[^.]*/] )
use Cwd;
my $os = $^O;
my $perl_dir = '/home/geoff/bin';
my $PATH_SEP = '/';
my $temp_dir = '/tmp';
if ($os =~ /win/i) {
    $perl_dir = 'C:\GTools\perl';
    $temp_dir = $perl_dir;
    $PATH_SEP = "\\";
}
unshift(@INC, $perl_dir);
require 'lib_utils.pl' or die "Unable to load 'lib_utils.pl' Check paths in \@INC...\n";
# log file stuff
our ($LF);
my $pgmname = $0;
if ($pgmname =~ /(\\|\/)/) {
    my @tmpsp = split(/(\\|\/)/,$pgmname);
    $pgmname = $tmpsp[-1];
}
my $outfile = $temp_dir.$PATH_SEP."temp.$pgmname.txt";
my $tmpout  = $temp_dir.$PATH_SEP."tempcmake.txt";

open_log($outfile);

# user variables
my $VERS = "0.0.2 2014-01-13";
my $load_log = 0;
my $in_dir = '';
my $verbosity = 0;
my $out_file = '';

# ### DEBUG ###
my $debug_on = 1;
my $def_dir = 'F:\Projects\xerces-c-3.1.1\src';

### program variables
my @warnings = ();
my $cwd = cwd();

sub VERB1() { return $verbosity >= 1; }
sub VERB2() { return $verbosity >= 2; }
sub VERB5() { return $verbosity >= 5; }
sub VERB9() { return $verbosity >= 9; }

sub show_warnings($) {
    my ($val) = @_;
    if (@warnings) {
        prt( "\nGot ".scalar @warnings." WARNINGS...\n" );
        foreach my $itm (@warnings) {
           prt("$itm\n");
        }
        prt("\n");
    } else {
        prt( "\nNo warnings issued.\n\n" ) if (VERB9());
    }
}

sub pgm_exit($$) {
    my ($val,$msg) = @_;
    if (length($msg)) {
        $msg .= "\n" if (!($msg =~ /\n$/));
        prt($msg);
    }
    show_warnings($val);
    close_log($outfile,$load_log);
    exit($val);
}


sub prtw($) {
   my ($tx) = shift;
   $tx =~ s/\n$//;
   prt("$tx\n");
   push(@warnings,$tx);
}

sub process_in_dir($$);

my %src_lists = ();
my %hdr_lists = ();
my %oth_lists = ();
my @name_stack = ();

sub process_in_dir($$) {
    my ($dir,$lev) = @_;
    my ($n,$d) = fileparse($dir);
    if (! opendir(DIR, $dir) ) {
        pgm_exit(1,"ERROR: Unable to open directory [$dir]\n"); 
    }
    my @files = readdir(DIR);
    closedir(DIR);
    my $lncnt = scalar @files;
    my $msg = "Got $lncnt items, from [$dir]... ";
    my ($file,$ff,$nsrc,$nhdr,$noth,$ra,$tmp);
    ut_fix_directory(\$dir);
    my @dirs = ();
    push(@name_stack,$n);
    $tmp = '';
    foreach $file (@name_stack) {
        $tmp .= "_" if (length($tmp));
        $tmp .= $file;
    }
    $nsrc = $tmp."_srcs";
    $nhdr = $tmp."_hdrs";
    $noth = $tmp."_others";
    $msg .= "name $nsrc";
    prt("$msg\n");
    foreach $file (@files) {
        next if ($file eq '.');
        next if ($file eq '..');
        $ff = $dir.$file;
        if (-d $ff) {
            push(@dirs,$ff);
        } elsif (-f $ff) {
            if (is_c_source($file)) {
                $src_lists{$nsrc} = [] if (!defined $src_lists{$nsrc});
                $ra = $src_lists{$nsrc};
                push(@{$ra},$file);
            } elsif (is_h_source($file)) {
                $hdr_lists{$nhdr} = [] if (!defined $hdr_lists{$nhdr});
                $ra = $hdr_lists{$nhdr};
                push(@{$ra},$file);
            } else {
                $oth_lists{$noth} = [] if (!defined $oth_lists{$noth});
                $ra = $oth_lists{$noth};
                push(@{$ra},$file);
            }
        } else {
            prtw("WARNING: What is this [$ff]!\n");
        }
    }
    foreach $dir (@dirs) {
        process_in_dir($dir,($lev+1));
    }
    pop(@name_stack);
}

sub show_files_found() {
    my (@arr,@arr2);
    my ($key,$file,$ra,$cnt,$i,$dir,$scnt);
    my ($keycnt,$srccnt,$ukey,$tmp,$last);
    my $cmake = '';
    my %ukeys = ();
    my %uniq  = ();
    ##############################################
    ### C/C++ SOURCES
    @arr = sort keys %src_lists;
    $keycnt = 0;
    $srccnt = 0;
    foreach $key (@arr) {
        $keycnt++;
        $ra = $src_lists{$key};
        $scnt = scalar @{$ra};
        $srccnt += $scnt;
    }
    $cmake .= "\n";
    $cmake .= "##############################################\n";
    $cmake .= "### C/C++ SOURCES keys $keycnt, sources $srccnt\n";

    foreach $key (@arr) {
        $ra = $src_lists{$key};
        $scnt = scalar @{$ra};
        next if ($scnt == 0);
        @arr2 = split("_",$key);
        $last = uc(pop(@arr2)); # drop last to build directory
        $cnt = scalar @arr2;
        $dir = '';
        for ($i = 0; $i < $cnt; $i++) {
            $dir .= '/' if (length($dir));
            $dir .= $arr2[$i];
        }
        $cmake .= "set(dir $dir)\n";
        $ukey = '';
        for ($i = ($cnt-1); $i >= 0; $i--) {
            $tmp = uc($arr2[$i]);
            if (length($ukey)) {
                $ukey = $tmp."_".$ukey;
            } else {
                $ukey = $tmp."_".$last;
            }
            if (( !defined $ukeys{$ukey} ) && ( !defined $uniq{$ukey} )) {
                $ukeys{$ukey} = 1;
                $uniq{$ukey}  = 1;
                last;
            }
        }
        $cmake .= "set( $ukey\n";
        foreach $file (@{$ra}) {
            $cmake .= "    \$\{dir\}/$file\n";
        }
        $cmake .= "    )\n";
    }
    @arr2 = sort keys %ukeys;
    $cmake .= "set( ALL_SRCS\n";
    foreach $key (@arr2) {
        $cmake .= "    \$\{".$key."\}\n";
    }
    $cmake .= "    )\n";

    ###############################################
    ### C/C++ HEADERS
    %ukeys = ();
    @arr = sort keys %hdr_lists;
    $keycnt = 0;
    $srccnt = 0;
    foreach $key (@arr) {
        $keycnt++;
        $ra = $hdr_lists{$key};
        $scnt = scalar @{$ra};
        $srccnt += $scnt;
    }
    $cmake .= "\n";
    $cmake .= "##############################################\n";
    $cmake .= "### C/C++ HEADERS keys $keycnt, sources $srccnt\n";
    foreach $key (@arr) {
        $ra = $hdr_lists{$key};
        $scnt = scalar @{$ra};
        next if ($scnt == 0);
        @arr2 = split("_",$key);
        $last = uc(pop(@arr2));
        $cnt = scalar @arr2;
        $dir = '';
        for ($i = 0; $i < $cnt; $i++) {
            $dir .= '/' if (length($dir));
            $dir .= $arr2[$i];
        }
        $cmake .= "set(dir $dir)\n";
        $ukey = '';
        for ($i = ($cnt-1); $i >= 0; $i--) {
            $tmp = uc($arr2[$i]);
            if (length($ukey)) {
                $ukey = $tmp."_".$ukey;
            } else {
                $ukey = $tmp."_".$last;
            }
            if (( !defined $ukeys{$ukey} ) && ( !defined $uniq{$ukey} )) {
                $ukeys{$ukey} = 1;
                $uniq{$ukey}  = 1;
                last;
            }
        }
        $cmake .= "set( $ukey\n";
        foreach $file (@{$ra}) {
            $cmake .= "    \$\{dir\}/$file\n";
        }
        $cmake .= "    )\n";
    }
    @arr2 = sort keys %ukeys;
    $cmake .= "set( ALL_HDRS\n";
    foreach $key (@arr2) {
        $cmake .= "    \$\{".$key."\}\n";
    }
    $cmake .= "    )\n";

    ###############################################
    ### Other files found
    if (VERB5()) {
        %ukeys = ();
        @arr = sort keys %oth_lists;
        $keycnt = 0;
        $srccnt = 0;
        foreach $key (@arr) {
            $keycnt++;
            $ra = $oth_lists{$key};
            $scnt = scalar @{$ra};
            $srccnt += $scnt;
        }
        $cmake .= "\n";
        $cmake .= "##############################################\n";
        $cmake .= "### OTHER FILES keys $keycnt, sources $srccnt\n";
        foreach $key (@arr) {
            $ra = $oth_lists{$key};
            $scnt = scalar @{$ra};
            next if ($scnt == 0);
            @arr2 = split("_",$key);
            $last = uc(pop(@arr2));
            $cnt = scalar @arr2;
            $dir = '';
            for ($i = 0; $i < $cnt; $i++) {
                $dir .= '/' if (length($dir));
                $dir .= $arr2[$i];
            }
            $cmake .= "set(dir $dir)\n";
            $ukey = '';
            for ($i = ($cnt-1); $i >= 0; $i--) {
                $tmp = uc($arr2[$i]);
                if (length($ukey)) {
                    $ukey = $tmp."_".$ukey;
                } else {
                    $ukey = $tmp."_".$last;
                }
                if (( !defined $ukeys{$ukey} ) && ( !defined $uniq{$ukey} )) {
                    $ukeys{$ukey} = 1;
                    $uniq{$ukey}  = 1;
                    last;
                }
            }
            $cmake .= "set( $ukey\n";
            foreach $file (@{$ra}) {
                $cmake .= "    \$\{dir\}/$file\n";
            }
            $cmake .= "    )\n";
        }
        @arr2 = sort keys %ukeys;
        $cmake .= "set( ALL_OTHERS\n";
        foreach $key (@arr2) {
            $cmake .= "    \$\{".$key."\}\n";
        }
        $cmake .= "    )\n";
    }

    if (length($out_file)) {
        write2file($cmake,$out_file);
        prt("Contents written to $out_file\n");
    } else {
        prt("$cmake\n");
        prt("No -o out_file given so written to console\n");
    }
}

#########################################
### MAIN ###
parse_args(@ARGV);
process_in_dir($in_dir,0);
show_files_found();
pgm_exit(0,"");
########################################

sub need_arg {
    my ($arg,@av) = @_;
    pgm_exit(1,"ERROR: [$arg] must have a following argument!\n") if (!@av);
}

sub parse_args {
    my (@av) = @_;
    my ($arg,$sarg);
    my $verb = VERB2();
    while (@av) {
        $arg = $av[0];
        if ($arg =~ /^-/) {
            $sarg = substr($arg,1);
            $sarg = substr($sarg,1) while ($sarg =~ /^-/);
            if (($sarg =~ /^h/i)||($sarg eq '?')) {
                give_help();
                pgm_exit(0,"Help exit(0)");
            } elsif ($sarg =~ /^v/) {
                if ($sarg =~ /^v.*(\d+)$/) {
                    $verbosity = $1;
                } else {
                    while ($sarg =~ /^v/) {
                        $verbosity++;
                        $sarg = substr($sarg,1);
                    }
                }
                $verb = VERB2();
                prt("Verbosity = $verbosity\n") if ($verb);
            } elsif ($sarg =~ /^l/) {
                if ($sarg =~ /^ll/) {
                    $load_log = 2;
                } else {
                    $load_log = 1;
                }
                prt("Set to load log at end. ($load_log)\n") if ($verb);
            } elsif ($sarg =~ /^o/) {
                need_arg(@av);
                shift @av;
                $sarg = $av[0];
                $out_file = $sarg;
                prt("Set out file to [$out_file].\n") if ($verb);
            } else {
                pgm_exit(1,"ERROR: Invalid argument [$arg]! Try -?\n");
            }
        } else {
            $in_dir = $arg;
            prt("Set input to [$in_dir]\n") if ($verb);
        }
        shift @av;
    }

    if ($debug_on) {
        prtw("WARNING: DEBUG is ON!\n");
        if (length($in_dir) ==  0) {
            if (-d $def_dir) {
                $in_dir = $def_dir;
                prt("Set DEFAULT input to [$in_dir]\n");
            }
        }
        if (length($out_file) ==  0) {
            $out_file = $tmpout;
        }
        $verbosity = 5;

    }
    if (length($in_dir) ==  0) {
        pgm_exit(1,"ERROR: No in directory found in command!\n");
    }
    if (! -d $in_dir) {
        pgm_exit(1,"ERROR: Unable to directory [$in_dir]! Check name, location...\n");
    }
}

sub give_help {
    prt("$pgmname: version $VERS\n");
    prt("Usage: $pgmname [options] in-directory\n");
    prt("Options:\n");
    prt(" --help  (-h or -?) = This help, and exit 0.\n");
    prt(" --verb[n]     (-v) = Bump [or set] verbosity. def=$verbosity\n");
    prt(" --load        (-l) = Load LOG at end. ($outfile)\n");
    prt(" --out <file>  (-o) = Write output to this file.\n");
}

# eof - gensrclist2.pl
