#!/usr/bin/perl -w
# NAME: gentidyyml.pl
# AIM: Scan the Tidy binary folder, and generate a tidy.yml file
# 21/05/2015 - some UI improvements
# 25/04/2015 geoff mclane http://geoffair.net/mperl
use strict;
use warnings;
use File::Basename;  # split path ($name,$dir,$ext) = fileparse($file [, qr/\.[^.]*/] )
use File::stat; # get file info if ($sb = stat($fil)){$dt = $sb->mtime; $sz = $sb->size;}
use Cwd;
my $os = $^O;
my $perl_dir = '/home/geoff/bin';
my $PATH_SEP = '/';
my $temp_dir = '/tmp';
if ($os =~ /win/i) {
    $perl_dir = 'C:\GTools\perl';
    $temp_dir = $perl_dir;
    $PATH_SEP = "\\";
}
unshift(@INC, $perl_dir);
require 'lib_utils.pl' or die "Unable to load 'lib_utils.pl' Check paths in \@INC...\n";
# log file stuff
our ($LF);
my $pgmname = $0;
if ($pgmname =~ /(\\|\/)/) {
    my @tmpsp = split(/(\\|\/)/,$pgmname);
    $pgmname = $tmpsp[-1];
}
my $outfile = $temp_dir.$PATH_SEP."temp.$pgmname.txt";
open_log($outfile);

# user variables
my $VERS = "0.0.5 2015-01-09";
my $load_log = 0;
my $in_dir = '';
my $verbosity = 0;
my $out_file = '';
my $show_all = 0;

# ### DEBUG ###
my $debug_on = 0;
my $def_dir = 'F:\Projects\tidy-bins';

### program variables
my @warnings = ();
my $cwd = cwd();

sub VERB1() { return $verbosity >= 1; }
sub VERB2() { return $verbosity >= 2; }
sub VERB5() { return $verbosity >= 5; }
sub VERB9() { return $verbosity >= 9; }

sub show_warnings($) {
    my ($val) = @_;
    if (@warnings) {
        prt( "\nGot ".scalar @warnings." WARNINGS...\n" );
        foreach my $itm (@warnings) {
           prt("$itm\n");
        }
        prt("\n");
    } else {
        prt( "\nNo warnings issued.\n\n" ) if (VERB9());
    }
}

sub pgm_exit($$) {
    my ($val,$msg) = @_;
    if (length($msg)) {
        $msg .= "\n" if (!($msg =~ /\n$/));
        prt($msg);
    }
    show_warnings($val);
    close_log($outfile,$load_log);
    exit($val);
}


sub prtw($) {
   my ($tx) = shift;
   $tx =~ s/\n$//;
   prt("$tx\n");
   push(@warnings,$tx);
}

sub process_in_file($) {
    my ($inf) = @_;
    if (! open INF, "<$inf") {
        pgm_exit(1,"ERROR: Unable to open file [$inf]\n"); 
    }
    my @lines = <INF>;
    close INF;
    my $lncnt = scalar @lines;
    prt("Processing $lncnt lines, from [$inf]...\n");
    my ($line,$inc,$lnn);
    $lnn = 0;
    foreach $line (@lines) {
        chomp $line;
        $lnn++;
        if ($line =~ /\s*#\s*include\s+(.+)$/) {
            $inc = $1;
            prt("$lnn: $inc\n");
        }
    }
}

sub get_yml_lines($) {
    my ($inf) = @_;
    if (! open INF, "<$inf") {
        pgm_exit(1,"ERROR: Unable to open file [$inf]\n"); 
    }
    my @lines = <INF>;
    close INF;
    my $lncnt = scalar @lines;
    prt("Got $lncnt lines, from [$inf]...\n");
    return \@lines;
}

sub get_files_in_dir($) {
    my $dir = shift;
    my %dfiles = ();
    if (! opendir(DIR,$dir)) {
        prt("Unable to open directory $dir\n");
        return \%dfiles;
    }
    my @files = readdir(DIR);
    closedir(DIR);
    my ($file,$ff,$sb);
    ut_fix_directory(\$dir);
    foreach $file (@files) {
        next if ($file eq '.');
        next if ($file eq '..');
        $ff = $dir.$file;
        if (-f $ff) {
            if ($sb = stat($ff)) {
                $dfiles{$file} = [$sb->mtime,$sb->size];
            }
        }
    }
    return \%dfiles;
}

sub get_dirs_and_files($) {
    my $dir = shift;
    my %dfiles = ();
    if (! opendir(DIR,$dir)) {
        prt("Unable to oopen directory $dir\n");
    }
    my @files = readdir(DIR);
    closedir(DIR);
    my ($file,$ff,$sb);
    ut_fix_directory(\$dir);
    foreach $file (@files) {
        next if ($file eq '.');
        next if ($file eq '..');
        $ff = $dir.$file;
        if (-d $ff) {
            $dfiles{$file} = get_files_in_dir($ff);
        }
    }
    return \%dfiles;

}


sub get_description($) {
    my $fil = shift;
    # - filename: tidy5-4.9.23-win64.msi
    if ($fil =~ /\.msi$/) {
        if ($fil =~ /win64/) {
            return "Win64 MSI installer package.";
        } else {
            return "Win32 MSI installer package.";
        }
    }
    #- filename: tidy5-4.9.23-win64.exe
    if ($fil =~ /\.exe$/) {
        if ($fil =~ /win64/) {
            return "Win64 NSIS installer package.";
        } else {
            return "Win32 NSIS installer package.";
        }
    }
    #- filename: tidy5-4.9.23-win64.zip
    #- filename: tidy5-4.9.23-Source.zip
    if ($fil =~ /\.zip$/) {
        if ($fil =~ /Source/) {
            return "Windows ZIP of source.";
        } elsif ($fil =~ /win64/) {
            return "Win64 ZIP package.";
        } else {
            return "Win32 ZIP package.";
        }
    }

    #tidy5-4.9.20-64bit.deb
    if ($fil =~ /\.deb$/) {
        if ($fil =~ /64bit/) {
            return "linux 64-bit DEB.";
        } else {
            return "linux 32-bit DEB.";
        }
    }
    # - filename: tidy5-4.9.20-64bit.rpm
    if ($fil =~ /\.rpm$/) {
        if ($fil =~ /64bit/) {
            return "linux 64-bit RPM.";
        } else {
            return "linux 32-bit RPM.";
        }
    }
    # - filename: tidy5-4.9.23-Source.tar.gz
    if ($fil =~ /\.tar\.gz$/) {
        if ($fil =~ /Source/) {
            return "Unix tar.gz of source.";
        }
    }


    pgm_exit(1,"TODO: Need description of $fil! *** FIX ME ***\n");
}

sub does_path_exist($$) {
    my ($yra,$key) = @_;
    my ($line,$tlin,$len,$tmp);
    foreach $line (@{$yra}) {
        $tlin = trim_all($line);
        $len = length($tlin);
        next if ($len == 0);
        if ($line =~ /^-\s+path:\s+(.+)$/) {
            $tmp = $1;
            return 1 if ($tmp eq $key);
        }
    }
    return 0;
}

sub does_file_exist($$) {
    my ($yra,$key) = @_;
    my ($line,$tlin,$len,$tmp);
    foreach $line (@{$yra}) {
        $tlin = trim_all($line);
        $len = length($tlin);
        next if ($len == 0);
        if ($line =~ /^-\s+filename:\s+(.+)\s*$/) {
            $tmp = $1;
            return 1 if ($tmp eq $key);
        }
    }
    return 0;
}


#- path: tidy-4.9.23
#  heading: "HTML Tidy version 4.9.23"
#  notes: http://www.example.com
#  binaries:
#    - filename: tidy5-4.9.23-win64.msi
#      filesize: 815K
#      modified: "20/03/2015  13:11"
#      describe: "Win64 MSI installer package."


sub process_in_dir($) {
    my $dir = shift;
    if (! -d $dir) {
        pgm_exit(1,"Input directory $dir does NOT exist!\n");
    }
    my $dirb = $dir.$PATH_SEP.'binaries';
    if (! -d $dirb) {
        pgm_exit(1,"Directory $dirb does NOT exist!\n");
    }
    my $diry = $dir.$PATH_SEP.'_data';
    if (! -d $diry) {
        pgm_exit(1,"Directory $diry does NOT exist!\n");
    }
    my $yfil = $diry.$PATH_SEP.'tidy.yml';
    if (! -f $yfil) {
        pgm_exit(1,"YML file $yfil does NOT exist here!\n");
    }


    my $yra = get_yml_lines($yfil);
    ##my ($line,$lnn,$tlin,$len,$path,$ra,$ind,$head,$note,$filn,$date,$desc,$size,$tmp,$ver);
    my ($len,$tmp,$ver,$ra);
    ##my $inbin = 0;
    ##$lnn = 0;
    my $rh = get_dirs_and_files($dirb);
    my @arr = keys %{$rh};
    my ($rh1,$key,@arr1,$key2,@arr2);
    my $yml = '';
    $key2 = 0;
    my $dcnt = 0;
    my $fcnt = 0;
    foreach $key (@arr) {
        $rh1 = ${$rh}{$key};
        @arr1 = keys %{$rh1};
        $fcnt += scalar @arr1;
    }
    $dcnt = scalar @arr;
    prt("Have $dcnt scanned directories... $fcnt files...\n");

    $fcnt = 0;
    $dcnt = 0;
    foreach $key (@arr) {
        @arr2 = split("-",$key);
        $len = scalar @arr2;
        pgm_exit(1,"Wrong length $len\n") if ($len != 2);
        $ver = $arr2[1];
        $rh1 = ${$rh}{$key};
        next if (!$show_all && does_path_exist($yra,$key));
        $yml .= "- path: $key\n";
        $yml .= "  heading: HTML Tidy version $ver\n";
        $yml .= "  notes: /binaries/binaries/tidy-$ver/$ver.html\n";
        $yml .= "  binaries:\n";
        @arr1 = keys %{$rh1};
        $dcnt++;
        foreach $key2 (@arr1) {
            next if ($key2 =~ /\.html/);    # ignore any HTML
            $ra = ${$rh1}{$key2};
            $tmp = ${$ra}[1];
            $tmp /= 1024;
            $tmp = int($tmp).'K';
            $yml .= "    - filename: $key2\n";
            $yml .= "      filesize: $tmp\n";
            $yml .= "      modified: ".lu_get_YYYYMMDD_hhmmss(${$ra}[0])."\n";
            $yml .= "      describe: ".get_description($key2)."\n";
            $yml .= "\n";
            $fcnt++;
        }
    }
    if ($fcnt) {
        prt("Did $dcnt directories, $fcnt files\n");
        if (length($out_file)) {
            write2file($yml,$out_file);
            prt("tidy.yml written to $out_file\n");
        } else {
            prt($yml);
        }
    } else {
        prt("No new directories found... Use -a to show all...\n");
    }
    return;

#    #- path: tidy-4.9.23
#    my %paths = ();
#    my %files = ();
#    foreach $line (@{$yra}) {
#        $lnn++;
#        $tlin = trim_all($line);
#        $len = length($tlin);
#        next if ($len == 0);
#        if ($line =~ /^-\s+path:\s+(.+)$/) {
#            $tmp = $1;
#            if ($inbin && $path && $head && $note) {
#                prt("- path: $path\n");
#                prt("  heading: $head\n");
#                prt("  notes: $note\n");
#                prt("  binaries:\n");
#            }
#            prt("$lnn: $tmp\n");
#
#            $path = $tmp;
#            $paths{$path} = [];
#            $inbin = 0;
#            $head = undef;
#            $note = undef;
#        } elsif ($line =~ /^(\s+)heading:\s+(.+)$/) {
#            $ind = $1;
#            $head = $2;
#        } elsif ($line =~ /^(\s+)notes:\s+(.+)$/) {
#            $ind = $1;
#            $note = $2;
#        } elsif ($line =~ /^(\s+)binaries:\s*$/) {
#            $ind = $1;
#            $inbin = 1;
#        } elsif ($inbin) {
#            if ($line =~ /^(\s+)-\s+filename:\s+(.+)$/) {
#                $ind = $1;
#                $tmp = $2;
#                if ($filn && $size && $date && $desc) {
#                    prt("    - filename: $filn\n");
#                    prt("      filesize: $size\n");
#                    prt("      modified: $date\n");
#                    prt("      describe: $desc\n");
#                }
#                $filn = $tmp;
#                $size = undef;
#                $date = undef;
#                $desc = undef;
#            } elsif ($line =~ /^(\s+)filesize:\s+(.+)$/) {
#                $ind = $1;
#                $size = $2;
#            } elsif ($line =~ /^(\s+)modified:\s+(.+)$/) {
#                $ind = $1;
#                $date = $2
#            } elsif ($line =~ /^(\s+)describe:\s+(.+)$/) {
#                $ind = $1;
#                $desc = $2;
#            } else {
#                pgm_exit(1,"$lnn: Line '$line' un-parsed!\n");
#            }
#        } else {
#            pgm_exit(1,"$lnn: Line '$line' un-parsed!\n");
#        }
#    }

}

#########################################
### MAIN ###
#$load_log = 1;
#$show_all = 1;
parse_args(@ARGV);
process_in_dir($in_dir);
pgm_exit(0,"");
########################################

sub need_arg {
    my ($arg,@av) = @_;
    pgm_exit(1,"ERROR: [$arg] must have a following argument!\n") if (!@av);
}

sub parse_args {
    my (@av) = @_;
    my ($arg,$sarg);
    my $verb = VERB2();
    while (@av) {
        $arg = $av[0];
        if ($arg =~ /^-/) {
            $sarg = substr($arg,1);
            $sarg = substr($sarg,1) while ($sarg =~ /^-/);
            if (($sarg =~ /^h/i)||($sarg eq '?')) {
                give_help();
                pgm_exit(0,"Help exit(0)");
            } elsif ($sarg =~ /^v/) {
                if ($sarg =~ /^v.*(\d+)$/) {
                    $verbosity = $1;
                } else {
                    while ($sarg =~ /^v/) {
                        $verbosity++;
                        $sarg = substr($sarg,1);
                    }
                }
                $verb = VERB2();
                prt("Verbosity = $verbosity\n") if ($verb);
            } elsif ($sarg =~ /^a/) {
                $show_all = 1;
            } elsif ($sarg =~ /^l/) {
                if ($sarg =~ /^ll/) {
                    $load_log = 2;
                } else {
                    $load_log = 1;
                }
                prt("Set to load log at end. ($load_log)\n") if ($verb);
            } elsif ($sarg =~ /^o/) {
                need_arg(@av);
                shift @av;
                $sarg = $av[0];
                $out_file = $sarg;
                prt("Set out file to [$out_file].\n") if ($verb);
            } else {
                pgm_exit(1,"ERROR: Invalid argument [$arg]! Try -?\n");
            }
        } else {
            $in_dir = $arg;
            prt("Set input to [$in_dir]\n") if ($verb);
        }
        shift @av;
    }

    if ($debug_on) {
        prtw("WARNING: DEBUG is ON!\n");
        if (length($in_dir) ==  0) {
            if (-d $def_dir) {
                $in_dir = $def_dir;
                prt("Set DEFAULT input to [$in_dir]\n");
            }
        }
    }
    if (length($in_dir) ==  0) {
        pgm_exit(1,"ERROR: No input directory found in command!\n");
    }
    if (! -d $in_dir) {
        pgm_exit(1,"ERROR: Unable to find directory [$in_dir]! Check name, location...\n");
    }
}

sub give_help {
    prt("$pgmname: version $VERS\n");
    prt("Usage: $pgmname [options] in-file\n");
    prt("Options:\n");
    prt(" --help  (-h or -?) = This help, and exit 0.\n");
    prt(" --all         (-a) = Output ALL files.\n");
    prt(" --verb[n]     (-v) = Bump [or set] verbosity. def=$verbosity\n");
    prt(" --load        (-l) = Load LOG at end. ($outfile)\n");
    prt(" --out <file>  (-o) = Write output to this file.\n");
}

# eof - gentidyyml.pl
