#!/usr/bin/perl -w
# NAME: geometry.pl
# AIM: Test Math::Geometry::Planar
use strict;
use warnings;
use File::Basename;  # split path ($name,$dir,$ext) = fileparse($file [, qr/\.[^.]*/] )
use Math::Geometry::Planar;
use Cwd;
my $os = $^O;
my $perl_dir = '/home/geoff/bin';
my $PATH_SEP = '/';
my $temp_dir = '/tmp';
if ($os =~ /win/i) {
    $perl_dir = 'C:\GTools\perl';
    $temp_dir = $perl_dir;
    $PATH_SEP = "\\";
}
unshift(@INC, $perl_dir);
require 'lib_utils.pl' or die "Unable to load 'lib_utils.pl' Check paths in \@INC...\n";
require 'fg_wsg84.pl' or die "Unable to load fg_wsg84.pl ...\n";
# log file stuff
our ($LF);
my $pgmname = $0;
if ($pgmname =~ /(\\|\/)/) {
    my @tmpsp = split(/(\\|\/)/,$pgmname);
    $pgmname = $tmpsp[-1];
}
my $outfile = $temp_dir.$PATH_SEP."temp.$pgmname.txt";
open_log($outfile);

# user variables
my $VERS = "0.0.2 2014-01-13";
my $load_log = 0;
my $in_file = '';
my $verbosity = 0;
my $out_file = '';

# ### DEBUG ###
my $debug_on = 0;
my $def_file = 'def_file';

### program variables
my @warnings = ();
my $cwd = cwd();

sub VERB1() { return $verbosity >= 1; }
sub VERB2() { return $verbosity >= 2; }
sub VERB5() { return $verbosity >= 5; }
sub VERB9() { return $verbosity >= 9; }

sub show_warnings($) {
    my ($val) = @_;
    if (@warnings) {
        prt( "\nGot ".scalar @warnings." WARNINGS...\n" );
        foreach my $itm (@warnings) {
           prt("$itm\n");
        }
        prt("\n");
    } else {
        prt( "\nNo warnings issued.\n\n" ) if (VERB9());
    }
}

sub pgm_exit($$) {
    my ($val,$msg) = @_;
    if (length($msg)) {
        $msg .= "\n" if (!($msg =~ /\n$/));
        prt($msg);
    }
    show_warnings($val);
    close_log($outfile,$load_log);
    exit($val);
}


sub prtw($) {
   my ($tx) = shift;
   $tx =~ s/\n$//;
   prt("$tx\n");
   push(@warnings,$tx);
}

sub process_in_file($) {
    my ($inf) = @_;
    if (! open INF, "<$inf") {
        pgm_exit(1,"ERROR: Unable to open file [$inf]\n"); 
    }
    my @lines = <INF>;
    close INF;
    my $lncnt = scalar @lines;
    prt("Processing $lncnt lines, from [$inf]...\n");
    my ($line,$inc,$lnn);
    $lnn = 0;
    foreach $line (@lines) {
        chomp $line;
        $lnn++;
        if ($line =~ /\s*#\s*include\s+(.+)$/) {
            $inc = $1;
            prt("$lnn: $inc\n");
        }
    }
}

#sub fg_geo_inverse_wgs_84 {
#    my ($lat1, $lon1, $lat2, $lon2, $az1, $az2, $s) = @_;
#sub fg_geo_direct_wgs_84 {
#    my ( $lat1, $lon1, $az1, $s, $lat2, $lon2, $az2 ) = @_;
#
# from planar.pm
#  my $d  = DotProduct([$points[0],$points[1],$points[0],$points[1]]);
#  my $dp = CrossProduct([$points[2],$points[0],$points[1]]) / sqrt $d;
#  return $dp;
# EXPORTS
# SegmentLength[$p1,$p2];
# Determinant(x1,y1,x2,y2);
# DotProduct($p1,$p2,$p3,$p4);
# CrossProduct($p1,$p2,$p3);
# DistanceToLine($p1,$p2,$p3);

sub run_test() {
    # $point = [$x_coord,$y_coord];
    my $lat1 = -31.696845765;
    my $lon1 = 148.636770758;
    my $lat2 = -33.949273000;
    my $lon2 = 151.181346833;
    my ($p3,$p123,$dist,$az1,$az2,$res,$clat,$clon,$nlat,$nlon,$factor);
    my $mdist = 10000;

    my $p1 = [$lon1,$lat1]; # YGIL
    my $p2 = [$lon2,$lat2]; # YSSY
    prt("p1 = ".join(",",@{$p1}).", p2 = ".join(",",@{$p2})."\n");
    my $tot_dist = 0;
    # anno 149.12444400 -33.38305600 NDB ORANGE NDB
    my $orlat = -33.38305600;
    my $orlon = 149.12444400;
    $p3 = [$orlon,$orlat]; # Orange
    $p123 = [$p1,$p2,$p3];
    ###my $dist = DistanceToLine($p1,$p2,$p3);
    $dist = DistanceToLine($p123);
    prt("ORANGE NDB: Dist $dist\n");
    $tot_dist += abs($dist);

    # anno 150.83103300 -32.03474700 NDB SCONE NDB
    my $sclat = -32.03474700;
    my $sclon = 150.83103300;
    $p3 = [$sclon,$sclat]; # Scone
    $p123 = [$p1,$p2,$p3];
    $dist = DistanceToLine($p123);
    prt("SCONE NDB: Dist $dist\n");
    $tot_dist += abs($dist);

    $res = fg_geo_inverse_wgs_84( $orlat, $orlon, $sclat, $sclon, \$az1, \$az2, \$dist );
    $dist /= 1000;  # to km
    $factor = $dist / $tot_dist;
    prt("fg dist $dist, sum $tot_dist, factor $factor\n");

    # anno 149.59368900 -32.56092800 VOR MUDGEE VOR
    my $mulat = -32.56092800;
    my $mulon = 149.59368900;
    $p3 = [$mulon,$mulat]; # Mudgee
    $p123 = [$p1,$p2,$p3];
    $dist = DistanceToLine($p123);
    prt("MUDGEE VOR: Dist $dist\n");

    $res = fg_geo_inverse_wgs_84( $lat1, $lon1, $lat2, $lon2, \$az1, \$az2, \$dist );
    $res = fg_geo_direct_wgs_84( $lat1, $lon1, $az1, ($dist/2), \$clat, \$clon, \$az2 );
    $p3 = [$clon,$clat]; # Center
    $p123 = [$p1,$p2,$p3];
    $dist = DistanceToLine($p123);
    prt("Center: Dist $dist\n");

    $az1 += 90;
    $az1 -= 360 if ($az1 >= 360);
    $res = fg_geo_direct_wgs_84( $clat, $clon, $az1, $mdist, \$nlat, \$nlon, \$az2 );
    $p3 = [$nlon,$nlat]; # Center
    $p123 = [$p1,$p2,$p3];
    $dist = abs(DistanceToLine($p123));
    $mdist /= 1000;
    $factor = abs($mdist / $dist);
    prt("Known: SG $mdist, Dist $dist, factor $factor\n");

}

#########################################
### MAIN ###
# parse_args(@ARGV);
run_test();
# process_in_file($in_file);
pgm_exit(0,"");
########################################

sub need_arg {
    my ($arg,@av) = @_;
    pgm_exit(1,"ERROR: [$arg] must have a following argument!\n") if (!@av);
}

sub parse_args {
    my (@av) = @_;
    my ($arg,$sarg);
    my $verb = VERB2();
    while (@av) {
        $arg = $av[0];
        if ($arg =~ /^-/) {
            $sarg = substr($arg,1);
            $sarg = substr($sarg,1) while ($sarg =~ /^-/);
            if (($sarg =~ /^h/i)||($sarg eq '?')) {
                give_help();
                pgm_exit(0,"Help exit(0)");
            } elsif ($sarg =~ /^v/) {
                if ($sarg =~ /^v.*(\d+)$/) {
                    $verbosity = $1;
                } else {
                    while ($sarg =~ /^v/) {
                        $verbosity++;
                        $sarg = substr($sarg,1);
                    }
                }
                $verb = VERB2();
                prt("Verbosity = $verbosity\n") if ($verb);
            } elsif ($sarg =~ /^l/) {
                if ($sarg =~ /^ll/) {
                    $load_log = 2;
                } else {
                    $load_log = 1;
                }
                prt("Set to load log at end. ($load_log)\n") if ($verb);
            } elsif ($sarg =~ /^o/) {
                need_arg(@av);
                shift @av;
                $sarg = $av[0];
                $out_file = $sarg;
                prt("Set out file to [$out_file].\n") if ($verb);
            } else {
                pgm_exit(1,"ERROR: Invalid argument [$arg]! Try -?\n");
            }
        } else {
            $in_file = $arg;
            prt("Set input to [$in_file]\n") if ($verb);
        }
        shift @av;
    }

    if ($debug_on) {
        prtw("WARNING: DEBUG is ON!\n");
        if (length($in_file) ==  0) {
            $in_file = $def_file;
            prt("Set DEFAULT input to [$in_file]\n");
        }
    }
    if (length($in_file) ==  0) {
        pgm_exit(1,"ERROR: No input files found in command!\n");
    }
    if (! -f $in_file) {
        pgm_exit(1,"ERROR: Unable to find in file [$in_file]! Check name, location...\n");
    }
}

sub give_help {
    prt("$pgmname: version $VERS\n");
    prt("Usage: $pgmname [options] in-file\n");
    prt("Options:\n");
    prt(" --help  (-h or -?) = This help, and exit 0.\n");
    prt(" --verb[n]     (-v) = Bump [or set] verbosity. def=$verbosity\n");
    prt(" --load        (-l) = Load LOG at end. ($outfile)\n");
    prt(" --out <file>  (-o) = Write output to this file.\n");
}

# eof - template.pl
