#!/usr/bin/perl -w
# NAME: getmagvar.pl
# AIM: Given a lat,lon,[alt,$date] report the computed magnetic variation
use strict;
use warnings;
use File::Basename;  # split path ($name,$dir,$ext) = fileparse($file [, qr/\.[^.]*/] )
use Time::gmtime;
use Cwd;
my $os = $^O;
my $perl_dir = '/home/geoff/bin';
my $PATH_SEP = '/';
my $temp_dir = '/tmp';
if ($os =~ /win/i) {
    $perl_dir = 'C:\GTools\perl';
    $temp_dir = $perl_dir;
    $PATH_SEP = "\\";
}
unshift(@INC, $perl_dir);
require 'lib_utils.pl' or die "Unable to load 'lib_utils.pl' Check paths in \@INC...\n";
require 'lib_magvar.pl' or die "Unable to load 'lib_magvar.pl' Check paths in \@INC...\n";
# log file stuff
our ($LF);
my $pgmname = $0;
if ($pgmname =~ /(\\|\/)/) {
    my @tmpsp = split(/(\\|\/)/,$pgmname);
    $pgmname = $tmpsp[-1];
}
my $outfile = $temp_dir.$PATH_SEP."temp.$pgmname.txt";
open_log($outfile);

# user variables
my $VERS = "0.0.2 2014-10-10";
my $load_log = 0;
my $in_file = '';
my $verbosity = 0;
my $out_file = '';
my $usr_lat = 400;
my $usr_lon = 400;
my $usr_alt = 0;

# ### DEBUG ###
my $debug_on = 1;
my $def_lat = 37.618674211;
my $def_lon = -122.375007609;

### program variables
my @warnings = ();
my $cwd = cwd();
my $tm = gmtime(time());
my $year = $tm->year() + 1900;
my $month= $tm->mon() + 1;
my $day = $tm->mday();

sub VERB1() { return $verbosity >= 1; }
sub VERB2() { return $verbosity >= 2; }
sub VERB5() { return $verbosity >= 5; }
sub VERB9() { return $verbosity >= 9; }

sub show_warnings($) {
    my ($val) = @_;
    if (@warnings) {
        prt( "\nGot ".scalar @warnings." WARNINGS...\n" );
        foreach my $itm (@warnings) {
           prt("$itm\n");
        }
        prt("\n");
    } else {
        prt( "\nNo warnings issued.\n\n" ) if (VERB9());
    }
}

sub pgm_exit($$) {
    my ($val,$msg) = @_;
    if (length($msg)) {
        $msg .= "\n" if (!($msg =~ /\n$/));
        prt($msg);
    }
    show_warnings($val);
    close_log($outfile,$load_log);
    exit($val);
}


sub prtw($) {
   my ($tx) = shift;
   $tx =~ s/\n$//;
   prt("$tx\n");
   push(@warnings,$tx);
}

sub process_in_file($) {
    my ($inf) = @_;
    if (! open INF, "<$inf") {
        pgm_exit(1,"ERROR: Unable to open file [$inf]\n"); 
    }
    my @lines = <INF>;
    close INF;
    my $lncnt = scalar @lines;
    prt("Processing $lncnt lines, from [$inf]...\n");
    my ($line,$inc,$lnn);
    $lnn = 0;
    foreach $line (@lines) {
        chomp $line;
        $lnn++;
        if ($line =~ /\s*#\s*include\s+(.+)$/) {
            $inc = $1;
            prt("$lnn: $inc\n");
        }
    }
}

sub process_inputs() {
    my $yy = $year;
    if ($year > 2000) {
        $yy = $year - 2000;
    } elsif ($year > 1900) {
        $yy = $year - 1900;
    }
    my @fields = ();
    my $jdate = yymmdd_to_julian_days($yy,$month,$day);
    my $var = calc_magvar($usr_lat, $usr_lon, $usr_alt, $jdate, \@fields);
    prt("For lat/lon/alt $usr_lat,$usr_lon,$usr_alt, on $day/$month/$yy, jdate $jdate\n");
    prt("Magnetic Variation is $var\n");
}


#########################################
### MAIN ###
parse_args(@ARGV);
process_inputs();
pgm_exit(0,"");
########################################

sub need_arg {
    my ($arg,@av) = @_;
    pgm_exit(1,"ERROR: [$arg] must have a following argument!\n") if (!@av);
}

sub is_decimal($) {
    my $num = shift;
    return 1 if ($num =~ /^[-+]?[0-9]*\.?[0-9]+$/);
    return 0;
}

sub lat_in_world($) {
    my $lat = shift;
    return 1 if (($lat >= -90)&&($lat <= 90));
    return 0;
}
sub lon_in_world($) {
    my $lon = shift;
    return 0 if (($lon < -180)||($lon > 180));
    return 1;
}

sub parse_args {
    my (@av) = @_;
    my ($arg,$sarg,$cnt);
    my $verb = VERB2();
    $cnt = 0;
    while (@av) {
        $arg = $av[0];
        if (($arg =~ /^-/)&&(!is_decimal($arg))) {
            $sarg = substr($arg,1);
            $sarg = substr($sarg,1) while ($sarg =~ /^-/);
            if (($sarg =~ /^h/i)||($sarg eq '?')) {
                give_help();
                pgm_exit(0,"Help exit(0)");
            } elsif ($sarg =~ /^v/) {
                if ($sarg =~ /^v.*(\d+)$/) {
                    $verbosity = $1;
                } else {
                    while ($sarg =~ /^v/) {
                        $verbosity++;
                        $sarg = substr($sarg,1);
                    }
                }
                $verb = VERB2();
                prt("Verbosity = $verbosity\n") if ($verb);
            } elsif ($sarg =~ /^l/) {
                if ($sarg =~ /^ll/) {
                    $load_log = 2;
                } else {
                    $load_log = 1;
                }
                prt("Set to load log at end. ($load_log)\n") if ($verb);
            } elsif ($sarg =~ /^o/) {
                need_arg(@av);
                shift @av;
                $sarg = $av[0];
                $out_file = $sarg;
                prt("Set out file to [$out_file].\n") if ($verb);
            } else {
                pgm_exit(1,"ERROR: Invalid argument [$arg]! Try -?\n");
            }
        } else {
            if ($cnt == 0) {
                pgm_exit(1,"Error: lat=$arg OUT OF RANGE!\n") if (!lat_in_world($arg));
                $usr_lat = $arg;
            } elsif ($cnt == 1) {
                $usr_lon = $arg;
                pgm_exit(1,"Error: lon=$arg OUT OF RANGE!\n") if (!lon_in_world($arg));
            } elsif ($cnt == 2) {
                $usr_alt = $arg;
            } else {
                pgm_exit(1,"Error: Already have lat=$usr_lat, lon=$usr_lon, alt=$usr_alt!\nWhat is this $arg\n");
            }
            $cnt++;
        }
        shift @av;
    }

    if ($debug_on) {
        prtw("WARNING: DEBUG is ON!\n");
        if ($usr_lat == 400) {
            $usr_lat = $def_lat;
            prt("Set default lat=$usr_lat\n");
        }
        if ($usr_lon == 400) {
            $usr_lon = $def_lon;
            prt("Set default lon=$usr_lon\n");
        }
    }
    if ($usr_lat == 400) {
        pgm_exit(1,"ERROR: No lat lon found in command!\n");
    }
    if ($usr_lon == 400) {
        pgm_exit(1,"ERROR: Have a lat=$usr_lat, but no lon found in command!\n");
    }
}

sub give_help {
    $month = '0'.$month if ($month < 10);
    $day = '0'.$day if ($day < 10);
    prt("$pgmname: version $VERS\n");
    prt("Usage: $pgmname [options] lat lon [alt]\n");
    prt("Options:\n");
    prt(" --help  (-h or -?) = This help, and exit 0.\n");
    prt(" --date <date> (-d) = Set the date. (def=$year/$month/$day)\n");
    prt(" --verb[n]     (-v) = Bump [or set] verbosity. def=$verbosity\n");
    prt(" --load        (-l) = Load LOG at end. ($outfile)\n");
    prt(" --out <file>  (-o) = Write output to this file.\n");
}

# eof - template.pl
