#!/usr/bin/perl -w
# NAME: osm-gpx.pl
# AIM: Read a *.gpx XML file, and list points
# 20/12/2011 geoff mclane http://geoffair.net/mperl
use strict;
use warnings;
use File::Basename;  # split path ($name,$dir,$ext) = fileparse($file [, qr/\.[^.]*/] )
use Cwd;
my $perl_dir = 'C:\GTools\perl';
unshift(@INC, $perl_dir);
require 'lib_utils.pl' or die "Unable to load 'lib_utils.pl' Check paths in \@INC...\n";
# log file stuff
our ($LF);
my $pgmname = $0;
if ($pgmname =~ /(\\|\/)/) {
    my @tmpsp = split(/(\\|\/)/,$pgmname);
    $pgmname = $tmpsp[-1];
}
my $outfile = $perl_dir."\\temp.$pgmname.txt";
open_log($outfile);

# user variables
my $VERS = "0.0.1 2011-11-03";
my $load_log = 0;
my $in_file = '';
my $verbosity = 0;
my $debug_on = 1;
my $def_file = 'C:\DTEMP\osm-1153716.gpx';
my $out_xml = '';
my $keep_spaces = 0;

### program variables
my @warnings = ();
my $cwd = cwd();
my $os = $^O;

sub VERB1() { return $verbosity >= 1; }
sub VERB2() { return $verbosity >= 2; }
sub VERB5() { return $verbosity >= 5; }
sub VERB9() { return $verbosity >= 9; }

sub show_warnings($) {
    my ($val) = @_;
    if (@warnings) {
        prt( "\nGot ".scalar @warnings." WARNINGS...\n" );
        foreach my $itm (@warnings) {
           prt("$itm\n");
        }
        prt("\n");
    } else {
        prt( "\nNo warnings issued.\n\n" ) if (VERB9());
    }
}

sub pgm_exit($$) {
    my ($val,$msg) = @_;
    if (length($msg)) {
        $msg .= "\n" if (!($msg =~ /\n$/));
        prt($msg);
    }
    show_warnings($val);
    close_log($outfile,$load_log);
    exit($val);
}


sub prtw($) {
   my ($tx) = shift;
   $tx =~ s/\n$//;
   prt("$tx\n");
   push(@warnings,$tx);
}

sub process_in_file_xml($) {
    my ($inf) = @_;
    if (! open INF, "<$inf") {
        pgm_exit(1,"ERROR: Unable to open file [$inf]\n"); 
    }
    my @lines = <INF>;
    close INF;
    my $lncnt = scalar @lines;
    prt("Processing $lncnt lines, from [$inf]...\n");
    my ($line,$inc,$lnn);
    my ($len,$ch,$pc,$nc,$i,$i2);
    my ($tag,$txt,$intag,$inquot,$qc,$ntag,$ltag,@arr,$indent,$scnt);
    my $nline = 0;
    my @tagstack = ();
    $lnn = 0;
    $ch = '';
    $intag = 0;
    $inquot = 0;
    $tag = '';
    $txt = '';
    $ntag = '';
    $scnt = 0;
    $indent = '';
    my $xml_msg = '';
    my $on_new_line = 1;
    my @xml_array = ();
    foreach $line (@lines) {
        chomp $line;
        $lnn++;
        next if ($line =~ /^\s*$/);
        $len = length($line);
        for ($i = 0; $i < $len; $i++) {
            $pc = $ch;
            $ch = substr($line,$i,1);
            $i2 = $i + 1;
            $nc = ($i2 < $len) ? substr($line,$i2,1) : "";
            if ($intag) {
                $tag .= $ch;
                if ($inquot) {
                    $inquot = 0 if ($ch eq $qc);
                } else {
                    if ($ch eq '>') {
                        $intag = 0;
                        $txt = trim_tailing($txt) if (!$keep_spaces);
                        if (length($txt)) {
                            if ($on_new_line) {
                                $indent = ' ' x $scnt;
                                prt($indent.$txt) if (VERB1());
                                $xml_msg .= $indent.$txt;
                            } else {
                                prt($txt) if (VERB1());
                                $xml_msg .= $txt;
                            }
                            push(@xml_array,$txt);
                            $on_new_line = 0;   # now NOT on a new line
                        }
                        if ($tag =~ /^<\//) {
                            $nline = 1;
                            $ntag =~ s/^\///;
                            # closing a tag
                            if (@tagstack) {
                                $ltag = pop @tagstack;
                                if ($ltag ne $ntag) {
                                    prtw("WARNING: $lnn: Closing tag [$ntag], not last [$ntag]! [$line]\n");
                                }
                                $scnt = scalar @tagstack;
                            } else {
                                prtw("WARNING: $lnn: Closing tag [$ntag], but no stack! [$line]\n");
                            }
                        } elsif ($tag =~ /\/>$/) {
                            $nline = 1;
                        } elsif ($tag =~ /^<\?/) {
                            $nline = 1;
                            $tag = trim_all($tag);
                        } elsif ($tag =~ /^<!/) {
                            $nline = 1;
                            $tag = trim_all($tag);
                        } else {
                            $nline = 0;
                            if ($ntag =~ /\s/) {
                                # contains a space, thus maybe has attributes
                                @arr = split(/\s/,$ntag);
                                $ntag = $arr[0];
                            }
                            push(@tagstack,$ntag);
                            #$scnt = scalar @tagstack;
                        }
                        $indent = '';
                        if (length($txt) == 0) {
                            $indent = ' ' x $scnt;
                        }
                        prt($indent."$tag") if (VERB1());
                        $xml_msg .= $indent."$tag";
                        push(@xml_array,$tag);
                        $on_new_line = 0;
                        if ($nline) {
                            prt("\n") if (VERB1());
                            $xml_msg .= "\n";
                            $on_new_line = 1;
                        }
                        $scnt = scalar @tagstack;
                        $tag = '';
                        $txt = '';
                    } else {
                        # continuing an IN TAG
                        $ntag .= $ch;
                        if (($ch eq '"')||($ch eq "'")) {
                            $qc = $ch;
                            $inquot = 1;
                        } elsif ($ch eq '[') {
                            $qc = ']';
                            $inquot = 1;
                        }
                    }
                }
            } else {
                # awaiting start of tag
                if ($ch eq '<') {
                    $tag = $ch;
                    $ntag = '';
                    $intag = 1;
                    if ((length($txt) == 0)&&($nline == 0)) {
                        prt("\n") if (VERB1());
                        $xml_msg .= "\n";
                        $on_new_line = 1;
                    }
                    # prt("$lnn: Begin tag\n");
                } else {
                    if ($ch =~ /\s/) {
                        if ($keep_spaces) {
                            $txt .= $ch if (length($txt));
                        } elsif (length($txt)) {
                            $txt .= $ch if ($txt =~ /\S$/);
                        }
                    } else {
                        $txt .= $ch;
                    }
                }
            }
        } # for length of each line
    }
#    if ((length($out_xml))&&(length($xml_msg))) {
#        write2file($xml_msg,$out_xml);
#        prt("Written XML to [$out_xml]\n");
#    } elsif (length($xml_msg)) {
#        prt("No output file given, so xml not written...\n");
#    }
    return \@xml_array;
}

sub process_in_file_NOT_USED($) {
    my ($inf) = @_;
    if (! open INF, "<$inf") {
        pgm_exit(1,"ERROR: Unable to open file [$inf]\n"); 
    }
    my @lines = <INF>;
    close INF;
    my $lncnt = scalar @lines;
    prt("Processing $lncnt lines, from [$inf]...\n");
    my ($line,$inc,$lnn);
    my ($i,$ln,$len,$ch,$qc,$inquot,$txt,$tag,$intag);
    $lnn = 0;
    $inquot = 0;
    $intag = 0;
    $txt = '';
    $tag = '';
    for ($ln = 0; $ln < $lncnt; $ln++) {
        $line = $lines[$ln];
        chomp $line;
        $lnn++;
        $line = trim_all($line);
        $len = length($line);
        for ($i = 0; $i < $len; $i++) {
            $ch = substr($line,$i,1);
            if ($inquot) {
                if ($intag) {
                    $tag .= $ch;
                } else {
                    $txt .= $ch;
                }
                $inquot = 0 if ($ch eq $qc);
            } else {
                if ($ch eq '"') {
                    $inquot = 1;
                    $qc = $ch;
                } elsif ($ch eq '<') {
                    $intag = 1;
                    prt("$tag") if (length($tag));
                    prt("$txt") if (length($txt));
                    #prt("\n") if (length($tag) || length($txt));
                    prt("\n") if ($tag =~ /^<\//);
                    $tag = '';
                    $txt = '';
                }
                if ($intag) {
                    $tag .= $ch;
                    if ($ch eq '>') {
                        $intag = 0;
                    }
                } else {
                    $txt .= $ch;
                }
            }
        }
    }
}

sub show_ref_arr($) {
    my ($ra) = @_;
    my ($item);
    foreach $item (@{$ra}) {
        prt("$item\n");
    }
}


#########################################
### MAIN ###
parse_args(@ARGV);
### prt( "$pgmname: in [$cwd]: Hello, World...\n" );
my $ref_arr = process_in_file_xml($in_file);
show_ref_arr($ref_arr);
pgm_exit(0,"");
########################################
sub give_help {
    prt("$pgmname: version $VERS\n");
    prt("Usage: $pgmname [options] in-file\n");
    prt("Options:\n");
    prt(" --help  (-h or -?) = This help, and exit 0.\n");
    prt(" --verb[n]     (-v) = Bump [or set] verbosity. def=$verbosity\n");
    prt(" --load        (-l) = Load LOG at end. ($outfile)\n");
    prt(" --out <file>  (-o) = Write output to this file.\n");
}

sub need_arg {
    my ($arg,@av) = @_;
    pgm_exit(1,"ERROR: [$arg] must have a following argument!\n") if (!@av);
}

sub parse_args {
    my (@av) = @_;
    my ($arg,$sarg);
    while (@av) {
        $arg = $av[0];
        if ($arg =~ /^-/) {
            $sarg = substr($arg,1);
            $sarg = substr($sarg,1) while ($sarg =~ /^-/);
            if (($sarg =~ /^h/i)||($sarg eq '?')) {
                give_help();
                pgm_exit(0,"Help exit(0)");
            } elsif ($sarg =~ /^v/) {
                if ($sarg =~ /^v.*(\d+)$/) {
                    $verbosity = $1;
                } else {
                    while ($sarg =~ /^v/) {
                        $verbosity++;
                        $sarg = substr($sarg,1);
                    }
                }
                prt("Verbosity = $verbosity\n") if (VERB1());
            } elsif ($sarg =~ /^l/) {
                $load_log = 1;
                prt("Set to load log at end.\n") if (VERB1());
            } elsif ($sarg =~ /^o/) {
                need_arg(@av);
                shift @av;
                $sarg = $av[0];
                $out_xml = $sarg;
                prt("Set out file to [$out_xml].\n") if (VERB1());
            } else {
                pgm_exit(1,"ERROR: Invalid argument [$arg]! Try -?\n");
            }
        } else {
            $in_file = $arg;
            prt("Set input to [$in_file]\n");
        }
        shift @av;
    }

    if ((length($in_file) ==  0) && $debug_on) {
        $in_file = $def_file;
    }
    if (length($in_file) ==  0) {
        pgm_exit(1,"ERROR: No input files found in command!\n");
    }
    if (! -f $in_file) {
        pgm_exit(1,"ERROR: Unable to find in file [$in_file]! Check name, location...\n");
    }
}

# eof - template.pl
