#!/usr/bin/perl -w
# NAME: showmakevars.pl
# AIM: Read a automake type file, accumulate varaiable, and show list - output in CMakeLists.txt form
# 30/08/2014 geoff mclane http://geoffair.net/mperl
use strict;
use warnings;
use File::Basename;  # split path ($name,$dir,$ext) = fileparse($file [, qr/\.[^.]*/] )
use Cwd;
my $os = $^O;
my $perl_dir = '/home/geoff/bin';
my $PATH_SEP = '/';
my $temp_dir = '/tmp';
if ($os =~ /win/i) {
    $perl_dir = 'C:\GTools\perl';
    $temp_dir = $perl_dir;
    $PATH_SEP = "\\";
}
unshift(@INC, $perl_dir);
require 'lib_utils.pl' or die "Unable to load 'lib_utils.pl' Check paths in \@INC...\n";
# log file stuff
our ($LF);
my $pgmname = $0;
if ($pgmname =~ /(\\|\/)/) {
    my @tmpsp = split(/(\\|\/)/,$pgmname);
    $pgmname = $tmpsp[-1];
}
my $outfile = $temp_dir.$PATH_SEP."temp.$pgmname.txt";
open_log($outfile);

# user variables
my $VERS = "0.0.2 2014-01-13";
my $load_log = 0;
my $in_file = '';
my $verbosity = 0;
my $out_file = '';

# ### DEBUG ###
my $debug_on = 0;
my $def_file = 'F:\Projects\tar-1.28\gnu\gnulib.mk';
my $def_out = $temp_dir.$PATH_SEP."tempcmake.txt";

### program variables
my @warnings = ();
my $cwd = cwd();

sub VERB1() { return $verbosity >= 1; }
sub VERB2() { return $verbosity >= 2; }
sub VERB5() { return $verbosity >= 5; }
sub VERB9() { return $verbosity >= 9; }

sub show_warnings($) {
    my ($val) = @_;
    if (@warnings) {
        prt( "\nGot ".scalar @warnings." WARNINGS...\n" );
        foreach my $itm (@warnings) {
           prt("$itm\n");
        }
        prt("\n");
    } else {
        prt( "\nNo warnings issued.\n\n" ) if (VERB9());
    }
}

sub pgm_exit($$) {
    my ($val,$msg) = @_;
    if (length($msg)) {
        $msg .= "\n" if (!($msg =~ /\n$/));
        prt($msg);
    }
    show_warnings($val);
    close_log($outfile,$load_log);
    exit($val);
}


sub prtw($) {
   my ($tx) = shift;
   $tx =~ s/\n$//;
   prt("$tx\n");
   push(@warnings,$tx);
}
my $max_line_len = 90;

sub get_max_line($) {
    my $line = shift;
    my $len = length($line);
    if ($len < $max_line_len) {
        return $line;
    }
    my $hl = $max_line_len / 2;
    my $nline = substr($line,0,$hl);
    $nline .= "...";
    $nline .= substr($line, ($len - $hl));
    return $nline;
}

sub process_in_file($) {
    my ($inf) = @_;
    if (! open INF, "<$inf") {
        pgm_exit(1,"ERROR: Unable to open file [$inf]\n"); 
    }
    my @lines = <INF>;
    close INF;
    my $lncnt = scalar @lines;
    my ($name,$dir) = fileparse($inf);
    prt("Processing $lncnt lines, from [$inf]...\n");
    my ($i,$line,$inc,$lnn,$len,$i2,$tmp,$bgn,$end);
    $lnn = 0;
    my @nlines = ();
    for ($i = 0; $i < $lncnt; $i++) {
        $line = $lines[$i];
        chomp $line;
        $line = trim_all($line);
        $lnn++;
        $i2 = $i + 1;
        $len = length($line);
        next if ($len == 0);
        next if ($line =~ /^\#/);
        $bgn = $lnn;
        $end = $lnn;
        while (($line =~ /\\$/)&&($i2 < $lncnt)) {
            $line =~ s /\\$/ /;
            $i++;
            $i2 = $i + 1;
            $tmp = $lines[$i];
            chomp $tmp;
            $tmp = trim_all($tmp);
            $lnn++;
            $end = $lnn;
            $len = length($tmp);
            next if ($len == 0);
            next if ($tmp =~ /^\#/);
            $line .= $tmp;
        }
        $tmp = get_max_line($line);
        prt("$bgn:$end: $tmp\n") if (VERB9());
        push(@nlines,[$line,$bgn,$end]);
    }
    $lncnt = scalar @nlines;
    prt("Processing $lncnt 'trimmed' lines...\n");
    my ($cond,$targ,$var,$val,$ra);
    my %varhash = ();
    for ($i = 0; $i < $lncnt; $i++) {
        $line = $nlines[$i][0];
        $bgn = $nlines[$i][1];
        $end = $nlines[$i][2];
        if ($line =~ /^if\s+(.+)$/) {
            $cond = $1;
        } elsif ($line =~ /^else/) {
        } elsif ($line =~ /^endif/) {
        } elsif ($line =~ /^([\w\.\-\/]+)\s*:\s*(.+)$/) {
            $targ = $1;
        } elsif ($line =~ /^(\w+)\s*=\s*(.*)$/) {
            $var = $1;
            $val = $2;
            $varhash{$var} = [] if (!defined $varhash{$var});
            $ra = $varhash{$var};
            @{$ra} = ();
            if ($val && length($val)) {
                push(@{$ra},$val);
            }
            $varhash{$var} = $ra;
        } elsif ($line =~ /^(\w+)\s*\+=\s*(.*)$/) {
            $var = $1;
            $val = $2;
            $varhash{$var} = [] if (!defined $varhash{$var});
            $ra = $varhash{$var};
            if ($val && length($val)) {
                push(@{$ra},$val);
            }
            $varhash{$var} = $ra;
        } elsif ($line =~ /rm\s+\-f\s+/) {
        } else {
            $tmp = get_max_line($line);
            prt("Unhandled:$bgn:$end: [$tmp]\n") if (VERB5());
        }
    }
    my @keys = sort keys %varhash;
    my (@arr,$ff,$hdr);
    $lncnt = scalar @keys;
    my $cmake = "# Got $lncnt keys in var hash...\n";
    prt($cmake);
    ut_fix_directory(\$dir);
    foreach $var (@keys) {
        $ra = $varhash{$var};
        $len = scalar @{$ra};
        $var =~ s/\//_/g;
        $var =~ s/\-/_/g;
        $var =~ s/\./_/g;
        if ($len) {
            # start a 'temp' cmake string, in case there are NONE
            my $tcmake .= "set( $var\n";
            my @nf = ();
            my @headers = ();
            my $cnt = 0;
            foreach $val (@{$ra}) {
                @arr = split(/\s+/,$val);
                foreach $tmp (@arr) {
                    $ff = $dir.$tmp;
                    if (-f $ff) {
                        if (is_c_source($tmp)) {
                            $tcmake .= "    \${dir}/$tmp\n";
                            $cnt++;
                            my ($n,$d,$e) = fileparse($tmp, qr/\.[^.]*/ );
                            $hdr = $n.".h";
                            $ff = $dir.$hdr;
                            if (-f $ff) {
                                push(@headers,$hdr);
                            }
                        } elsif (is_h_source($tmp)) {
                            push(@headers,$tmp);
                        } else {
                            $tcmake .= "    \${dir}/$tmp\n";
                            $cnt++;
                        }
                    } else {
                        push(@nf,$tmp);
                    }
                }
            }
            if ($cnt) {
                $cmake .= $tcmake;
                $cmake .= "    )\n";
            }
            if (@headers) {
                $cmake .= "set( $var"."_HDRS\n";
                foreach $hdr (@headers) {
                     $cmake .= "    \${dir}/$hdr\n";
                }
                $cmake .= "    )\n";
            }
            if (@nf) {
                $cmake .= "set( $var"."_NF\n";
                foreach $hdr (@nf) {
                     $cmake .= "    \${dir}/$hdr\n";
                }
                $cmake .= "    )\n";
            }
        }
    }
    if (length($out_file) == 0) {
        prt($cmake);
        prt("No -o file found in command so written to stdout\n");
    } else {
        write2file($cmake,$out_file);
        prt("List written to $out_file\n");
    }
}

sub check_regex() {
    my $line = "EXTRA_DIST += xgetcwd.h";
    my ($var,$val);
    if ($line =~ /^(\w+)\s*\+=\s*(.*)$/) {
        $var = $1;
        $val = $2;
        prt("Got var $var with added value $val\n");
    } else {
        prt("regex FAILED on '$line'\n");
    }
    pgm_exit(1,"");
}

#########################################
### MAIN ###
##check_regex();
parse_args(@ARGV);
process_in_file($in_file);
pgm_exit(0,"");
########################################

sub need_arg {
    my ($arg,@av) = @_;
    pgm_exit(1,"ERROR: [$arg] must have a following argument!\n") if (!@av);
}

sub parse_args {
    my (@av) = @_;
    my ($arg,$sarg);
    my $verb = VERB2();
    while (@av) {
        $arg = $av[0];
        if ($arg =~ /^-/) {
            $sarg = substr($arg,1);
            $sarg = substr($sarg,1) while ($sarg =~ /^-/);
            if (($sarg =~ /^h/i)||($sarg eq '?')) {
                give_help();
                pgm_exit(0,"Help exit(0)");
            } elsif ($sarg =~ /^v/) {
                if ($sarg =~ /^v.*(\d+)$/) {
                    $verbosity = $1;
                } else {
                    while ($sarg =~ /^v/) {
                        $verbosity++;
                        $sarg = substr($sarg,1);
                    }
                }
                $verb = VERB2();
                prt("Verbosity = $verbosity\n") if ($verb);
            } elsif ($sarg =~ /^l/) {
                if ($sarg =~ /^ll/) {
                    $load_log = 2;
                } else {
                    $load_log = 1;
                }
                prt("Set to load log at end. ($load_log)\n") if ($verb);
            } elsif ($sarg =~ /^o/) {
                need_arg(@av);
                shift @av;
                $sarg = $av[0];
                $out_file = $sarg;
                prt("Set out file to [$out_file].\n") if ($verb);
            } else {
                pgm_exit(1,"ERROR: Invalid argument [$arg]! Try -?\n");
            }
        } else {
            $in_file = $arg;
            prt("Set input to [$in_file]\n") if ($verb);
        }
        shift @av;
    }

    if ($debug_on) {
        prtw("WARNING: DEBUG is ON!\n");
        if (length($in_file) ==  0) {
            $in_file = $def_file;
            prt("Set DEFAULT input to [$in_file]\n");
        }
        if (length($out_file) == 0) {
            $out_file = $def_out;
            prt("Set DEFAULT output to [$out_file]\n");
        }
    }
    if (length($in_file) ==  0) {
        pgm_exit(1,"ERROR: No input files found in command!\n");
    }
    if (! -f $in_file) {
        pgm_exit(1,"ERROR: Unable to find in file [$in_file]! Check name, location...\n");
    }
}

sub give_help {
    prt("$pgmname: version $VERS\n");
    prt("Usage: $pgmname [options] in-file\n");
    prt("Options:\n");
    prt(" --help  (-h or -?) = This help, and exit 0.\n");
    prt(" --verb[n]     (-v) = Bump [or set] verbosity. def=$verbosity\n");
    prt(" --load        (-l) = Load LOG at end. ($outfile)\n");
    prt(" --out <file>  (-o) = Write output to this file.\n");
}

# eof - template.pl
