#!/usr/bin/perl -w
# NAME: shwoffset.pl
# AIM: Given a file and an offset, show line with that offset
# 20150108 - Limit output unless -v5
# 01/03/2014 geoff mclane http://geoffair.net/mperl 
use strict;
use warnings;
use File::Basename;  # split path ($name,$dir,$ext) = fileparse($file [, qr/\.[^.]*/] )
use Cwd;
my $os = $^O;
my $perl_dir = '/home/geoff/bin';
my $PATH_SEP = '/';
my $temp_dir = '/tmp';
if ($os =~ /win/i) {
    $perl_dir = 'C:\GTools\perl';
    $temp_dir = $perl_dir;
    $PATH_SEP = "\\";
}
unshift(@INC, $perl_dir);
require 'lib_utils.pl' or die "Unable to load 'lib_utils.pl' Check paths in \@INC...\n";
# log file stuff
our ($LF);
my $pgmname = $0;
if ($pgmname =~ /(\\|\/)/) {
    my @tmpsp = split(/(\\|\/)/,$pgmname);
    $pgmname = $tmpsp[-1];
}
my $outfile = $temp_dir.$PATH_SEP."temp.$pgmname.txt";
open_log($outfile);

# user variables
my $VERS = "0.0.2 2014-03-01";
my $load_log = 0;
my $in_file = '';
my $verbosity = 0;
my $out_file = '';
my $offset = -1;
my $def_span = 32;

# ### DEBUG ###
my $debug_on = 0;
my $def_file = 'def_file';

### program variables
my @warnings = ();
my $cwd = cwd();

sub VERB1() { return $verbosity >= 1; }
sub VERB2() { return $verbosity >= 2; }
sub VERB5() { return $verbosity >= 5; }
sub VERB9() { return $verbosity >= 9; }

sub show_warnings($) {
    my ($val) = @_;
    if (@warnings) {
        prt( "\nGot ".scalar @warnings." WARNINGS...\n" );
        foreach my $itm (@warnings) {
           prt("$itm\n");
        }
        prt("\n");
    } else {
        prt( "\nNo warnings issued.\n\n" ) if (VERB9());
    }
}

sub pgm_exit($$) {
    my ($val,$msg) = @_;
    if (length($msg)) {
        $msg .= "\n" if (!($msg =~ /\n$/));
        prt($msg);
    }
    show_warnings($val);
    close_log($outfile,$load_log);
    exit($val);
}


sub prtw($) {
   my ($tx) = shift;
   $tx =~ s/\n$//;
   prt("$tx\n");
   push(@warnings,$tx);
}

sub process_in_file($) {
    my ($inf) = @_;
    if (! open INF, "<$inf") {
        pgm_exit(1,"ERROR: Unable to open file [$inf]\n"); 
    }
    my @lines = <INF>;
    close INF;
    my $lncnt = scalar @lines;
    prt("Processing $lncnt lines, from [$inf], up to offet $offset...\n");
    my ($i,$line,$inc,$lnn,$len,$tmp,$skip,$tlen1,$tlen2,$tlen,$rem);
    $lnn = 0;
    my $total_len = 0;
    my $prev_tot = 0;
    for ($i = 0; $i < $lncnt; $i++) {
        $line = $lines[$i];
        $len = length($line);
        $prev_tot = $total_len;
        $total_len += $len;
        chomp $line;
        $lnn++;
        if ($total_len >= $offset) {
            if (VERB9()) {
                if ($i > 4) {
                    $tmp = $lines[$i-5];
                    chomp $tmp;
                    prt(($i - 4).": $tmp\n");
                }
            }
            if (VERB5()) {
                if ($i > 3) {
                    $tmp = $lines[$i-4];
                    chomp $tmp;
                    prt(($i - 3).": $tmp\n");
                }
            }
            if (VERB2()) {
                if ($i > 2) {
                    $tmp = $lines[$i-3];
                    chomp $tmp;
                    prt(($i - 2).": $tmp\n");
                }
            }
            if (VERB1()) {
                if ($i > 1) {
                    $tmp = $lines[$i-2];
                    chomp $tmp;
                    prt(($i - 1).": $tmp\n");
                }
            }
            if ($i) {
                $tmp = $lines[$i-1];
                chomp $tmp;
                prt("$i: $tmp\n");
            }
            ################################################
            # the line containing this offset
            if (VERB5()) {
                # whow WHOLE line
                prt("$lnn> $line\n");
            } else {
                # show only around the OFFSET
                # offset is after $prev_tot, and before or at $total_len
                $len = length($line);
                $tlen1 = $total_len - $offset;
                if ($tlen1 > $def_span) {
                    $tlen1 = $def_span;
                }
                $tlen2 = $offset - $prev_tot;
                if ($tlen2 > $def_span) {
                    $tlen2 = $def_span;
                }
                $tlen = $tlen1 + $tlen2;
                $skip = $offset - $tlen2;
                $tmp = substr($line,$skip,$tlen);
                prt("Offset $offset prev $prev_tot next $total_len substr $skip,$tlen\n");
                prt("$lnn> $tmp\n");
                $tmp = ' ' x ($tlen2-1);
                $tmp .= '^';
                prt("$lnn> $tmp\n");

            }

            if (($i + 1) < $lncnt) {
                $tmp = $lines[$i+1];
                chomp $tmp;
                prt(($i+2).": $tmp\n");
            }
            if (VERB1()) {
                if (($i + 2) < $lncnt) {
                    $tmp = $lines[$i+2];
                    chomp $tmp;
                    prt(($i+3).": $tmp\n");
                }
            }
            if (VERB2()) {
                if (($i + 3) < $lncnt) {
                    $tmp = $lines[$i+3];
                    chomp $tmp;
                    prt(($i+4).": $tmp\n");
                }
            }
            if (VERB5()) {
                if (($i + 4) < $lncnt) {
                    $tmp = $lines[$i+4];
                    chomp $tmp;
                    prt(($i+5).": $tmp\n");
                }
            }
            if (VERB9()) {
                if (($i + 5) < $lncnt) {
                    $tmp = $lines[$i+5];
                    chomp $tmp;
                    prt(($i+6).": $tmp\n");
                }
            }
            last;
        }
    }
}

#########################################
### MAIN ###
parse_args(@ARGV);
process_in_file($in_file);
pgm_exit(0,"");
########################################

sub need_arg {
    my ($arg,@av) = @_;
    pgm_exit(1,"ERROR: [$arg] must have a following argument!\n") if (!@av);
}

sub parse_args {
    my (@av) = @_;
    my ($arg,$sarg,$cnt);
    my $verb = VERB2();
    $cnt = 0;
    while (@av) {
        $arg = $av[0];
        if ($arg =~ /^-/) {
            $sarg = substr($arg,1);
            $sarg = substr($sarg,1) while ($sarg =~ /^-/);
            if (($sarg =~ /^h/i)||($sarg eq '?')) {
                give_help();
                pgm_exit(0,"Help exit(0)");
            } elsif ($sarg =~ /^v/) {
                if ($sarg =~ /^v.*(\d+)$/) {
                    $verbosity = $1;
                } else {
                    while ($sarg =~ /^v/) {
                        $verbosity++;
                        $sarg = substr($sarg,1);
                    }
                }
                $verb = VERB2();
                prt("Verbosity = $verbosity\n") if ($verb);
            } elsif ($sarg =~ /^l/) {
                if ($sarg =~ /^ll/) {
                    $load_log = 2;
                } else {
                    $load_log = 1;
                }
                prt("Set to load log at end. ($load_log)\n") if ($verb);
            } elsif ($sarg =~ /^o/) {
                need_arg(@av);
                shift @av;
                $sarg = $av[0];
                $out_file = $sarg;
                prt("Set out file to [$out_file].\n") if ($verb);
            } else {
                pgm_exit(1,"ERROR: Invalid argument [$arg]! Try -?\n");
            }
        } else {
            if ($cnt == 0) {
                $in_file = $arg;
                if (-f $in_file) {
                    prt("Set input to [$in_file]\n") if ($verb);
                } else {
                    pgm_exit(1,"Command error! Expect first bear argument to be a valid file.\nFile [$in_file] NOT found.\n");
                }
            } elsif ($cnt == 1) {
                $offset = $arg;
                if ($offset =~ /^\d+$/) {
                    prt("Set offset to [$offset]\n") if ($verb);
                } else {
                    pgm_exit(1,"Command error! Expected integer offset and 2nd bare argument, not [$arg]\n");
                }
            } else {
                pgm_exit(1,"Command error! Already have file [$in_file], and offset [$offset]!\nWhat is this [$arg]\n");
            }
            $cnt++;
        }
        shift @av;
    }

    if ($debug_on) {
        prtw("WARNING: DEBUG is ON!\n");
        if (length($in_file) ==  0) {
            $in_file = $def_file;
            prt("Set DEFAULT input to [$in_file]\n");
        }
    }
    if (length($in_file) ==  0) {
        pgm_exit(1,"ERROR: No input files found in command! Try -?\n");
    }
    if ($offset == -1) {
        pgm_exit(1,"ERROR: No offset found in command! Try -?\n");
    }
    if ($offset >= 0) {
    } else {
        pgm_exit(1,"ERROR: Bad offset [$offset]! Must be 0 to 99999999999! Try -?\n");
    }
    if (! -f $in_file) {
        pgm_exit(1,"ERROR: Unable to find in file [$in_file]! Check name, location...\n");
    }
}

sub give_help {
    prt("$pgmname: version $VERS\n");
    prt("Usage: $pgmname [options] in-file offset\n");
    prt("Options:\n");
    prt(" --help  (-h or -?) = This help, and exit 0.\n");
    prt(" --verb[n]     (-v) = Bump [or set] verbosity. def=$verbosity\n");
    prt(" --load        (-l) = Load LOG at end. ($outfile)\n");
    prt(" --out <file>  (-o) = Write output to this file.\n");
}

# eof - template.pl
