#!/usr/bin/perl -w
# NAME: cmp2dsw.pl
# AIM: Compare 2 DSW files
# 29/09/2010 geoff mclane http://geoffair.net/mperl
use strict;
use warnings;
use File::Basename;  # split path ($name,$dir,$ext) = fileparse($file [, qr/\.[^.]*/] )
use Cwd;
my $perl_dir = 'C:\GTools\perl';
unshift(@INC, $perl_dir);
require 'logfile.pl' or die "Unable to load logfile.pl ...\n";
# log file stuff
my ($LF);
my $pgmname = $0;
if ($pgmname =~ /(\\|\/)/) {
    my @tmpsp = split(/(\\|\/)/,$pgmname);
    $pgmname = $tmpsp[-1];
}
my $outfile = $perl_dir."\\temp.$pgmname.txt";
open_log($outfile);

# user variables
my $load_log = 0;
my $in_file1 = '';
my $in_file2 = '';

# debug
my $debug_on = 0;
my $def_file1 = 'temp\temp.glib.dsw';
my $def_file2 = 'temp\temp.glib.dsw.bak';

my $dbg_01 = 0;

### program variables
my @warnings = ();
my $cwd = cwd();
my $os = $^O;

sub show_warnings($) {
    my ($val) = @_;
    if (@warnings) {
        prt( "\nGot ".scalar @warnings." WARNINGS...\n" );
        foreach my $itm (@warnings) {
           prt("$itm\n");
        }
        prt("\n");
    } else {
        prt( "No warnings issued.\n\n" );
    }
}

sub pgm_exit($$) {
    my ($val,$msg) = @_;
    if (length($msg)) {
        $msg .= "\n" if (!($msg =~ /\n$/));
        prt($msg);
    }
    show_warnings($val);
    close_log($outfile,$load_log);
    exit($val);
}


sub prtw($) {
   my ($tx) = shift;
   $tx =~ s/\n$//;
   prt("$tx\n");
   push(@warnings,$tx);
}

sub get_anon_dep() {
    my @dep = ();
    return \@dep;
}

sub get_DSW_hash($$) {
    my ($fil,$rlines) = @_;
    my $max = scalar @{$rlines};
    my ($i,$line,$proj,$dsp,$rdeps,$dep);
    my %hash = ();
    prt("Processing $max lines, from [$fil]\n");

    for ($i = 0; $i < $max; $i++) {
        $line = ${$rlines}[$i];
        chomp $line;
        # Project: "unicode_normalize"="unicode_normalize.dsp" - Package Owner=<4>
        if ($line =~ /^Project:\s+"(\w+)"="([-\w\.]+)"\s+-/) {
            $proj = $1;
            $dsp = $2;
            $rdeps = get_anon_dep();
            $hash{$proj} = [$dsp,$rdeps];
            prt("[01] Project [$proj] [$dsp]\n") if ($dbg_01);
        } elsif ($line =~ /\s*Project_Dep_Name\s+([-\w\.]+)\s*$/) {
            # Project_Dep_Name 
            $dep = $1;
            push(@{$rdeps},$dep);
            prt(" [01] Deps: $dep\n") if ($dbg_01);
        } elsif ($line =~ /Project:/) {
            prtw("WARNING: Missed line [$line]! WHY?\n");
        } elsif ($line =~ /Project_Dep_Name/) {
            prtw("WARNING: Missed line [$line]! WHY?\n");
        }
    }
    return \%hash;
}

sub compare_h1_with_h2($$) {
    my ($rh1,$rh2) = @_;
    my $cnt1 = scalar keys(%{$rh1});
    my $cnt2 = scalar keys(%{$rh2});
    my ($ky1,$ky2,$dsp1,$dsp2,$rdeps1,$rdeps2,$v1,$v2,$dep1,$dep2,$fnd);
    my $changes = 0;

    prt("Comparing 1 $cnt1, with 2 $cnt2...\n");
    foreach $ky1 (keys %{$rh1}) {
        if (!defined ${$rh2}{$ky1}) {
            prt("Project [$ky1] NOT in 2\n");
            $changes++;
        } else {
            $v1 = ${$rh1}{$ky1};
            $v2 = ${$rh2}{$ky1};
            $dsp1 = ${$v1}[0];
            $dsp2 = ${$v2}[0];
            $rdeps1 = ${$v1}[1];
            $rdeps2 = ${$v2}[1];
            # compare the ref arrays
            foreach $dep1 (@{$rdeps1}) {
                $fnd = 0;
                foreach $dep2 (@{$rdeps2}) {
                    if ($dep1 eq $dep2) {
                        $fnd = 1;
                        last;
                    }
                }
                if (!$fnd) {
                    prt(" $ky1 depends on [$dep1], NOT in 2\n");
                    $changes++;
                }
            }
            foreach $dep2 (@{$rdeps2}) {
                $fnd = 0;
                foreach $dep1 (@{$rdeps1}) {
                    if ($dep1 eq $dep2) {
                        $fnd = 1;
                        last;
                    }
                }
                prt(" $ky1 depends on [$dep1], NOT in 1\n") if (!$fnd);
            }

        }
    }
    prt("Comparing 2 $cnt2, with 1 $cnt1...\n");
    foreach $ky2 (keys %{$rh2}) {
        if (!defined ${$rh1}{$ky2}) {
            prt("Project [$ky2] NOT in 1\n");
            $changes++;
        } else {
            $v1 = ${$rh1}{$ky2};
            $v2 = ${$rh2}{$ky2};
            $dsp1 = ${$v1}[0];
            $dsp2 = ${$v2}[0];
            $rdeps1 = ${$v1}[1];
            $rdeps2 = ${$v2}[1];
            # compare the ref arrays
            foreach $dep1 (@{$rdeps1}) {
                $fnd = 0;
                foreach $dep2 (@{$rdeps2}) {
                    if ($dep1 eq $dep2) {
                        $fnd = 1;
                        last;
                    }
                }
                prt(" $ky1 depends on [$dep1], NOT in 2\n") if (!$fnd);
            }
            foreach $dep2 (@{$rdeps2}) {
                $fnd = 0;
                foreach $dep1 (@{$rdeps1}) {
                    if ($dep1 eq $dep2) {
                        $fnd = 1;
                        last;
                    }
                }
                if (!$fnd) {
                    prt(" $ky1 depends on [$dep1], NOT in 1\n");
                    $changes++;
                }
            }
        }
    }
    if ($changes) {
        prt("Have $changes changes...\n");
    } else {
        prt("Have NO changes... these two DSW files appear the SAME...\n");
    }
}

sub process_in_files($$) {
    my ($inf1,$inf2) = @_;
    if (! open INF, "<$inf1") {
        pgm_exit(1,"ERROR: Unable to open file [$inf1]\n"); 
    }
    my @lines1 = <INF>;
    close INF;
    my $lncnt1 = scalar @lines1;
    prt("Got $lncnt1 lines, from [$inf1]...\n");
    if (! open INF, "<$inf2") {
        pgm_exit(1,"ERROR: Unable to open file [$inf2]\n"); 
    }
    my @lines2 = <INF>;
    close INF;
    my $lncnt2 = scalar @lines2;
    prt("Got $lncnt2 lines, from [$inf2]...\n");
    my $rh1 = get_DSW_hash($inf1,\@lines1);
    my $rh2 = get_DSW_hash($inf2,\@lines2);
    compare_h1_with_h2($rh1,$rh2);

}

#########################################
### MAIN ###
parse_args(@ARGV);
# prt( "$pgmname: in [$cwd]: Hello, World...\n" );
process_in_files($in_file1,$in_file2);
pgm_exit(0,"Normal exit(0)");
########################################
sub give_help {
    prt("$pgmname: version 0.0.1 2010-09-29\n");
    prt("Usage: $pgmname [options] in-file1 in-file2\n");
    prt("Options:\n");
    prt(" --help (-h or -?) = This help, and exit 0.\n");
    prt("Purpose:\n");
    prt(" Compare tow DSW files, and report difference...\n");
}
sub need_arg {
    my ($arg,@av) = @_;
    pgm_exit(1,"ERROR: [$arg] must have following argument!\n") if (!@av);
}

sub parse_args {
    my (@av) = @_;
    my ($arg,$sarg,$cnt);
    $cnt = 0;
    while (@av) {
        $arg = $av[0];
        if ($arg =~ /^-/) {
            $sarg = substr($arg,1);
            $sarg = substr($sarg,1) while ($sarg =~ /^-/);
            if (($sarg =~ /^h/i)||($sarg eq '?')) {
                give_help();
                pgm_exit(0,"Help exit(0)");
            } else {
                pgm_exit(1,"ERROR: Invalid argument [$arg]! Try -?\n");
            }
        } else {
            if ($cnt == 0) {
                $in_file1 = $arg;
                prt("Set input file 1 to [$in_file1]\n");
            } elsif ($cnt == 1) {
                $in_file2 = $arg;
                prt("Set input file 2 to [$in_file2]\n");
            } else {
                pgm_exit(2,"ERROR: Already have [$in_file1], and [$in_file2]! WHAT IS THIS [$arg]?\n");
            }
            $cnt++;
        }
        shift @av;
    }

    if ((length($in_file1) ==  0) && $debug_on) {
        $in_file1 = $def_file1;
    }
    if ((length($in_file2) ==  0) && $debug_on) {
        $in_file2 = $def_file2;
    }
    if (length($in_file1) ==  0) {
        pgm_exit(1,"ERROR: No first input files found in command!\n");
    }
    if (length($in_file2) ==  0) {
        pgm_exit(1,"ERROR: No second input files found in command!\n");
    }
    if (! -f $in_file1) {
        pgm_exit(1,"ERROR: Unable to find in file [$in_file1]! Check name, location...\n");
    }
    if (! -f $in_file2) {
        pgm_exit(1,"ERROR: Unable to find in file [$in_file2]! Check name, location...\n");
    }
}

# eof - template.pl
