// load-scene.cxx
// load a scenery tile
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <simgear/compiler.h>
#include <simgear/bucket/newbucket.hxx>
#include <simgear/io/lowlevel.hxx>
#include <simgear/misc/sg_path.hxx>
#include <simgear/misc/strutils.hxx>
#include <simgear/io/sg_binobj.hxx>
#include <simgear/misc/sgstream.hxx>
#include <string>

#include "load-scene.hxx"
#include "lib_sprtf.hxx"
#include "poly-utils.hxx"
#include "poly-map.hxx"
#include "poly-out.hxx"
#include "poly-outGL.hxx"
#include "poly-text.hxx" // for paint_next_line_text(char*)

#include <sstream>   // for ostingstream

using namespace std;

// DEBUG
//#define sprtf_each_pt3  sprtf
#define sprtf_each_pt3

extern void set_ref_xyz( sgVec3 xyz );

void Show_Tiles_Loaded( void );

float autoscale = 0.0f;                 // 0.0f == no autoscale
bool global = false;
bool doublebuffer = true, headless = false, create_jpeg = false;
bool smooth_shade = true;
bool textured_fonts = false; // true;
int features = MapMaker::DO_SHADE | MapMaker::DO_VERBOSE;
int jpeg_quality = 75;
int rescale_factor = 1;
char * font_name = NULL;
MapMaker mapobj;  // defaults int size = 256, int scale = 100000
OutputGL mapout;

typedef vector<TGTileLoad> vec_tgtl;

void set_mapobj_mapsize( int size   ) { mapobj.setSize(size);   }
void set_mapobj_fgroot( char * root ) { mapobj.setFGRoot(root); }
bool load_mapobj_materials( char * file )
{
   bool curr = mapobj.palette_loaded;  // get any current load
   mapobj.palette_loaded = false;      // set it FALSE
   mapobj.read_materials( file );      // load this new file
   bool load = mapobj.palette_loaded;  // get what happened
   if( load && !curr )  // if it worked, and
     curr = load; // !previous, then set loaded
     mapobj.palette_loaded = curr;
   return load;   // and advise what happend to this file.
}
bool is_mapobj_palette_loaded(void) { return mapobj.palette_loaded; }
MapMaker & get_map_obj(void) { return mapobj; }
const float *get_map_palette_color_ptr( std::string & mats )
{
    return get_map_obj().string_to_rgba_color(mats);
}
vec_tgtl tiles_loaded;

int not_in_token( vector<string> & token, string & fp )
{
   size_t len = token.size();
   size_t i;
   for( i = 0; i < len; i++ ) {
      if( token[i] == fp )
         return 0;
   }
   return 1;
}

vector<string> pts_mat_list;
vector<string> tris_mat_list;
vector<string> strips_mat_list;
vector<string> fans_mat_list;

int add_to_list( string & stg, vector<string> & list )
{
   size_t len = list.size();
   size_t i;
   for( i = 0; i < len; i++ ) {
      if( list[i] == stg )
         return 0;
   }
   list.push_back(stg);
   return 1;
}

int add_to_pts_mat_list( string & stg )    { return add_to_list( stg, pts_mat_list ); }
int add_to_tris_mat_list( string & stg )   { return add_to_list( stg, tris_mat_list ); }
int add_to_strips_mat_list( string & stg ) { return add_to_list( stg, strips_mat_list ); }
int add_to_fans_mat_list( string & stg )   { return add_to_list( stg, fans_mat_list ); }

void clear_mat_list( string & desc, vector<string> & list )
{
   sprtf( "%s %d...\n", desc.c_str(), list.size() );
   for( vector<string>::iterator it = list.begin();
      it != list.end(); it++ ) {
         string s = *it;
         sprtf( "%s ", s.c_str() );
   }
   if( list.size() )
      sprtf("\n");
   list.clear();
}

void clear_all_mat_lists(void)
{
   string s;
   s = "pts materials saved";
   clear_mat_list( s, pts_mat_list );
   s = "tris materials saved";
   clear_mat_list( s, tris_mat_list );
   s = "strips materials saved";
   clear_mat_list( s, strips_mat_list );
   s = "fans materials saved";
   clear_mat_list( s, fans_mat_list );
}

void clear_tiles_loaded(void)
{
   sprtf( "Clearing %d tiles loaded...\n", tiles_loaded.size() );
   for( vec_tgtl::iterator it = tiles_loaded.begin(); it != tiles_loaded.end(); it++ )
   {
      TGTileLoad tgtl = *it;
      tgtl.vec_bin_files.clear();
      tgtl.vec_bin_objs.clear();
      tgtl.vec_bin_types.clear();
      tgtl.vec_pt3_name.clear();
      tgtl.vec_tg_objs.clear();
   }
   tiles_loaded.clear();
   clear_all_mat_lists();
}

void load_scenery_tile( string path, SGBucket & b )
{
   int   fnd = 0;
   SGPath fp = path;
   SGPath p  = path;
   double dlat = b.get_center_lat();
   double dlon = b.get_center_lon();
   vector<string> token;
   fp.append("*");
   //load_via_mapmaker( dlat, dlon, (char *)path.c_str() );

#ifdef _MSC_VER
   WIN32_FIND_DATA fd;
   HANDLE hfind = FindFirstFile( fp.str().c_str(), &fd );
   if( hfind && (hfind != (HANDLE)-1 ) ) {
      string bp = b.gen_base_path();
      token.clear();
      token = simgear::strutils::split( bp, "/" );
      do {
         if(( strcmp(fd.cFileName, "." ) == 0 ) ||
            ( strcmp(fd.cFileName, "..") == 0 )) {
            // skip
         } else {
            // have file or directory, of this directory...
            if( strcmp(fd.cFileName, "Terrain") == 0 ) {
               fnd = 1;
               p.append(fd.cFileName);
               p.append(bp);
               break;
            } else if( strcmp(fd.cFileName, token[0].c_str()) == 0 ) {
               fnd = 2;
               p.append(bp);
               break;
            }
            // need to handle other options
         }
      } while ( FindNextFile( hfind, &fd ) );
      FindClose(hfind);
      token.clear();
      if(fnd) {
         string base = p.str();
         p.append( b.gen_index_str() + ".*");
         hfind = FindFirstFile( p.str().c_str(), &fd );
         if( hfind && (hfind != (HANDLE)-1 ) ) {
            do {
               if(( strcmp(fd.cFileName, "." ) == 0 ) ||
                  ( strcmp(fd.cFileName, "..") == 0 )) {
                  // skip
               } else {
                  string file = fd.cFileName;
                  size_t len = file.size();
                  if( file.find_last_of(".stg") == (len - 1) ) {
                     SGPath fp(base);
                     fp.append(fd.cFileName);
                     if( not_in_token( token, fp.str() ) ) {
                        token.push_back(fp.str()); // accumulate files
                        tiles_loaded.push_back( TGTileLoad( b, path ) );
                     }
                  }
               }
            } while ( FindNextFile( hfind, &fd ) );
            FindClose(hfind);
         }
      }
      if( token.size() > 0 ) {
         // we have a/some files to party with... *.stg
         for( int i = 0; i < token.size(); i++ ) {
            sprtf( "File: %s\n", token[i].c_str() );
         }
         Show_Tiles_Loaded();
      }
   }

#else // !_MSC_VER
#pragma message("ERROR: NOT YET CODED")
.error ERROR not yet coded...

#endif // _MSC_VER y/n
}

TGTileLoad::TGTileLoad( SGBucket& b, string & base_path_in )
{
    is_valid = false;
    tile_bucket = b;
    string b_base = b.gen_base_path();
    string b_index = b.gen_index_str();
     mapobj.tile_bucket = b;
     // open the STG file
     //SGPath path(base_path_in);
     SGPath path;
     path = base_path_in;
     if( path.str().find("/Terrain") == -1 )
         path.append("Terrain");
     path.append(b_base);
     base_path = path.str();
     path.append(b_index + ".stg");
     tileFileName = path.str();
     if( path.exists() ) {
         //if( is_file_or_directory( tileFileName ) == 2 ) {
         is_valid = true;
     } else {
         path = base_path_in;
         path.append(b_base);
         base_path = path.str();
         path.append(b_index + ".stg");
         tileFileName = path.str();
         if( path.exists() ) {
             //if( is_file_or_directory( tileFileName ) == 2 ) {
             is_valid = true;
         } else {
             sprtf("\nWARNING: File [%s] NOT FOUND! (nor by adding '/Terrain'!\n",
                 path.c_str() );
         }
     }

     if(is_valid)
         loadSTGfile();

}

bool TGTileLoad::loadSTGfile( void )
{
   ostringstream msg;
   SGBinObject object;
   bool found_base = false;
   bool has_base = false;
   bool ret = false;
   float clon = (float)(tile_bucket.get_center_lon() * SG_DEGREES_TO_RADIANS);
   float clat = (float)(tile_bucket.get_center_lat() * SG_DEGREES_TO_RADIANS);
   sgVec3 xyz;
   float r_e;
   string current_path = base_path;
   int  i;
   sg_gzifstream in( tileFileName );   // open STG file
   msg.str("");
   msg << "STG: Loading [" << tileFileName << "]";
   if ( in.is_open() )
      msg << " open";
   else
      msg << " FAILED";
   paint_next_line_text( (char *)msg.str().c_str() );
   if ( in.is_open() ) {
      xyz_lat( clat, clon, xyz, &r_e );
      {
         double dbl[3];
         sgVec3 dst;
         dbl[0] = xyz[0];
         dbl[1] = xyz[1];
         dbl[2] = xyz[2];
         sprtf("From:d: rlon %f, rlat %f, got xyz %0.1f,%0.1f,%0.1f from xyz_lat(%0.4f,%0.4f)\n",
            clon, clat,
            dbl[0], dbl[1], dbl[2],
            clon * SG_RADIANS_TO_DEGREES, clat * SG_RADIANS_TO_DEGREES);
         sprtf("floats: rlon %f, rlat %f, got xyz %0.1f,%0.1f,%0.1f from xyz_lat(%0.4f,%0.4f)\n",
            clon, clat,
            xyz[0], xyz[1], xyz[2],
            clon * SG_RADIANS_TO_DEGREES, clat * SG_RADIANS_TO_DEGREES);
         ab_xy( xyz, xyz, dst );
         sprtf("ab_xy(xyz,xyz,dest) = %0.1f,%0.1f,%0.1f (%d,%d)\n",
            dst[0], dst[1], dst[2],
            win_int(dst[0]), win_int(dst[1]) );
         float scale = mapobj.getScale();
         float lat, lon;
         lat_ab( scale / 2, scale / 2, clat, clon, &lat, &lon );
         sprtf("Using scale %f, got lon,lat, %f,%f (%f,%f)\n", scale,
            lon, lat, (lon * SG_RADIANS_TO_DEGREES), (lat * SG_RADIANS_TO_DEGREES));
      }
      //OutputGL output( "temp1.png", mapobj.getSize(), smooth_shade, 
	   //    textured_fonts, font_name, create_jpeg, jpeg_quality, rescale_factor );
      //mapobj.output = &output;
      //output.ref_xyz[0] = xyz[0];   // set reference xyz
      //output.ref_xyz[1] = xyz[1];   // set reference xyz
      //output.ref_xyz[2] = xyz[2];   // set reference xyz
      mapobj.output = &mapout;   // use a DEFAULT construction

      mapout.ref_xyz[0] = xyz[0];   // set reference xyz
      mapout.ref_xyz[1] = xyz[1];   // set reference xyz
      mapout.ref_xyz[2] = xyz[2];   // set reference xyz
      set_ref_xyz( xyz );
      if (!mapobj.palette_loaded) {
         sprtf( "WARNING: loading palette! This should have been done.\n");
         mapobj.read_materials();
      }
      if (!mapobj.palette_loaded) {
#ifdef POLY_VIEW
         sprtf( "WARNING: No palette loaded. Aborting read of STG file\n");
#else // #ifdef POLY_VIEW
         fprintf(stderr, "No palette loaded.\n");
#endif // #ifdef POLY_VIEW
         return false;
      }

      string token, name;
      while ( ! in.eof() ) {
         in >> token;
         if ( token == "OBJECT_BASE" ) {
            ret = true;
            in >> name >> ::skipws;
            if ( !found_base ) {
               msg.str("");
               msg << "STG:OBJECT_BASE: loading [" << name << "]";
               paint_next_line_text( (char *)msg.str().c_str() );
               found_base =true;
               has_base = true;
               SGPath custom_path = current_path;
               custom_path.append( name );
               if( object.read_bin( custom_path.str() ) ) {
                  ret = true;
                  vec_bin_objs.push_back(object);
                  vec_bin_types.push_back(token);
                  vec_bin_files.push_back(custom_path.str());
               }
               mapobj.setFeatures(features);
               mapobj.process_binary_file( (char *)custom_path.str().c_str(), xyz );
               sprtf("STG:OBJECT_BASE: name %s loaded.\n", name.c_str() );
            } else {
               sprtf("STG:OBJECT_BASE: but base already found!\n" );
            }
         } else if ( token == "OBJECT" ) {
            ret = true;
            in >> name >> ::skipws;
            msg.str("");
            msg << "STG:OBJECT: loading [" << name << "]";
            paint_next_line_text( (char *)msg.str().c_str() );
            if ( !found_base || has_base ) {
               SGPath custom_path = current_path;
               custom_path.append( name );
               if( object.read_bin( custom_path.str() ) ) {
                  ret = true;
                  vec_bin_objs.push_back(object);
                  vec_bin_types.push_back(token);
                  vec_bin_files.push_back(custom_path.str());
                  i = mapobj.process_binary_file( (char *)custom_path.str().c_str(), xyz );
                  sprtf("STG:OBJECT: file [%s] load %s.\n", custom_path.c_str(),
                      (i ? "ok" : "FAILED") );
               } else {
                  sprtf("WARNING:STG: read_bin FAILED on %s! [%s]\n",
                      name.c_str(), custom_path.c_str() );
               }
            } else {
               sprtf("WARNING: STG:NOT found base, or has base!!!\n" );
            }
         } else if ( token == "OBJECT_STATIC" ) {
            ret = true;
            // load object info
            TGObject o;
            o.shared = false;
            o.agl = false;
            in >> o.name >> o.lon >> o.lat >> o.height >> o.heading >> ::skipws;
            sprtf("STG:OBJECT_STATIC: loading %s...\n", o.name.c_str() );
            vec_tg_objs.push_back(o);
            vec_obj_types.push_back(token);
            sprtf("STG:OBJECT_STATIC: name %s loaded...\n", o.name.c_str() );
         } else if ( token == "OBJECT_SHARED" ) {
            ret = true;
            // load object info
            TGObject o;
            o.shared = false;
            o.agl = false;
            in >> o.name >> o.lon >> o.lat >> o.height >> o.heading >> ::skipws;
            sprtf("STG:OBJECT_SHARED: loading %s...\n", o.name.c_str() );
            vec_tg_objs.push_back(o);
            vec_obj_types.push_back(token);
            sprtf("STG:OBJECT_SHARED: name %s loaded...\n", o.name.c_str() );
         } 
/*                else if ( token == "OBJECT_TAXI_SIGN" ) {
                     ret = true;
                  } else if ( token == "OBJECT_RUNWAY_SIGN" ) {
                     ret = true;
                  } else if ( token == "RWY_LIGHTS" ) {
                     ret = true;
                  }
 */
          else {
             // unknown or unsupported token
             sprtf("WARNING:STG: token %s: unknown or unsupported token!\n",
                token.c_str() );
             in >> ::skipws;
          }
      }
   }
   return ret;
}

TGTileLoad::~TGTileLoad()
{
   //sprtf("TGTileLoad: destructor.\n");
}

double g_wgs_min_lon = 2000.0;
double g_wgs_max_lon = -2000.0;
double g_wgs_min_lat = 2000.0;
double g_wgs_max_lat = -2000.0;
double g_wgs_min_hgt = 99999.0;
double g_wgs_max_hgt = -99999.0;

Point3D g_pt3d_max, g_pt3d_min;
static int done_init_globals = 0;

#define set_pt3d_max(a) \
   a.setlon(-2000.0); \
   a.setlat(-2000.0); \
   a.setelev(-99999.0)

#define set_pt3d_min(a) \
   a.setlon(2000.0); \
   a.setlat(2000.0); \
   a.setelev(99999.0)

#define set_pt3d_gt(a,b) \
   if( b.lon() > a.lon() ) a.setlon(b.lon()); \
   if( b.lat() > a.lat() ) a.setlat(b.lat()); \
   if( b.elev() > a.elev() ) a.setelev(b.elev())

#define set_pt3d_lt(a,b) \
   if( b.lon() < a.lon() ) a.setlon(b.lon()); \
   if( b.lat() < a.lat() ) a.setlat(b.lat()); \
   if( b.elev() < a.elev() ) a.setelev(b.elev())

#define sprtf_showverts
//#define sprtf_showverts sprtf

void Compare_WGS84_with_Normals( SGBinObject * o )
{
   SGVec3d v;
   unsigned short vers = o->get_version();
   SGVec3d gbs_cent = o->get_gbs_center2();
   mapobj.gbs_cent = gbs_cent;
   float rad = o->get_gbs_radius();
   Point3D cgbs, vt, nm; // this is cartesian = o->get_gbs_center();
   v[0] = 0.0;    // so zero the offset,
   v[1] = 0.0;
   v[2] = 0.0;    // and convert, I hope
   cgbs = cartv3d_2_p3dllh( gbs_cent, v );
   sprtf( "Vers:%d - cent lon,lat,elev=%f,%f,%f, xyz=%f,%f,%f, rad %f\n",
      vers, cgbs.lon(), cgbs.lat(), cgbs.elev(),
      gbs_cent[0], gbs_cent[1], gbs_cent[2],
      rad );
   Point3D pt3d_max;
   Point3D pt3d_min;
   set_pt3d_max(pt3d_max);
   set_pt3d_min(pt3d_min);
   if( !done_init_globals ) {
      done_init_globals = 1;
      set_pt3d_max(g_pt3d_max);
      set_pt3d_min(g_pt3d_min);
   }
   vector<SGVec3d> vertices = o->get_wgs84_nodes();
   int wgs_size = vertices.size();
   vector<SGVec3f> normals = o->get_normals();
   int norm_size = normals.size();
   if( wgs_size && ( wgs_size == norm_size )) {
      sprtf( "Compare List of %d vertices and normals...\n", wgs_size );
      for( int i = 0; i < wgs_size; i++ ) {
         SGVec3d vert = vertices[i];
         SGVec3f norm = normals[i];
         vt = cartv3d_2_p3dllh( vert, gbs_cent );
         //nm = cartv3f_2_p3dllh( norm, gbs_cent );
         sprtf_showverts("%3d: %s = %f,%f,%f vs %f,%f,%f\n", (i + 1),
            pt3d2stg(vt),
            vert[0], vert[1], vert[2],
            norm[0], norm[1], norm[2] );
         set_pt3d_gt(pt3d_max, vt);
         set_pt3d_lt(pt3d_min, vt);
      }
      set_pt3d_gt(g_pt3d_max, pt3d_max);
      set_pt3d_lt(g_pt3d_min, pt3d_min);
      sprtf( "DONE List %d - min %s, max %s\n", wgs_size,
         pt3d2stg(pt3d_min), pt3d2stg(pt3d_max) );
   } else {
      sprtf( "WARNING: Not same size - vertices = %d, normals = %d\n",
         wgs_size, norm_size );
      if(wgs_size) {
         sprtf( "Show List of %d vertices...\n", wgs_size );
         for( int i = 0; i < wgs_size; i++ ) {
            SGVec3d vert = vertices[i];
            vt = cartv3d_2_p3dllh( vert, gbs_cent );
            sprtf_showverts("%3d: %s = %f,%f,%f \n", (i + 1),
               pt3d2stg(vt),
               vert[0], vert[1], vert[2] );
            set_pt3d_gt(pt3d_max, vt);
            set_pt3d_lt(pt3d_min, vt);
         }
         set_pt3d_gt(g_pt3d_max, pt3d_max);
         set_pt3d_lt(g_pt3d_min, pt3d_min);
         sprtf( "DONE List %d - min %s, max %s\n", wgs_size,
            pt3d2stg(pt3d_min), pt3d2stg(pt3d_max) );
      } else {
         sprtf( "WARNING: vertices ZERO\n" );
      }
   }
}


void Show_Tiles_Loaded( void )
{
   size_t max = tiles_loaded.size();
   size_t i;
   size_t k, g_tot;
   SGVec3d v;
   SGVec3f vf;
   double x,y,z;
   double min_lat, min_lon, max_lat, max_lon, max_alt, min_alt;
   vec_pt3d wgs_pts, norm_pts;
   int_list ilist;
   Point3D pt3d;
   Point3D pt3d3[3];
   sprtf("load-scene:Show_Tiles_Loaded: Got %d tiles loaded...\n", max );
   g_tot = 0;
   min_lat = min_lon = min_alt = 2000000.0;
   max_lat = max_lon = max_alt = -2000000.0;
   for(i = 0; i < max; i++) {
      TGTileLoad tgtl = tiles_loaded[i];  // make copy
      size_t bmax = tgtl.vec_bin_objs.size();
      size_t j;
      for( j = 0; j < bmax; j++ )
      {
         SGBinObject * o = &tgtl.vec_bin_objs[j];
         Compare_WGS84_with_Normals(o);
         unsigned short vers = o->get_version();
         float rad = o->get_gbs_radius();
         SGVec3d gbs_cent = o->get_gbs_center2();
         mapobj.gbs_cent = gbs_cent;
         Point3D cgbs; // this is cartesian = o->get_gbs_center();
         v[0] = 0.0;    // so zero the offset,
         v[1] = 0.0;
         v[2] = 0.0;    // and convert, I hope
         cgbs = cartv3d_2_p3dllh( gbs_cent, v );
         sprtf( "Version %d - center x.y.z %f,%f,%f, radius %f\n",
            vers, 
            cgbs.lon(), 
            cgbs.lat(),
            cgbs.elev(),
            rad );
#define MX_SZS 24
         static size_t szs[MX_SZS];
         static char * nms[MX_SZS];
         int total = 0;
         size_t wgs_size, norm_size;
         vector<SGVec3d> wgs_list = o->get_wgs84_nodes();
         szs[0] = wgs_size = wgs_list.size();
         nms[0] = "wgs84_nodes";
         wgs_pts.clear();
         if(szs[0]) {
            for(k = 0; k < szs[0]; k++) {
               v = wgs_list[k];
               pt3d = cartv3d_2_p3dllh( v, gbs_cent );
               wgs_pts.push_back(pt3d);
               ilist.push_back(0);
               x = pt3d.lon();  // in degrees, so NO SG_RADIANS_TO_DEGREES;
               y = pt3d.lat();  // DEGREES;
               z = pt3d.elev(); // meters
               if( x > max_lon ) max_lon = x;
               if( x < min_lon ) min_lon = x;
               if( y > max_lat ) max_lat = y;
               if( y < min_lat ) min_lat = y;
               if( z > max_alt ) max_alt = z;
               if( z < min_alt ) min_alt = z;
            }
            sprtf( "wsg84:%3d: x,y,z min %f,%f,%f, max %f,%f,%f\n", szs[0],
               min_lon, min_lat, min_alt,
               max_lon, max_lat, max_alt );
            // add to original, not copy
            tiles_loaded[i].m_vv_pt3d.push_back(wgs_pts);      // save set of points
            tiles_loaded[i].vec_pt3_name.push_back(nms[0]);// and their name/type
         }
         vector<SGVec4f> c_list = o->get_colors();
         szs[1] = c_list.size();
         nms[1] = "colors";
         vector<SGVec3f> norm_list = o->get_normals();
         szs[2] = norm_size = norm_list.size();
         nms[2] = "normals";
         norm_pts.clear();
         if(norm_list.size() > 0) { // szs[2]
            for(k = 0; k < szs[2]; k++) {
               vf = norm_list[k];   // get vector (floats)
               v[0] = vf[0];  // ugly conversion to doubles
               v[1] = vf[1];
               v[2] = vf[2];
               //Point3D pt3d = cartv3d_2_p3dllh( v, gbs_cent );
               Point3D pt3d = cartv3f_2_p3dllh( vf, gbs_cent );
               norm_pts.push_back(pt3d);
               x = pt3d.lon();  // in degrees, so NO SG_RADIANS_TO_DEGREES;
               y = pt3d.lat();  // DEGREES;
               z = pt3d.elev(); // meters
               if( x > max_lon ) max_lon = x;
               if( x < min_lon ) min_lon = x;
               if( y > max_lat ) max_lat = y;
               if( y < min_lat ) min_lat = y;
               if( z > max_alt ) max_alt = z;
               if( z < min_alt ) min_alt = z;
            }
            sprtf( "norms:%d: x,y,z min %f,%f,%f, max %f,%f,%f\n", szs[0],
               min_lon, min_lat, min_alt,
               max_lon, max_lat, max_alt );
            // add to original, not copy
            tiles_loaded[i].m_vv_pt3d.push_back(norm_pts);      // save set of points
            tiles_loaded[i].vec_pt3_name.push_back(nms[2]);// and their name/type
         }
         vector<SGVec2f> tex_list = o->get_texcoords();
         szs[3] = tex_list.size();
         nms[3] = "texcoords";
         group_list ptsv_list = o->get_pts_v();
         szs[4] = ptsv_list.size();
         nms[4] = "pts_v";
         group_list ptsn_list = o->get_pts_n();
         szs[5] = ptsn_list.size();
         nms[5] = "pts_n";
         group_list ptsc_list = o->get_pts_c();
         szs[6] = ptsc_list.size();
         nms[6] = "pts_c";
         group_list ptstc_list = o->get_pts_tc();
         szs[7] = ptstc_list.size();
         nms[7] = "pts_tc";
         string_list ptMat_list = o->get_pt_materials();
         szs[8] = ptMat_list.size();
         nms[8] = "pt_mats";
         group_list tris_v_list = o->get_tris_v();
         szs[9] = tris_v_list.size();
         nms[9] = "tris_v";
         group_list tris_n_list = o->get_tris_n();
         szs[10] = tris_n_list.size();
         nms[10] = "tris_n";
         group_list tris_c_list = o->get_tris_c();
         szs[11] = tris_c_list.size();
         nms[11] = "tris_c";
         group_list tris_tc_list = o->get_tris_tc();
         szs[12] = tris_tc_list.size();
         nms[12] = "tris_tc";
         string_list tri_mats = o->get_tri_materials();
         szs[13] = tri_mats.size();
         nms[13] = "tri_mats";
         group_list strips_v = o->get_strips_v();
         szs[14] = strips_v.size();
         nms[14] = "strip_v";
         group_list strips_n = o->get_strips_n();
         szs[15] = strips_n.size();
         nms[15] = "strip_n";
         group_list strc_list = o->get_strips_c();
         szs[16] = strc_list.size();
         nms[16] = "strip_c";
         group_list strtc_list = o->get_strips_tc();
         szs[17] = strtc_list.size();
         nms[17] = "strip_tc";
         string_list strmat_list = o->get_strip_materials();
         szs[18] = strmat_list.size();
         nms[18] = "strip_mats";
         group_list fans_v = o->get_fans_v();
         szs[19] = fans_v.size();
         nms[19] = "fans_v";
         group_list fans_n = o->get_fans_n();
         szs[20] = fans_n.size();
         nms[20] = "fans_n";
         group_list fans_c_list = o->get_fans_c();
         szs[21] = fans_c_list.size();
         nms[21] = "fans_c";
         group_list fans_tc_list = o->get_fans_tc();
         szs[22] = fans_tc_list.size();
         nms[22] = "fans_tc";
         string_list fans_mat = o->get_fan_materials();
         szs[23] = fans_mat.size();
         nms[23] = "fans_mats";
         // add some warnings - from obj.cxx
         unsigned int grp;
         int vert0, vert1, vert2;
         int norm0, norm1, norm2;
         int_list vert, norm;
         std::string materialName;
         int   ii;
         //if (o->get_pts_v().size() != o->get_pts_n().size()) {
         if( ptsv_list.size() != ptsn_list.size() ) {
            sprtf("WARNING: pts_v %d, NE pts_n size!\n", ptsv_list.size(), ptsn_list.size() );
            // o->get_pts_v().size(), o->get_pts_n().size() );
         } else {
            //for (grp = 0; grp < o->get_pts_v().size(); ++grp) {
            for (grp = 0; grp < ptsv_list.size(); ++grp) {
               //materialName = o->get_pt_materials()[grp];
               materialName = ptMat_list[grp];
               add_to_pts_mat_list( materialName );
               vert = ptsv_list[grp];
               //norm = tris_n_list[grp];
               for(ii = 0; ii < vert.size(); ii++) {
                  vert0 = vert[ii];
                     if(vert0 < wgs_size) {
                        //if(vert1 < wgs_size) {
                        //   if(vert2 < wgs_size) {
                              //pv_draw_a_tri(vert0, vert1, vert2, norm0, norm1, norm2, v, n, vp, np, col);
                              pt3d3[0] = wgs_pts[vert0];      // save set of points
                        //      pt3d3[1] = wgs_pts[vert1];      // save set of points
                        //      pt3d3[2] = wgs_pts[vert2];      // save set of points
                              ilist[vert0]++;
                        //      ilist[vert1]++;
                        //      ilist[vert2]++;
                              sprtf_each_pt3("%d:%d:pts:%03d: %0.6f,%0.6f\n",
                                 i, j,
                                 vert0, // vert1, vert2,
                                 pt3d3[0].lon(), pt3d3[0].lat() );
                         //  } else {
                         //     sprtf("WARNING: offset2 %d OUT OF RANGE %d\n", vert2, wgs_size );
                         //  }
                        //} else {
                        //   sprtf("WARNING: offset1 %d OUT OF RANGE %d\n", vert1, wgs_size );
                        //}
                     } else {
                        sprtf("WARNING: offset0 %d OUT OF RANGE %d\n", vert0, wgs_size );
                     }
               }
#if 0
               if (3 <= materialName.size() && materialName.substr(0, 3) != "RWY") {
                  // Just plain lights. Not something for the runway.
                  //addPointGeometry(tileLights, obj.get_wgs84_nodes(), color,
                  //       obj.get_pts_v()[grp]);
               } else if (materialName == "RWY_BLUE_TAXIWAY_LIGHTS" || materialName == "RWY_GREEN_TAXIWAY_LIGHTS") {
                  // addPointGeometry(taxiLights, obj.get_wgs84_nodes(), obj.get_normals(),
                  //       color, obj.get_pts_v()[grp], obj.get_pts_n()[grp]);
               } else if (materialName == "RWY_VASI_LIGHTS") {
                  // vasiLights.push_back(SGDirectionalLightBin());
                  //  addPointGeometry(vasiLights.back(), obj.get_wgs84_nodes(),
                  //    obj.get_normals(), color, obj.get_pts_v()[grp],
                  //       obj.get_pts_n()[grp]);
               } else if (materialName == "RWY_SEQUENCED_LIGHTS") {
                  // rabitLights.push_back(SGDirectionalLightBin());
                  // addPointGeometry(rabitLights.back(), obj.get_wgs84_nodes(),
                  //       obj.get_normals(), color, obj.get_pts_v()[grp],
                  //       obj.get_pts_n()[grp]);
               } else if (materialName == "RWY_ODALS_LIGHTS") {
                  // odalLights.push_back(SGLightBin());
                  //  addPointGeometry(odalLights.back(), obj.get_wgs84_nodes(),
                  //       color, obj.get_pts_v()[grp]);
               } else if (materialName == "RWY_REIL_LIGHTS") {
                  // reilLights.push_back(SGDirectionalLightBin());
                  // addPointGeometry(reilLights.back(), obj.get_wgs84_nodes(),
                  //   obj.get_normals(), color, obj.get_pts_v()[grp],
                  //   obj.get_pts_n()[grp]);
               } else {
                  // what is left must be runway lights
                  // addPointGeometry(runwayLights, obj.get_wgs84_nodes(),
                  // obj.get_normals(), color, obj.get_pts_v()[grp],
                  // obj.get_pts_n()[grp]);
               }
#endif // 0
            }
         }

         // tris_v_list
         // ===========
         //assert((tris_v_list.size() % 3) == 0);
         if (tris_v_list.size() != tris_n_list.size()) {
            sprtf("WARNING: tris_v %d, NE tris_n size!\n",
            tris_v_list.size(), tris_n_list.size() );
         } else {
            for (grp = 0; grp < tris_v_list.size(); grp++) {
               materialName = o->get_tri_materials()[grp];
               add_to_tris_mat_list( materialName );
               vert = tris_v_list[grp];
               norm = tris_n_list[grp];
               if((vert.size() % 3) == 0) {
                  for(ii = 0; ii < vert.size(); ii += 3) {
                     if((ii + 2) < vert.size()) {
                        vert0 = vert[ii];
                        vert1 = vert[ii+1];
                        vert2 = vert[ii+2];
                        if(norm.size() >= ii + 2) {
                           norm0 = norm[ii];
                           norm1 = norm[ii+1];
                           norm2 = norm[ii+2];
                        } else {
                           norm0 = vert0;
                           norm1 = vert1;
                           norm2 = vert2;
                        }
                     }
                     if(vert0 < wgs_size) {
                        if(vert1 < wgs_size) {
                           if(vert2 < wgs_size) {
                              //pv_draw_a_tri(vert0, vert1, vert2, norm0, norm1, norm2, v, n, vp, np, col);
                              pt3d3[0] = wgs_pts[vert0];      // save set of points
                              pt3d3[1] = wgs_pts[vert1];      // save set of points
                              pt3d3[2] = wgs_pts[vert2];      // save set of points
                              ilist[vert0]++;
                              ilist[vert1]++;
                              ilist[vert2]++;
                              sprtf_each_pt3("%d:%d:tris:%03d,%03d,%03d: %0.6f,%0.6f - %0.6f,%0.6f - %0.6f,%0.6f \n",
                                 i, j,
                                 vert0, vert1, vert2,
                                 pt3d3[0].lon(), pt3d3[0].lat(), 
                                 pt3d3[1].lon(), pt3d3[1].lat(), 
                                 pt3d3[2].lon(), pt3d3[2].lat() );
                           } else {
                              sprtf("WARNING: offset2 %d OUT OF RANGE %d\n", vert2, wgs_size );
                           }
                        } else {
                           sprtf("WARNING: offset1 %d OUT OF RANGE %d\n", vert1, wgs_size );
                        }
                     } else {
                        sprtf("WARNING: offset0 %d OUT OF RANGE %d\n", vert0, wgs_size );
                     }
                  }
               } else {
                  sprtf("WARNING: vert %d NOT DIVISABLE BY 3!\n", vert.size() );
               }
            }
         }

         // strips
         // ======
         if (strips_v.size() != strips_n.size()) {
            sprtf("WARNING: strips_v %d, NE strips_n size!\n",
            strips_v.size(), strips_n.size() );
         } else {
            for (grp = 0; grp < strips_v.size(); ++grp) {
               materialName = o->get_strip_materials()[grp];
               add_to_strips_mat_list( materialName );
               vert = strips_v[grp];
               norm = strips_v[grp];
               if( vert.size() > 2 ) {
                  vert0 = vert[0];
                  vert1 = vert[1];
                  if(norm.size() > 2) {
                     norm0 = norm[0];
                     norm1 = norm[1];
                  } else {
                     norm0 = vert0;
                     norm1 = vert1;
                  }
                  for (ii = 2; ii < vert.size(); ii++) {
                     vert2 = vert[ii];
                     if(ii < norm.size())
                        norm2 = norm[ii];
                     else
                        norm2 = vert2;
                     if(vert0 < wgs_size) {
                        if(vert1 < wgs_size) {
                           if(vert2 < wgs_size) {
                              //pv_draw_a_tri(cvert, vert1, vert2, cnorm, norm1, norm2, v, n, vp, np, col);
                              // tiles_loaded[i].m_vv_pt3d.push_back(pts);      // save set of points
                              pt3d3[0] = wgs_pts[vert0];      // save set of points
                              pt3d3[1] = wgs_pts[vert1];      // save set of points
                              pt3d3[2] = wgs_pts[vert2];      // save set of points
                              ilist[vert0]++;
                              ilist[vert1]++;
                              ilist[vert2]++;
                              sprtf_each_pt3("%d:%d:strips:%03d,%03d,%03d: %0.6f,%0.6f - %0.6f,%0.6f - %0.6f,%0.6f \n",
                                 i, j,
                                 vert0, vert1, vert2,
                                 pt3d3[0].lon(), pt3d3[0].lat(), 
                                 pt3d3[1].lon(), pt3d3[1].lat(), 
                                 pt3d3[2].lon(), pt3d3[2].lat() );
                           } else {
                              sprtf("WARNING: offset2 %d OUT OF RANGE %d\n", vert2, wgs_size );
                           }
                        } else {
                           sprtf("WARNING: offset1 %d OUT OF RANGE %d\n", vert1, wgs_size );
                        }
                     } else {
                        sprtf("WARNING: offset0 %d OUT OF RANGE %d\n", vert0, wgs_size );
                     }
                     vert1 = vert2;
                     norm1 = norm2;
                  }
               } else {
                  sprtf("WARNING: vert %d LESS THAN 3!\n", vert.size() );
               }
            }
         }
         // fans_v
         // ======
         if (fans_v.size() != fans_n.size()) {
            sprtf("WARNING: fans_v %d, NE fans_n size!\n",
            fans_v.size(), fans_n.size() );
         } else { 
            for (grp = 0; grp < fans_v.size(); ++grp) {
               materialName = o->get_fan_materials()[grp];
               add_to_fans_mat_list( materialName );
               vert = fans_v[grp];
               norm = fans_n[grp];
               if( vert.size() > 2 ) {
                  vert0 = vert[0];
                  vert1 = vert[1];
                  if(norm.size() > 2) {
                     norm0 = norm[0];
                     norm1 = norm[1];
                  } else {
                     norm0 = vert0;
                     norm1 = vert1;
                  }
                  for (ii = 2; ii < vert.size(); ii++) {
                     vert2 = vert[ii];
                     if(ii < norm.size())
                        norm2 = norm[ii];
                     else
                        norm2 = vert2;
                     if(vert0 < wgs_size) {
                        if(vert1 < wgs_size) {
                           if(vert2 < wgs_size) {
                              //pv_draw_a_tri(cvert, vert1, vert2, cnorm, norm1, norm2, v, n, vp, np, col);
                              // tiles_loaded[i].m_vv_pt3d.push_back(pts);      // save set of points
                              pt3d3[0] = wgs_pts[vert0];      // save set of points
                              pt3d3[1] = wgs_pts[vert1];      // save set of points
                              pt3d3[2] = wgs_pts[vert2];      // save set of points
                              ilist[vert0]++;
                              ilist[vert1]++;
                              ilist[vert2]++;
                              sprtf_each_pt3("%d:%d:fans:%03d,%03d,%03d: %0.6f,%0.6f - %0.6f,%0.6f - %0.6f,%0.6f \n",
                                 i, j,
                                 vert0, vert1, vert2,
                                 pt3d3[0].lon(), pt3d3[0].lat(), 
                                 pt3d3[1].lon(), pt3d3[1].lat(), 
                                 pt3d3[2].lon(), pt3d3[2].lat() );
                           } else {
                              sprtf("WARNING: offset2 %d OUT OF RANGE %d\n", vert2, wgs_size );
                           }
                        } else {
                           sprtf("WARNING: offset1 %d OUT OF RANGE %d\n", vert1, wgs_size );
                        }
                     } else {
                        sprtf("WARNING: offset0 %d OUT OF RANGE %d\n", vert0, wgs_size );
                     }
                     vert1 = vert2;
                     norm1 = norm2;
                  }
               } else {
                  sprtf("WARNING: vert %d LESS THAN 3!\n", vert.size() );
               }
            } // for each group
         }

         total = 0;
         int pcnt = 0;
         for(k = 0; k < MX_SZS; k++)
            total += szs[k];
         sprtf("load-scene: Summary of %d items collected...\n", total);
         total = 0;
         for(k = 0; k < MX_SZS; k++) {
            if( k && szs[k] && !pcnt )
               sprtf("\n");
            sprtf( "Item: %s %d ", nms[k], szs[k] );
            //if( k && (( k % 8 ) == 0 ))
            if( szs[k] )
               sprtf("\n");
            pcnt = szs[k];
            total += pcnt;
         }
         if(!pcnt)
            sprtf("\n");
         sprtf("load-scene: Total points %d\n", total );
         g_tot += total;
         size_t not_used = 0;
         for(ii = 0; ii < wgs_size; ii++) {
            if( ilist[ii] == 0 ) {
               sprtf( "%d ", ii );
               not_used++;
            }
         }
         if(not_used)
            sprtf( "\nWARNING:load-scene: Indexes NOT USED: %d\n", not_used );
         else
            sprtf( "load-scene: All wgs indexed vertices used\n" );

      } // for each bin_object per tile loaded
   } // for each TILE LOADED
   sprtf("load-scene:Show_Tiles_Loaded: Done...\n");
}

class WGS_NODES_str
{
public:
   WGS_NODES_str() { max = 0; };
   ~WGS_NODES_str() { max = 0; };
   size_t max;
   size_t currrent;
   TGTileLoad * ptgtl;
   size_t ms;
   size_t i;
};

typedef struct tagWGS_NODES {
   size_t max;
   size_t currrent;
   TGTileLoad * ptgtl;
   size_t ms;
   size_t i;
}WGS_NODES, * PWGS_NODES;

WGS_NODES wgs_nodes;

vec_pt3d * get_first_node( string & str )
{
   WGS_NODES * pwgs = &wgs_nodes;
   pwgs->max = tiles_loaded.size();
   pwgs->currrent = 0;
   while( pwgs->currrent < pwgs->max ) {
      pwgs->ptgtl = &tiles_loaded[pwgs->currrent];
      pwgs->ms = pwgs->ptgtl->vec_pt3_name.size();
      pwgs->ms = pwgs->ptgtl->m_vv_pt3d.size();
      for(pwgs->i = 0; pwgs->i < pwgs->ms; pwgs->i++) {
         if( pwgs->ptgtl->vec_pt3_name[pwgs->i] == str )
            return &pwgs->ptgtl->m_vv_pt3d[pwgs->i];
      }
      pwgs->currrent++;
   }
   return NULL;
}

vec_pt3d * get_next_node( string & str )
{
   WGS_NODES * pwgs = &wgs_nodes;
   if( pwgs->max ) {
      pwgs->currrent++;
      while( pwgs->currrent < pwgs->max ) {
         pwgs->ptgtl = &tiles_loaded[pwgs->currrent];
         pwgs->ms = pwgs->ptgtl->vec_pt3_name.size();
         pwgs->ms = pwgs->ptgtl->m_vv_pt3d.size();
         for(pwgs->i = 0; pwgs->i < pwgs->ms; pwgs->i++) {
            if( pwgs->ptgtl->vec_pt3_name[pwgs->i] == str )
               return &pwgs->ptgtl->m_vv_pt3d[pwgs->i];
         }
         pwgs->currrent++;
      }
   }
   return NULL;
}


vec_pt3d * get_first_wgs84_nodes( void )
{
   string str = "wgs84_nodes";
   return get_first_node(str);
}

vec_pt3d * get_next_wgs84_nodes( void )
{
   string str = "wgs84_nodes";
   return get_next_node(str);
}

vec_pt3d * get_first_normal_nodes( void )
{
   string str = "normals";
   return get_first_node(str);
}

vec_pt3d * get_next_normal_nodes( void )
{
   string str = "normals";
   return get_next_node(str);
}

// ========================================================
// eof - load-scene.cxx
