// poly-array.hxx

// array.hxx -- Array management class
//
// Written by Curtis Olson, started March 1998.
//
// Copyright (C) 1998 - 1999  Curtis L. Olson  - http://www.flightgear.org/~curt
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation; either version 2 of the
// License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
//
// $Id: array.hxx,v 1.18 2005-11-10 16:26:59 curt Exp $


#ifndef _POLY_ARRAY_HXX_
#define _POLY_ARRAY_HXX_

#ifndef __cplusplus                                                          
# error This library requires C++
#endif                                   

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <simgear/compiler.h>

#include <simgear/bucket/newbucket.hxx>
#include <simgear/math/point3d.hxx>
#include <simgear/math/sg_types.hxx>
#include <simgear/misc/sgstream.hxx>

#define ARRAY_SIZE_1 1201

#define pvFLG_SET  0x00000001
#define pvFLG_SET2 0x00000002
#define pvFLG_TRI1 0x00000004
#define pvFLG_TRI2 0x00000008
#define pvFLG_TRI3 0x00000010

#define M_NEG_ELEV -9999

typedef struct tagpvFGS {
    int elevation;
    Point3D p[3];
    unsigned int flag;
    unsigned int used;
}pvFGS, * PpvFGS;

// a point is an x,y grid location
typedef struct tagpvPT {
    int x;
    int y;
}pvPT;

// a vectored point, is a 3D point plus a grid point
typedef struct tagpvPTV {
    SGVec3d vp;
    pvPT pt;
}pvPTV;

typedef struct tagpvTRIV {
    SGVec3d p1;
    SGVec3d p2;
    SGVec3d p3;
    int p1_x, p1_y;
    int p2_x, p2_y;
    int p3_x, p3_y;
}pvTRIV;

typedef struct tagpvTRIV3 {
    pvPTV ptv[3];
}pvTRIV3;

class PVArray {

public:

    // array file pointer
    sg_gzifstream *array_in;
    string in_array;

    // fitted file pointer
    sg_gzifstream *fitted_in;
    string in_fitted;

    // coordinates (in arc seconds) of south west corner
    double originx, originy;
    double min_lon, max_lon, min_lat, max_lat;
    // number of columns and rows
    int cols, rows;
    int max_height, min_height;

    // Distance between column and row data points (in arc seconds)
    double col_step, row_step;
    
    int fitted_size;

    // pointers to the actual grid data allocated here
    int (*in_data)[ARRAY_SIZE_1];
    // float (*out_data)[ARRAY_SIZE_1];

    // divide fitted list into triangles
    vector<pvTRIV> fit_tris;
    std::vector<pvTRIV3> fit_tris3;

    pvFGS (*fit_str)[ARRAY_SIZE_1];
    void free_mem(void);
    int allocate_mem(void);

    void fit_to_array_grid(void);
    void extract_tris(void);
    void add_tri(pvTRIV & t);

    // output nodes
    point_list corner_list;
    point_list fitted_list;

    // Constructor
    PVArray( void );
    PVArray( const string& file );

    // Destructor
    ~PVArray( void );

    // open an Array file (use "-" if input is coming from stdin)
    bool open ( const string& file_base );

    // return if array was successfully opened or not
    inline bool is_open() { 
      if ( array_in != NULL ) {
	return array_in->is_open();
      } else {
	return false;
      }
    }

    // close a Array file
    bool close();

    // parse a Array file
    int parse( SGBucket& b );

    // write an Array file
    bool write( const string root_dir, SGBucket& b );

    // do our best to remove voids by picking data from the nearest
    // neighbor.
    void remove_voids();

    // add a node to the output corner node list
    void add_corner_node( int i, int j, double val );

    // add a node to the output fitted node list
    void add_fit_node( int i, int j, double val );

    // Return the elevation of the closest non-void grid point to lon, lat
    double closest_nonvoid_elev( double lon, double lat ) const;

    // return the current altitude based on grid data.  We should
    // rewrite this to interpolate exact values, but for now this is
    // good enough
    double altitude_from_grid( double lon, double lat ) const;

    // Informational methods
    inline double get_originx() const { return originx; }
    inline double get_originy() const { return originy; }
    inline int get_cols() const { return cols; }
    inline int get_rows() const { return rows; }
    inline double get_col_step() const { return col_step; }
    inline double get_row_step() const { return row_step; }

    inline point_list get_corner_list() const { return corner_list; }
    inline point_list get_fitted_list() const { return fitted_list; }

    inline int get_array_elev( int col, int row ) {
        return in_data[col][row];
    }
    inline void set_array_elev( int col, int row, int val ) {
        in_data[col][row] = val;
    }
    inline Point3D get_fitted_pt( int i ) {
        return fitted_list[i];
    }
};

#endif // _POLY_ARRAY_HXX_
// eof - poly-array.hxx



