// poly-grid.cxx
// paint lat,lon grid onto map
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif
#include "poly-view.hxx"
#include "poly-paint.hxx"
#include "poly-utils.hxx"
#include "poly-ptools.hxx"
#include "poly-grid.hxx"
#include "poly-pmacs.hxx"  // some paint macros
#include "poly-text.hxx"

typedef struct tagTGPt {
    double lon,lat,alt;
    int x,y,z;
}TGPt, * PTGPt;

typedef struct tagTGPt4 {
    TGPt pt[4];
}TGPt4, * PTGPt4;

// ==========================================================
// this one paints all types of LAT/LON lines
// whole values are solid lines
// intermediate lines are dotted
// ==========================================================
//char * pszLonLat = " %0.6f,%0.6f ";
char * pszLonLat = "%0.4f,%0.4f";
static char _s_tmpbuf[1024];
// default - between 3 and 6 lines
int   target_min = 3;
int   target_max = 6;

// DEBUG ITEMS
// to show the painted grid
//#define sprtf_showgrid  sprtf
#define sprtf_showgrid
#define get_ll2pt win_ll2pt

#define LLE_WRAPNORTH   1
#define LLE_WRAPSOUTH   2
#define LLE_WRAPWEST    4
#define LLE_WRAPEAST    8

#define norm_lon(a) (a > 180.0) ? 180.0 : (a < -180.0) ? -180.0 : a
#define norm_lat(a) (a > 90.0) ? 90.0 : (a < -90.0) ? -90.0 : a

typedef vector<SGBucket> vSGBList;
typedef vSGBList::iterator iSGB;

vSGBList buck_list;

int get_bucket_count(double min_lat, double max_lat, double min_lon, double max_lon)
{
    int buckets = 0;
   	SGBucket b_cur;
	int dx, dy, i, j;
    buck_list.clear();

    SGBucket b_min(min_lon, min_lat);
    SGBucket b_max(max_lon, max_lat);
    if (b_min == b_max) {
        buck_list.push_back(b_min);
        return 1;
    }
    sgBucketDiff(b_min, b_max, &dx, &dy);
    if ( (dx > 2880) || (dy > 1440) || (dx < 0) || (dy < 0)) {
        static int _s_shown_error = 0;
        if (_s_shown_error < 3) {
            _s_shown_error++;
            char * cp = GetNxtBuf();
            sprintf(cp,"Some stupid error on diffing two Buckets!\ndx=%d, dy=%d? (%d of 3)",
                dx,dy,_s_shown_error);
            MB2("BUCKET ERROR",cp);
        }
        return 0;
    }
    // iterate through the buckets...
	for ( j = 0; j <= dy; j++ ) {
        for ( i = 0; i <= dx; i++ ) {
            b_cur = sgBucketOffset(min_lon, min_lat, i, j);
            buck_list.push_back(b_cur);
            buckets++;
        }
    }
    return buckets;
}

void set_bucket_pts( SGBucket & b, PTGPt4 ppt )
{
    double lat,lon,wid,hgt;
    lat = b.get_center_lat();
    lon = b.get_center_lon();
    wid = b.get_width() / 2.0;
    hgt = b.get_height() / 2.0;
    // bottom left point
    ppt->pt[0].lat = lat + hgt;
    ppt->pt[0].lon = lon - wid;
    // bottom right point
    ppt->pt[1].lat = lat + hgt;
    ppt->pt[1].lon = lon + wid;
    // top right point
    ppt->pt[2].lat = lat - hgt;
    ppt->pt[2].lon = lon + wid;
    // top left point
    ppt->pt[3].lat = lat - hgt;
    ppt->pt[3].lon = lon - wid;
}

#ifndef NDEBUG
#define sprtf_bucket sprtf
#else
#define sprtf_bucket
#endif

void paint_bucket_outlines(PPPSET pps)
{
    iSGB it;
    HPEN hpLBlue = Get_LBlue_Pen();
    HPEN hpOld = NULL;
    TGPt4 pt;
    double dx,dy,lat,lon;
    int i, j, cnt, x, y, iLen;
    string index;
    COLORREF crOld;
    SIZE sz;
    char * ps;
    HFONT hf = Get_CN8_Font( pps->hdc );
    HFONT hfOld = NULL;
    crOld = SetTextColor( pps->hdc, get_std_text_color() );
    if(hpLBlue)
       hpOld = (HPEN)SelectObject(pps->hdc, hpLBlue);
    if (hf)
        hfOld = (HFONT)SelectObject(pps->hdc, hf);
    cnt = 0;
    for (it = buck_list.begin(); it != buck_list.end(); it++) {
        cnt++;
        SGBucket b = *it;
        index = b.gen_index_str();
        set_bucket_pts( b, &pt );
        // convert bucket points
        for (j = 0; j < 4; j++) {
            lat = pt.pt[j].lat;
            lon = pt.pt[j].lon;
            win_ll2pt( lat, lon,
                pps->c_lat, pps->c_lon, pps->map_size, pps->map_zoom,
                &dx, &dy );
            pt.pt[j].x = win_int(dx);
            pt.pt[j].y = win_int(dy);
            sprtf_bucket("Pt%d:%d: %f,%f, and %d,%d\n", cnt, (j+1), lat, lon,
                pt.pt[j].x, pt.pt[j].y );
        }
        // draw the bucket points
        for (i = 0; i < 4; i++) {
            j = i + 1;
            if (j == 4) j = 0;
             draw_line( pps->hdc,
                 pt.pt[i].x, pt.pt[i].y,
                 pt.pt[j].x, pt.pt[j].y );
             sprtf_bucket("Ln%d:%d: %d,%d to %d,%d\n", cnt, (i+1),
                 pt.pt[i].x, pt.pt[i].y,
                 pt.pt[j].x, pt.pt[j].y );
        }
        x = (pt.pt[0].x + pt.pt[1].x) / 2;
        y = (pt.pt[1].y + pt.pt[2].y) / 2;
        ps = (char *)index.c_str();
        iLen = index.size();
        GetTextExtentPoint32( pps->hdc, ps, iLen, &sz );
        TextOut( pps->hdc, x - (sz.cx / 2), y - (sz.cy / 2), ps, iLen );


    }

    if (hfOld)
        SelectObject(pps->hdc, hfOld);
    SetTextColor( pps->hdc, crOld );
    if (hpOld)
        SelectObject(pps->hdc, hpOld);
}

void get_screen_extents( void )
{
   PPPSET pps = get_act_paint_set();
   PPolyGlobs ppg = pps->ppg; // get_poly_globs();
   PRECT prcp = get_corner_rect(); // store the corner paints
   double dx,dy, rdx, rdy;
   //double min_slat, max_slat, min_slon, max_slon;
   int iLen;
   char * ps = _s_tmpbuf;  // temporary (SHARED) buffer
   SIZE sz;
   POINT pT;
   RECT  rc = *pps->prc;
   double c_lat = mm_c_lat;
   double c_lon = mm_c_lon;
   COLORREF crOld;
   HFONT hf = Get_CN8_Font( pps->hdc );
   HFONT hfOld;
   HPEN hpGrid  = Get_GRID_Pen();
   HPEN hpGrid2 = Get_GRID2_Pen();
   HPEN hpOld;
   double rlat, rlon;


   max_north = c_lat + 90.0;
   max_south = c_lat - 90.0;
   max_west  = c_lon - 90.0;
   max_east  = c_lon + 90.0;
   // but these may need wrapping
   unsigned int flag = 0;
   //  L   clon    R
   // -90 =   0 =  90
   // -80 =  10 = 100
   // -10 =  80 = 170
   //   0 =  90 = 180
   //  10 = 100 =-170
   //  80 = 170 =-100
   //  90 = 180 = -90
   // 100 =-170 = -80
   // 170 =-100 = -10
   // 180 = -90 =   0
   //-170 = -80 =  10
   //================

   if( c_lat > 0.0 ) {
      max_north = 90.0 - c_lat;
      flag |= LLE_WRAPNORTH;
   } else if( c_lat < 0.0 ) {
      max_south = -(90.0 + c_lat);
      flag |= LLE_WRAPSOUTH;
   }
   if( max_west < -180.0 ) {
      max_west += 360.0;
      flag |= LLE_WRAPWEST;
   } else if( max_east > 180.0 ) {
      max_east -= 360.0;
      flag |= LLE_WRAPEAST;
   }
   sprtf("NS Extents: MN=%f, MS=%f %s\n", max_north, max_south,
      ((flag & LLE_WRAPNORTH) ? "through North Pole" :
      (flag & LLE_WRAPSOUTH) ? "through South Pole" : "center"));
   sprtf("EW Extents: MW=%f, ME=%f %s\n", max_west, max_east,
      ((flag & LLE_WRAPWEST) ? "WRAPWEST -180" :
      (flag & LLE_WRAPEAST) ? "WRAPEAST +180" : "center"));

   // CENTER OF SCREEN
   // ===================================
   dx = pps->map_size / 2.0;
   dy = pps->map_size / 2.0;
   win_pt2ll( &pps->dlat, &pps->dlon, pps->c_lat, pps->c_lon, pps->map_size, pps->map_zoom, dx, dy );
   iLen = sprintf(ps, pszLonLat, pps->dlon, pps->dlat );
   win_ll2pt( pps->dlat, pps->dlon, pps->c_lat, pps->c_lon, pps->map_size, pps->map_zoom, &rdx, &rdy );
   sprtf( "Center: %0.1f,%0.1f, put %s(%d) at (r %g,%g %d,%d)\n",
      dx, dy, ps, iLen,
      rdx, rdy, win_int(rdx), win_int(rdy));
   rlat = norm_lat(pps->dlat);
   rlon = norm_lon(pps->dlon);
   win_ll2pt( pps->dlat, pps->dlon, pps->c_lat, pps->c_lon, pps->map_size, pps->map_zoom, &rdx, &rdy );
   sprtf( "Norm: %0.6f,%0.6f, at %d,%d\n", rlon, rlat, win_int(dx), win_int(dy));
   // ===================================

   // get screen extents
   // =========================================
   min_slat = 2000.0;
   max_slat = -2000.0;
   min_slon = 2000.0;
   max_slon = -2000.0;

   crOld = SetTextColor( pps->hdc, get_std_text_color() );
   if(hf)
      hfOld = (HFONT)SelectObject(pps->hdc, hf);
   if(hpGrid)
      hpOld = (HPEN)SelectObject(pps->hdc, hpGrid);

   // TOP LEFT
   dx = 0.0;
   dy = 0.0;
   win_pt2ll( &pps->dlat, &pps->dlon, pps->c_lat, pps->c_lon, pps->map_size, pps->map_zoom, dx, dy );
   iLen = sprintf(ps, pszLonLat, pps->dlon, pps->dlat );

   win_ll2pt( pps->dlat, pps->dlon, pps->c_lat, pps->c_lon, pps->map_size, pps->map_zoom, &rdx, &rdy );
   GetTextExtentPoint32( pps->hdc, ps, iLen, &sz );
   pT.x = 0;
   pT.y = 0;
   TextOut( pps->hdc, pT.x, pT.y, ps, iLen );
   prcp[RC_TOP_LEFT].left   = pT.x;
   prcp[RC_TOP_LEFT].right  = prcp->left + sz.cx;
   prcp[RC_TOP_LEFT].top    = pT.y;
   prcp[RC_TOP_LEFT].bottom = prcp->top + sz.cy;

   sprtf( "For %0.1f,%0.1f, put %s(%d) at %d,%d (sz %d,%d)(r %g,%g %d,%d)\n", dx, dy, ps, iLen,
      pT.x, pT.y, sz.cx, sz.cy, rdx, rdy, win_int(rdx), win_int(rdy));
   rlat = norm_lat(pps->dlat);
   rlon = norm_lon(pps->dlon);
   win_ll2pt( pps->dlat, pps->dlon, pps->c_lat, pps->c_lon, pps->map_size, pps->map_zoom, &rdx, &rdy );
   sprtf( "Norm: %0.6f,%0.6f, at %d,%d\n", rlon, rlat, win_int(dx), win_int(dy));

   if( pps->dlon < min_slon )
      min_slon = pps->dlon;
   if( pps->dlon > max_slon )
      max_slon = pps->dlon;
   if( pps->dlat < min_slat )
      min_slat = pps->dlat;
   if( pps->dlat > max_slat )
      max_slat = pps->dlat;

   // TOP RIGHT
   dx = (double) rc.right;
   dy = 0.0;
   win_pt2ll( &pps->dlat, &pps->dlon, pps->c_lat, pps->c_lon, pps->map_size, pps->map_zoom, dx, dy );
   iLen = sprintf(ps, pszLonLat, pps->dlon, pps->dlat );
   win_ll2pt( pps->dlat, pps->dlon, pps->c_lat, pps->c_lon, pps->map_size, pps->map_zoom, &rdx, &rdy );
   GetTextExtentPoint32( pps->hdc, ps, iLen, &sz );
   pT.x = rc.right - sz.cx - 2;
   pT.y = 0;
   TextOut( pps->hdc, pT.x, pT.y, ps, iLen );
   prcp[RC_TOP_RIGHT].left   = pT.x;
   prcp[RC_TOP_RIGHT].right  = prcp->left + sz.cx;
   prcp[RC_TOP_RIGHT].top    = pT.y;
   prcp[RC_TOP_RIGHT].bottom = prcp->top + sz.cy;
   sprtf( "For %0.1f,%0.1f, put %s(%d) at %d,%d (sz %d,%d)(r %g,%g %d,%d)\n", dx, dy, ps, iLen,
      pT.x, pT.y, sz.cx, sz.cy, rdx, rdy, win_int(rdx), win_int(rdy));
   rlat = norm_lat(pps->dlat);
   rlon = norm_lon(pps->dlon);
   win_ll2pt( pps->dlat, pps->dlon, pps->c_lat, pps->c_lon, pps->map_size, pps->map_zoom, &rdx, &rdy );
   sprtf( "Norm: %0.6f,%0.6f, at %d,%d\n", rlon, rlat, win_int(dx), win_int(dy));

   if( pps->dlon < min_slon )
      min_slon = pps->dlon;
   if( pps->dlon > max_slon )
      max_slon = pps->dlon;
   if( pps->dlat < min_slat )
      min_slat = pps->dlat;
   if( pps->dlat > max_slat )
      max_slat = pps->dlat;

   // BOTTOM RIGHT
   dx = (double) rc.right;
   dy = (double) rc.bottom;
   win_pt2ll( &pps->dlat, &pps->dlon, pps->c_lat, pps->c_lon, pps->map_size, pps->map_zoom, dx, dy );
   iLen = sprintf(ps, pszLonLat, pps->dlon, pps->dlat );
   win_ll2pt( pps->dlat, pps->dlon, pps->c_lat, pps->c_lon, pps->map_size, pps->map_zoom, &rdx, &rdy );
   GetTextExtentPoint32( pps->hdc, ps, iLen, &sz );
   pT.x = rc.right - sz.cx - 2;
   pT.y = rc.bottom - sz.cy - 2;
   TextOut( pps->hdc, pT.x, pT.y, ps, iLen );
   prcp[RC_BOTTOM_RIGHT].left   = pT.x;
   prcp[RC_BOTTOM_RIGHT].right  = prcp->left + sz.cx;
   prcp[RC_BOTTOM_RIGHT].top    = pT.y;
   prcp[RC_BOTTOM_RIGHT].bottom = prcp->top + sz.cy;
   sprtf( "For %0.1f,%0.1f, put %s(%d) at %d,%d (sz %d,%d)(r %g,%g %d,%d)\n", dx, dy, ps, iLen,
      pT.x, pT.y, sz.cx, sz.cy, rdx, rdy, win_int(rdx), win_int(rdy));
   rlat = norm_lat(pps->dlat);
   rlon = norm_lon(pps->dlon);
   win_ll2pt( pps->dlat, pps->dlon, pps->c_lat, pps->c_lon, pps->map_size, pps->map_zoom, &rdx, &rdy );
   sprtf( "Norm: %0.6f,%0.6f, at %d,%d\n", rlon, rlat, win_int(dx), win_int(dy));

   if( pps->dlon < min_slon )
      min_slon = pps->dlon;
   if( pps->dlon > max_slon )
      max_slon = pps->dlon;
   if( pps->dlat < min_slat )
      min_slat = pps->dlat;
   if( pps->dlat > max_slat )
      max_slat = pps->dlat;

   // BOTTOM LEFT
   dx = 0.0;
   dy = (double) rc.bottom;
   win_pt2ll( &pps->dlat, &pps->dlon, pps->c_lat, pps->c_lon, pps->map_size, pps->map_zoom, dx, dy );
   iLen = sprintf(ps, pszLonLat, pps->dlon, pps->dlat );
   win_ll2pt( pps->dlat, pps->dlon, pps->c_lat, pps->c_lon, pps->map_size, pps->map_zoom, &rdx, &rdy );
   GetTextExtentPoint32( pps->hdc, ps, iLen, &sz );
   pT.x = 0;
   pT.y = rc.bottom - sz.cy;
   TextOut( pps->hdc, pT.x, pT.y, ps, iLen );
   prcp[RC_BOTTOM_LEFT].left   = pT.x;
   prcp[RC_BOTTOM_LEFT].right  = prcp->left + sz.cx;
   prcp[RC_BOTTOM_LEFT].top    = pT.y;
   prcp[RC_BOTTOM_LEFT].bottom = prcp->top + sz.cy;
   sprtf( "For %0.1f,%0.1f, put %s(%d) at %d,%d (sz %d,%d)(r %g,%g %d,%d)\n", dx, dy, ps, iLen,
      pT.x, pT.y, sz.cx, sz.cy, rdx, rdy, win_int(rdx), win_int(rdy));
   rlat = norm_lat(pps->dlat);
   rlon = norm_lon(pps->dlon);
   win_ll2pt( pps->dlat, pps->dlon, pps->c_lat, pps->c_lon, pps->map_size, pps->map_zoom, &rdx, &rdy );
   sprtf( "Norm: %0.6f,%0.6f, at %d,%d\n", rlon, rlat, win_int(dx), win_int(dy));

   if( pps->dlon < min_slon )
      min_slon = pps->dlon;
   if( pps->dlon > max_slon )
      max_slon = pps->dlon;
   if( pps->dlat < min_slat )
      min_slat = pps->dlat;
   if( pps->dlat > max_slat )
      max_slat = pps->dlat;
   prcp[RC_CENTER_SPACE].bottom = rc.bottom - sz.cy;
   prcp[RC_CENTER_SPACE].top    = sz.cy;

   // these value can be used to paint the bucket boundaries
   ppg->g_bucket_count = get_bucket_count(min_slat, max_slat, min_slon, max_slon);
   sprtf("min/max lat %f %f, min/max lon %f %f - %d bucket\n",
       min_slat, max_slat, min_slon, max_slon,
       ppg->g_bucket_count );
   // done later -  AFTER grid so matching GRID lines do NOT overwrite Blue bucket lines
   //if ( ppg->g_paint_buckets && (ppg->g_bucket_count > 1)) {
        // add bucket outline paint
   //    paint_bucket_outlines(pps);
   //}
   // clean up
   // ========
   crOld = SetTextColor( pps->hdc, crOld );
   if(hf)
      hfOld = (HFONT)SelectObject(pps->hdc, hfOld);
   if(hpGrid)
      hpOld = (HPEN)SelectObject(pps->hdc, hpOld);
   c_lat = mm_c_lat;
   c_lon = mm_c_lon;
   win_ll2pt( pps->dlat, pps->dlon, c_lat, c_lon, pps->map_size, pps->map_zoom, &rdx, &rdy );
   sprtf("Done get_screen_extents: center=%0.6f,%0.6f (%d,%d)\n", c_lon, c_lat,
       win_int(rdx), win_int(rdy));
}

//void paint_lat_lon_grid( HDC pps->hdc, double c_lon_in, double c_lat_in,
//                        double map_size_in, double map_zoom_in, PRECT prc_in )
void paint_lat_lon_grid( HDC hdc_in )
{
   PPPSET pps = get_act_paint_set();
   pps->hdc = hdc_in;
   COLORREF crOld;
   HFONT hf = Get_CN8_Font( pps->hdc );
   HFONT hfOld;
   HPEN hpGrid  = Get_GRID_Pen();
   HPEN hpGrid2 = Get_GRID2_Pen();
   HPEN hpOld;
   double dx, dy;
   double rdx,rdy;
   // double min_slat, max_slat, min_slon, max_slon;
   RECT  rc = *pps->prc;
   int   x, y;
   int   last_x, last_y;
   PRECT prcp = get_corner_rect(); // store the corner paints
   int   i;
   int   total_polys = get_loaded_poly_count();
   int   enab_polys  = get_poly_enabled_count();

   min_slat = 2000.0;
   max_slat = -2000.0;
   min_slon = 2000.0;
   max_slon = -2000.0;
   SIZE sz;
   POINT pT;
   char * ps = _s_tmpbuf;  // temporary (SHARED) buffer
   int   iLen;

   get_screen_extents();

   get_CN8_Text_Size( pps->hdc, &sz );
   prcp[RC_CENTER_SPACE].left = sz.cx;
   prcp[RC_CENTER_SPACE].right = pps->prc->right - sz.cx;

   sprtf( "paint_lat_lon_grid: center lon,lat %0.4f,%0.4f, size %f, zoom(MM) %f\n",
      pps->c_lon, pps->c_lat, pps->map_size,
      (pps->map_zoom * 1000000.0) );

   crOld = SetTextColor( pps->hdc, get_std_text_color() );
   if(hf)
      hfOld = (HFONT)SelectObject(pps->hdc, hf);
   if(hpGrid)
      hpOld = (HPEN)SelectObject(pps->hdc, hpGrid);

#if 0 // sep func
   // get screen extents
   // =========================================
   // TOP LEFT
   dx = 0.0;
   dy = 0.0;
   win_pt2ll( &pps->dlat, &pps->dlon, pps->c_lat, pps->c_lon, pps->map_size, pps->map_zoom, dx, dy );
   iLen = sprintf(ps, pszLonLat, pps->dlon, pps->dlat );
   win_ll2pt( pps->dlat, pps->dlon, pps->c_lat, pps->c_lon, pps->map_size, pps->map_zoom, &rdx, &rdy );
   GetTextExtentPoint32( pps->hdc, ps, iLen, &sz );
   pT.x = 0;
   pT.y = 0;
   TextOut( pps->hdc, pT.x, pT.y, ps, iLen );
   prcp[RC_TOP_LEFT].left   = pT.x;
   prcp[RC_TOP_LEFT].right  = prcp->left + sz.cx;
   prcp[RC_TOP_LEFT].top    = pT.y;
   prcp[RC_TOP_LEFT].bottom = prcp->top + sz.cy;

   sprtf( "For %0.1f,%0.1f, put %s(%d) at %d,%d (sz %d,%d)(r %0.6f,%o.6f\n", dx, dy, ps, iLen,
      pT.x, pT.y, sz.cx, sz.cy, rdx, rdy);
   if( pps->dlon < min_slon )
      min_slon = pps->dlon;
   if( pps->dlon > max_slon )
      max_slon = pps->dlon;
   if( pps->dlat < min_slat )
      min_slat = pps->dlat;
   if( pps->dlat > max_slat )
      max_slat = pps->dlat;

   // TOP RIGHT
   dx = (double) rc.right;
   dy = 0.0;
   win_pt2ll( &pps->dlat, &pps->dlon, pps->c_lat, pps->c_lon, pps->map_size, pps->map_zoom, dx, dy );
   iLen = sprintf(ps, pszLonLat, pps->dlon, pps->dlat );
   win_ll2pt( pps->dlat, pps->dlon, pps->c_lat, pps->c_lon, pps->map_size, pps->map_zoom, &rdx, &rdy );
   GetTextExtentPoint32( pps->hdc, ps, iLen, &sz );
   pT.x = rc.right - sz.cx - 2;
   pT.y = 0;
   TextOut( pps->hdc, pT.x, pT.y, ps, iLen );
   prcp[RC_TOP_RIGHT].left   = pT.x;
   prcp[RC_TOP_RIGHT].right  = prcp->left + sz.cx;
   prcp[RC_TOP_RIGHT].top    = pT.y;
   prcp[RC_TOP_RIGHT].bottom = prcp->top + sz.cy;
   sprtf( "For %0.1f,%0.1f, put %s(%d) at %d,%d (sz %d,%d)(r %0.6f,%o.6f\n", dx, dy, ps, iLen,
      pT.x, pT.y, sz.cx, sz.cy, rdx, rdy);
   if( pps->dlon < min_slon )
      min_slon = pps->dlon;
   if( pps->dlon > max_slon )
      max_slon = pps->dlon;
   if( pps->dlat < min_slat )
      min_slat = pps->dlat;
   if( pps->dlat > max_slat )
      max_slat = pps->dlat;

   // BOTTOM RIGHT
   dx = (double) rc.right;
   dy = (double) rc.bottom;
   win_pt2ll( &pps->dlat, &pps->dlon, pps->c_lat, pps->c_lon, pps->map_size, pps->map_zoom, dx, dy );
   iLen = sprintf(ps, pszLonLat, pps->dlon, pps->dlat );
   win_ll2pt( pps->dlat, pps->dlon, pps->c_lat, pps->c_lon, pps->map_size, pps->map_zoom, &rdx, &rdy );
   GetTextExtentPoint32( pps->hdc, ps, iLen, &sz );
   pT.x = rc.right - sz.cx - 2;
   pT.y = rc.bottom - sz.cy - 2;
   TextOut( pps->hdc, pT.x, pT.y, ps, iLen );
   prcp[RC_BOTTOM_RIGHT].left   = pT.x;
   prcp[RC_BOTTOM_RIGHT].right  = prcp->left + sz.cx;
   prcp[RC_BOTTOM_RIGHT].top    = pT.y;
   prcp[RC_BOTTOM_RIGHT].bottom = prcp->top + sz.cy;
   sprtf( "For %0.1f,%0.1f, put %s(%d) at %d,%d (sz %d,%d)(r %0.6f,%o.6f\n", dx, dy, ps, iLen,
      pT.x, pT.y, sz.cx, sz.cy, rdx, rdy);
   if( pps->dlon < min_slon )
      min_slon = pps->dlon;
   if( pps->dlon > max_slon )
      max_slon = pps->dlon;
   if( pps->dlat < min_slat )
      min_slat = pps->dlat;
   if( pps->dlat > max_slat )
      max_slat = pps->dlat;

   // BOTTOM LEFT
   dx = 0.0;
   dy = (double) rc.bottom;
   win_pt2ll( &pps->dlat, &pps->dlon, pps->c_lat, pps->c_lon, pps->map_size, pps->map_zoom, dx, dy );
   iLen = sprintf(ps, pszLonLat, pps->dlon, pps->dlat );
   win_ll2pt( pps->dlat, pps->dlon, pps->c_lat, pps->c_lon, pps->map_size, pps->map_zoom, &rdx, &rdy );
   GetTextExtentPoint32( pps->hdc, ps, iLen, &sz );
   pT.x = 0;
   pT.y = rc.bottom - sz.cy;
   TextOut( pps->hdc, pT.x, pT.y, ps, iLen );
   prcp[RC_BOTTOM_LEFT].left   = pT.x;
   prcp[RC_BOTTOM_LEFT].right  = prcp->left + sz.cx;
   prcp[RC_BOTTOM_LEFT].top    = pT.y;
   prcp[RC_BOTTOM_LEFT].bottom = prcp->top + sz.cy;
   sprtf( "For %0.1f,%0.1f, put %s(%d) at %d,%d (sz %d,%d)(r %0.6f,%o.6f\n", dx, dy, ps, iLen,
      pT.x, pT.y, sz.cx, sz.cy, rdx, rdy);
   if( pps->dlon < min_slon )
      min_slon = pps->dlon;
   if( pps->dlon > max_slon )
      max_slon = pps->dlon;
   if( pps->dlat < min_slat )
      min_slat = pps->dlat;
   if( pps->dlat > max_slat )
      max_slat = pps->dlat;
   prcp[RC_CENTER_SPACE].bottom = rc.bottom - sz.cy;
   prcp[RC_CENTER_SPACE].top    = sz.cy;

#endif // 0

   // =========================================
   sprintf(ps, "Enab(%d): ", total_polys);
   for( i = 0; i < 9; i++ ) {
      char c[2];
      c[1] = 0;
      if( is_poly_enabled(i) ) {
         c[0] = (i + '1');
      } else {
         c[0] = '-';
      }
      strcat(ps, c);
   }
   if( i < total_polys )
      strcat(ps,"+");
   sprintf(EndBuf(ps),"(%d)", enab_polys);
   sprintf( EndBuf(ps), " Zoom(M) %G", (pps->map_zoom * 1000) );

   paint_top_center_text(ps);

   // =========================================
   // PAINT LON, LAT GRID
   // ----------------------
   if(hpGrid2)
      SelectObject(pps->hdc, hpGrid2);

   double dfirst_lat, dstep_lat;
   double dfirst_lon, dstep_lon;
   int   lat_dec, lon_dec;
   int   lat_steps, lon_steps;
   int lat_ok = poly_get_dstep2( min_slat, max_slat,  // from here to here
               target_min, target_max,  // target min and max
               &dfirst_lat, &dstep_lat, // first value, and step
               &lat_steps, &lat_dec );  // last 2 can be NULL
   int lon_ok = poly_get_dstep2( min_slon, max_slon,  // from here to here
               target_min, target_max,  // target min and max
               &dfirst_lon, &dstep_lon, // first value, and step
               &lon_steps, &lon_dec );  // last 2 can be NULL

   int lat_count = 0;
   int lon_count = 0;
   int is_int;
   double step_dlat, step_dlon;
   // paint all grid stuff within these limits
   PRECT prcLimits = &prcp[RC_CENTER_SPACE];
   sprtf( "Rectangle limits left,top,right,bottom %s\n",
      get_rectangle_string(prcLimits) );

   //int imin_lon = (int)min_slon;
   //if((double)imin_lon < min_slon)
   //   imin_lon++;
   //int imin_lat = (int)min_slat;
   //if( (double)imin_lat < min_slat )
   //   imin_lat++;

   get_CN8_Text_Size( pps->hdc, &sz );
   if(lon_ok)
   {
      for( step_dlon = dfirst_lon; step_dlon < max_slon; step_dlon += dstep_lon )
      {
         is_int = double_value_is_integer( step_dlon );
         get_ll2pt( max_slat, step_dlon, pps->c_lat, pps->c_lon, pps->map_size, pps->map_zoom, &dx, &dy );
         x = win_int(dx);
         y = win_int(dy);
         last_x = x;
         last_y = y;
         get_ll2pt( min_slat, step_dlon, pps->c_lat, pps->c_lon, pps->map_size, pps->map_zoom, &dx, &dy );
         x = win_int(dx);
         y = win_int(dy);
         if( is_int && hpGrid )
            SelectObject(pps->hdc, hpGrid);
         else if( hpGrid2 )
            SelectObject(pps->hdc, hpGrid2);
         // draw vertical lon grid line, but start it one
         // line down from the top, and stop one line from bottom
         if( y < prcLimits->top )
            y = prcLimits->top;
         if( y > prcLimits->bottom )
            y = prcLimits->bottom;
         if( last_y < prcLimits->top )
            last_y = prcLimits->top;
         if( last_y > prcLimits->bottom )
            last_y = prcLimits->bottom;

         draw_line( pps->hdc, last_x, last_y, x, y );
         sprtf_showgrid( "Painted GRID lon %0.12g from %d,%d to %d,%d\n",
            step_dlon, last_x, last_y, x, y );
         if( is_int )
            iLen = sprintf(ps, " %d ", win_int(step_dlon));
         else
            iLen = sprintf(ps, " %g ", step_dlon);
         GetTextExtentPoint32( pps->hdc, ps, iLen, &sz );
         pT.x = last_x - (sz.cx / 2);  // centered on top lon last_x
         pT.y = rc.top + sz.cy;        // 2nd line from top
         TextOut( pps->hdc, pT.x, pT.y, ps, iLen );
         sprtf_showgrid( "Painted lon [%s] at %d,%d\n",
            ps, pT.x, pT.y );
         pT.x = x - (sz.cx / 2);          // centered on bottom lon x
         pT.y = rc.bottom - (2 * sz.cy);  // 2nd line from bottom
         TextOut( pps->hdc, pT.x, pT.y, ps, iLen );
         sprtf_showgrid( "Painted lon [%s] at %d,%d\n",
            ps, pT.x, pT.y );
         lon_count++;   // count an integer lon painted
      }
   }
   if(lat_ok)
   {
      for( step_dlat = dfirst_lat; step_dlat < max_slat; step_dlat += dstep_lat )
      {
         is_int = double_value_is_integer( step_dlat );
         get_ll2pt( step_dlat, max_slon, pps->c_lat, pps->c_lon, pps->map_size, pps->map_zoom, &dx, &dy );
         x = win_int(dx);
         y = win_int(dy);
         last_x = x;
         last_y = y;
         get_ll2pt( step_dlat, min_slon, pps->c_lat, pps->c_lon, pps->map_size, pps->map_zoom, &dx, &dy );
         x = win_int(dx);
         y = win_int(dy);
         // avoid the bottom horizontal lat grid cutting up
         // the 2 bottom lines...
         if(( y >= (prcLimits->bottom - sz.cy))||
            ( last_y >= (prcLimits->bottom - sz.cy)))
            continue;
         // avoid the top horizontal lat grid cutting up
         // the 2 top lines...
         if(( y <= (prcLimits->top + sz.cy))||
            ( last_y <= (prcLimits->top + sz.cy)))
            continue;

         if( is_int && hpGrid )
            SelectObject(pps->hdc, hpGrid);
         else if( hpGrid2 )
            SelectObject(pps->hdc, hpGrid2);
         draw_line( pps->hdc, last_x, last_y, x, y );
         sprtf_showgrid( "Painted GRID lat %0.12g from %d,%d to %d,%d\n",
            step_dlat, last_x, last_y, x, y );
         if( is_int )
            iLen = sprintf(ps, " %d ", win_int(step_dlat) );
         else
            iLen = sprintf(ps, " %g ", step_dlat );
         GetTextExtentPoint32( pps->hdc, ps, iLen, &sz );
         pT.x = 0;                     // at left x
         pT.y = last_y - (sz.cy / 2);  // centered on lat line
         TextOut( pps->hdc, pT.x, pT.y, ps, iLen );
         sprtf_showgrid( "Painted lat [%s] at %d,%d\n",
            ps, pT.x, pT.y );
         pT.x = rc.right - sz.cx;      // at right - length
         pT.y = y - (sz.cy / 2);       // centered on lat line
         TextOut( pps->hdc, pT.x, pT.y, ps, iLen );
         sprtf_showgrid( "Painted lat [%s] at %d,%d\n",
            ps, pT.x, pT.y );
         lat_count++;   // count an integer lat painted
      }
   }

   // Done here, after the GRID, so any matching lines will be the
   // light blue of the bucket lines
   if ( pps->ppg->g_paint_buckets && (pps->ppg->g_bucket_count > 1)) {
        // add bucket outline paint
       paint_bucket_outlines(pps);
   }
   // clean up
   // ========
   crOld = SetTextColor( pps->hdc, crOld );
   if(hf)
      hfOld = (HFONT)SelectObject(pps->hdc, hfOld);
   if(hpGrid)
      hpOld = (HPEN)SelectObject(pps->hdc, hpOld);
   sprtf("Done paint_lat_lon_grid:\n");
}


extern void paint_line_text_to_DC( HDC hdc, char * ps, int line );
extern void re_paint_lines_to_DC( HDC hdc );
extern int had_scenery_paint; // only after first POLY paint
extern void paint_scenery( HDC hdc_in );
extern void paint_bounding_box( HDC hdc, PRECT pmm );
extern void paint_airports( HDC hdc, double c_lat, double c_lon,
                       double map_size, double map_zoom,
                       PRECT prc);
extern void clear_center_bottom_text(void);
extern int paint_all_points;  // this seems best, but NOT for polygon fill!
extern void fill_polygon( HDC hdc, PRECT prc );
extern int add_last_join;


void paint_polys(void)
{
   PPPSET pps = get_act_paint_set();
   TGPolyLoad * polyload = NULL;
   int points_painted = 0;
   int polys_painted = 0;
   int   cnum;
   sGETPT sgp;
   int last_contour;
   HPEN  hp, hpOld;
    int in_scene_range;

   if( !done_arguments ) {
      sprtf("paint_polys: Not done arguments...\n");
      paint_line_text_to_DC( pps->hdc, "Not done arguments...", 1 );
      re_paint_lines_to_DC( pps->hdc );
      return;
   }
   polyload = get_first_loaded_poly();
   if( !polyload ) {
      sprtf("paint_polys: No polys loaded...\n");
      paint_line_text_to_DC( pps->hdc, "No polys loaded...", 1 );
      re_paint_lines_to_DC( pps->hdc );
      return;
   }
   if( get_poly_enabled_count() == 0 ) {
      sprtf("paint_polys: No polys enabled...\n");
      paint_line_text_to_DC( pps->hdc, "No polys enabled...", 1 );
      re_paint_lines_to_DC( pps->hdc );
      return;
   }

   sprtf( "\nPaint: Got window size x,y %s, lon,lat center %f,%f\n",
      get_rectangle_string(pps->prc),
      pps->c_lon, pps->c_lat );

   int points_outside = 0;
   int pts_in_scene = 0;
   int pts_out_scene = 0;
   int verb9 = ((VERB9) ? 1 : 0);
   had_scenery_paint = 1;

   paint_scenery( pps->hdc ); // this is essentially a BACKGROUND image only

   paint_lat_lon_grid( pps->hdc );  // put LAT, LON over this background

   pps->mm.bottom = -10000;
   pps->mm.left   =  10000;
   pps->mm.right  = -10000;
   pps->mm.top    =  10000;
   sgp.pcnum = &cnum;
   sgp.px    = &pps->dlon;
   sgp.py    = &pps->dlat;
   g_map_next_color = 0;
   points_painted = 0;
   points_outside = 0;
   clear_painted_points();
   clear_paintPoints();
   int  join_pts = pps->ppg->g_join_poly_points;
   int  join_fl  = pps->ppg->g_join_first_to_last;
   int  spread   = (pps->ppg->g_paint_X ? pps->spread_x : 0);
   sprtf("paint_polys: Join points [%s], first-to-last [%s]\n",
       (join_pts ? "enabled" : "disabled"),
       (join_fl ? "enabled" : "disabled") );
   polyload = get_first_loaded_poly();
   do {
      PPolyCounter ppc = Get_Poly_Counts();
      COLORREF cr = get_next_color();  //RGB(255,0,0);
      //hp = Get_Solid_Pen_1(cr);
      hp = Get_Act_Pen();
      if(hp)
         hpOld = (HPEN)SelectObject(pps->hdc,hp);

      if ( ppc->total && is_current_poly_enabled() )
      {
         int to_paint;
         polys_painted++;
         // === BEGIN PAINT POINTS ===
         last_contour = -1; // restart contour number
         void * vp = get_first_point( &sgp );
         if ( vp ) {
            pps->last_x = pps->last_y = pps->first_x = pps->first_y = -1;
            do {
               win_ll2pt( pps->dlat, pps->dlon, pps->c_lat, pps->c_lon, pps->map_size, pps->map_zoom, &pps->dx, &pps->dy );
               pps->x = win_int(pps->dx);
               pps->y = win_int(pps->dy);
               in_scene_range = (P_IN_SCREEN_RANGE((pps->x),(pps->y),(pps->prc))) ? 1 : 0;
               to_paint = ( paint_all_points || in_scene_range );
               if (in_scene_range)
                   pts_in_scene++;
               else
                   pts_out_scene++;
               if( cnum != last_contour ) {
                  // things to do BEFORE dealing with this NEW point
                  if (verb9) sprtf("paint_polys: Changed contour %d (from %d)\n", cnum, last_contour);
                  if( join_fl ) {
                     if(P_VALID_FIRST_LAST) {
                        draw_line( pps->hdc, pps->first_x, pps->first_y, pps->last_x, pps->last_y ); // join with pen
                        sprtf_showpoint( "Pln: %d,%d to %d,%d (1st2last) in %s\n",
                           pps->first_x, pps->first_y, pps->last_x, pps->last_y,
                           g_act_color ); 
                     }
                     if(pps->ppg->g_fill_polygon) {
                        fill_polygon(pps->hdc, pps->prc);
                     }
                  }
                  pps->first_x = pps->x;
                  pps->first_y = pps->y;
                  pps->last_x = -1;
                  pps->last_y = -1;
                  last_contour = cnum; // not same contour, just update number
                  clear_painted_points();
               }
               if( to_paint ) {
                  paint_pixel_X( pps->hdc, pps->x, pps->y, cr, spread );
                  add_painted_point( pps->x, pps->y, pps->dlon, pps->dlat );
                  sprtf_showpoint( "Pxy: %0.6f,%0.6f at %d,%d in %s\n",
                     pps->dlon, pps->dlat, pps->x, pps->y, g_act_color ); 
                  points_painted++; // count a painted point
                  if( join_pts ) {
                     if((pps->last_x != -1)&&(pps->last_y != -1)) {
                        draw_line( pps->hdc, pps->last_x, pps->last_y, pps->x, pps->y ); // join with pen
                        sprtf_showpoint( "Pln: %d,%d to %d,%d in %s (c=%d)\n",
                           pps->last_x, pps->last_y, pps->x, pps->y,
                           g_act_color, cnum ); 
                     }
                  }
                  pps->last_x = pps->x;
                  pps->last_y = pps->y;
                  if( pps->x < pps->mm.left )
                     pps->mm.left = pps->x;
                  if( pps->x > pps->mm.right )
                     pps->mm.right = pps->x;
                  if( pps->y < pps->mm.top )
                     pps->mm.top = pps->y;
                  if( pps->y > pps->mm.bottom )
                     pps->mm.bottom = pps->y;
               } else {
                  last_contour = -1;
                  sprtf_showpoint( "Pxy: OFF SCREEN %0.6f,%0.6f at %d,%d\n",
                     pps->dlon, pps->dlat, pps->x, pps->y ); 
                  points_outside++; // same group, and outside
                  pps->last_x = -1;
                  pps->last_y = -1;
               }
            } while (( vp = get_next_point( vp, &sgp ) ) != NULL );
            // OUT OF POINTS
            // ======== try for last join, if ok =======
            if( join_pts && join_fl && add_last_join )  {
               if( first_last_all_in_range(pps) ) {
                  draw_line( pps->hdc, pps->first_x, pps->first_y, pps->last_x, pps->last_y ); // join with pen
                  sprtf_showpoint( "Pln: %d,%d to %d,%d (1st2last) in %s\n",
                     pps->first_x, pps->first_y, pps->last_x, pps->last_y, g_act_color );
               }
            }
            // =============================================================
         }  // if vp - got first point
         // === END PAINT POINTS ===
      }
      if(hp) {
         hpOld = (HPEN)SelectObject(pps->hdc,hpOld);
         //DeleteObject(hp);
      }
      // skip to next poly set loaded
   } while( (polyload = get_next_loaded_poly()) != NULL );

   if( points_painted ) {
      if( pps->ppg->g_paint_bounding_box )
         paint_bounding_box( pps->hdc, &pps->mm );
      if( pps->ppg->g_paint_airports ) {
         paint_airports( pps->hdc,
            pps->c_lat, pps->c_lon,
            pps->map_size, pps->map_zoom,
            pps->prc ); 
      }
   } else {
      paint_top_center_text_wcolor( "No points painted!", RGB(255,0,0) );
   }

   clear_center_bottom_text(); // we are bitblt-ing this HDC, so NO bottom text
   sprtf("paint_polys:end: painted %u points (%u in, %u out)\n", points_painted,
       pts_in_scene, pts_out_scene );
}

// eof - poly-grid.cxx

