// poly-help.cxx - parse command line input
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation; either version 2 of the
// License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
//

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

extern void Do_InvalidateRect(void * vp);

#ifdef _MSC_VER
extern void show_memory_usage( void );
#include <sys\types.h>
#include <sys\stat.h>
#include <io.h>
#include <direct.h>     // for getcwd()
#else
#  include <sys/types.h>	// for directory reading
#  include <dirent.h>		// for directory reading
#endif

#ifdef HAVE_SYS_TIME_H
#  include <sys/time.h>		// set mem allocation limit
#endif
#ifndef _MSC_VER
#  include <sys/resource.h>	// set mem allocation limit
#  include <unistd.h>		// set mem allocation limit
#endif

#include <simgear/compiler.h>

#include <iostream>
#include <string>
#include <vector>

#include <plib/sg.h>

#include <simgear/constants.h>
#include <simgear/bucket/newbucket.hxx>
#include <simgear/debug/logstream.hxx>
#include <simgear/math/polar3d.hxx>

#include <Geometry/poly_support.hxx>
#include <Array/array.hxx>
#include <Clipper/clipper.hxx>
#include <GenOutput/genobj.hxx>
#include <Match/match.hxx>
#include <Triangulate/triangle.hxx>
#include <landcover/landcover.hxx>
#include <Geometry/util.hxx>
#include <sstream>   // for ostingstream

using std::cout;
using std::cerr;
using std::endl;
using std::string;
using std::vector;
using std::ostringstream;

#include "poly-view.hxx"
#include "load-poly.hxx"      // class TGPolyLoad
#include "poly-utils.hxx"
#include "poly-text.hxx"
#include "poly-load.hxx"      // loaded poly functions
#include "poly-ini.hxx"       // for IsYes and IsNo
#ifdef ADD_SCENERY_LOAD
#include "load-scene.hxx"
#endif
#include "load-arr.hxx"

char * exe_name = "poly-view.exe";

static int exit_if_no_files = 0;

#define PREPROCESS_DIRECTORIES   // convert ALL directories to files

extern void set_mapobj_fgroot( char * root );
extern bool load_mapobj_materials( char * file );

// forward references
void  Out_Poly_Counts( ostringstream & msg );

static char _s_tmpbuf[1024];
static char _s_tmpbuf2[1024];

int g_clipping_enabled = 0;   // this take ENORMOUS amount of time
#ifdef NDEBUG
int g_use_thread = 1;   // turn on and off thread use
#else
int g_use_thread = 0;   // turn on and off thread use
#endif

int add_bucket_neighbors = 1; // load scenery AROUND a bucket

// some DEBUG switches
int debug_show_extra_info = 0;   // go through polys again
int debug_show_zero_str   = 0;     // maybe show materials
int debug_cmd_args        = 1;   // output cmd from input file
// ===================

typedef vector<SGBucket> sg_bucket_list;

sg_bucket_list bucket_list;
int add_to_bucket_list( SGBucket & b )
{
   size_t max = bucket_list.size();
   size_t i;
   for( i = 0; i < max; i++ ) {
      if( bucket_list[i] == b )
         return 0;
   }
   bucket_list.push_back(b);
   return 1;
}

// features
int load_into_one   = 1;  // load ALL files into ONE PTGPolyLoad (ppl)
int separate_chunks = 1;   // but if the CHUNK changes, use NEW ppl
// ========

string fg_scenery = "";

int done_arguments = 0;
static int bShowPolyCount = 0;   // mainly for diagnostics
// it usually shows a LOT of missing poly files, and is boisterous

// display usage and exit
void get_usage_stg( string name, ostringstream & stg )
{
    char * cp = GetNxtBuf();
    sprintf(cp,", compiled on %s, at %s", __DATE__, __TIME__);
    stg << "Usage: " << name << " [options] file/dir" << endl;
    stg << "Version " << PV_VERSION << " date " << PV_DATE;
    stg << cp << endl;
//    stg << "  --output-dir=<directory>" << endl;
    stg << "OPTIONS:" << endl;
    stg << "  --nudge=<float> = this will be applied to following files/dirs." << endl;
    stg << "  --work-file=<poly-file> = alternate to list following..." << endl;
    stg << "  --work-dir=<directory>  =         <ditto>" << endl;
//    stg << "  --cover=<path to land-cover raster>" << endl;
#ifdef ADD_WORK_LIMITS
    stg << "Set work limits..." << endl;
    stg << "  --chunk=<chunk> = load 10x10 degree chunk." << endl;
    stg << "  --tile=<tile> = load 1x1 degree tile." << endl;
    stg << "  --tile-id=<id>  = load one bucket, if found." << endl;
    stg << "The following, if given, must be all 4 items in degrees..." << endl;
    stg << "  --lon=<d> --lat=<d> --xdist=<d> --ydist=<d>" << endl;
#endif
    stg << "Map options, overriding default, the INI values..." << endl;
    stg << "  --autozoom=<yes|no> or --enable-autozoom or --disable-autozoom" << endl;
    stg << "  --zoom=<float> = will also set autozoom off" << endl;
    stg << "  --color=<yes|no> or --disable-color or --enable-color = no color changes" << endl;
    stg << "  --join-points=<yes|no> or --disable-joint-points or --enable-joint-points" << endl;
    stg << "  --join-last=<yes|no> or --disable-joint-last or --enable-joint-last" << endl;
    stg << "  --fg_scenery=<path> - path to FG scenery root." << endl;
    stg << "  --neighbors=<yes|no>, --disable-neighbors or --enable-neighbors" << endl;
    stg << "  --fg-root=<path>    - path to FG base data." << endl;
    stg << "  --palette=<file>    - palette file for map coloring." << endl;
    stg << "  --disable-scene-fill or --enable-scene-fill(def)" << endl;
    stg << "  --disable-poly-fill  or --enable-poly-fill(def)" << endl;
    stg << "  --disable-paint-X or --enable-paint-X - to paint each point as an X" << endl;
    stg << "  --set-spread-x=<num> - If above enables, use this spread value." << endl;
    stg << "  --center-lon-lat=<float:float> = Center the scene on these coordinates." << endl;
    stg << "  --disable-line-size or --enable-line-size (def) - vary line size per zoom." << endl;
    stg << "  --disable-scenery/--enable-scenery = g_use_fg_scenery = 0/1(def)." << endl;
    stg << "  --disable-scenery-paint/--enable-scenery-paint = g_enable_scenery = 0/1(def)." << endl;
    stg << "  --enable-scenery-wsg84 and/or --enable-scenery-normals = set these addition on. default = off." << endl;
    stg << "  --enable-join-points/--disable-join-points = g_join_poly_points = 1/0(def) 'j' toggle." << endl;
    stg << "  --enable-paint-fit-points plus --enable-paint-fit-tris, or disable (def)." << endl;
    stg << "  --enable-paint-buckets, or disable (def)." << endl;
    stg << "  --verbosity=<num>   - set LOG output verbosity. Default = 0." << endl;
//    stg << "  --useUKgrid" << endl;
    stg << endl;
    stg << "Arguments following the options will be taken as a load directory and/or file list..." << endl;
    stg << "NOTE: At least one VALID file, or directory MUST be given." << endl;
}

static void show_and_exit( ostringstream & stg )
{
#if defined(CONSOLE) || !defined(_MSC_VER)
   cout << stg.str();
#else
   stg << "On OK, will exit application." << endl;
   MessageBox(NULL, stg.str().c_str(), "poly-view error",
      MB_OK | MB_ICONINFORMATION );
#endif
   exit(1);
}

void show_usage_dialogue(void)
{
   ostringstream stg;
   get_usage_stg( exe_name, stg );
   MessageBox(NULL, stg.str().c_str(), "poly-view help",
      MB_OK | MB_ICONINFORMATION );
}

static void usage( const string name, const string arg, int ret ) {
   ostringstream stg;
   get_usage_stg( name, stg );
   if (ret)
       stg << endl << "Error: [" << arg << "]" << endl;

#if defined(CONSOLE) || !defined(_MSC_VER)
   cout << stg.str();
#else
   char * cp = GetNxtBuf();
   sprintf(cp, " (with exit value %d)",ret);
   stg << "On OK, will exit application." << cp << endl;
   sprtf("poly-view help\n");
   sprtf((char *)stg.str().c_str());
   MessageBox(NULL, stg.str().c_str(), "poly-view help",
      MB_OK | MB_ICONINFORMATION );
#endif
   exit(ret);
}

#define  ISNUM(a) (( a >= '0' ) && ( a <= '9' ))

static int is_all_numeric( string item )
{
   string::iterator si = item.begin();
   if( si != item.end() ) {
      for( ; si < item.end(); si++ ) {
         if( !ISNUM(*si) )
            return 0;
      }
      return 1;
   }
   return 0;
}

#ifdef   ADD_POLY_CLIPPING
static void clip_polys_for_bucket( SGBucket& b, TGPolyLoad * ppl )
{
    point2d min, max;
    double bc_lon    = b.get_center_lon();
    double bc_lat    = b.get_center_lat();
    double bh_width = 0.5 * b.get_width();
    double bh_height = 0.5 * b.get_height();

    min.x = bc_lon - bh_width;
    min.y = bc_lat - bh_height;
    max.x = bc_lon + bh_width;
    max.y = bc_lat + bh_height;

    // do clipping
    sprtf( "clipping polygons\n" );
    ppl->clip_all(min,max);

}
#endif // #ifdef   ADD_POLY_CLIPPING

#ifndef PREPROCESS_DIRECTORIES

int   debug_count = 0;
// Scan a directory and load polygon files.
static int load_polys_for_bucket( const string& dir,
			      SGBucket& b,
			      TGPolyLoad * ppl,
               ostringstream & msg,
               int level ) {
   int counter = 0;
   string base = b.gen_base_path();
   string tile_str = b.gen_index_str();
   string ext;
   string file, f_index, full_path;
   int pos, i;
   vector<string> dir_list;

   ulDir *d;
   struct ulDirEnt * de;

   if ( ( d = ulOpenDir( dir.c_str() ) ) == NULL ) {
	   msg << "cannot open directory " << dir << endl;
	   return 0;
   }

   while ( (de = ulReadDir(d)) != NULL ) {
      file = de->d_name;
      if ( de->d_isdir ) {
         if (( file == "." )||( file == ".." ))
            continue;
         file = dir + "\\" + de->d_name;
         dir_list.push_back(file);
         continue;
      }
      // process the file
      pos = file.find(".");
      if ( pos == -1 )
         continue;
      f_index = file.substr(0, pos);
      ext = file.substr(pos + 1);
      if ( tile_str == f_index ) {
         msg << file << "  " << f_index << "  '" << ext << "'" << endl;
         full_path = dir + "/" + file;
         if ( (ext == "arr") || (ext == "arr.gz") || 
                 (ext == "fit") || (ext == "fit.gz") || (ext == "ind")) {
                // skip
         } else if (ext == "osgb36") {
            msg << "Loading osgb36 poly definition file\n";
            //clipper.load_osgb36_polys( full_path, msg );
            ppl->load_osgb36_polys( full_path, msg );
            ++counter;
         } else {
            if ( msg.str().size() > 0 ) {
               out_to_log( (char *)msg.str().c_str() );
               msg.str("");
            }
            msg << "loading index " << tile_str << ", ext = '" << ext << "' for bucket " << b;
            paint_next_line_text( (char *)msg.str().c_str() );
            msg << endl;
            //clipper.load_polys( full_path, msg );
            ppl->load_polys( full_path, msg );
            ++counter;
            if ( msg.str().size() > 256 ) {
               out_to_log( (char *)msg.str().c_str() );
               msg.str("");
            }
         }
      }
   }

   for ( i = 0; i < (int)dir_list.size(); ++i ) {
      counter += load_polys_for_bucket( dir_list[i], b, ppl,
         msg, (level + 1) );
   }
   if( level == 0 ) {
#ifdef   ADD_POLY_CLIPPING
      if ( counter && g_clipping_enabled ) {
         show_memory_usage();
         clip_polys_for_bucket( b, ppl );
         show_memory_usage();
         debug_count++;
         //if( debug_count > 20 )
         //   exit(0);
      }
#endif // #ifdef   ADD_POLY_CLIPPING
      if ( bShowPolyCount ) {
         if ( counter ) {
            msg << "Got " << counter << " poly ";
         } else {
            msg << "NO poly ";
         }
         msg << " for index " << tile_str << endl;
      }
   }
   return counter;
}

#endif // #ifndef PREPROCESS_DIRECTORIES

typedef struct tagSTGLST {
   string file;
   double nudge;
}STGLST, * PSTGLST;

typedef vector<STGLST> vec_stglst;

typedef struct tagCMDARGS {
   double nudge;
   double center_lat, center_lon;
   double xdist;		// 1/2 degrees in each direction
   double ydist;
   long tile_id;
   int total_count;
   //vector<string> load_files; // individual files to load
   //vector<string> load_dirs;  // directories to search
   vec_stglst  file_items;
   vec_stglst  dir_items;
   int act_arg;
   TGPolyLoad * ppl;
}CMDARGS, * PCMDARGS;

// forward references
int parse_command_line(int argc, char **argv, ostringstream & msg,
                       STGLST & sl, PCMDARGS pca);

static int scan_directory_for_poly_files( const string& dir,
                                         PCMDARGS pca,
                                         ostringstream & msg,
                                         int level ) {
   int counter = 0;
   string ext;
   string file, f_index, full_path;
   int pos, i;
   vector<string> dir_list;
   STGLST sl;

   ulDir *d;
   struct ulDirEnt * de;

   msg << "Scanning [" << dir << "]" << endl;
   if ( ( d = ulOpenDir( dir.c_str() ) ) == NULL ) {
      msg << "WARNING: cannot open directory " << dir << endl;
	   return 0;
   }

   while ( (de = ulReadDir(d)) != NULL ) {
      file = de->d_name;
      if ( de->d_isdir ) {
         if (( file == "." )||( file == ".." ))
            continue;
         file = dir + "\\" + de->d_name;
         dir_list.push_back(file);
         continue;
      }
      // process the file
      full_path = dir + "/" + file;
      pos = file.find(".");
      if ( pos == -1 ) {
         msg << "Skipping [" << full_path << "] file. No point." << endl;
         continue;
      }
      f_index = file.substr(0, pos);
      if ( !is_all_numeric(f_index) ) {
         msg << "Skipping [" << full_path << "] file. Non-numeric." << endl;
         continue;
      }
      ext = file.substr(pos + 1);
      // msg << file << "  " << f_index << "  '" << ext << "'" << endl;
      if ((ext == "arr") || (ext == "arr.gz") || 
          (ext == "fit") || (ext == "fit.gz") ||
          (ext == "btg") || (ext == "btg.gz") ||
          (ext == "ind") || (ext == "stg"   ) ) {
             // skip
         msg << "Skipping [" << full_path << "] file ext." << endl;
      } else {
         msg << "Adding [" << full_path << "] to file list." << endl;
         // pca->load_files.push_back(full_path);
         sl.file = full_path;
         sl.nudge = pca->nudge;
         pca->file_items.push_back(sl);
         ++counter;
         if ( msg.str().size() > 256 ) {
            out_to_log( (char *)msg.str().c_str() );
            msg.str("");
         }
      }
   }

   for ( i = 0; i < (int)dir_list.size(); ++i ) {
      if ( msg.str().size() > 256 ) {
         out_to_log( (char *)msg.str().c_str() );
         msg.str("");
      }
      counter += scan_directory_for_poly_files( dir_list[i], pca, msg, (level + 1));
   }
   return counter;
}

vector<string> path_list;

static int load_polys_for_file( const string& file,
                               const string & ext,
                               SGBucket& b,
                               ostringstream & msg,
                               PCMDARGS pca)
{
   int counter = 0;
   string btg = ".btg";
   string arr = ".arr";
   string fit = ".fit";
   TGPolyLoad * ppl;
   // initialize loader
   if( load_into_one ) {
      int pos = file.rfind('/');
      string path = file.substr(0, pos);
      if( !pca->ppl ) {
         // get first
         pca->ppl = get_new_loaded_poly();
         path_list.push_back(path);
      } else {
         if( separate_chunks ) {
            int i;
            for( i = 0; i < path_list.size(); i++ ) {
               if( path_list[i] == path ) {
                  break;
               }
            }
            if( i == path_list.size() ) {
               // not found - set an show counts
               // this should be done later, NOT per file
               // Out_Poly_Counts( msg ); // for currently ACTIVE polys
               pca->ppl = get_new_loaded_poly(); // get a NEW loader
               path_list.push_back(path); // and store this path
            }
         }
      }
      ppl = pca->ppl;
   } else {
      ppl = get_new_loaded_poly();
   }
   ppl->nudge = pca->nudge;
   if ( msg.str().size() > 0 ) {
      out_to_log( (char *)msg.str().c_str() );
      msg.str("");
   }
   msg << "Loading file [" << file << "] for bucket " << b << ", nudge=" << pca->nudge;
   paint_next_line_text( (char *)msg.str().c_str() );
   msg << endl;
   if ( msg.str().size() > 0 ) {
      out_to_log( (char *)msg.str().c_str() );
      msg.str("");
   }
   size_t st = file.find(btg);
   if ( st != std::string::npos ) {
       ppl->load_btg_polys( file, msg );
   } else {
       st = file.find(arr);
       if ( st != std::string::npos ) {
           load_arr_polys( file, ppl );
       } else {
           st = file.find(fit);
           if ( st != std::string::npos ) {
               load_arr_polys( file, ppl );
           } else {
                // ASSUME IT _IS_ A POLY FILE
                //clipper.load_polys( full_path, msg );
                ppl->load_polys( file, msg );
           }
       }
   }
   if( !load_into_one ) {
      Set_Poly_Counts( ppl );
   }
   ++counter;
   sprtf("Done file %s\n", file.c_str());
   return counter;
}

static CMDARGS cmdargs;

static int load_nearby_buckets( const string& file,
			      SGBucket& b,
               ostringstream & msg,
               PCMDARGS pca)
{
   int load_cnt = 0;
   int dx,dy;
   int   i, j;
   double bc_c_lon = b.get_center_lon();
   double bc_c_lat = b.get_center_lat();
   // we need the bucket around this bucket
   double tl_lon = bc_c_lon - b.get_width();
   double tl_lat = bc_c_lat - b.get_height();
   double br_lon = bc_c_lon + b.get_width();
   double br_lat = bc_c_lat + b.get_height();
   SGBucket b_min(tl_lon, tl_lat);
   SGBucket b_max(br_lon, br_lat);
   sgBucketDiff(b_min, b_max, &dx, &dy);
   //cout << "  construction area spans tile boundaries" << endl;
   msg << "Add buckets dx = " << dx << "  dy = " << dy << endl;
   for ( j = 0; j <= dy; j++ ) {
      for ( i = 0; i <= dx; i++ ) {
         SGBucket b_cur = sgBucketOffset(tl_lon, tl_lat, i, j);
         if( add_to_bucket_list( b_cur ) )
         {
            msg << "Neighbor: " << b_cur << endl;
         }
      }
   }
   return load_cnt;
}

int is_bucket_path( string work_file, string & base_name )
{
    char ew = 0;
    char ns = 0;
    int lat = 200;
    int lon = 200;
    int pos = work_file.rfind("/");
    if (pos > 0) {
        string path_name = work_file.substr(0,pos);
        pos = path_name.rfind("/");
        if (pos > 0) {
            string last_path = path_name.substr(pos+1);
            if (sscanf(last_path.c_str(), "%c%3d%c%2d", &ew, &lon, &ns, &lat ) == 4) {
                if (((ew == 'e')||(ew == 'w'))&&
                    ((ns == 'n')||(ns == 's'))&&
                    ((lat >= 0)&&(lat <= 90))&&
                    ((lon >= 0)&&(lon <= 180))) {
                    if (ew == 'w')
                        lon = -lon;
                    if (ns == 's')
                        lat = -lat;
                    SGBucket b((double)lon,(double)lat);
                    long index = b.gen_index();
                    char buff[128];
                    sprintf(buff,"%ld.btg.gz",index);
                    base_name = buff;
                    return 1;
                }
            }
        }
    }
    return 0;
}
//unsigned int __stdcall process_dirs_or_file( void * vp )
DWORD WINAPI process_dirs_or_file( void * vp )
{
   int arg_pos, max, add_buck;
   string work_dir, work_file;
   ostringstream msg;
   char * ps = _s_tmpbuf2;
   PCMDARGS pca = (PCMDARGS)vp;
   STGLST sl;
   PPolyGlobs ppg = get_poly_globs();

   paint_next_line_text("Begin of thread...");
   pca->total_count = 0;
#ifdef PREPROCESS_DIRECTORIES
   //max = pca->load_dirs.size();
   max = pca->dir_items.size();
   for( arg_pos = 0; arg_pos < max; arg_pos++ )
   {
      int this_count;
      //work_dir = pca->v_load_dirs[arg_pos];
      sl = pca->dir_items[arg_pos];
      work_dir = sl.file;
      msg << "Processing directory " << work_dir << endl;
      out_to_log( (char *)msg.str().c_str() );
      msg.str("");
      sprintf(ps, "Processing %s... nudge %f, moment...", work_dir.c_str(), sl.nudge );
      paint_top_red_text(ps);  // paint to TOP line (in red)
      pca->act_arg = arg_pos;
      pca->nudge = sl.nudge;
      this_count = scan_directory_for_poly_files( work_dir, pca, msg, 0);
   }
   //pca->load_dirs.clear();
   pca->dir_items.clear();
#endif

   //max = pca->load_files.size();
   max = pca->file_items.size();
   for( arg_pos = 0; arg_pos < max; arg_pos++ )
   {
      int this_count = 0;
      //work_file = pca->load_files[arg_pos];
      sl = pca->file_items[arg_pos];
      work_file = sl.file;
      // this shows nudge...
      msg << endl << "Processing file " << work_file << ", nudge= " << sl.nudge << endl;
      out_to_log( (char *)msg.str().c_str() );
      msg.str("");
      // this shows nudge...
      sprintf(ps, "Processing %s, nudge=%f... moment...", work_file.c_str(), sl.nudge );
      paint_top_red_text(ps);  // paint to TOP line (in red)
      // determine bucket for this (assumed) polygon file
      set_unix_sep(work_file);
      int pos = work_file.rfind("/");
      string file_name = work_file.substr(pos + 1);
      msg << "file name = " << file_name << endl;
      pos = file_name.find(".");
      string base_name = file_name.substr(0, pos);
      string extension = file_name.substr(pos);
      msg << "base_name = " << base_name << ", ext = " << extension << endl;
      if( !is_all_numeric( base_name ) ) {
          if ( !is_bucket_path( work_file, base_name ) ) {
              msg << "ALERT: ERROR: base name [" << base_name << "] is NOT all numeric!" << endl;
              msg << "File: " << work_file << " NOT LOADED!" << endl;
              continue;
          }
         //show_and_exit(msg);
         //return 0;
      }
      long index;
      sscanf( base_name.c_str(), "%ld", &index);
      SGBucket b(index);
      msg << "bucket = " << b << endl;
      pca->act_arg = arg_pos;
      pca->nudge   = sl.nudge;   // set nudge for this file
      load_polys_for_file( work_file, extension, b, msg, pca );
      pca->total_count++;
      add_buck = add_to_bucket_list( b );
      if ( msg.str().size() > 0 ) {
         out_to_log( (char *)msg.str().c_str() );
         msg.str("");
      }
      if( add_buck && ppg->g_add_neighbors ) { // add_bucket_neighbors
         load_nearby_buckets( work_file, b, msg, pca );
      }
   }

   if ( msg.str().size() > 0 ) {
      out_to_log( (char *)msg.str().c_str() );
      msg.str("");
   }

   Out_Poly_Counts( msg ); // scan all polys loaded, and set limits
   if( msg.str().size() > 0 ) {
      msg << endl;
      out_to_log( (char *)msg.str().c_str() );
      msg.str("");
   }

#ifdef ADD_SCENERY_LOAD
   if( ppg->g_got_fg_scenery && ppg->g_use_fg_scenery ) {
      max = bucket_list.size();
      sprtf( "\nLoading scenery tiles for %d buckets...\n", max );
      for( arg_pos = 0; arg_pos < max; arg_pos++ ) {
         load_scenery_tile( ppg->g_fg_scenery, bucket_list[arg_pos] );
      }
   }
#endif
   msg << "[Finished successfully] with count " <<  pca->total_count << endl << endl;
   out_to_log( (char *)msg.str().c_str() );
   msg.str("");

   sprintf(ps, "End of thread, with count %d", pca->total_count );
   paint_next_line_text(ps);
   // sprtf("%s\n",ps);
   done_arguments = 1;
   Do_InvalidateRect(NULL);
   return pca->total_count;
}

int process_dirs_on_thread( PCMDARGS pca )
{
   DWORD dwThreadId;
   if( g_use_thread ) {
      HANDLE hThread = CreateThread( 
            NULL,              // default security attributes
            0,                 // use default stack size  
            process_dirs_or_file,     // thread function 
            pca,             // argument to thread function 
            0,                 // use default creation flags 
            &dwThreadId);   // returns the thread identifier 
   
      if ( hThread == NULL ) {
         sprtf( "ERROR: Thread FAILED...\n" );
         exit(1);
         return 0;
      }
   } else {
      process_dirs_or_file(pca); // thread function 
   }
   return 1;
}

// void set_yes_no_option( arg.substr(11), &get_poly_globs()->g_auto_zoom, "--autozoom" );
void set_yes_no_option( string & val, int * pi, char * pitem, ostringstream & msg )
{
   if( IsYes((char *)val.c_str()) )
      *pi = 1;
   else if( IsNo((char *)val.c_str()) )
      *pi = 0;
   else {
      msg << "ERROR: INVALID ITEM! [" << val << "]" << endl;
      msg << "Only 'yes', 'on', 'off' or 'no' can follow " << pitem << "..." << endl;
      show_and_exit(msg);
   }
}

int load_input_file( char * exe, string & file, ostringstream & msg, STGLST & sl, PCMDARGS pca)
{
   int iret = 0;
   size_t size = get_file_size(file);
   FILE * inp = fopen( file.c_str(), "rb" );
   if( !inp )
      return 1;
   if(size == 0) {
      fclose(inp);
      return 0;   // silently ignore empty files
   }
   char * buf = (char *)malloc(size + 1);
   if( !buf ) {
      msg << "Memory allocation of " << (size+1) << " bytes FAILED!" << endl;
      fclose(inp);
      return 2;
   }
   int count = fread( buf, 1, size, inp );
   fclose(inp);   // finished with file handle - close it
   if( count != size ) {
      msg << "Read of " << size << " bytes FAILED!" << endl;
      free(buf);
      return 3;
   }
   size_t i, i2, ibgn;
   char * cmd;
   vector<char *> cmd_list;
   cmd_list.clear();
   for(i = 0; i < size; i++) {
      char c = buf[i];
      if(( c == ';' )||( c == '#' )) {
         i++;
         for(; i < size; i++) {
            c = buf[i];
            if(( c < ' ')&&( c != '\t' ))
               break;
         }
      } else if( c > ' ' ) {
         // commence parameter
         ibgn = i;
         cmd = &buf[i]; // keep start
         i2 = i;
         i++; // bump, and go to end of command
         for(; i < size; i++) {
            c = buf[i];
            if((( c == ';' )||( c == '#' ))||(( c < ' ')&&( c != '\t' ))) {
               // reached end of command
               i2 = i;
               buf[i] = 0; // zero the END
               while(i2 > ibgn) {
                  if( buf[i2] > ' ' )
                     break;
                  buf[i2] = 0;   // zero any trailing spacey
                  i2--; // and back up
               }
               break;
            }
         }
         if( i2 > ibgn )
            cmd_list.push_back(cmd);
         // silently skip if no command on line
         // if stopped by comment, go to END OF LINE
         if(( c == ';' )||( c == '#' )) {
            i++;
            for(; i < size; i++) {
               c = buf[i];
               if(( c < ' ')&&( c != '\t' ))
                  break;
            }
         }
      }
   }
   size = cmd_list.size();
   if(size) {  // we have some commands
      char ** argv = (char **)malloc( sizeof(char *) * (size + 2) );
      if(!argv) {
         msg << "Memory allocation of " << ( sizeof(char *) * (size + 2) ) << " bytes FAILED!" << endl;
         free(buf);
         return 4;
      }
      for( i = 0; i <= size; i++ ) {
         if( i == 0 )
            argv[i] = exe; // or could be just say "dummy";
         else {
            argv[i] = cmd_list[i - 1];
            if(debug_cmd_args) {
               cmd = cmd_list[i - 1];
               msg << "INP" << i << ": [" << cmd << "]" << endl;
               out_to_log( (char *)msg.str().c_str() );
               msg.str("");
            }
         }
      }
      argv[i] = 0;   // not sure this extra NULL is required
      // pass the command list to the parser, plus 1 argument
      i = parse_command_line(size+1, argv, msg, sl, pca);
      if( !i )
         iret = 5;
   }
   free(buf);
   return iret;
}

int set_center_lon_lat( string & arg )
{
    size_t len = arg.size();
    size_t off = arg.find(',');
    if (len < 3)
        return 1;
    if (off == std::string::npos)
        return 1;
    if (off < 2)
        return 1;
    if (off >= len - 1)
        return 1;
    string slon = arg.substr(0,off);
    string slat = arg.substr(off+1);
    double lon = atof(slon.c_str());
    double lat = atof(slat.c_str());
    if ( !in_world_range(lon,lat) )
        return 1;
    set_user_center_lon_lat( lon, lat );
    sprtf("Set USER center - lon=%0.6f, lat=%0.6f\n", lon, lat );

    return 0;
}
int parse_command_line(int argc, char **argv, ostringstream & msg,
                       STGLST & sl, PCMDARGS pca)
{
   // Parse the command-line arguments.
   int arg_pos;
   string arg, work_file, work_dir, tmp;
   double dbl;
   PPolyGlobs ppg = get_poly_globs();
   for (arg_pos = 1; arg_pos < argc; arg_pos++)
   {
      arg = argv[arg_pos];
      tmp = arg;
      while (tmp.at(0) == '-')
          tmp = tmp.substr(1);
      if ((tmp == "help")||(tmp == "version")||(tmp == "?")) {
        usage(argv[0],arg,0);
      }
      if (argv[arg_pos][0] == '@') {
         work_file = arg.substr(1);
         if ((is_file_or_directory( work_file ) != 2 ) ||
            load_input_file( argv[0], work_file, msg, sl, pca)) {
            msg << "ERROR: INPUT FILE [" << work_file << "] NOT VALID!" << endl;
            msg << "Current directory is [" << getcwd(NULL,0) << "]" << endl;
            show_and_exit(msg);
            return 0;
         }
      } else if (arg.find("--verbosity=") == 0) {
         tmp = arg.substr(12);
         ppg->g_verbosity = atoi(tmp.c_str());
      } else if (arg.find("--work-file=") == 0) {
         work_file = arg.substr(12);
         if (is_file_or_directory( work_file ) != 2 ) {
            msg << "ERROR: NOT VALID FILE! [" << work_file << "]" << endl;
            msg << "Current directory is [" << getcwd(NULL,0) << "]" << endl;
            show_and_exit(msg);
            return 0;
         }
         //pca->load_files.push_back(work_file);
         sl.file = work_file;
         sl.nudge = pca->nudge;
         pca->file_items.push_back(sl);
      } else if (arg.find("--work-dir=") == 0) {
         work_dir = arg.substr(11);
         if (is_file_or_directory( work_dir ) != 1 ) {
            msg << "ERROR: NOT VALID DIRECTORY! [" << work_dir << "]" << endl;
            msg << "Current directory is [" << getcwd(NULL,0) << "]" << endl;
            show_and_exit(msg);
            return 0;
         }
         //pca->load_dirs.push_back(work_dir);
         sl.file = work_dir;
         sl.nudge = pca->nudge;
         pca->dir_items.push_back(sl);
      } else if (arg.find("--nudge=") == 0) {
          pca->nudge = atof(arg.substr(8).c_str());   // *SG_EPSILON ???;
#ifdef ADD_WORK_LIMITS
      } else if (arg.find("--tile-id=") == 0) {
          pca->tile_id = atol(arg.substr(10).c_str());
      } else if (arg.find("--lon=") == 0) {
          pca->center_lon = atof(arg.substr(6).c_str());
      } else if (arg.find("--lat=") == 0) {
          pca->center_lat = atof(arg.substr(6).c_str());
      } else if (arg.find("--xdist=") == 0) {
          pca->xdist = atof(arg.substr(8).c_str());
      } else if (arg.find("--ydist=") == 0) {
          pca->ydist = atof(arg.substr(8).c_str());
//      } else if (arg.find("--cover=") == 0) {
//          cover = arg.substr(8);
//      } else if (arg.find("--useUKgrid") == 0) {
//	      useUKgrid = true;
      } else if (arg.find("--chunk=") == 0) {
         tg::Rectangle rectangle = tg::parseChunk(arg.substr(8).c_str(),10.0);
         double min_lon, min_lat, max_lon, max_lat;
         min_lon = rectangle.getMin().x();
         min_lat = rectangle.getMin().y();
         max_lon = rectangle.getMax().x();
         max_lat = rectangle.getMax().y();
         pca->center_lon = ( min_lon + max_lon ) / 2.0;
         pca->center_lat = ( min_lat + max_lat ) / 2.0;
         pca->xdist = max_lon - pca->center_lon;
         pca->ydist = max_lat - pca->center_lat;
      } else if (arg.find("--tile=") == 0) {
         tg::Rectangle rectangle = tg::parseChunk(arg.substr(7).c_str(),1.0);
         double min_lon, min_lat, max_lon, max_lat;
         min_lon = rectangle.getMin().x();
         min_lat = rectangle.getMin().y();
         max_lon = rectangle.getMax().x();
         max_lat = rectangle.getMax().y();
         pca->center_lon = ( min_lon + max_lon ) / 2.0;
         pca->center_lat = ( min_lat + max_lat ) / 2.0;
         pca->xdist = max_lon - pca->center_lon;
         pca->ydist = max_lat - pca->center_lat;
#endif
      } else if (arg.find("--zoom=") == 0) {
         work_file = arg.substr(7);
         dbl = atof(work_file.c_str());
         if( dbl > 0.0 ) {
            ppg->g_auto_zoom = 0;
            set_g_map_zoom( dbl / 1000.0 );
         } else {
            msg << "ERROR: INVALID ITEM! [" << work_dir << "]" << endl;
            msg << "A double greate than zero must follow --zoom=..." << endl;
            show_and_exit(msg);
            return 0;
         }
      } else if (arg.find("--enable-autozoom") == 0) {
            ppg->g_auto_zoom = 1;
      } else if (arg.find("--disable-autozoom") == 0) {
            ppg->g_auto_zoom = 0;
      } else if (arg.find("--autozoom=") == 0) {
         set_yes_no_option( arg.substr(11),
            &ppg->g_auto_zoom, "--autozoom=", msg );
      } else if (arg == "--disable-scene-fill") {
          ppg->g_fill_scene = 0;
      } else if (arg == "--enable-scene-fill") {
          ppg->g_fill_scene = 1;
      } else if (arg == "--disable-poly-fill") {
          ppg->g_fill_polygon = 0;
      } else if (arg == "--enable-poly-fill") {
          ppg->g_fill_polygon = 1;
      } else if (arg == "--enable-color") {
         ppg->g_paint_colors = 1;
      } else if (arg == "--disable-color") {
         ppg->g_paint_colors = 0;
      } else if (arg.find("--color=") == 0) {
         set_yes_no_option( arg.substr(8),
            &ppg->g_paint_colors, "--color=", msg );
      } else if (arg == "--enable-join-points") {   // g_join_poly_points
         ppg->g_join_poly_points = 1;
      } else if (arg == "--disable-join-points") {
         ppg->g_join_poly_points = 0;
      } else if (arg == "--enable-paint-fit-points") {   // g_paint_fit_points
         ppg->g_paint_fit_points = 1;
      } else if (arg == "--disable-paint-fit-points") {
         ppg->g_paint_fit_points = 0;
      } else if (arg == "--enable-paint-fit-tris") {   // g_paint_fit_tris
         ppg->g_paint_fit_tris = 1;
      } else if (arg == "--disable-paint-fit-tris") {
         ppg->g_paint_fit_tris = 0;
      } else if (arg.find("--join-points=") == 0) {
         set_yes_no_option( arg.substr(14),
            &ppg->g_join_poly_points, "--join-points=", msg );
      } else if (arg == "--enable-join-last") {
         ppg->g_join_first_to_last = 1;
      } else if (arg == "--disable-join-last") {
         ppg->g_join_first_to_last = 0;
      } else if (arg == "--enable-paint-buckets") {
         ppg->g_paint_buckets = 1;
      } else if (arg == "--disable-paint-buckets") {
         ppg->g_paint_buckets = 0;
      } else if (arg.find("--join-last=") == 0) {
         set_yes_no_option( arg.substr(12),
            &ppg->g_join_first_to_last, "--join-last=", msg );
#ifdef ADD_SCENERY_LOAD
      } else if (arg.find("--fg-scenery=") == 0) {
         work_dir = arg.substr(13);
         if (is_file_or_directory( work_dir ) != 1 ) {
            msg << "ERROR: NOT VALID DIRECTORY! [" << work_dir << "]" << endl;
            msg << "Argument [" << arg << "] FAILED!" << endl;
            msg << "Current directory is [" << getcwd(NULL,0) << "]" << endl;
            show_and_exit(msg);
            return 0;
         }
         ppg->g_fg_scenery = work_dir;
         ppg->g_got_fg_scenery = 1;
      } else if (arg.find("--disable-scenery") == 0) {
          ppg->g_use_fg_scenery = 0;
      } else if (arg.find("--enable-scenery") == 0) {
          ppg->g_use_fg_scenery = 1;
      } else if (arg.find("--disable-scenery-paint") == 0) {
          ppg->g_enable_scenery = 0;
      } else if (arg.find("--enable-scenery-paint") == 0) {
          ppg->g_enable_scenery = 1;
      } else if (arg.find("--enable-scenery-wsg84") == 0) {
          ppg->g_enable_scenery_wsg84 = 1;
      } else if (arg.find("--enable-scenery-normals") == 0) {
          ppg->g_enable_scenery_normals = 1;
      } else if (arg.find("--disable-scenery-wsg84") == 0) {
          ppg->g_enable_scenery_wsg84 = 0;
      } else if (arg.find("--disable-scenery-normals") == 0) {
          ppg->g_enable_scenery_normals = 0;
      } else if (arg.find("--neighbors=") == 0) { // if load scenery, load neighboring buckets
         set_yes_no_option( arg.substr(12),
            &ppg->g_add_neighbors, "--neighbors=", msg );
      } else if (arg == "--enable-neighbors") {
         ppg->g_add_neighbors = 1;
      } else if (arg == "--disable-neighbors") {
         ppg->g_add_neighbors = 0;
#endif
      } else if (arg.find("--fg-root=") == 0) {
         work_dir = arg.substr(10);
         if (is_file_or_directory( work_dir ) != 1 ) {
            msg << "ERROR: NOT VALID DIRECTORY! [" << work_dir << "]" << endl;
            msg << "Argument [" << arg << "] FAILED!" << endl;
            msg << "Current directory is [" << getcwd(NULL,0) << "]" << endl;
            show_and_exit(msg);
            return 0;
         }
         set_mapobj_fgroot( (char *)work_dir.c_str() );
      } else if (arg.find("--center-lon-lat=") == 0) {
          // 12345678901234567
          // --center-lon-lat=-52.731165,47.603897
          if (set_center_lon_lat( arg.substr(17) )) {
              msg << "ERROR: NOT VALID LON,LAT! [" << arg << "]" << endl;
              show_and_exit(msg);
              return 0;
          }
      } else if (arg.find("--disable-paint-X") == 0) { // ppg->g_paint_X
          ppg->g_paint_X = 0;
      } else if (arg.find("--enable-paint-X") == 0) { // ppg->g_paint_X
          ppg->g_paint_X = 1;
      } else if (arg.find("--set-spread-x=") == 0) { // ppg->g_paint_X
          tmp = arg.substr(15);
          ppg->g_fixed_spread = atoi(tmp.c_str());
      } else if (arg.find("--disable-line-size") == 0) {
          ppg->g_var_line_size = 0;
      } else if (arg.find("--enable-line-size") == 0) {
          ppg->g_var_line_size = 1;
      } else if (arg.find("--palette=") == 0) {
         work_dir = arg.substr(10);
         if (is_file_or_directory( work_dir ) != 2 ) {
            msg << "ERROR: NOT VALID FILE! [" << work_dir << "]" << endl;
            msg << "Argument [" << arg << "] FAILED!" << endl;
            msg << "Current directory is [" << getcwd(NULL,0) << "]" << endl;
            show_and_exit(msg);
            return 0;
         }
         if( !load_mapobj_materials( (char *)work_dir.c_str() ) ) {
            msg << "WARNING: palette file [" << work_dir << "] FAILED!" << endl;
            msg << "ERROR: NOT VALID FILE! [" << work_dir << "]" << endl;
            msg << "Argument [" << arg << "] FAILED!" << endl;
            msg << "Current directory is [" << getcwd(NULL,0) << "]" << endl;
            show_and_exit(msg);
            return 0;
         }
      } else if (arg.find("-") == 0) {
          usage(argv[0],arg,1);
      } else {
          break;
      }
   }
   return arg_pos;
}

int main2(int argc, char **argv, ostringstream & msg)
{
   PPolyGlobs ppg = get_poly_globs();
   string work_file = "";
   string work_dir = ".";
   string cover = "";
   string arg;
   PCMDARGS pca = &cmdargs;
   long index;
   double   dbl;
   STGLST   sl;

   pca->xdist = -1;		// 1/2 degree in each direction
   pca->ydist = -1;
   pca->tile_id = -1;
   pca->nudge = 0.0;
   pca->center_lat = NO_LATLON;
   pca->center_lon = NO_LATLON;
   pca->ppl = NULL;
  
   char * ps = _s_tmpbuf;

   // flag indicating whether UK grid should be used for in-UK
   // texture coordinate generation
   bool useUKgrid = false;
    
   // not used in this WINDOWS application but several
   // of the TG libraries, and SG do, so...
   sglog().setLogLevels( SG_ALL, SG_DEBUG );

   // Parse the command-line arguments.
   int arg_pos = parse_command_line( argc, argv, msg, sl, pca );
   if(arg_pos == 0)
      return 0;

   // eat up any ramaining arguments as files or directories
   for ( ; arg_pos < argc; arg_pos++)
   {
      arg = argv[arg_pos];
      if (is_file_or_directory( arg ) == 2 ) {
         //pca->load_files.push_back(arg);
         sl.file = arg;
         sl.nudge = pca->nudge;
         pca->file_items.push_back(sl);
      } else if (is_file_or_directory( arg ) == 1 ) {
         //pca->load_dirs.push_back(arg);
         sl.file = arg;
         sl.nudge = pca->nudge;
         pca->dir_items.push_back(sl);
      } else {
         msg << "ERROR: NOT VALID DIRECTORY OR FILE! [" << arg << "]" << endl;
         msg << "Current directory is [" << getcwd(NULL,0) << "]" << endl;
         show_and_exit(msg);
         return 0;
      }
   }
   //if(( pca->load_dirs.size() == 0 )&&(pca->load_files.size() == 0)) {
   if(exit_if_no_files && ( pca->dir_items.size() == 0 )&&(pca->file_items.size() == 0)) {
      msg << "ERROR: NO VALID DIRECTORY OR FILES! " << endl;
      show_and_exit(msg);
      return 0;
   }

   // if we have files - test them
   //if( pca->load_files.size() > 0 ) {
   if( pca->file_items.size() > 0 ) {
      msg << "Working file(s):" << endl;
      //for( arg_pos = 0; arg_pos < pca->load_files.size(); arg_pos++ )
      for( arg_pos = 0; arg_pos < pca->file_items.size(); arg_pos++ )
      {
         //work_file = pca->load_files[arg_pos];
         sl = pca->file_items[arg_pos];
         work_file = sl.file;
         // determine bucket for this (assumed) polygon file
         set_unix_sep(work_file);
         int pos = work_file.rfind("/");
         string file_name = work_file.substr(pos + 1);
         msg << "file name = " << file_name << ", nudge=" << sl.nudge << endl;
         pos = file_name.find(".");
         string base_name = file_name.substr(0, pos);
         msg << "base_name = " << base_name << endl;
         if( !is_all_numeric( base_name ) ) {
              if ( !is_bucket_path( work_file, base_name ) ) {
                    msg << "ALERT: ERROR: base name [" << base_name << "] is NOT all numeric! aborting..." << endl;
                    show_and_exit(msg);
                    return 0;
              }
         }
         sscanf( base_name.c_str(), "%ld", &index);
         SGBucket b(index);
         msg << "bucket = " << b << endl;

      }
   }
   // if we have directories
   //if( pca->load_dirs.size() > 0 ) {
   if( pca->dir_items.size() > 0 ) {
      msg << "Working directory(s):" << endl;
      //for( arg_pos = 0; arg_pos < pca->load_dirs.size(); arg_pos++ )
      for( arg_pos = 0; arg_pos < pca->dir_items.size(); arg_pos++ )
      {
         //work_dir = pca->load_dirs[arg_pos];
         sl = pca->dir_items[arg_pos];
         work_dir = sl.file;
         msg << work_dir << ", with nudge=" << sl.nudge << endl;
      }
   }

#ifdef ADD_WORK_LIMITS
   // if given a tile ID, center on this
   // ==================================
   if ( pca->tile_id > 0 ) {
      SGBucket tb( pca->tile_id );
      msg << "Tile id is " << pca->tile_id << endl;
      pca->center_lon = tb.get_center_lon();
      pca->center_lat = tb.get_center_lat();
   } else if (pca->xdist == -1 || pca->ydist == -1) {
      // this is a BIT confused!!!
      // do I need any limits,
      // since I am loading files and/or directories
   } else {
      msg << "Center longitude is " << pca->center_lon << endl;
      msg << "Center latitude is " << pca->center_lat << endl;
      msg << "X distance is " << pca->xdist << endl;
      msg << "Y distance is " << pca->ydist << endl;
   }
#endif

   msg << "Nudge is " << pca->nudge << endl;

   int total_count = 0;
   total_count = process_dirs_on_thread(pca);
   return total_count;
}

void Show_Zoom_Set( ostringstream & msg,
                   double mapsize, double zoom,
                   double c_lat, double c_lon,
                   double ydist, double xdist )
{
   char * ptmp = _s_tmpbuf;
   double tlat, tlon;
   double   px, py;
   double nlat, nlon;
   msg << "Using s=" << mapsize << " z=" << zoom << endl;

   tlat = c_lat;
   tlon = c_lon;
   win_ll2pt( tlat, tlon, c_lat, c_lon, mapsize, zoom, &px, &py );
   win_pt2ll( &nlat, &nlon, c_lat, c_lon, mapsize, zoom, px, py );
   sprintf(ptmp, "CT %0.6f,%0.6f is x,y %3d,%3d. (r=%0.6f,%0.6f)",
      tlon, tlat, win_int(px), win_int(py), nlon, nlat);
   msg << ptmp << endl;

   tlat = c_lat + xdist;
   tlon = c_lon + ydist;
   win_ll2pt( tlat, tlon, c_lat, c_lon, mapsize, zoom, &px, &py );
   win_pt2ll( &nlat, &nlon, c_lat, c_lon, mapsize, zoom, px, py );
   sprintf(ptmp, "BL %0.6f,%0.6f is x,y %3d,%3d. (r=%0.6f,%0.6f)",
      tlon, tlat, win_int(px), win_int(py), nlon, nlat);
   msg << ptmp << endl;

   tlat = c_lat - xdist;
   tlon = c_lon - ydist;
   win_ll2pt( tlat, tlon, c_lat, c_lon, mapsize, zoom, &px, &py );
   win_pt2ll( &nlat, &nlon, c_lat, c_lon, mapsize, zoom, px, py );
   sprintf(ptmp, "TR %0.6f,%0.6f is x,y %3d,%3d. (r=%0.6f,%0.6f)",
      tlon, tlat, win_int(px), win_int(py), nlon, nlat);
   msg << ptmp << endl;

   out_to_log( (char *)msg.str().c_str() );
   msg.str("");
}

void  Out_Poly_Counts( ostringstream & msg )
{
   char * ptmp = _s_tmpbuf;
   double mapsize = 600;
   double zoom = 1.0;
   double c_lat, c_lon, xdist, ydist;
   int   x, y, i, cnt2;
   double  px, py;
   double tlat, tlon, nlat, nlon;
   PPolyGlobs ppg = get_poly_globs();
   TGPolyLoad * polyload = get_first_loaded_poly();
   if( !polyload ) {
      msg << "ADVICE: NO LOADED POLYS!" << endl;
      return;
   }
   double min_lon, min_lat, max_lon, max_lat, ctr_lat, ctr_lon;
   min_lon = min_lat = 2000;
   max_lon = max_lat = -2000;
   ctr_lat = ctr_lon = 0.0;
   cnt2 = 0;
   // process group of polys loaded - 1 or more
   do {
      cnt2++;
      PPolyCounter ppc = Get_Poly_Counts();
      double minx, miny, maxx, maxy;
      minx = ppc->pmin.lon();
      miny = ppc->pmin.lat();
      maxx = ppc->pmax.lon();
      maxy = ppc->pmax.lat();
      if ( minx < min_lon )
         min_lon = minx;
      if ( miny < min_lat )
         min_lat = miny;
      if ( maxx > max_lon )
         max_lon = maxx;
      if ( maxy > max_lat )
         max_lat = maxy;
      ctr_lon += ppc->pcenter.lon();
      ctr_lat += ppc->pcenter.lat();
   } while( (polyload = get_next_loaded_poly()) != NULL );

   // done processing ALL loaded polys
   c_lat = ctr_lat / (double)cnt2;
   c_lon = ctr_lon / (double)cnt2;
   if ( !get_g_user_center() ) {
       ppg->g_set_c_lat = c_lat;
       ppg->g_set_c_lon = c_lon;
       // set the offset center
       set_g_center_lon_lat( c_lon, c_lat );
   }
   //c_lat = ppg->g_ctr_lat = ppg->g_offset_lat + ppg->g_set_c_lat;
   //c_lon = ppg->g_ctr_lon = ppg->g_offset_lon + ppg->g_set_c_lon;
   ppg->g_min_lon = min_lon;
   ppg->g_max_lon = max_lon;
   ppg->g_min_lat = min_lat;
   ppg->g_max_lat = max_lat;
   ppg->g_xdist = xdist = c_lon - min_lon;
   ppg->g_ydist = ydist = c_lat - min_lat;

   // show these poly count results
   sprtf( "Range:%d: lon,lat min %0.6f,%0.6f to max %0.6f,%0.6f CTR: %0.6f,%0.6f.\n",
      cnt2,
      min_lon, min_lat, max_lon, max_lat,
      ((min_lon + max_lon) / 2.0),
      ((min_lat + max_lat) / 2.0) );

   if ( get_g_user_center() ) {
       sprtf( "Would CENTER lon,lat %0.6f,%0.6f - spread x,y %0.6f,%0.6f\n but got USER %0.6f,%0.6f. AutoZoom=%s\n",
          c_lon, c_lat,
          xdist, ydist,
          ppg->g_set_c_lon, ppg->g_set_c_lat,
          (ppg->g_auto_zoom ? "On" : "Off") );
   } else {
       sprtf( "Set CENTER lon,lat %0.6f,%0.6f - spread x,y %0.6f,%0.6f. AutoZoom=%s\n",
          c_lon, c_lat,
          xdist, ydist,
          (ppg->g_auto_zoom ? "On" : "Off") );
   }
   // now have GLOBAL min,max points
   mapsize = get_g_map_size(); // get current size
   zoom = get_g_map_zoom();    // get current zoom
   if( ppg->g_auto_zoom ) {
      // attempt to find a ZOOM factor which will FILL a 600x600 screen
      double change = get_g_zoom_change();   // { return zoom_change; }
      double tzoom[5];
      tlat = c_lat;
      tlon = c_lon;
      win_ll2pt( tlat, tlon, c_lat, c_lon, mapsize, zoom, &px, &py );
      x = win_int(px);
      y = win_int(py);
      // convert lat,lon = x,y to ZOOM
      win_llpt2zoom( min_lat, min_lon,
           c_lat, c_lon,
           mapsize,  &tzoom[0],
           14.0, 28.0 );
           //0.0, 0.0 );
      win_llpt2zoom( max_lat, max_lon,
           c_lat, c_lon,
           mapsize,  &tzoom[1],
           mapsize - 14.0, mapsize - 28.0);
           //mapsize, mapsize );
      win_llpt2zoom( min_lat, max_lon,
           c_lat, c_lon,
           mapsize,  &tzoom[2],
           14.0, mapsize - 28.0 );
           //0.0, mapsize );
      win_llpt2zoom( max_lat, min_lon,
           c_lat, c_lon,
           mapsize,  &tzoom[3],
           mapsize - 14.0, 28.0 );
           //mapsize, 0.0 );
      tzoom[4] = tzoom[0];
      if ( tzoom[1] > tzoom[4] )
         tzoom[4] = tzoom[1];
      if ( tzoom[2] > tzoom[4] )
         tzoom[4] = tzoom[2];
      if ( tzoom[3] > tzoom[4] )
         tzoom[4] = tzoom[3];
      sprintf( ptmp, "most %g, range 0,0=%g, ms,ms=%g, 0,ms=%g, ms.0=%g Was %g",
         tzoom[4], tzoom[0], tzoom[1], tzoom[2], tzoom[3], zoom );
      msg << "autozoom: " << ptmp;  // show the CALCULATED ZOOM factor
      set_g_map_zoom(tzoom[4]); // SET the CALCULATED ZOOM factor
      msg << " ZOOM=" << get_g_map_zoom() << endl;
      mapsize = get_g_map_size(); // get current size
      zoom = get_g_map_zoom();    // get current zoom
      Show_Zoom_Set( msg, mapsize, zoom, c_lat, c_lon, ydist, xdist );
   }

   // ================
   calculate_scroll_sizes();
   // ================

   if( debug_show_extra_info ) {
      // now one by one
      // ==============
      polyload = get_first_loaded_poly(); // already checked to yield a result!!!
      do {
         PPolyCounter ppc = Get_Poly_Counts();
         ostringstream stg;
         double min_lat, max_lat, min_lon, max_lon;
         min_lat = ppc->pmin.lat();
         min_lon = ppc->pmin.lon();
         max_lat = ppc->pmax.lat();
         max_lon = ppc->pmax.lon();
         c_lat   = ppc->pcenter.lat(); // (min_lat + max_lat) / 2.0;
         c_lon   = ppc->pcenter.lon(); // (min_lon + max_lon) / 2.0;
         xdist   = ppc->pdist.lon();   // max_lon - c_lon;
         ydist   = ppc->pdist.lat();   // max_lat - c_lat;

         msg << "Total " << ppc->total << " polys loaded." << endl;
         msg << "with total of " << ppc->points << " points defined." << endl;
         msg << "in range MIN lon,lat " << ppc->pmin.lon() << "," << ppc->pmin.lat() <<
            ", MAX lon,lat " << ppc->pmax.lon() << "," << ppc->pmax.lat() << endl;
         msg << "Center lon,lat " << c_lon << "," << c_lat << " spread x,y " <<
            xdist << "," << ydist << endl;
         out_to_log( (char *)msg.str().c_str() );
         msg.str("");

         if( ppg->g_auto_zoom ) {
            // just show the chosen ZOOM size
            mapsize = get_g_map_size(); // get current size
            zoom = get_g_map_zoom();    // get current zoom
            Show_Zoom_Set( msg, mapsize, zoom, c_lat, c_lon, ydist, xdist );
         } else {
            // show a set of ZOOM sizes
            zoom = 0.000005;
            Show_Zoom_Set( msg, mapsize, zoom, c_lat, c_lon, ydist, xdist );
            zoom = 0.00001;
            Show_Zoom_Set( msg, mapsize, zoom, c_lat, c_lon, ydist, xdist );
            zoom = 0.000013;
            Show_Zoom_Set( msg, mapsize, zoom, c_lat, c_lon, ydist, xdist );
            zoom = 0.000015;
            Show_Zoom_Set( msg, mapsize, zoom, c_lat, c_lon, ydist, xdist );
            zoom = 0.0001;
            Show_Zoom_Set( msg, mapsize, zoom, c_lat, c_lon, ydist, xdist );
            zoom = 0.001;
            Show_Zoom_Set( msg, mapsize, zoom, c_lat, c_lon, ydist, xdist );
         }
         cnt2 = 0;
         for ( i = 0; i < TG_MAX_AREA_TYPES; ++i ) {
            if ( ppc->size[i] ) {
               msg << get_area_name((AreaType)i) << "(" << i << ") = " << ppc->size[i] << endl;
            } else {
               stg << i << " ";
               cnt2++;
               if( cnt2 == 10 ) {
                  stg << endl;
                  cnt2 = 0;
               }
            }
         }
         if( debug_show_zero_str ) {
            if( cnt2 )
               stg << endl;
            if ( stg.str().size() )
               msg << "Zero: " << stg.str() << endl;
         }
         msg << "Total " << ppc->total << endl;
         out_to_log( (char *)msg.str().c_str() );
         msg.str("");
         // while there are loaded polys
      } while( (polyload = get_next_loaded_poly()) != NULL );
   }
}

int ParseArgs( int argc, char ** argv )
{
   int iret = 1;
   ostringstream stg;
   done_arguments = 0;
   sprtf( "%s: parsing %d commands...\n", argv[0], argc - 1 );
   show_memory_usage();
   exe_name = argv[0];
   int cnt = main2( argc, argv, stg );
   if( stg.str().size() ) {
      out_to_log( (char *)stg.str().c_str() );
      stg.str("");
   }
   //Out_Poly_Counts( stg );
   //out_to_log( (char *)stg.str().c_str() );
   if(cnt)
      iret = 0;   // continue
   show_memory_usage();
   return iret;
}

// poly-help.cxx
