// poly-utils.cxx
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif
#include <sys\types.h>
#include <sys\stat.h>
#include <simgear/compiler.h>
#include <simgear/constants.h>
#include <math.h>
#include "poly-utils.hxx"
#include "lib_sprtf.hxx"
#include "poly-load.hxx"   // for mm_ MACROS

extern HWND g_hWnd;

// DEBUG - define as 'sprtf' to get output
#define sprtf_distance

// =====================================
// set of (static) buffers
#define  MX_BUFFER_SIZE    264
#define  MX_BUFFER_COUNT   64
static int nxt_gen_buf = 0;
static char _s_nxt_buf[MX_BUFFER_SIZE * MX_BUFFER_COUNT];
char * GetNxtBuf(void)
{
   nxt_gen_buf++;
   if(nxt_gen_buf >= MX_BUFFER_COUNT)
      nxt_gen_buf = 0;
   return &_s_nxt_buf[(MX_BUFFER_SIZE * nxt_gen_buf)];
}

#ifdef _MSC_VER
#define M_ISDIR _S_IFDIR
#else
#define M_ISDIR __S_IFDIR
#endif

int is_file_or_directory( string cur_item )
{
   struct stat buf;
   // stat seems to FAIL on a UNC (network) path \\DELL01\something
   if( stat( cur_item.c_str(), &buf ) == 0 ) {
      if ( buf.st_mode & M_ISDIR )
         return 1; // is directory
      else
         return 2; // is file
   }
#ifdef _MSC_VER
   // this works on a UNC (network) path \\DELL01\something
   // but NOT for a raw DIRECTORY - must add \* to it...
   WIN32_FIND_DATA fd;
   HANDLE hfind = FindFirstFile( cur_item.c_str(), &fd );
   if( hfind && (hfind != (HANDLE)-1) ) {
      FindClose(hfind);
      if( fd.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY )
         return 1;   // is a direcotry
      else
         return 2;
   } else {
      char * cp = GetNxtBuf();
      strcpy(cp, cur_item.c_str());
      strcat(cp,"\\*");
      // note: network finds ONLY work with wild card added
      hfind = FindFirstFile( cp, &fd );
      if( hfind && (hfind != (HANDLE)-1) ) {
         FindClose(hfind);
         return 1;   // is a direcotry
      }
   }
#endif
   return 0;
}

size_t get_file_size( string cur_item )
{
   struct stat buf;
   // stat seems to FAIL on a UNC (network) path \\DELL01\something
   if( stat( cur_item.c_str(), &buf ) == 0 ) {
      return buf.st_size;
   }
#ifdef _MSC_VER
   // this works on a UNC (network) path \\DELL01\something
   // but NOT for a raw DIRECTORY - must add \* to it...
   WIN32_FIND_DATA fd;
   HANDLE hfind = FindFirstFile( cur_item.c_str(), &fd );
   if( hfind && (hfind != (HANDLE)-1) ) {
      FindClose(hfind);
      return fd.nFileSizeLow;
      if(fd.nFileSizeHigh)
         return -1;
   }
#endif
   return 0;
}


//inline double degs2rad( double x ) { return x * SG_DEGREES_TO_RADIANS; }
//inline double rad2degs( double x ) { return x * SG_RADIANS_TO_DEGREES; }
#define degs2rad(x) (x * SG_DEGREES_TO_RADIANS)
#define rad2degs(x) (x * SG_RADIANS_TO_DEGREES)

#define cos_degs(a)  cos( degs2rad(a) )

double er_equ = 6378137.0;   // earth radius, equator (6,378.1370 km?)
double er_pol = 6356752.314; // earth radius, polar   (6,356.7523 km?)
double get_erad( double lat )
{
	// case SANSON_FLAMSTEED:
	double a = cos(lat) / er_equ;
	double b = sin(lat) / er_pol;
	return (1.0 / sqrt( a * a + b * b ));
}

double get_erad_degs( double lat_degs )
{
   return get_erad( degs2rad(lat_degs) );
}

// dst[2] = earth_radius_lat( lat_r );
// dst[0] = dst[2] * cos(lat)*(lon-lon_r);
// dst[1] = dst[2] * (lat-lat_r);
// ============================================
// NOTE: For windows, the y pixel is inverted
// 0,0            MAX_X,0
//    ------------
//    |           |
//    |           |
//    |           |
//    -------------
// 0, MAX_Y       MAX_X,MAX_Y
// =============================================
void win_ll2pt( double obj_latr, double obj_lonr,
           double cent_latr, double cent_lonr,
           double map_size,  double map_zoom,
           double * px_pixel, double * py_pixel )
{
	double x,y,r;
	r = get_erad_degs(cent_latr); // case SANSON_FLAMSTEED:
   x = r * cos_degs(obj_latr)*(obj_lonr - cent_lonr);
   y = r * (obj_latr - cent_latr);
	*px_pixel = (map_size / 2.0) + (x * map_zoom);
	*py_pixel = (map_size - ((map_size / 2.0) + (y * map_zoom)));
}

void win_pt2ll( double * pobj_latr, double * pobj_lonr,
           double cent_latr, double cent_lonr,
           double map_size,  double map_zoom,
           double x_pixel, double y_pixel )
{
	double x,y,r,latr,lonr;
	r = get_erad_degs(cent_latr); // case SANSON_FLAMSTEED:
	x = (x_pixel - (map_size / 2.0)) / map_zoom;
	y = ((map_size - y_pixel) - (map_size / 2.0)) / map_zoom;
   latr = (y / r) + cent_latr;
   lonr = (x / (r * cos_degs(latr))) + cent_lonr;
   *pobj_latr = latr;
   *pobj_lonr = lonr; 
}

// convert lat,lon = x,y to ZOOM
void win_llpt2zoom( double obj_latr, double obj_lonr,
           double cent_latr, double cent_lonr,
           double map_size,  double * pmap_zoom,
           double px_pixel, double py_pixel )
{
	double x,y,r;
	r = get_erad_degs(cent_latr); // case SANSON_FLAMSTEED:
   x = r * cos_degs(obj_latr)*(obj_lonr - cent_lonr);
   y = r * (obj_latr - cent_latr);
	double zoom1 = abs(((map_size / 2.0) - px_pixel) / x);
	double zoom2 = abs((map_size - py_pixel - (map_size / 2.0)) / y);
   //*pmap_zoom = (zoom1 + zoom2) / 2.0; // return average
   *pmap_zoom = (zoom1 < zoom2) ? zoom1 : zoom2; // return smallest
   sprtf( "llpt2zoom: zoom1=%0.6fM, zoom2=%0.6fM, return=%0.6fM\n",
      zoom1 * 1000.0, zoom2 * 1000.0, *pmap_zoom * 1000.0 );
}


static double round_it( double d, int dec )
{
   double rval = 0.0;
   double val;
   int   i;
   double div = 1;
   switch(dec)
   {
   case 0:
      div = 1.0;
      break;
   case 1:
      div = 10.0;
      break;
   case 2:
      div = 100.0;
      break;
   case 3:
      div = 1000.0;
      break;
   case 4:
      div = 10000.0;
      break;
   case 5:
      div = 100000.0;
      break;
   case 6:
      div = 1000000.0;
      break;
   case 7:
      div = 10000000.0;
      break;
   case 8:
      div = 100000000.0;
      break;
   case 9:
      div = 1000000000.0;
      break;
   default:
      div = 10000000000.0;
      break;
   }
   val = d * div;    // bump number of decimal places
   i = win_int(val); // get the integer
   rval = ((double)i / div);  // and back into double
   return rval;
}

// aim - to get close to target number of division
// between two doubles, using largest (integral) value
// of step from the first... return 1 for succes, 0 = FAILED
int poly_get_dstep2( double min, double max,
               int target_min, int target_max,
               double *pfirst, double *pstep,
               int * pcnt, int * pdec )
{
   int i;
   double   av = (double)(target_min + target_max) / 2.0;
   double diff = max - min;   // get difference
   double tstep = diff / av;   // get steps to produce av. target divs
   double best_step;
   //sprtf( "From %0.12g to %0.12g, min %d, max %d\n", min, max, target_min, target_max );
   if( max <= min )
      return 0;

   for( i = 0; i < 10; i++ )
   {
      best_step = round_it( tstep, i );
      if(best_step > 0.0)
      {
         double r_min;
         int   cnt = 0;
         double best_first = round_it( min, i );
         while( best_first < min ) {
            cnt++;
            best_first = round_it( min + (best_step * (double)cnt), i );
         }
         cnt = 0;
         for(r_min = best_first; r_min < max; r_min += best_step) {
            cnt++;
         }
         if(( cnt >= target_min ) && ( cnt <= target_max ))
         {
            *pstep = best_step;
            *pfirst = best_first;
            if(pcnt)
               *pcnt = cnt;
            if(pdec)
               *pdec = i;
            return 1;   // success
         }
      }
   }
   // sprtf( "FAILED!\n" );
   return 0;
}

int double_value_is_integer( double val )
{
   int ival = win_int(val);
   if ( fabs((double)ival - val) < SG_EPSILON )
      return 1;
   return 0;
}

static int nxt_file_counter = 0;
char * Get_Next_FileName( char * pbase, char * pext )
{
   char * file_name = GetNxtBuf();
   sprintf(file_name, "%s%d.%s", pbase, nxt_file_counter, pext);
   while( is_file_or_directory((string)file_name) ) {
      nxt_file_counter++;
      sprintf(file_name, "%s%d.%s", pbase, nxt_file_counter, pext);
   }
   return file_name;
}

char * get_rectangle_string( PRECT prc )
{
   char * cp = GetNxtBuf();
   sprintf( cp, "[%d,%d,%d,%d]",
      prc->left, prc->top, prc->right, prc->bottom );
   return cp;
}


#ifdef   ADD_OLD_BITMAP_SERVICES
// =======================================================
// write CLIENT screen to BITMAP file
// WIDTHBYTES takes # of bits in a scan line and rounds up to nearest
//  dword (32-bits). The # of bits in a scan line would typically be
//  the pixel width (bmWidth) times the BPP (bits-per-pixel = bmBitsPixel)
#define WIDTHBYTES(bits)      (((bits) + 31) / 32 * 4)

#define  DVGlobalAlloc  GlobalAlloc
#define  DVGlobalLock   GlobalLock
#define  DVGlobalFree   GlobalFree
#define  DVGlobalUnlock GlobalUnlock

#define  MLPTR *
#define	Win30_Bmp	(sizeof (BITMAPINFOHEADER))

#define IS_WIN30_DIB(lpbi)  ( (*(LPDWORD) (lpbi)) == Win30_Bmp )
//--------------  DIB header Marker Define -------------------------
#define DIB_HEADER_MARKER   ((WORD) ('M' << 8) | 'B')	/* Simple "BM" ... */

DWORD	DIBBitCount( LPSTR lpbi )
{
	DWORD dwBC;
	if( IS_WIN30_DIB(lpbi) )
		dwBC = ((LPBITMAPINFOHEADER) lpbi)->biBitCount;
	else
		dwBC = ((LPBITMAPCOREHEADER) lpbi)->bcBitCount;
	return( dwBC );
}

DWORD DIBBPP( LPSTR lpDIB )
{
   DWORD dwBPP;
   if( IS_WIN30_DIB( lpDIB ) ) {
		//wCompression = (WORD) ((LPBITMAPINFOHEADER) lpDIB)->biCompression;
		dwBPP = ((LPBITMAPINFOHEADER) lpDIB)->biBitCount;
	} else {
      //wCompression = BI_PM;
      dwBPP = ((LPBITMAPCOREHEADER) lpDIB)->bcBitCount;
	}
   return dwBPP;
}


// Only certain bitmaps have a color count
// That is a COLORREF table
DWORD	GetColorCnt( DWORD wBitCount )
{
	DWORD	wClrCnt = 0;
	switch( wBitCount )
	{
	case 1:
		wClrCnt = 2;
		break;
	case 4:
		wClrCnt = 16;
		break;
	case 8:
		wClrCnt = 256;
		break;
	}
	return wClrCnt;
}


DWORD	CalcDIBColors( LPSTR lpbi )
{
	DWORD	dwClrCnt, dwBitCount;
	dwBitCount = DIBBitCount( lpbi);
	dwClrCnt   = GetColorCnt( dwBitCount );
	return dwClrCnt;
}

DWORD DIBNumColors (LPSTR lpbi)
{
   DWORD wClrCount;
   // If this is a Windows style DIB, the number of colors in the
   //  color table can be less than the number of bits per pixel
   //  allows for (i.e. lpbi->biClrUsed can be set to some value).
   //  If this is the case, return the appropriate value.
   if (IS_WIN30_DIB (lpbi)) {
      DWORD dwClrUsed = ((LPBITMAPINFOHEADER) lpbi)->biClrUsed;
      if (dwClrUsed)
         return (DWORD) dwClrUsed;
   }
   // Calculate the number of colors in the color table based on
   //  the number of bits per pixel for the DIB.
	wClrCount = CalcDIBColors( lpbi );
	return wClrCount;
}

DWORD PaletteSize (LPSTR lpbi)
{
   if (IS_WIN30_DIB (lpbi))
      return (DIBNumColors (lpbi) * sizeof (RGBQUAD));
   else
      return (DIBNumColors (lpbi) * sizeof (RGBTRIPLE));
}

/****************************************************************************
  FUNCTION   : WinDibFromBitmap()
  PURPOSE    : Will create a global memory block in DIB format that
		 represents the Device-dependent bitmap (DDB) passed in.
		      dwStyle -> DIB format     ==  RGB, RLE
		      wBits  -> Bits per pixel ==  1,4,8,24
  RETURNS    : A handle to the DIB
 ****************************************************************************/
HANDLE WinDibFromBitmap( HBITMAP hBitmap,
						DWORD dwStyle,
						WORD wBits, 
                  HPALETTE hPal )
{
	BITMAP               bm;
	BITMAPINFOHEADER     bi;
	BITMAPINFOHEADER MLPTR lpbi;
	DWORD                dwLen;
	HANDLE               hDIB;
	HANDLE               h;
	HDC                  hDC;
	WORD                 wB;
	if( !hBitmap )
		return NULL;

	if( hPal == NULL )
		hPal = (HPALETTE)GetStockObject( DEFAULT_PALETTE );

   if( !GetObject( hBitmap, sizeof(BITMAP), (LPSTR)&bm ) )
		return NULL;
	if( wBits == 0 ) {
		wB =  bm.bmPlanes * bm.bmBitsPixel;
	} else {
		wB = wBits;
	}
	// Condition the BIT size ... is this really required???
	if( wB <= 1 )
		wB = 1;
	else if( wB <= 4 )
		wB = 4;
	else if( wB <= 8 )
		wB = 8;
	else
		wB = 24;
   // setup the info header
	bi.biSize               = sizeof (BITMAPINFOHEADER);
	bi.biWidth              = bm.bmWidth;
	bi.biHeight             = bm.bmHeight;
	bi.biPlanes             = 1;
	bi.biBitCount           = wB;
	bi.biCompression        = dwStyle;
	bi.biSizeImage          = 0;
	bi.biXPelsPerMeter      = 0;
	bi.biYPelsPerMeter      = 0;
	bi.biClrUsed            = 0;
	bi.biClrImportant       = 0;
	dwLen  = bi.biSize + PaletteSize( (LPSTR)&bi );
	hDIB = DVGlobalAlloc( GHND, dwLen );
	if( !hDIB )
		return( NULL );

	lpbi   = (BITMAPINFOHEADER MLPTR)DVGlobalLock(hDIB);  // LOCK DIB HANDLE
   if( !lpbi ) {
		DVGlobalFree( hDIB );
      return( NULL );
   }
	*lpbi  = bi;
	hDC    = GetDC (NULL);
	hPal   = SelectPalette (hDC, hPal, FALSE);
	RealizePalette(hDC);
	// call GetDIBits with a NULL lpBits param, so it
	// will calculate the biSizeImage field for us.
	GetDIBits( hDC,
		hBitmap,
		0,
		(WORD) bi.biHeight,
		NULL,
		(LPBITMAPINFO) lpbi,
		DIB_RGB_COLORS );

	bi = *lpbi;
	DVGlobalUnlock( hDIB ); // UNLOCK DIB HANDLE
	// If the driver did not fill in the biSizeImage field,
	// make one up
	if( bi.biSizeImage == 0 ) {
		bi.biSizeImage = WIDTHBYTES((DWORD)bm.bmWidth * wBits) *
			bm.bmHeight;
		if( dwStyle != BI_RGB )
			bi.biSizeImage = (bi.biSizeImage * 3) / 2;
	}

	// Realloc the buffer big enough to hold all the bits.
	dwLen = bi.biSize + PaletteSize((LPSTR)&bi) + bi.biSizeImage;
	if( h = GlobalReAlloc( hDIB, dwLen, 0 ) ) {
		hDIB = h;	// Return the HANDLE
	} else {
		DVGlobalFree( hDIB );
		hDIB = NULL;
		SelectPalette( hDC, hPal, FALSE );
		ReleaseDC( NULL, hDC );
		return( hDIB );
	}

	// call GetDIBits with a NON-NULL lpBits param,
	// and actualy get the bits this time.
	lpbi = (BITMAPINFOHEADER MLPTR)DVGlobalLock(hDIB); // LOCK DIB HANDLE
   if( !lpbi ) {
		DVGlobalFree( hDIB );
		SelectPalette( hDC, hPal, FALSE );
		ReleaseDC( NULL, hDC );
      return NULL;
   }

	if( GetDIBits( hDC,
		hBitmap,
		0,
		(WORD) bi.biHeight,
		(LPSTR) lpbi + (WORD) lpbi->biSize + PaletteSize((LPSTR) lpbi),
		(LPBITMAPINFO) lpbi, DIB_RGB_COLORS) == 0 )
	{
		DVGlobalUnlock (hDIB);  // UNLOCK DIB HANDLE - for error exit
		hDIB = NULL;
		SelectPalette (hDC, hPal, FALSE);
		ReleaseDC (NULL, hDC);
		return NULL;
	}
	bi = *lpbi;
	DVGlobalUnlock( hDIB ); // UNLOCK DIB HANDLE
	SelectPalette (hDC, hPal, FALSE);
	ReleaseDC (NULL, hDC);
	return hDIB;
}

void write_client_windows_to_bitmap( void )
{
   int   succeeded = 0;
   RECT  rc;
   HWND  hWnd = g_hWnd;
   char * file_name = Get_Next_FileName( "tempbmp", ".bmp" );

   if(hWnd) {
      GetClientRect( hWnd, &rc );
      int   cx = rc.right;
      int   cy = rc.bottom;
      HDC hdc = GetDC(hWnd);
      if(hdc) {
         HDC memDC = CreateCompatibleDC ( hdc );
         if(memDC) {
            HBITMAP oldBM;
            HPALETTE hPal = (HPALETTE)GetStockObject( DEFAULT_PALETTE );
            HBITMAP memBM = CreateCompatibleBitmap ( hdc, cx, cy );
            if(memBM) {
               oldBM = (HBITMAP)SelectObject ( memDC, memBM );
               hPal = (HPALETTE)SelectPalette ( memDC, hPal, FALSE );
            	RealizePalette(memDC);
               if ( BitBlt(memDC, 0,0, cx, cy, hdc, 0,0, SRCCOPY) ) {
                  SelectObject ( memDC, oldBM );
                  // DWORD dwStyle, maybe BI_RGB BI_RLE8, BI_RLE4, BI_BITFIELDS, BI_JPEG or BI_PNG
                  HANDLE hDIB = WinDibFromBitmap( memBM,
                     BI_RGB,  // DWORD dwStyle,
                     0,       // WORD wBits, 
                     NULL );  // HPALETTE hPal )
                  if( hDIB ) {
                     DWORD dws = GlobalSize(hDIB);
                     BITMAPFILEHEADER	   hdr;
                     LPBITMAPINFOHEADER   lpbi;
                     FILE * pf = fopen( file_name, "wb" );
                     if(pf) {
                        // Lock down the DIB
                        lpbi = (LPBITMAPINFOHEADER)DVGlobalLock( hDIB );   // LOCK DIB HANDLE
                        if( lpbi )  // GOT LOCKED DIB HANDLE
                        {
                           // Fill in the fields of the file header
                           ZeroMemory( &hdr, sizeof(BITMAPFILEHEADER) );
                           hdr.bfType		 = DIB_HEADER_MARKER;
                           hdr.bfSize		 = dws + sizeof (BITMAPFILEHEADER);
                           hdr.bfReserved1 = 0;
                           hdr.bfReserved2 = 0;
                           hdr.bfOffBits   = (DWORD)sizeof(BITMAPFILEHEADER) +   // file header plus
                                 lpbi->biSize +    // size of 	LPBITMAPINFOHEADER
                                 PaletteSize((LPSTR)lpbi ); // plus PALETTE (if any)
                           // write header to file
                           fwrite( &hdr, 1, sizeof(BITMAPFILEHEADER), pf );
                           fwrite( lpbi, 1, dws, pf );
                           fclose(pf);
                           DVGlobalUnlock( hDIB );
                           succeeded = 1;
                        }
                     }
               		DVGlobalFree( hDIB );
                  }
               }
             	SelectPalette (memDC, hPal, FALSE);
               DeleteObject(memBM);
            }
            DeleteDC(memDC);
         }
         ReleaseDC(g_hWnd,hdc);
      }
   }
   if( succeeded )
      sprtf("Written client screen to %s file.\n", file_name );
   else
      sprtf("WARNING: filed to write client to BMP file!\n" );
}

#endif // #ifdef   ADD_OLD_BITMAP_SERVICES

//   dist = get_distance_between( lprcClip->left, lprcClip->top,
//      lprcClip->right, lprcClip->bottom, &direction );
double get_distance_between( int left, int top,
                             int right, int bottom,
                             double * pdirection )
{
   double dist = 0.0;
   double dir  = 0.0;
   double tllat, tllon, brlat, brlon;
   double az1;
   mm_pt2ll( &tllat, &tllon, left, top );
   mm_pt2ll( &brlat, &brlon, right, bottom );
   double xdist = abs(tllon - brlon);
   double ydist = abs(tllat - brlat);
   if((xdist > SG_EPSILON)&&(ydist > SG_EPSILON)) {
      int i = geo_inverse_wgs_84( tllat, tllon, brlat, brlon,
         &az1, &dir, &dist );
   }
   if(pdirection)
      *pdirection = dir;
   sprtf_distance("distance: %0.6f, in %0.1f\n", dist, dir );
   return dist;
}

#ifdef _MSC_VER
/* =========================================================================
// New rint() - December, 2008 - Geoff R. McLane
//
// In WIN32, the largest unsigned value is __int64, 64-bits, which has a 
// maximum hex 0xffffffffffffffff, decimal 18446744073709551615,
// scientific 1.84467E+019, and the Tor rint(d) implementation below will FAIL
// if given a double larger (or smaller) than this MAXIMUM __int64!
//
// So have implemented this all double rint(), which works ;=))
// It matches the value from Ubuntu 64-bit rint() for all examples tested.
// ======================================================================== */
double rint( double x )
{
   if ( x > 0.0 )
      return floor( x + 0.5 );
   if ( x < 0.0 )
      return ceil( x - 0.5 );
   return 0.0;
}

// =======================================================
#endif // _MSC_VER

Point3D cartv3d_2_p3dllh( SGVec3d cart, const SGVec3d &c )
{
   cart += c;
   SGGeod geod;
   SGGeodesy::SGCartToGeod( cart, geod );
   Point3D pt( geod.getLongitudeDeg(), geod.getLatitudeDeg(), geod.getElevationM() );
   return pt;
}

Point3D cartv3f_2_p3dllh( SGVec3f cart, const SGVec3d &c )
{
   SGVec3d v;
   v[0] = cart[0];
   v[1] = cart[1];
   v[2] = cart[2];
   return cartv3d_2_p3dllh( v, c );
}

char * pt3d2stg( Point3D & pt )
{
   char * cp = GetNxtBuf();
   sprintf(cp, "%11.6f,%10.6f,%6.1f",
      pt.lon(), pt.lat(), pt.elev() );
   return cp;
}

char * sgv22stg( sgVec2 & pt )
{
   char * cp = GetNxtBuf();
   sprintf(cp, "%11.6f,%10.6f",
      pt[0], pt[1] );
   return cp;
}

/* ==========================================
   Bucket corners
   Center = lon,lat   -122.3750,   37.5625
   BL Corner 0:   -122.5000,   37.5000
   BR Corner 1:   -122.2500,   37.5000
   TR Corner 2:   -122.2500,   37.6250
   TL Corner 3:   -122.5000,   37.6250
   ========================================== */
#define  SG_BLC   0
#define  SG_BRC   1
#define  SG_TRC   2
#define  SG_TLC   3

bool out_of_bucket_range( double lon_in, double lat_in, SGBucket & b )
{
   SGGeod sgg[4];
   int i;
   double min_lon = 2000.0;
   double max_lon = -2000.0;
   double min_lat = 2000.0;
   double max_lat = -2000.0;
   double lon, lat;
   for(i = 0; i < 4; i++) {
      sgg[i] = b.get_corner(i);
      lon = sgg[i].getLongitudeDeg();
      lat = sgg[i].getLatitudeDeg();
      if(lon < min_lon) min_lon = lon;
      if(lon > max_lon) max_lon = lon;
      if(lat < min_lat) min_lat = lat;
      if(lat > max_lat) max_lat = lat;
   }
   if( lon_in < min_lon )
      return 1;
   if( lon_in > max_lon )
      return 1;
   if( lat_in < min_lat )
      return 1;
   if( lat_in > max_lat )
      return 1;

   //if( pt[0] < sgg[SG_BLC].getLongitudeDeg() ) //||(pt[0] < sgg[SG_TLC].getLongitudeDeg() ))
   //   return 1;
   //if( pt[0] > sgg[SG_BRC].getLongitudeDeg() )
   //   return 1;
   //if( pt[1] < sgg[SG_BLC].getLatitudeDeg() )
   //   return 1;
   //if( pt[1] > sgg[SG_TLC].getLatitudeDeg() )
   //   return 1;
   return 0;
}

bool sgv2_out_of_bucket_range( sgVec2 & pt, SGBucket & b )
{
   return out_of_bucket_range( pt[0], pt[1], b );
}

bool Pt3d_out_of_bucket_range( Point3D & pt, SGBucket & b )
{
   return out_of_bucket_range( pt.lon(), pt.lat(), b );
}

bool sgg_out_of_bucket_range( SGGeod & pt, SGBucket & b )
{
   return out_of_bucket_range( pt.getLongitudeDeg(), pt.getLatitudeDeg(), b );
}

// 2009-11-21
void split_file_path(string in_file,
                     string & out_dir, string & out_name, string & out_ext)
{
    string in_path = in_file;
    string file_name;
    set_unix_sep(in_path);
    int pos = in_path.rfind("/"); // get LAST 
    if (pos >= 0) {
        out_dir   = in_file.substr(0, pos);
        file_name = in_file.substr(pos + 1);
    } else {
        out_dir   = "";
        file_name = in_file;
    }
    pos = file_name.find(".");
    if (pos > 0) {
        out_name = file_name.substr(0, pos);
        out_ext  = file_name.substr(pos);
    } else {
        // no extent
        out_name = file_name;
        out_ext  = "";
    }
}

///////////////////////////////////////////////////////////////////////////////
// FUNCTION   : InStr
// Return type: int 
// Arguments  : char * lpb
//            : char * lps
// Description: Return the position of the FIRST instance of the string in lps
//              Emulates the Visual Basic function.
///////////////////////////////////////////////////////////////////////////////
int   InStr( char * lpb, char * lps )  // extracted from FixFObj.c, dc4wUtil.c
{
   int   iRet = 0;
   int   i, j, k, l, m;
   char  c;
   i = strlen(lpb);
   j = strlen(lps);
   if( i && j && ( i >= j ) ) {
      c = *lps;   // get the first we are looking for
      l = i - ( j - 1 );   // get the maximum length to search
      for( k = 0; k < l; k++ ) {    // search for first
         if( lpb[k] == c ) {
            // found the FIRST char so check until end of compare string
            for( m = 1; m < j; m++ ) {
               if( lpb[k+m] != lps[m] )   // on first NOT equal
                  break;   // out of here
            }
            if( m == j ) {  // if we reached the end of the search string
               iRet = k + 1;  // return NUMERIC position (that is LOGICAL + 1)
               break;   // and out of the outer search loop
            }
         }
      }  // for the search length
   }
   return iRet;
}

int in_world_range( double lon, double lat )
{
    if ((lon <= 180.0)&&(lon >= -180.0)&&(lat <= 90.0)&&(lat >= -90.0))
        return 1;
    return 0;
}

char * float3p2rgbstg( float * fp, int typ )
{
    char * cp = GetNxtBuf();
    COLORREF cr = RGB((int)(fp[0] * 255.0),(int)(fp[1] * 255.0),(int)(fp[2] * 255.0));
    if (typ) {
        sprintf(cp, "(%03d,%03d,%03d)=(%f,%f,%f)",
            GetRValue(cr), GetGValue(cr), GetBValue(cr),
            fp[0], fp[1], fp[2]);
    } else {
        sprintf(cp, "(%03d,%03d,%03d)",
            GetRValue(cr), GetGValue(cr), GetBValue(cr) );
    }
    return cp;
}

// eof - poly-utils.cxx
