// parseargs.cpp
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation; either version 2 of the
// License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
//

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

extern void Do_InvalidateRect(void * vp);

#ifdef _MSC_VER
extern void show_memory_usage( void );
#include <sys\types.h>
#include <sys\stat.h>
#include <io.h>
#include <direct.h>     // for getcwd()
#else
#  include <sys/types.h>	// for directory reading
#  include <dirent.h>		// for directory reading
#endif

#ifdef HAVE_SYS_TIME_H
#  include <sys/time.h>		// set mem allocation limit
#endif
#ifndef _MSC_VER
#  include <sys/resource.h>	// set mem allocation limit
#  include <unistd.h>		// set mem allocation limit
#endif

#include <simgear/compiler.h>

#include <iostream>
#include <string>
#include <vector>

#include <plib/sg.h>

#include <simgear/constants.h>
#include <simgear/bucket/newbucket.hxx>
#include <simgear/debug/logstream.hxx>
#include <simgear/math/polar3d.hxx>

#include <Geometry/poly_support.hxx>
#include <Array/array.hxx>
#include <Clipper/clipper.hxx>
#include <GenOutput/genobj.hxx>
#include <Match/match.hxx>
#include <Triangulate/triangle.hxx>
#include <landcover/landcover.hxx>
#include <Geometry/util.hxx>
#include <sstream>   // for ostingstream

using std::cout;
using std::cerr;
using std::endl;
using std::string;
using std::vector;
using std::ostringstream;

#include "poly-view.h"
#include "load-poly.hxx"      // class TGPolyLoad
#include "poly-utils.hxx"
#include "poly-paint.hxx"
#include "poly-load.hxx"      // loaded poly functions

#define PREPROCESS_DIRECTORIES   // convert ALL directories to files

// forward references
void  Out_Poly_Counts( ostringstream & msg );

static char _s_tmpbuf[1024];
static char _s_tmpbuf2[1024];

int g_clipping_enabled = 0;   // this take ENORMOUS amount of time
#ifdef NDEBUG
int g_use_thread = 1;   // turn on and off thread use
#else
int g_use_thread = 0;   // turn on and off thread use
#endif

int load_into_one = 1;

int done_arguments = 0;
static int bShowPolyCount = 0;   // mainly for diagnostics
// it usually shows a LOT of missing poly files, and is boisterous

// display usage and exit
void get_usage_stg( string name, ostringstream & stg )
{
    stg << "Usage: " << name << " [" << endl;
//    stg << "  --output-dir=<directory>" << endl;
    stg << "  --work-file=<poly-file>" << endl;
    stg << "  --work-dir=<directory>" << endl;
//    stg << "  --cover=<path to land-cover raster>" << endl;
    stg << "  --tile-id=<id>" << endl;
    stg << "  --lon=<degrees>" << endl;
    stg << "  --lat=<degrees>" << endl;
    stg << "  --xdist=<degrees>" << endl;
    stg << "  --ydist=<degrees>" << endl;
    stg << "  --nudge=<float>" << endl;
    stg << "  --autozoom=<yes|no>" << endl;
    stg << "  --zoom=<float> - will also set autozoom off" << endl;
    stg << "  --disable-color - no color changes" << endl;
//    stg << "  --useUKgrid" << endl;
    stg << " ] <load directory or file...>" << endl;
}

static void show_and_exit( ostringstream & stg )
{
#if defined(CONSOLE) || !defined(_MSC_VER)
   cout << stg.str();
#else
   stg << "On OK, will exit application." << endl;
   MessageBox(NULL, stg.str().c_str(), "poly-view error",
      MB_OK | MB_ICONINFORMATION );
#endif
   exit(1);
}

static void usage( const string name ) {
   ostringstream stg;
   get_usage_stg( name, stg );
#if defined(CONSOLE) || !defined(_MSC_VER)
   cout << stg.str();
#else
   stg << "On OK, will exit application." << endl;
   MessageBox(NULL, stg.str().c_str(), "poly-view help",
      MB_OK | MB_ICONINFORMATION );
#endif
   exit(1);
}

#ifdef _MSC_VER
#define M_ISDIR _S_IFDIR
#else
#define M_ISDIR __S_IFDIR
#endif

static int is_file_or_directory( string cur_item )
{
   struct stat buf;
   if( stat( cur_item.c_str(), &buf ) == 0 ) {
      if ( buf.st_mode & M_ISDIR )
         return 1; // is directory
      else
         return 2; // is file
   }
   return 0;
}

#define set_unix_sep(s) {    \
   size_t fnd = s.find('\\'); \
   while( fnd != -1 ) {       \
      s.replace(fnd, 1, "/"); \
      fnd = s.find('\\'); }}

#define  ISNUM(a) (( a >= '0' ) && ( a <= '9' ))

static int is_all_numeric( string item )
{
   string::iterator si = item.begin();
   if( si != item.end() ) {
      for( ; si < item.end(); si++ ) {
         if( !ISNUM(*si) )
            return 0;
      }
      return 1;
   }
   return 0;
}

#ifdef   ADD_POLY_CLIPPING
static void clip_polys_for_bucket( SGBucket& b, TGPolyLoad * ppl )
{
    point2d min, max;
    double bc_lon    = b.get_center_lon();
    double bc_lat    = b.get_center_lat();
    double bh_width = 0.5 * b.get_width();
    double bh_height = 0.5 * b.get_height();

    min.x = bc_lon - bh_width;
    min.y = bc_lat - bh_height;
    max.x = bc_lon + bh_width;
    max.y = bc_lat + bh_height;

    // do clipping
    sprtf( "clipping polygons\n" );
    ppl->clip_all(min,max);

}
#endif // #ifdef   ADD_POLY_CLIPPING

#ifndef PREPROCESS_DIRECTORIES

int   debug_count = 0;
// Scan a directory and load polygon files.
static int load_polys_for_bucket( const string& dir,
			      SGBucket& b,
			      TGPolyLoad * ppl,
               ostringstream & msg,
               int level ) {
   int counter = 0;
   string base = b.gen_base_path();
   string tile_str = b.gen_index_str();
   string ext;
   string file, f_index, full_path;
   int pos, i;
   vector<string> dir_list;

   ulDir *d;
   struct ulDirEnt * de;

   if ( ( d = ulOpenDir( dir.c_str() ) ) == NULL ) {
	   msg << "cannot open directory " << dir << endl;
	   return 0;
   }

   while ( (de = ulReadDir(d)) != NULL ) {
      file = de->d_name;
      if ( de->d_isdir ) {
         if (( file == "." )||( file == ".." ))
            continue;
         file = dir + "\\" + de->d_name;
         dir_list.push_back(file);
         continue;
      }
      // process the file
      pos = file.find(".");
      if ( pos == -1 )
         continue;
      f_index = file.substr(0, pos);
      ext = file.substr(pos + 1);
      if ( tile_str == f_index ) {
         msg << file << "  " << f_index << "  '" << ext << "'" << endl;
         full_path = dir + "/" + file;
         if ( (ext == "arr") || (ext == "arr.gz") || 
                 (ext == "fit") || (ext == "fit.gz") || (ext == "ind")) {
                // skip
         } else if (ext == "osgb36") {
            msg << "Loading osgb36 poly definition file\n";
            //clipper.load_osgb36_polys( full_path, msg );
            ppl->load_osgb36_polys( full_path, msg );
            ++counter;
         } else {
            if ( msg.str().size() > 0 ) {
               out_to_log( (char *)msg.str().c_str() );
               msg.str("");
            }
            msg << "loading index " << tile_str << ", ext = '" << ext << "' for bucket " << b;
            paint_next_line_text( (char *)msg.str().c_str() );
            msg << endl;
            //clipper.load_polys( full_path, msg );
            ppl->load_polys( full_path, msg );
            ++counter;
            if ( msg.str().size() > 256 ) {
               out_to_log( (char *)msg.str().c_str() );
               msg.str("");
            }
         }
      }
   }

   for ( i = 0; i < (int)dir_list.size(); ++i ) {
      counter += load_polys_for_bucket( dir_list[i], b, ppl,
         msg, (level + 1) );
   }
   if( level == 0 ) {
#ifdef   ADD_POLY_CLIPPING
      if ( counter && g_clipping_enabled ) {
         show_memory_usage();
         clip_polys_for_bucket( b, ppl );
         show_memory_usage();
         debug_count++;
         //if( debug_count > 20 )
         //   exit(0);
      }
#endif // #ifdef   ADD_POLY_CLIPPING
      if ( bShowPolyCount ) {
         if ( counter ) {
            msg << "Got " << counter << " poly ";
         } else {
            msg << "NO poly ";
         }
         msg << " for index " << tile_str << endl;
      }
   }
   return counter;
}

#endif // #ifndef PREPROCESS_DIRECTORIES

typedef struct tagCMDARGS {
   double nudge;
   double center_lat, center_lon;
   double xdist;		// 1/2 degrees in each direction
   double ydist;
   long tile_id;
   int total_count;
   vector<string> load_files; // individual files to load
   vector<string> load_dirs;  // directories to search
   TGPolyLoad * ppl;
}CMDARGS, * PCMDARGS;

static int scan_directory_for_poly_files( const string& dir,
                                         PCMDARGS pca,
                                         ostringstream & msg,
                                         int level ) {
   int counter = 0;
   string ext;
   string file, f_index, full_path;
   int pos, i;
   vector<string> dir_list;

   ulDir *d;
   struct ulDirEnt * de;

   msg << "Scanning [" << dir << "]" << endl;
   if ( ( d = ulOpenDir( dir.c_str() ) ) == NULL ) {
      msg << "WARNING: cannot open directory " << dir << endl;
	   return 0;
   }

   while ( (de = ulReadDir(d)) != NULL ) {
      file = de->d_name;
      if ( de->d_isdir ) {
         if (( file == "." )||( file == ".." ))
            continue;
         file = dir + "\\" + de->d_name;
         dir_list.push_back(file);
         continue;
      }
      // process the file
      full_path = dir + "/" + file;
      pos = file.find(".");
      if ( pos == -1 ) {
         msg << "Skipping [" << full_path << "] file. No point." << endl;
         continue;
      }
      f_index = file.substr(0, pos);
      if ( !is_all_numeric(f_index) ) {
         msg << "Skipping [" << full_path << "] file. Non-numeric." << endl;
         continue;
      }
      ext = file.substr(pos + 1);
      // msg << file << "  " << f_index << "  '" << ext << "'" << endl;
      if ((ext == "arr") || (ext == "arr.gz") || 
          (ext == "fit") || (ext == "fit.gz") ||
          (ext == "btg") || (ext == "btg.gz") ||
          (ext == "ind") || (ext == "stg"   ) ) {
             // skip
         msg << "Skipping [" << full_path << "] file ext." << endl;
      } else {
         msg << "Adding [" << full_path << "] to file list." << endl;
         pca->load_files.push_back(full_path);
         ++counter;
         if ( msg.str().size() > 256 ) {
            out_to_log( (char *)msg.str().c_str() );
            msg.str("");
         }
      }
   }

   for ( i = 0; i < (int)dir_list.size(); ++i ) {
      if ( msg.str().size() > 256 ) {
         out_to_log( (char *)msg.str().c_str() );
         msg.str("");
      }
      counter += scan_directory_for_poly_files( dir_list[i], pca, msg, (level + 1));
   }
   return counter;
}


static int load_polys_for_file( const string& file,
			      SGBucket& b,
               ostringstream & msg,
               PCMDARGS pca)
{
   int counter = 0;
   TGPolyLoad * ppl;
   // initialize loader
   if( load_into_one ) {
      if( !pca->ppl )
         pca->ppl = get_new_loaded_poly();
      ppl = pca->ppl;
   } else {
      ppl = get_new_loaded_poly();
   }
   ppl->nudge = pca->nudge;
   if ( msg.str().size() > 0 ) {
      out_to_log( (char *)msg.str().c_str() );
      msg.str("");
   }
   msg << "loading file [" << file << "] for bucket " << b;
   paint_next_line_text( (char *)msg.str().c_str() );
   msg << endl;
   //clipper.load_polys( full_path, msg );
   ppl->load_polys( file, msg );
   Set_Poly_Counts( ppl );
   ++counter;
   return counter;
}

static CMDARGS cmdargs;

//unsigned int __stdcall process_dirs_or_file( void * vp )
DWORD WINAPI process_dirs_or_file( void * vp )
{
   int arg_pos, max;
   string work_dir;
   ostringstream msg;
   char * ps = _s_tmpbuf2;
   PCMDARGS pca = (PCMDARGS)vp;

   paint_next_line_text("Begin of thread...");
   pca->total_count = 0;
#ifdef PREPROCESS_DIRECTORIES
   max = pca->load_dirs.size();
   for( arg_pos = 0; arg_pos < max; arg_pos++ )
   {
      int this_count;
      work_dir = pca->load_dirs[arg_pos];
      msg << "Processing directory " << work_dir << endl;
      out_to_log( (char *)msg.str().c_str() );
      msg.str("");
      sprintf(ps, "Processing %s... moment...", work_dir.c_str() );
      paint_center_text(ps);  // paint to TOP line (in red)
      this_count = scan_directory_for_poly_files( work_dir, pca, msg, 0);
   }
   pca->load_dirs.clear();
#endif

   max = pca->load_files.size();
   for( arg_pos = 0; arg_pos < max; arg_pos++ )
   {
      int this_count = 0;
      string work_file = pca->load_files[arg_pos];
      msg << "Processing file " << work_file << endl;
      out_to_log( (char *)msg.str().c_str() );
      msg.str("");
      sprintf(ps, "Processing %s... moment...", work_file.c_str() );
      paint_center_text(ps);  // paint to TOP line (in red)
      // determine bucket for this (assumed) polygon file
      set_unix_sep(work_file);
      int pos = work_file.rfind("/");
      string file_name = work_file.substr(pos + 1);
      msg << "file name = " << file_name << endl;
      pos = file_name.find(".");
      string base_name = file_name.substr(0, pos);
      msg << "base_name = " << base_name << endl;
      if( !is_all_numeric( base_name ) ) {
         msg << "ALERT: ERROR: base name [" << base_name << "] is NOT all numeric!" << endl;
         msg << "File: " << work_file << " NOT LOADED!" << endl;
         continue;
         //show_and_exit(msg);
         //return 0;
      }
      long index;
      sscanf( base_name.c_str(), "%ld", &index);
      SGBucket b(index);
      msg << "bucket = " << b << endl;
      load_polys_for_file( work_file, b, msg, pca );
      pca->total_count++;
   }

#ifndef PREPROCESS_DIRECTORIES
   max = pca->load_dirs.size();
   for( arg_pos = 0; arg_pos < max; arg_pos++ )
   {
      int this_count = 0;
      work_dir = pca->load_dirs[arg_pos];
      msg << "Processing directory " << work_dir << endl;
      out_to_log( (char *)msg.str().c_str() );
      msg.str("");
      sprintf(ps, "Processing %s... moment...", work_dir.c_str() );
      paint_center_text(ps);  // paint to TOP line (in red)
      // initialize loader
      TGPolyLoad * ppl = get_new_loaded_poly();
      
      ppl->nudge = pca->nudge;

      if (pca->tile_id == -1) {
	      if (pca->xdist == -1 || pca->ydist == -1) {
	         // construct the tile around the specified location
	         msg << "Building single tile at " << pca->center_lat << ',' << pca->center_lon << endl;
	         SGBucket b( pca->center_lon, pca->center_lat );
            this_count = load_polys_for_bucket( work_dir, b, ppl, msg, 0 );
         } else {
            // build all the tiles in an area
            int ix, jy;
	         msg << "Building tile(s) at " 
               << pca->center_lat << ',' << pca->center_lon
		         << " with x distance " << pca->xdist
		         << " and y distance " << pca->ydist << endl;
	         double min_x = pca->center_lon - pca->xdist;
	         double min_y = pca->center_lat - pca->ydist;
	         SGBucket b_min( min_x, min_y );
	         SGBucket b_max( pca->center_lon + pca->xdist, pca->center_lat + pca->ydist );

	         //SGBucket b_start(550401L);
	         bool do_tile = true;

	         if ( b_min == b_max ) {
               this_count = load_polys_for_bucket( work_dir, b_min, ppl, msg, 0 );
            } else {
		         SGBucket b_cur;
		         int dx, dy, i, j;
   	    
		         sgBucketDiff(b_min, b_max, &dx, &dy);
		         msg << "  construction area spans tile boundaries" << endl;
		         msg << "  dx = " << dx << "  dy = " << dy << endl;
               if ( msg.str().size() > 256 ) {
                  out_to_log( (char *)msg.str().c_str() );
                  msg.str("");
               }

      		   for ( jy = 0; jy <= dy; jy++ ) {
                  for ( ix = 0; ix <= dx; ix++ ) {
                     b_cur = sgBucketOffset(min_x, min_y, ix, jy);

                     string base = b_cur.gen_base_path();
                     string tile_str = b_cur.gen_index_str();

         			   if ( do_tile ) {
                         this_count += load_polys_for_bucket( work_dir, b_cur, ppl, msg, 0 );
			            } else {
			               msg << "skipping " << b_cur << endl;
			            }
		            }
		         }
	         }
	      }
      } else {
	      // construct the specified tile
         msg << "Building tile " << pca->tile_id << endl;
	      SGBucket b( pca->tile_id );
         this_count += load_polys_for_bucket( work_dir, b, ppl, msg, 0 );
      }
      Set_Poly_Counts( ppl );
      pca->total_count += this_count;
   }
#endif   // for NOT pre-process directories

   msg << "[Finished successfully] with count " <<  pca->total_count << endl;

   if ( msg.str().size() > 0 ) {
      out_to_log( (char *)msg.str().c_str() );
      msg.str("");
   }

   Out_Poly_Counts( msg );

   if( msg.str().size() > 0 ) {
      out_to_log( (char *)msg.str().c_str() );
      msg.str("");
   }

   sprintf(ps, "End of thread, with count %d", pca->total_count );
   paint_next_line_text(ps);
   // sprtf("%s\n",ps);
   done_arguments = 1;
   Do_InvalidateRect(NULL);
   return pca->total_count;
}

int process_dirs_on_thread( PCMDARGS pca )
{
   DWORD dwThreadId;
   if( g_use_thread ) {
      HANDLE hThread = CreateThread( 
            NULL,              // default security attributes
            0,                 // use default stack size  
            process_dirs_or_file,     // thread function 
            pca,             // argument to thread function 
            0,                 // use default creation flags 
            &dwThreadId);   // returns the thread identifier 
   
      if ( hThread == NULL ) {
         sprtf( "ERROR: Thread FAILED...\n" );
         exit(1);
         return 0;
      }
   } else {
      process_dirs_or_file(pca); // thread function 
   }
   return 1;
}


int main2(int argc, char **argv, ostringstream & msg)
{
   string work_file = "";
   string work_dir = ".";
   string cover = "";
   string arg;
   PCMDARGS pca = &cmdargs;
   long index;
   double   dbl;

   pca->xdist = -1;		// 1/2 degree in each direction
   pca->ydist = -1;
   pca->tile_id = -1;
   pca->nudge = 0.0;
   pca->center_lat = NO_LATLON;
   pca->center_lon = NO_LATLON;
   pca->ppl = NULL;
  
   char * ps = _s_tmpbuf;

   // flag indicating whether UK grid should be used for in-UK
   // texture coordinate generation
   bool useUKgrid = false;
    
   sglog().setLogLevels( SG_ALL, SG_DEBUG );

   // Parse the command-line arguments.
   int arg_pos;
   for (arg_pos = 1; arg_pos < argc; arg_pos++)
   {
      arg = argv[arg_pos];
      if (arg.find("--work-file=") == 0) {
         work_file = arg.substr(12);
         if (is_file_or_directory( work_file ) != 2 ) {
            msg << "ERROR: NOT VALID FILE! [" << work_file << "]" << endl;
            msg << "Current directory is [" << getcwd(NULL,0) << "]" << endl;
            show_and_exit(msg);
            return 0;
         }
         pca->load_files.push_back(work_file);
      } else if (arg.find("--work-dir=") == 0) {
         work_dir = arg.substr(11);
         if (is_file_or_directory( work_dir ) != 1 ) {
            msg << "ERROR: NOT VALID DIRECTORY! [" << work_dir << "]" << endl;
            msg << "Current directory is [" << getcwd(NULL,0) << "]" << endl;
            show_and_exit(msg);
            return 0;
         }
         pca->load_dirs.push_back(work_dir);
      } else if (arg.find("--tile-id=") == 0) {
          pca->tile_id = atol(arg.substr(10).c_str());
      } else if (arg.find("--lon=") == 0) {
          pca->center_lon = atof(arg.substr(6).c_str());
      } else if (arg.find("--lat=") == 0) {
          pca->center_lat = atof(arg.substr(6).c_str());
      } else if (arg.find("--xdist=") == 0) {
          pca->xdist = atof(arg.substr(8).c_str());
      } else if (arg.find("--ydist=") == 0) {
          pca->ydist = atof(arg.substr(8).c_str());
      } else if (arg.find("--nudge=") == 0) {
          pca->nudge = atof(arg.substr(8).c_str())*SG_EPSILON;
//      } else if (arg.find("--cover=") == 0) {
//          cover = arg.substr(8);
//      } else if (arg.find("--useUKgrid") == 0) {
//	      useUKgrid = true;
      } else if (arg.find("--chunk=") == 0) {
         tg::Rectangle rectangle = tg::parseChunk(arg.substr(8).c_str(),10.0);
         double min_lon, min_lat, max_lon, max_lat;
         min_lon = rectangle.getMin().x();
         min_lat = rectangle.getMin().y();
         max_lon = rectangle.getMax().x();
         max_lat = rectangle.getMax().y();
         pca->center_lon = ( min_lon + max_lon ) / 2.0;
         pca->center_lat = ( min_lat + max_lat ) / 2.0;
         pca->xdist = max_lon - pca->center_lon;
         pca->ydist = max_lat - pca->center_lat;
      } else if (arg.find("--zoom=") == 0) {
         work_file = arg.substr(7);
         dbl = atof(work_file.c_str());
         if( dbl > 0.0 ) {
            get_poly_globs()->g_auto_zoom = 0;
            set_g_map_zoom( dbl / 1000.0 );
         } else {
            msg << "ERROR: INVALID ITEM! [" << work_dir << "]" << endl;
            msg << "A double greate than zero must follow --zoom=..." << endl;
            show_and_exit(msg);
            return 0;
         }
      } else if (arg.find("--autozoom=") == 0) {
         work_file = arg.substr(11);
         if(work_file == "no")
            get_poly_globs()->g_auto_zoom = 0;
         else if(work_file == "yes")
            get_poly_globs()->g_auto_zoom = 1;
         else {
            msg << "ERROR: INVALID ITEM! [" << work_dir << "]" << endl;
            msg << "Only 'yes' or 'no' can follow --autozoom=..." << endl;
            show_and_exit(msg);
            return 0;
         }
      } else if (arg == "--disable-color") {
         get_poly_globs()->g_paint_colors = 0;
      } else if (arg.find("--") == 0) {
          usage(argv[0]);
      } else {
          break;
      }
   }

   // eat up any ramaining arguments as files or directories
   for ( ; arg_pos < argc; arg_pos++)
   {
      arg = argv[arg_pos];
      if (is_file_or_directory( arg ) == 2 ) {
         pca->load_files.push_back(arg);
      } else if (is_file_or_directory( arg ) == 1 ) {
         pca->load_dirs.push_back(arg);
      } else {
         msg << "ERROR: NOT VALID DIRECTORY OR FILE! [" << arg << "]" << endl;
         msg << "Current directory is [" << getcwd(NULL,0) << "]" << endl;
         show_and_exit(msg);
         return 0;
      }
   }
   if(( pca->load_dirs.size() == 0 )&&(pca->load_files.size() == 0)) {
      msg << "ERROR: NO VALID DIRECTORY OR FILES! " << endl;
      show_and_exit(msg);
      return 0;
   }

   // if we have files - test them
   if( pca->load_files.size() > 0 ) {
      msg << "Working file(s):" << endl;
      for( arg_pos = 0; arg_pos < pca->load_files.size(); arg_pos++ )
      {
         work_file = pca->load_files[arg_pos];
         // determine bucket for this (assumed) polygon file
         set_unix_sep(work_file);
         int pos = work_file.rfind("/");
         string file_name = work_file.substr(pos + 1);
         msg << "file name = " << file_name << endl;
         pos = file_name.find(".");
         string base_name = file_name.substr(0, pos);
         msg << "base_name = " << base_name << endl;
         if( !is_all_numeric( base_name ) ) {
             msg << "ALERT: ERROR: base name [" << base_name << "] is NOT all numeric! aborting..." << endl;
             show_and_exit(msg);
             return 0;
         }
         sscanf( base_name.c_str(), "%ld", &index);
         SGBucket b(index);
         msg << "bucket = " << b << endl;
      }
   }
   // if we have directories
   if( pca->load_dirs.size() > 0 ) {
      msg << "Working directory(s):" << endl;
      for( arg_pos = 0; arg_pos < pca->load_dirs.size(); arg_pos++ )
      {
         work_dir = pca->load_dirs[arg_pos];
         msg << work_dir << endl;
      }
   }

   // if given a tile ID, center on this
   // ==================================
   if ( pca->tile_id > 0 ) {
      SGBucket tb( pca->tile_id );
      msg << "Tile id is " << pca->tile_id << endl;
      pca->center_lon = tb.get_center_lon();
      pca->center_lat = tb.get_center_lat();
   } else if (pca->xdist == -1 || pca->ydist == -1) {
      // this is a BIT confused!!!
      // do I need any limits,
      // since I am loading files and/or directories
   } else {
      msg << "Center longitude is " << pca->center_lon << endl;
      msg << "Center latitude is " << pca->center_lat << endl;
      msg << "X distance is " << pca->xdist << endl;
      msg << "Y distance is " << pca->ydist << endl;
   }

   msg << "Nudge is " << pca->nudge << endl;

   int total_count = 0;
   total_count = process_dirs_on_thread(pca);
   return total_count;
}

void Show_Zoom_Set( ostringstream & msg,
                   double mapsize, double zoom,
                   double c_lat, double c_lon,
                   double ydist, double xdist )
{
   char * ptmp = _s_tmpbuf;
   double tlat, tlon;
   double   px, py;
   double nlat, nlon;
   msg << "Using s=" << mapsize << " z=" << zoom << endl;

   tlat = c_lat;
   tlon = c_lon;
   win_ll2pt( tlat, tlon, c_lat, c_lon, mapsize, zoom, &px, &py );
   win_pt2ll( &nlat, &nlon, c_lat, c_lon, mapsize, zoom, px, py );
   sprintf(ptmp, "CT %0.6f,%0.6f is x,y %3d,%3d. (r=%0.6f,%0.6f)",
      tlon, tlat, win_int(px), win_int(py), nlon, nlat);
   msg << ptmp << endl;

   tlat = c_lat + xdist;
   tlon = c_lon + ydist;
   win_ll2pt( tlat, tlon, c_lat, c_lon, mapsize, zoom, &px, &py );
   win_pt2ll( &nlat, &nlon, c_lat, c_lon, mapsize, zoom, px, py );
   sprintf(ptmp, "BL %0.6f,%0.6f is x,y %3d,%3d. (r=%0.6f,%0.6f)",
      tlon, tlat, win_int(px), win_int(py), nlon, nlat);
   msg << ptmp << endl;

   tlat = c_lat - xdist;
   tlon = c_lon - ydist;
   win_ll2pt( tlat, tlon, c_lat, c_lon, mapsize, zoom, &px, &py );
   win_pt2ll( &nlat, &nlon, c_lat, c_lon, mapsize, zoom, px, py );
   sprintf(ptmp, "TR %0.6f,%0.6f is x,y %3d,%3d. (r=%0.6f,%0.6f)",
      tlon, tlat, win_int(px), win_int(py), nlon, nlat);
   msg << ptmp << endl;

   out_to_log( (char *)msg.str().c_str() );
   msg.str("");
}

void  Out_Poly_Counts( ostringstream & msg )
{
   char * ptmp = _s_tmpbuf;
   double mapsize = 600;
   double zoom = 1.0;
   double c_lat, c_lon, xdist, ydist;
   int   x, y, i, cnt2;
   double  px, py;
   double tlat, tlon, nlat, nlon;
   PPolyGlobs ppg = get_poly_globs();
   TGPolyLoad * polyload = get_first_loaded_poly();
   if( !polyload ) {
      msg << "ADVICE: NO LOADED POLYS!" << endl;
      return;
   }
   double min_lon, min_lat, max_lon, max_lat, ctr_lat, ctr_lon;
   min_lon = min_lat = 2000;
   max_lon = max_lat = -2000;
   ctr_lat = ctr_lon = 0.0;
   cnt2 = 0;
   do {
      cnt2++;
      PPolyCounter ppc = Get_Poly_Counts();
      double minx, miny, maxx, maxy;
      minx = ppc->pmin.lon();
      miny = ppc->pmin.lat();
      maxx = ppc->pmax.lon();
      maxy = ppc->pmax.lat();
      if ( minx < min_lon )
         min_lon = minx;
      if ( miny < min_lat )
         min_lat = miny;
      if ( maxx > max_lon )
         max_lon = maxx;
      if ( maxy > max_lat )
         max_lat = maxy;
      ctr_lon += ppc->pcenter.lon();
      ctr_lat += ppc->pcenter.lat();
   } while( (polyload = get_next_loaded_poly()) != NULL );
   c_lat = ctr_lat / (double)cnt2;
   c_lon = ctr_lon / (double)cnt2;
   ppg->g_set_c_lat = c_lat;
   ppg->g_set_c_lon = c_lon;
   // set the offset center
   c_lat = ppg->g_c_lat = ppg->g_offset_lat + ppg->g_set_c_lat;
   c_lon = ppg->g_c_lon = ppg->g_offset_lon + ppg->g_set_c_lon;
   sprtf( "Range:%d: lon,lat min %0.6f,%0.6f to max %0.6f,%0.6f CTR: %0.6f,%0.6\n",
      cnt2,
      min_lon, min_lat, max_lon, max_lat,
      ((min_lon + max_lon) / 2.0),
      ((min_lat + max_lat) / 2.0) );
   xdist = c_lon - min_lon;
   ydist = c_lat - min_lat;
   sprtf( "Set CENTER lon,lat %0.6f,%0.6f - spread x,y %0.6f,0.6f\n",
      c_lon, c_lat,
      xdist, ydist );
   // now have GLOBAL min,max points
   if( ppg->g_auto_zoom ) {
      // attempt to find a ZOOM factor which will FILL a 600x600 screen
      double change = get_g_zoom_change();   // { return zoom_change; }
      double tzoom[5];
      mapsize = get_g_map_size(); // get current size
      zoom = get_g_map_zoom();    // get current zoom
      tlat = c_lat;
      tlon = c_lon;
      win_ll2pt( tlat, tlon, c_lat, c_lon, mapsize, zoom, &px, &py );
      x = win_int(px);
      y = win_int(py);
      // convert lat,lon = x,y to ZOOM
      win_llpt2zoom( min_lat, min_lon,
           c_lat, c_lon,
           mapsize,  &tzoom[0],
           0.0, 0.0 );
      win_llpt2zoom( max_lat, max_lon,
           c_lat, c_lon,
           mapsize,  &tzoom[1],
           mapsize, mapsize );
      win_llpt2zoom( min_lat, max_lon,
           c_lat, c_lon,
           mapsize,  &tzoom[2],
           0.0, mapsize );
      win_llpt2zoom( max_lat, min_lon,
           c_lat, c_lon,
           mapsize,  &tzoom[3],
           mapsize, 0.0 );
      tzoom[4] = tzoom[0];
      if ( tzoom[1] > tzoom[4] )
         tzoom[4] = tzoom[1];
      if ( tzoom[2] > tzoom[4] )
         tzoom[4] = tzoom[2];
      if ( tzoom[3] > tzoom[4] )
         tzoom[4] = tzoom[3];
      sprintf( ptmp, "most %g, range 0,0=%g, ms,ms=%g, 0,ms=%g, ms.0=%g Was %g",
         tzoom[4], tzoom[0], tzoom[1], tzoom[2], tzoom[3], zoom );
      msg << "autozoom: " << ptmp;
      set_g_map_zoom(tzoom[4]);
      msg << " ZOOM=" << get_g_map_zoom() << endl;
      mapsize = get_g_map_size(); // get current size
      zoom = get_g_map_zoom();    // get current zoom
      Show_Zoom_Set( msg, mapsize, zoom, c_lat, c_lon, ydist, xdist );
   }

   // now one by one
   // ==============
   polyload = get_first_loaded_poly(); // already checked to yield a result!!!
   do {
      PPolyCounter ppc = Get_Poly_Counts();
      ostringstream stg;
      double min_lat, max_lat, min_lon, max_lon;
      min_lat = ppc->pmin.lat();
      min_lon = ppc->pmin.lon();
      max_lat = ppc->pmax.lat();
      max_lon = ppc->pmax.lon();
      c_lat   = ppc->pcenter.lat(); // (min_lat + max_lat) / 2.0;
      c_lon   = ppc->pcenter.lon(); // (min_lon + max_lon) / 2.0;
      xdist   = ppc->pdist.lon();   // max_lon - c_lon;
      ydist   = ppc->pdist.lat();   // max_lat - c_lat;

      msg << "Total " << ppc->total << " polys loaded." << endl;
      msg << "with total of " << ppc->points << " points defined." << endl;
      msg << "in range MIN lon,lat " << ppc->pmin.lon() << "," << ppc->pmin.lat() <<
         ", MAX lon,lat " << ppc->pmax.lon() << "," << ppc->pmax.lat() << endl;
      msg << "Center lon,lat " << c_lon << "," << c_lat << " spread x,y " <<
         xdist << "," << ydist << endl;
      out_to_log( (char *)msg.str().c_str() );
      msg.str("");

      if( ppg->g_auto_zoom ) {
         // just show the chosen ZOOM size
         mapsize = get_g_map_size(); // get current size
         zoom = get_g_map_zoom();    // get current zoom
         Show_Zoom_Set( msg, mapsize, zoom, c_lat, c_lon, ydist, xdist );
      } else {
         // show a set of ZOOM sizes
         zoom = 0.000005;
         Show_Zoom_Set( msg, mapsize, zoom, c_lat, c_lon, ydist, xdist );
         zoom = 0.00001;
         Show_Zoom_Set( msg, mapsize, zoom, c_lat, c_lon, ydist, xdist );
         zoom = 0.000013;
         Show_Zoom_Set( msg, mapsize, zoom, c_lat, c_lon, ydist, xdist );
         zoom = 0.000015;
         Show_Zoom_Set( msg, mapsize, zoom, c_lat, c_lon, ydist, xdist );
         zoom = 0.0001;
         Show_Zoom_Set( msg, mapsize, zoom, c_lat, c_lon, ydist, xdist );
         zoom = 0.001;
         Show_Zoom_Set( msg, mapsize, zoom, c_lat, c_lon, ydist, xdist );
      }
      cnt2 = 0;
      for ( i = 0; i < TG_MAX_AREA_TYPES; ++i ) {
         if ( ppc->size[i] ) {
            msg << get_area_name((AreaType)i) << "(" << i << ") = " << ppc->size[i] << endl;
         } else {
            stg << i << " ";
            cnt2++;
            if( cnt2 == 10 ) {
               stg << endl;
               cnt2 = 0;
            }
         }
      }
      if( cnt2 )
         stg << endl;
      if ( stg.str().size() )
         msg << "Zero: " << stg.str() << endl;

      msg << "Total " << ppc->total << endl;
      out_to_log( (char *)msg.str().c_str() );
      msg.str("");
      // while there are loaded polys
   } while( (polyload = get_next_loaded_poly()) != NULL );
}

int ParseArgs( int argc, char ** argv )
{
   int iret = 1;
   ostringstream stg;
   done_arguments = 0;
   sprtf( "%s: parsing %d commands...\n", argv[0], argc - 1 );
   show_memory_usage();

   int cnt = main2( argc, argv, stg );
   if( stg.str().size() ) {
      out_to_log( (char *)stg.str().c_str() );
      stg.str("");
   }
   //Out_Poly_Counts( stg );
   //out_to_log( (char *)stg.str().c_str() );
   if(cnt)
      iret = 0;   // continue
   show_memory_usage();
   return iret;
}

// parseargs.cpp
