/* ==================================================
   qt_osm_map project
   Created: Geoff R. McLane - Sep 2011
   License: GPL2 (or later)
   With special thanks to Yves for FGx, and its map widget
   ================================================== */

#include "app_config.h"
#include "mainwindow.h"
#include "osm_map.h"
#include "testdialog.h"
#include "moveDialog.h"
#include "pathdialog.h"
#include "utilities/utilities.h"

MainWindow::MainWindow(QWidget *parent)
    : QMainWindow(parent)
{
    QString tmp;
    debug_mode = false;

#ifdef USE_INI_FILE
    tmp = get_data_file(DEF_INI_FILE);
    settings = new QSettings(tmp,QSettings::IniFormat,this);
    tmp = "MainWindow: Using INI file ["+tmp+"]";
#else // !USE_INI_FILE
    settings = new QSettings(this);
    tmp = "MainWindow: Using 'generic' global settings";
#endif // USE_INI_FILE y/b

    util_setStdLogFile();   // create 'standard' LOG file
    outLog(tmp);            // advise type of settings used

    // restore windows size and postion - saved at closeEvent(QCloseEvent*)
    restoreGeometry(settings->value("mainWindowGeometry").toByteArray());

#ifdef LOADAIRPORTS_H
    loadair = 0;
#endif // #ifdef LOADAIRPORTS_H
    widget = new QWidget(this);
    setCentralWidget(widget);

    int m = 10;
    mainLayout = new QVBoxLayout(this);
    mainLayout->setContentsMargins(m,m,m,m);
    mainLayout->setSpacing(0);
    widget->setLayout(mainLayout);
    //centralWidget()->setLayout(mainLayout);

    menuBar = new QMenuBar(this);

    mainLayout->addWidget(menuBar);

    menuFile = new QMenu(tr("&File"),this);
    exitAct = menuFile->addAction(tr("&Quit"),this,SLOT(on_exit()));
    //connect(exitAct,SIGNAL(triggered()),this,SLOT(on_exit()));
    menuBar->addMenu(menuFile);

    menuTest = new QMenu(tr("&Test"),this);
    testAct5 = menuTest->addAction(tr("&Set Paths..."),this,SLOT(on_test5()));
    testAct = menuTest->addAction(tr("&Move Map..."),this,SLOT(on_test1()));
    testAct2 = menuTest->addAction(tr("&Get File..."),this,SLOT(on_test2()));
    testAct3 = menuTest->addAction(tr("&Get Dir..."),this,SLOT(on_test3()));
    testAct4 = menuTest->addAction(tr("&Native Gets..."),this,SLOT(on_test4()));
    testAct6 = menuTest->addAction(tr("&Load Airports..."),this,SLOT(on_test6()));
    menuBar->addMenu(menuTest);

    menuHelp = new QMenu(tr("&Help"),this);
    helpAct = menuHelp->addAction(tr("&About"),this,SLOT(on_about()));
    abtqtAct = menuHelp->addAction(tr("&About Qt"),this,SLOT(on_about_qt()));
    menuBar->addMenu(menuHelp);

    osmmap = new osmMap(this);
    osmmap->load_map("airport");
    // osmmap->zoom_to_airport("YGIL"); - this FAILS
    // and this??? osmmap->zoom_to_latlon("-31.699", "148.635", 12);

    mainLayout->addWidget(osmmap);

    // == A test results group
    resultsGroup = new QGroupBox(tr("Test Results"),this);
    resultsLayout = new QVBoxLayout(this);
    resultsLayout->setContentsMargins(10, 2, 10, 2);
    resultsLayout->setSpacing(2);


    tmp = settings->value(S_FILENAME, "NewProfile.ini").toString();
    infoLabel = new QLabel("FILE: "+tmp,this);

    tmp = settings->value(S_DIRNAME, util_getCurrentWorkDirectory()).toString();
    infoLabel2 = new QLabel("DIR: "+tmp,this);

    tmp = settings->value(S_ROOT, "").toString();
    infoLabel3 = new QLabel("FG_ROOT: "+tmp,this);
    tmp = settings->value(S_SCENE, "").toString();
    infoLabel4 = new QLabel("Scenery: "+tmp,this);
    infoLabel5 = new QLabel("Load: ",this);

    root_list = settings->value(S_LISTROOT,"").toStringList();
    scene_list = settings->value(S_LISTSCENE,"").toStringList();

    resultsGroup->setLayout(resultsLayout); // set vertical layout
    resultsLayout->addWidget(infoLabel);    // add widgets
    resultsLayout->addWidget(infoLabel2);
    resultsLayout->addWidget(infoLabel3);
    resultsLayout->addWidget(infoLabel4);
    resultsLayout->addWidget(infoLabel5);
    mainLayout->addWidget(resultsGroup);    // add to window


    statusBar = new QStatusBar(this);
    labelTime = new QLabel(tr("00:00:00"),this);
    labelTime->setFrameStyle(QFrame::Panel | QFrame::Raised);
    statusBar->addPermanentWidget(labelTime);
    connect(statusBar,SIGNAL(messageChanged(QString)),this,SLOT(on_message_changed(QString)));

    mainLayout->addWidget(statusBar);

    done_setmap = false;
    timer = new QTimer(this);
    connect(timer, SIGNAL(timeout()), this, SLOT(on_time_out()));
    timer->start(1000);

    testdialog = new testDialog(this);

#ifdef USE_ALLOC_DIALOG
    movedialog = new moveDialog(this);
    connect(movedialog,SIGNAL(set_position(QString,QString,int)),this,SLOT(move_map_position(QString,QString,int)));
#endif // #ifdef USE_ALLOC_DIALOG

    m_time.start(); // start a time counter

}

MainWindow::~MainWindow()
{
#ifdef LOADAIRPORTS_H
    if (loadair)
        delete loadair;
#endif // #ifdef LOADAIRPORTS_H
}

void MainWindow::on_exit()
{
    close();
}

void MainWindow::closeEvent(QCloseEvent *event)
{
    settings->setValue("mainWindowGeometry", saveGeometry());
    outLog(util_getDateTimestg()+" - Application close",0x8001);
    event->accept();

}

void MainWindow::on_about()
{
    QString msg;
    msg = "Version: ";
    msg.append(APP_VERS);
    msg.append(", dated ");
    msg.append(APP_DATE);
    msg.append("\n");
    msg.append("Built: ");
    msg.append(__DATE__);
    msg.append(" at ");
    msg.append(__TIME__);
    msg.append("\n\n");
    msg.append("Qt Test Gui shows various aspects of Qt, and\n");
    msg.append("to load and show an OpenStreetMap map\n");
    msg.append("License: GPL v2 or later\n\n");
    // msg.append("Source: http://geoffair.org/projects/qt_osm_map.htm");
    msg.append("The session log file is [");
    msg.append(util_getLogFile());
    msg.append("]\n");
    QMessageBox::about(this, tr("About Qt OSM Map"), msg);
}

void MainWindow::on_about_qt()
{
    QMessageBox::aboutQt(this, "About Qt");
}

void MainWindow::move_map_position(QString lat, QString lon, int zoom)
{
    QString msg;
    msg.sprintf("%d",zoom);
    msg = "move_map_postion: "+lat+", "+lon+", "+msg;
    osmmap->zoom_to_latlon(lat, lon, zoom);
    osmmap->map_set_coords(lat,lon);
    settings->setValue(S_MAPLAT,lat);
    settings->setValue(S_MAPLON,lon);
    settings->setValue(S_MAPZOOM,zoom);
    outLog(msg);
    setStatusMessage(msg,DEF_TIMEOUT);
}

void MainWindow::on_time_out()
{
    QString msg = util_getTimestg();
    labelTime->setText(msg);
    if (!done_setmap) {
        //int ms = m_time.elapsed();
        //if (ms >= 1000) {
            done_setmap = true;
#ifdef USE_DEFAULT_KSFO
            QString slat = settings->value(S_MAPLAT,KSFO_LAT).toString();
            QString slon = settings->value(S_MAPLON,KSFO_LON).toString();
            int zoom = settings->value(S_MAPZOOM,KSFO_ZOOM).toInt();
#else // !USE_DEFAULT_KSFO = YGIL
            QString slat = settings->value(S_MAPLAT,YGIL_LAT).toString();
            QString slon = settings->value(S_MAPLON,YGIL_LAT).toString();
            int zoom = settings->value(S_MAPZOOM,YGIL_ZOOM).toInt();
#endif // USE_DEFAULT_KSFO y/n
            osmmap->zoom_to_latlon(slat, slon, zoom);
            osmmap->map_set_coords(slat,slon);
        //}
    }
}

void MainWindow::setStatusMessage(QString msg, int timeout)
{
    statusBar->showMessage(msg,timeout);
    outLog("STATUS: "+msg);
}

void MainWindow::on_message_changed(QString msg)
{
    if (msg.isEmpty())
        statusBar->showMessage("Ready");
}

//===========================================================================
//** Data File eg default.ini
//===========================================================================
/** \brief Path to a data file eg get_data_file("default.ini")
 * \return Absolute path to the file
 */
QString MainWindow::get_data_file(QString file_name)
{
    QString storedir = QDir(QDesktopServices::storageLocation(QDesktopServices::DataLocation)).absolutePath();

    // create path is not exist
    if(!QFile::exists(storedir)){
        QDir *dir = new QDir("");
        dir->mkpath(storedir);
    }
    return storedir.append("/").append(file_name);
}



#ifdef USE_ALLOC_DIALOG

void MainWindow::on_test1()
{
    // get a new location
    OSMPOS pos;
    osmmap->get_osm_pos(&pos);  // get OSM map position
    QString msg;
    msg.sprintf("%d",pos.zoom);

    // set current values in dialog
    movedialog->latEd->setText(pos.lat);
    movedialog->lonEd->setText(pos.lon);
    movedialog->zoomEd->setText(msg);
    movedialog->select_table_row();

    movedialog->exec();
}

#else // !#ifdef USE_ALLOC_DIALOG

void MainWindow::on_test1()
{
    // get a new location
    //QDialog *d = new QDialog(this);
    //moveDialog *d = new moveDialog(this);
    moveDialog movedialog(this);
    OSMPOS pos;
    osmmap->get_osm_pos(&pos);  // get OSM map position
    QString msg;
    msg.sprintf("%d",pos.zoom);

    // set default values in dialog
    movedialog.latEd->setText(pos.lat);
    movedialog.lonEd->setText(pos.lon);
    movedialog.zoomEd->setText(msg);
    movedialog.select_table_row(pos.lat,pos.lon); // does nothing if no #ifdef ADD_LIST_WIDGET

    connect(&movedialog,SIGNAL(set_position(QString,QString,int)),this,SLOT(move_map_position(QString,QString,int)));

    movedialog.exec();
    //d->exec();
    //delete d;
}

#endif // #ifdef USE_ALLOC_DIALOG y/n

void MainWindow::on_test2()
{
    QString title = "Select File Name";
    QString key = S_FILENAME;
    QString prev = settings->value(key, "NewProfile.ini").toString();
    QStringList filt;
    filt += "*.ini";
    QString file = util_getFileName( (QWidget *)this,title, prev, filt);
    QString msg = "Got new file ["+file+"]";
    if (file.length()) {
        settings->setValue(key,file);
        infoLabel->setText("FILE: "+file);
    }
    setStatusMessage(msg,DEF_TIMEOUT);

}

void MainWindow::on_test3()
{
    QString title = "Select Existing Directory";
    QString key = S_DIRNAME;
    QString prev = settings->value(key, "").toString();
    //// prev.append("/");   // JUST FOR TESTING
    QString dir = util_getDirName( (QWidget *)this,title, prev);
    QString msg = "Got new dir ["+dir+"]";
    if (dir.length()) {
        settings->setValue(key,dir);
        infoLabel2->setText("DIR: "+dir);
    }
    setStatusMessage(msg,DEF_TIMEOUT);
}

void MainWindow::on_test4()
{
    //testDialog *d = new testDialog(this);
    testdialog->exec();

}

void MainWindow::on_test5()
{
    QString tmp, msg, root, scene;
    outLog("Doing test5 - Get root and scenery paths");
    pathDialog d(this);
    root = settings->value(S_ROOT, "").toString();
    scene = settings->value(S_SCENE, "").toString();
    d.init(root,scene,root_list,scene_list);

    d.exec();
    if (d.exit_ok) {
        //tmp = settings->value(S_ROOT, "").toString();
        tmp = d.get_root();
        msg = "FG_ROOT: "+tmp;
        if (d.root_valid && tmp.length()) {
            if (tmp != root) // is it a NEW root
                settings->setValue(S_ROOT,tmp);
            if (!root_list.contains(tmp)) {
                root_list += tmp;
                settings->setValue(S_LISTROOT,root_list);
                msg.append(" (+list)");
            }
            infoLabel3->setText(msg);
            outLog("test5: "+msg);
        }
        //tmp = settings->value(S_SCENE, "").toString();
        tmp = d.get_scene();
        msg = "Scenery: "+tmp;
        if (d.scene_valid && tmp.length()) {
            if (tmp != scene)   // is it a NEW value
                settings->setValue(S_SCENE,tmp);
            if (!scene_list.contains(tmp)) {
                scene_list += tmp;
                settings->setValue(S_LISTSCENE,scene_list);
                msg.append(" (+list)");
            }
            infoLabel4->setText(msg);
            outLog("test5: "+msg);
        }
    }
}

// load airports
void MainWindow::on_test6()
{
    QTime tm;
    QString msg;
    tm.start();
    int file_count, thresh_count, air_count, run_count;
    file_count = thresh_count = air_count = run_count = 0;
    QString path = settings->value(S_SCENE,"").toString();   // sceneEd->text();
#ifdef LOADAIRPORTS_H
    if (!loadair)
        loadair = new loadAirports(this);
    air_count = loadair->readThreshold(path);
    file_count = loadair->file_count;
    thresh_count = loadair->thresh_count;
    run_count = loadair->run_count;
#else // !#ifdef LOADAIRPORTS_H


    if (util_isValidFGSceneryDir(path)) {
        QString xFileName;
        int idxd, max;
        path.append("/Airports");
        QDir dir(path);
        if (dir.exists()) {

            QDirIterator loopFiles( path, QDirIterator::Subdirectories );
            while (loopFiles.hasNext()) {
                //= Get file handle if there is one
                xFileName = loopFiles.next();
                file_count++;
                //= Check if file entry is a *.threshold.xml - cos this is what we want
                if(xFileName.endsWith(".threshold.xml") ) {

                    thresh_count++;
                    //= Split out "CODE.threshold.xml" with a "."
                    QFileInfo fileInfoThreshold(xFileName);
                    QString airport_code = fileInfoThreshold.fileName().split(".").at(0);
                    //==============================================================
                    // Parse the <CODE>.threshold.xml file to get runways
                    //==============================================================
                    /*
                    <?xml version="1.0"?>
                    <PropertyList>
                      <runway>
                        <threshold>
                          <lon>0.044298885776989</lon>
                          <lat>51.505569223906</lat>
                          <rwy>10</rwy>
                          <hdg-deg>92.88</hdg-deg>
                          <displ-m>95</displ-m>
                          <stopw-m>55</stopw-m>
                        </threshold>
                        <threshold>
                          <lon>0.065996952433288</lon>
                          <lat>51.5048897753222</lat>
                          <rwy>28</rwy>
                          <hdg-deg>272.88</hdg-deg>
                          <displ-m>71</displ-m>
                          <stopw-m>90</stopw-m>
                        </threshold>
                      </runway>
                    </PropertyList>
                    */

                    //* Get the contents of the file
                    // QString threshold_file( airport_dir.append("/").append(airport_code).append(".threshold.xml") );
                    QFile fileXmlThrehsold(xFileName);
                    if ( !fileXmlThrehsold.open(QIODevice::ReadOnly) ) {
                        outLog("WARNING: Failed to open "+xFileName);
                        continue;
                    }

                    //= Make file contents into a string from bytearray
                    QString xmlThresholdString = fileXmlThrehsold.readAll();

                    fileXmlThrehsold.close();   // done with file - close it...

                    if (xmlThresholdString.length() == 0) {
                        outLog("WARNING: Empty file "+xFileName);
                        continue;
                    }
                    air_count++;

                    QString fullpath = fileInfoThreshold.absoluteDir().absolutePath();
                    outLog("Path: "+fullpath);

                    //= Create domDocument - important dont pass string in  QDomConstrucor(string) as ERRORS.. took hours DONT DO IT
                    QDomDocument dom;
                    dom.setContent(xmlThresholdString); //* AFTER dom has been created, then set the content from a string from the file

                    //==================================
                    //= Get <runway> nodes
                    QDomNodeList nodeRunways = dom.elementsByTagName("runway");
                    max = nodeRunways.count();
                    run_count += max;
                    if (max > 0) {
                        for( idxd = 0; idxd < max; idxd++ ) {
                            // loops the <runway> nodes
                            QDomNode nodeRunway = nodeRunways.at(idxd);

                            //= Runway threshold 0
                            QString rw1 = nodeRunway.childNodes().at(0).firstChildElement("rwy").text();
                            QString rw1lat = nodeRunway.childNodes().at(0).firstChildElement("lat").text();
                            QString rw1lon = nodeRunway.childNodes().at(0).firstChildElement("lon").text();
                            QString rw1hdg = nodeRunway.childNodes().at(0).firstChildElement("hdg-deg").text();

                            //= Runway threshold 1
                            QString rw2 = nodeRunway.childNodes().at(1).firstChildElement("rwy").text();
                            QString rw2lat = nodeRunway.childNodes().at(1).firstChildElement("lat").text();
                            QString rw2lon = nodeRunway.childNodes().at(1).firstChildElement("lon").text();
                            QString rw2hdg = nodeRunway.childNodes().at(1).firstChildElement("hdg-deg").text();

                            QString rwyid = rw1+"-"+rw2;

                            msg = airport_code;
                            msg.append(" ");
                            msg.append(rwyid);
                            msg.append(": ");
                            msg.append(rw1+" "+rw1lat+","+rw1lon+" "+rw1hdg);
                            msg.append(" ");
                            msg.append(rw2+" "+rw2lat+","+rw2lon+" "+rw2hdg);
                            outLog(msg);

                        }   // foreach 'runway' node
                    } else {
                        outLog("WARNING: No runway nodes in file "+xFileName);
                    }
                }   // deal with only threshold.xml
            }   // while directory has files
        }   // found 'Airports' directory
    }   // is valid scenery directory
#endif // #ifdef LOADAIRPORTS_H
    msg.sprintf("airportsdata load:  %d (%d) threshold.xml, %d airports, with %d runways.",
                file_count, thresh_count, air_count, run_count);
    msg.append(", in "+getElapTimeStg(tm.elapsed()));
    outLog(msg);
    infoLabel5->setText(msg);

}

// eof - mainwindow.cpp
