/* ==================================================
   qt_osm_map project
   Created: Geoff R. McLane - Sep 2011
   License: GPL2 (or later)
   With special thanks to Yves for FGx, and its map widget
   ================================================== */

#include "moveDialog.h"
#include "osm_map.h"        // for min/max slider range
#include "utilities/utilities.h"    // outLog()
#include <math.h>

typedef struct tagMAPTABLE {
    QString icao;
    QString name;
    QString lat;
    QString lon;
    int zoom;
}MAPTABLE, * PMAPTABLE;

#define MX_ITEMS    19
MAPTABLE mapTable[MX_ITEMS] = {
    { "YGIL", "Gilgandra", YGIL_LAT, YGIL_LON, YGIL_ZOOM },
    { "YTWN", "Tooraweenah", "-31.441689", "148.900009", 14 },
    { "YSDU", "Dubbo"      , "-32.2174865", "148.57727", 14 },
    { "YNRM", "Narromine"  , "-32.2158175", "148.2266455", 14 },
    { "YCBB", "Coonabarabran", "-31.3316275", "149.2673545", 10 },
    { "YWRN", "Warren"       , "-31.7354785", "147.8039035", 14 },
    { "YCNM", "Coonamble"    , "-30.9803715", "148.3759295", 14 },
    { "LFPZ", "Saint Cyr l Ecole", "48.8136535", "2.0665855", 14 },
    { "LFPN", "Toussus Le Noble", "48.751923", "2.106138", 14 },
    { "LFPG", "Paris Charles De Gaulle", "49.0097445", "2.5626195", 12 },
    { "LFPO", "Paris Orly", "48.72698", "2.370448", 12 },
    { "VHSK", "SEK KONG", "22.436592", "114.080399", 14 },
    { "VHHH", "Hong Kong Intl", "22.3089945", "113.91461", 14 },
    { "VHXX", "X CLOSED Kai Tak", "22.314607", "114.204051", 14 },
    { "VMMC", "Macau Intl", "22.148675", "113.591412", 14 },
    { "KSFO", "San Francisco Intl", KSFO_LAT, KSFO_LON, KSFO_ZOOM },
    { "KTEX", "Telluride Rgnl", "37.953758", "-107.908481", 14 },
    { "KDRO", "DURANGO LA PLATA CO", "37.1515170", "-107.7537680", 14 },
    { "PASI", "SITKA ROCKY GUTIERREZ", "57.0471280", "-135.3615980", 14 }
};

moveDialog::moveDialog(QWidget *parent) :
    QDialog(parent)
{
    setWindowTitle("GET LATITUDE, LONGITUDE and ZOOM");

    int m = 10;
    int row = 0;
    QString msg;

    //QVBoxLayout *mainLayout = new QVBoxLayout(this);
    mainLayout = new QGridLayout(this);
    mainLayout->setContentsMargins(m,m,m,m);
    mainLayout->setSpacing(10);
    setLayout(mainLayout);

    infoLab = new QLabel("Enter new lat/lon location",this);
    mainLayout->addWidget(infoLab,row,0,1,4);
    row++;

    latLab = new QLabel("Lat:",this);
    latEd = new QLineEdit(this);
    lonLab = new QLabel("Lon:",this);
    lonEd = new QLineEdit(this);

    mainLayout->addWidget(latLab,row,0,1,1);
    mainLayout->addWidget(latEd,row,1,1,1);
    mainLayout->addWidget(lonLab,row,2,1,1);
    mainLayout->addWidget(lonEd,row,3,1,1);
    row++;

    zoom1Lab = new QLabel("Zoom:",this);
    zoomEd = new QLineEdit(this);
    msg.sprintf("Range %d to %d",MIN_SRNG,MAX_SRNG);
    zoom2Lab = new QLabel(msg,this);
    mainLayout->addWidget(zoom1Lab,row,0,1,1);
    mainLayout->addWidget(zoomEd,row,1,1,1);
    mainLayout->addWidget(zoom2Lab,row,2,1,3);
    row++;

#ifdef ADD_LIST_WIDGET
    //listWidget = new QListWidget(this);
    //listWidget->setBaseSize(200,16*14);
    //listWidget->setSelectionMode(QAbstractItemView::SingleSelection);
    // no, use a 'table'
    listWidget = new QTableWidget(this);
    listWidget->setColumnCount(4);
    QStringList header;
    header << "ICAO" << "Latitude" << "Longitude" << "Name";
    listWidget->setHorizontalHeaderLabels(header);
    listWidget->verticalHeader()->hide();
    listWidget->setAlternatingRowColors(true); // hard to see
    listWidget->setShowGrid(false);
    listWidget->horizontalHeader()->setClickable(false);
    listWidget->setSelectionMode(QAbstractItemView::SingleSelection);
    listWidget->setSelectionBehavior(QAbstractItemView::SelectRows);

    mainLayout->addWidget(listWidget,row,0,24,4);
    row += 24;

#endif // ADD_LIST_WIDGET

    row++;
    cancelBut = new QPushButton(tr("Cancel"),this);
    okButton = new QPushButton(tr("Ok"),this);

    mainLayout->addWidget(cancelBut,row,1,1,2);
    mainLayout->addWidget(okButton, row,3,1,2);

    connect(cancelBut, SIGNAL(clicked()), this, SLOT(on_cancel()));
    connect(okButton,  SIGNAL(clicked()), this, SLOT(on_ok()));

    finalise();
}

void moveDialog::finalise()
{
#ifdef ADD_LIST_WIDGET
    //listWidget->addItem(new QListWidgetItem("KSFO"));
    listWidget->setUpdatesEnabled(false);
    // clear the current list
    while (listWidget->rowCount() != 0) {
        listWidget->removeRow(0);
    }
    // update list
    int i;
    QTableWidgetItem *item;
    QString txt;
    for (i = 0; i < MX_ITEMS; i++) {
        listWidget->insertRow(i);
        txt = mapTable[i].icao;
        item = new QTableWidgetItem(txt,0);
        item->setFlags(Qt::ItemFlags(Qt::ItemIsSelectable | Qt::ItemIsEnabled));
        listWidget->setItem(i,0,item);
        txt = mapTable[i].lat;
        item = new QTableWidgetItem(txt,0);
        item->setFlags(Qt::ItemFlags(Qt::ItemIsSelectable | Qt::ItemIsEnabled));
        listWidget->setItem(i,1,item);
        txt = mapTable[i].lon;
        item = new QTableWidgetItem(txt,0);
        item->setFlags(Qt::ItemFlags(Qt::ItemIsSelectable | Qt::ItemIsEnabled));
        listWidget->setItem(i,2,item);
        txt = mapTable[i].name;
        item = new QTableWidgetItem(txt,0);
        item->setFlags(Qt::ItemFlags(Qt::ItemIsSelectable | Qt::ItemIsEnabled));
        listWidget->setItem(i,3,item);
    }

    listWidget->resizeRowsToContents();

    connect(listWidget,SIGNAL(itemSelectionChanged()),this,SLOT(on_selection_changed()));

    listWidget->setUpdatesEnabled(true);

#endif // #ifdef ADD_LIST_WIDGET
}

void moveDialog::select_table_row(QString clat, QString clon)
{
#ifdef ADD_LIST_WIDGET
    int i;
    double lat = clat.toDouble();
    double lon = clon.toDouble();
    for (i = 0; i < MX_ITEMS; i++) {
        double tlat = mapTable[i].lat.toDouble();
        double tlon = mapTable[i].lon.toDouble();
        if ((mapTable[i].lat == clat)&&
            (mapTable[i].lon == clon)) {
            // strings match - select this row - HOW?
            break;
        } else if ((fabs(tlat - lat) < 0.0001)&&
                   (fabs(tlon - lon) < 0.0001)) {
            break;
        }
    }
    if (i < MX_ITEMS) {
        listWidget->selectRow(i);
        listWidget->setFocus();
    }
#endif // #ifdef ADD_LIST_WIDGET
}

void moveDialog::on_ok()
{
    QString lat = latEd->text().trimmed();
    QString lon = lonEd->text().trimmed();
    int zoom = zoomEd->text().trimmed().toInt();
    QString msg;
    msg.sprintf("%d",zoom);
    if (lat.length() && lon.length()) {
        double dlat = lat.toDouble();
        double dlon = lon.toDouble();
        if ((dlat >= -90.0)&&(dlat <= 90.0)&&(dlon >= -180.0)&&(dlon <= 180.0)) {
            outLog("on_ok: emit set_position: lat="+lat+", lon="+lon+", zoom="+msg);
            emit set_position(lat,lon,zoom);
        } else {
            msg.sprintf("on_ok: lat/lon failed (%f,%f)",dlat,dlon);
            outLog(msg);
        }
    } else {
        outLog("on_ok: pos failed "+lat+", "+lon);
    }
    close();
}

void moveDialog::on_cancel()
{
    outLog("moveDialog::on_cancel()");
    close();
}

#ifdef ADD_LIST_WIDGET
void moveDialog::on_selection_changed()
{
    //= No Selection
    if(listWidget->selectionModel()->hasSelection() == false) {
        return;
    }
    QTableWidgetItem *item = listWidget->currentItem();
    QString txt = item->text();
    int ccol = listWidget->currentColumn();
    int crow = listWidget->currentRow();
    int i = MX_ITEMS;
    if (crow < MX_ITEMS) {
        if (ccol == 0) {
            if (mapTable[crow].icao == txt)
                i = crow;
        } else if (ccol == 1) {
            if (mapTable[crow].lat == txt)
                i = crow;
        } else if (ccol == 2) {
            if (mapTable[crow].lon == txt)
                i = crow;
        }
    }
    if ( !(i < MX_ITEMS) ) {
        // search for text
        for (i = 0; i < MX_ITEMS; i++) {
            if ((mapTable[i].icao == txt)||
                (mapTable[i].lat == txt)||
                (mapTable[i].lon == txt)){
                break;
            }
        }
    }
    if (i < MX_ITEMS) {
        // found selection
        latEd->setText(mapTable[i].lat);
        lonEd->setText(mapTable[i].lon);
        zoomEd->setText(QString::number(mapTable[i].zoom));
    }
}
#endif // #ifdef ADD_LIST_WIDGET

// eof - moveDialog.cpp
