/* ==================================================
   qt_osm_map project
   Created: Geoff R. McLane - Sep 2011
   License: GPL2 (or later)
   With special thanks to Yves for FGx, and its map widget
   ================================================== */

#include "app_config.h"
#include <QApplication>
#include "mainwindow.h"
#include "osm_map.h"
#include "testdialog.h"
#include "moveDialog.h"
#include "pathdialog.h"
#include "searchDialog.h"
#include "getAptDat.h"
#include "utilities/utilities.h"

MainWindow::MainWindow(QWidget *parent)
    : QMainWindow(parent)
{
    QString tmp;
    debug_mode = false;
    airdata = 0;
    done_init = false;
    test7_thread_done = false;
    done_init6 = false;
    test6_thread_done = false;

#ifdef USE_INI_FILE
    tmp = get_data_file(DEF_INI_FILE);
    settings = new QSettings(tmp,QSettings::IniFormat,this);
    tmp = "MainWindow: Using INI file ["+tmp+"]";
#else // !USE_INI_FILE
    settings = new QSettings(this);
    tmp = "MainWindow: Using 'generic' global settings";
#endif // USE_INI_FILE y/b

    util_setStdLogFile();   // create 'standard' LOG file
    outLog(tmp);            // advise type of settings used

    // restore windows size and postion - saved at closeEvent(QCloseEvent*)
    restoreGeometry(settings->value("mainWindowGeometry").toByteArray());

#ifdef LOADAIRPORTS_H
    loadair = 0;
#endif // #ifdef LOADAIRPORTS_H
    widget = new QWidget(this);
    setCentralWidget(widget);

    int m = 10;
    mainLayout = new QVBoxLayout(this);
    mainLayout->setContentsMargins(m,m,m,m);
    mainLayout->setSpacing(0);
    widget->setLayout(mainLayout);
    //centralWidget()->setLayout(mainLayout);

    menuBar = new QMenuBar(this);

    mainLayout->addWidget(menuBar);

    menuFile = new QMenu(tr("&File"),this);
    exitAct = menuFile->addAction(tr("&Quit"),this,SLOT(on_exit()));
    //connect(exitAct,SIGNAL(triggered()),this,SLOT(on_exit()));
    menuBar->addMenu(menuFile);

    menuTest = new QMenu(tr("&Test"),this);
    testAct5 = menuTest->addAction(tr("Set Root &Paths..."),this,SLOT(on_test5())); // &P
    testAct9 = menuTest->addAction(tr("Set D&ata files..."),this,SLOT(on_test9())); // &A
    testAct = menuTest->addAction(tr("&Move Map..."),this,SLOT(on_test1()));        // &M
    testAct2 = menuTest->addAction(tr("Get &File..."),this,SLOT(on_test2()));       // &F
    testAct3 = menuTest->addAction(tr("Get &Dir..."),this,SLOT(on_test3()));        // &D
    testAct4 = menuTest->addAction(tr("&Native Gets..."),this,SLOT(on_test4()));    // &N
    testAct6 = menuTest->addAction(tr("Load &Airports XML"),this,SLOT(on_test6())); // &A
    testAct7 = menuTest->addAction(tr("&Load Apt.dat"),this,SLOT(on_test7()));      // &L
    testAct8 = menuTest->addAction(tr("&Search ICAO..."),this,SLOT(on_test8()));    // &S
    menuBar->addMenu(menuTest);

    menuHelp = new QMenu(tr("&Help"),this);
    helpAct = menuHelp->addAction(tr("&About"),this,SLOT(on_about()));
    abtqtAct = menuHelp->addAction(tr("&About Qt"),this,SLOT(on_about_qt()));
    menuBar->addMenu(menuHelp);

    osmmap = new osmMap(this);
    osmmap->load_map("airport");
    // osmmap->zoom_to_airport("YGIL"); - this FAILS
    // and this??? osmmap->zoom_to_latlon("-31.699", "148.635", 12);

    mainLayout->addWidget(osmmap);

    // == A test results group
    resultsGroup = new QGroupBox(tr("Test Results"),this);
    resultsLayout = new QVBoxLayout(this);
    resultsLayout->setContentsMargins(10, 2, 10, 2);
    resultsLayout->setSpacing(2);


    tmp = settings->value(S_FILENAME, "NewProfile.ini").toString();
    infoLabel = new QLabel("FILE: "+tmp,this);

    tmp = settings->value(S_DIRNAME, util_getCurrentWorkDirectory()).toString();
    infoLabel2 = new QLabel("DIR: "+tmp,this);

    tmp = settings->value(S_FGAPTDAT,"").toString();
    fg_apt_dat_file = tmp;
    infoLabel7 = new QLabel("FG Apt.dat: "+tmp,this);

    tmp = settings->value(S_ROOT, "").toString();
    infoLabel3 = new QLabel("FG_ROOT: "+tmp,this);
    tmp = settings->value(S_SCENE, "").toString();
    infoLabel4 = new QLabel("Scenery: "+tmp,this);
    infoLabel5 = new QLabel("Load airport xml: ",this);
    infoLabel6 = new QLabel("Load Apt.dat: ",this);

    root_list = settings->value(S_LISTROOT,"").toStringList();
    scene_list = settings->value(S_LISTSCENE,"").toStringList();
    aptdat_list = settings->value(S_LISTFGAPTDAT,"").toStringList();
    saveAptDatList(fg_apt_dat_file);    // ensure current is in list

    resultsGroup->setLayout(resultsLayout); // set vertical layout
    resultsLayout->addWidget(infoLabel);    // add widgets
    resultsLayout->addWidget(infoLabel2);
    resultsLayout->addWidget(infoLabel3);
    resultsLayout->addWidget(infoLabel4);
    resultsLayout->addWidget(infoLabel7);
    resultsLayout->addWidget(infoLabel5);
    resultsLayout->addWidget(infoLabel6);
    mainLayout->addWidget(resultsGroup);    // add to window


    statusBar = new QStatusBar(this);
    labelTime = new QLabel(tr("00:00:00"),this);
    labelTime->setFrameStyle(QFrame::Panel | QFrame::Raised);
    statusBar->addPermanentWidget(labelTime);
    connect(statusBar,SIGNAL(messageChanged(QString)),this,SLOT(on_message_changed(QString)));

    mainLayout->addWidget(statusBar);

    done_setmap = false;
    timer = new QTimer(this);
    connect(timer, SIGNAL(timeout()), this, SLOT(on_time_out()));
    timer->start(1000);

    testdialog = new testDialog(this);

#ifdef USE_ALLOC_DIALOG
    movedialog = new moveDialog(this);
    connect(movedialog,SIGNAL(set_position(QString,QString,int)),this,SLOT(move_map_position(QString,QString,int)));
#endif // #ifdef USE_ALLOC_DIALOG

    m_time.start(); // start a time counter

}

MainWindow::~MainWindow()
{
#ifdef LOADAIRPORTS_H
    if (loadair)
        delete loadair;
#endif // #ifdef LOADAIRPORTS_H
#ifdef LOADAPTDAT_H
    if (airdata) {
        airdata->clear_list();
        delete airdata;
    }
#endif // #ifdef LOADAPTDAT_H

}

void MainWindow::on_exit()
{
    close();
}

void MainWindow::closeEvent(QCloseEvent *event)
{
    settings->setValue("mainWindowGeometry", saveGeometry());
    outLog(util_getDateTimestg()+" - Application close",0x8001);
    event->accept();

}

void MainWindow::on_about()
{
    QString msg;
    msg = "Version: ";
    msg.append(APP_VERS);
    msg.append(", dated ");
    msg.append(APP_DATE);
    msg.append("\n");
    msg.append("Built: ");
    msg.append(__DATE__);
    msg.append(" at ");
    msg.append(__TIME__);
    msg.append("\n\n");
    msg.append("Qt Test Gui shows various aspects of Qt, and\n");
    msg.append("to load and show an OpenStreetMap map\n");
    msg.append("License: GPL v2 or later\n\n");
    // msg.append("Source: http://geoffair.org/projects/qt_osm_map.htm");
    msg.append("The session log file is [");
    msg.append(util_getLogFile());
    msg.append("]\n");
    QMessageBox::about(this, tr("About Qt OSM Map"), msg);
}

void MainWindow::on_about_qt()
{
    QMessageBox::aboutQt(this, "About Qt");
}

void MainWindow::move_map_position(QString lat, QString lon, int zoom)
{
    QString msg;
    msg.sprintf("%d",zoom);
    msg = "move_map_postion: "+lat+", "+lon+", "+msg;
    osmmap->zoom_to_latlon(lat, lon, zoom);
    osmmap->map_set_coords(lat,lon);
    settings->setValue(S_MAPLAT,lat);
    settings->setValue(S_MAPLON,lon);
    settings->setValue(S_MAPZOOM,zoom);
    outLog(msg);
    setStatusMessage(msg,DEF_TIMEOUT);
}

void MainWindow::on_time_out()
{
    QString msg = util_getTimestg();
    bool to = false;
    if (airdata && airdata->isThreadInFunction())
        to = true;
    else if (loadair && loadair->isThreadInFunction())
        to = true;
    if (to)
        msg.append("*");
    labelTime->setText(msg);

    // any events scheduled
    if (!done_setmap) {
        //int ms = m_time.elapsed();
        //if (ms >= 1000) {
            done_setmap = true;
            QString slat = settings->value(S_MAPLAT,KSFO_LAT).toString();
            QString slon = settings->value(S_MAPLON,KSFO_LON).toString();
            int zoom = settings->value(S_MAPZOOM,KSFO_ZOOM).toInt();
            osmmap->zoom_to_latlon(slat, slon, zoom);
            osmmap->map_set_coords(slat,slon);
        //}
    }

    if (done_init) {
        if (test7_thread_done) {
            if (!done_init6) {
                done_init6 = true;
                on_test6_thread();  // run the xml load, on a thread
            }
        }
    } else {
        done_init = true;
        on_test7_thread();  // use thread to LOAD apt.dat, if a valid file
    }
}

void MainWindow::setStatusMessage(QString msg, int timeout)
{
    statusBar->showMessage(msg,timeout);
    QString tmp;
    tmp.sprintf(" (to=%d secs)", (int)(timeout / 1000));
    outLog("STATUS: "+msg+tmp);
}

void MainWindow::on_message_changed(QString msg)
{
    if (msg.isEmpty()) {
        QString tmp("Ready");
        bool to = false;
        if (airdata && airdata->isThreadInFunction()) {
            tmp.append(" - apt.dat thread running");
            to = true;
        }
        if (loadair && loadair->isThreadInFunction()) {
            tmp.append(" - scan xml thread running");
            to = true;
        }
        if (to)
            setStatusMessage(tmp,1000); // make it short lived, like 1 sec
        else
            statusBar->showMessage(tmp);
    }
}

//===========================================================================
//** Data File eg default.ini
//===========================================================================
/** \brief Path to a data file eg get_data_file("default.ini")
 * \return Absolute path to the file
 */
QString MainWindow::get_data_file(QString file_name)
{
    QString storedir = QDir(QDesktopServices::storageLocation(QDesktopServices::DataLocation)).absolutePath();

    // create path is not exist
    if(!QFile::exists(storedir)){
        QDir *dir = new QDir("");
        dir->mkpath(storedir);
    }
    return storedir.append("/").append(file_name);
}



#ifdef USE_ALLOC_DIALOG

void MainWindow::on_test1()
{
    // get a new location
    OSMPOS pos;
    osmmap->get_osm_pos(&pos);  // get OSM map position
    QString msg;
    msg.sprintf("%d",pos.zoom);

    movedialog->finalise(); // repopulate with list

    // set current values in dialog
    movedialog->latEd->setText(pos.lat);
    movedialog->lonEd->setText(pos.lon);
    movedialog->zoomEd->setText(msg);
    movedialog->select_table_row(pos.lat,pos.lon);

    movedialog->exec();
}

#else // !#ifdef USE_ALLOC_DIALOG

void MainWindow::on_test1()
{
    // get a new location
    //QDialog *d = new QDialog(this);
    //moveDialog *d = new moveDialog(this);
    moveDialog movedialog(this);
    OSMPOS pos;
    osmmap->get_osm_pos(&pos);  // get OSM map position
    QString msg;
    msg.sprintf("%d",pos.zoom);

    // set default values in dialog
    movedialog.latEd->setText(pos.lat);
    movedialog.lonEd->setText(pos.lon);
    movedialog.zoomEd->setText(msg);
    movedialog.select_table_row(pos.lat,pos.lon); // does nothing if no #ifdef ADD_LIST_WIDGET

    connect(&movedialog,SIGNAL(set_position(QString,QString,int)),this,SLOT(move_map_position(QString,QString,int)));

    movedialog.exec();
    //d->exec();
    //delete d;
}

#endif // #ifdef USE_ALLOC_DIALOG y/n

void MainWindow::on_test2()
{
    QString title = "Select File Name";
    QString key = S_FILENAME;
    QString prev = settings->value(key, "NewProfile.ini").toString();
    QStringList filt;
    filt += "*.ini";
    QString file = util_getFileName( (QWidget *)this,title, prev, filt);
    QString msg = "Got new file ["+file+"]";
    if (file.length()) {
        settings->setValue(key,file);
        infoLabel->setText("FILE: "+file);
    }
    setStatusMessage(msg,DEF_TIMEOUT);

}

void MainWindow::on_test3()
{
    QString title = "Select Existing Directory";
    QString key = S_DIRNAME;
    QString prev = settings->value(key, "").toString();
    //// prev.append("/");   // JUST FOR TESTING
    QString dir = util_getDirName( (QWidget *)this,title, prev);
    QString msg = "Got new dir ["+dir+"]";
    if (dir.length()) {
        settings->setValue(key,dir);
        infoLabel2->setText("DIR: "+dir);
    }
    setStatusMessage(msg,DEF_TIMEOUT);
}

void MainWindow::on_test4()
{
    //testDialog *d = new testDialog(this);
    testdialog->exec();

}

void MainWindow::on_test5()
{
    QString tmp, msg, root, scene;
    outLog("Doing test5 - Get root and scenery paths");
    pathDialog d(this);
    root = settings->value(S_ROOT, "").toString();
    scene = settings->value(S_SCENE, "").toString();
    d.init(root,scene,root_list,scene_list);

    d.exec();
    if (d.exit_ok) {
        //tmp = settings->value(S_ROOT, "").toString();
        tmp = d.get_root();
        msg = "FG_ROOT: "+tmp;
        if (d.root_valid && tmp.length()) {
            if (tmp != root) // is it a NEW root
                settings->setValue(S_ROOT,tmp);
            if (!root_list.contains(tmp)) {
                root_list += tmp;
                settings->setValue(S_LISTROOT,root_list);
                msg.append(" (+list)");
            }
            infoLabel3->setText(msg);
            outLog("test5: "+msg);
        }
        //tmp = settings->value(S_SCENE, "").toString();
        tmp = d.get_scene();
        msg = "Scenery: "+tmp;
        if (d.scene_valid && tmp.length()) {
            if (tmp != scene)   // is it a NEW value
                settings->setValue(S_SCENE,tmp);
            if (!scene_list.contains(tmp)) {
                scene_list += tmp;
                settings->setValue(S_LISTSCENE,scene_list);
                msg.append(" (+list)");
            }
            infoLabel4->setText(msg);
            outLog("test5: "+msg);
        }
    }
}

// load airports using ICAO.threshold.xml files
// testAct6 = menuTest->addAction(tr("Load &Airports XML"),this,SLOT(on_test6())); // &A
void MainWindow::on_test6()
{
    QTime tm;
    QString msg;
    tm.start();
    int file_count, thresh_count, air_count, run_count;
    file_count = thresh_count = air_count = run_count = 0;
    QString path = settings->value(S_SCENE,"").toString();   // sceneEd->text();
    if (!loadair)
        loadair = new loadAirports(this);
    loadair->def_xr_flag = 0;   // xrf_Debug1;
    msg = "Scanning ["+path+"] for ICAO.threshold.xml files...";
    outLog(msg);
    infoLabel5->setText(msg);
    setStatusMessage(msg);
    // update
    qApp->processEvents();  // see if this UPDATES the UI??? - appears to be ok
    // =====================================================
    air_count = loadair->readThreshold(path); // DO THE WORK ON THIS MAIN THREAD
    // =====================================================
    file_count = loadair->file_count;
    thresh_count = loadair->thresh_count;
    run_count = loadair->run_count;
    if (air_count == file_count) {
        msg.sprintf("airports XML load:  %d (%d) threshold.xml, %d airports, with %d runways.",
                file_count, thresh_count, air_count, run_count);
    } else {
        msg.sprintf("airports XML load:  %d files, %d threshold.xml, %d airports, with %d runways.",
                file_count, thresh_count, air_count, run_count);
    }
    msg.append(", in "+getElapTimeStg(tm.elapsed()));
    outLog(msg);
    infoLabel5->setText(msg);
    testAct6->setText("Re-&Load &Apt XML");   // &A
}

void MainWindow::on_test6_thread()
{
    //QTime tm;
    QString msg;
    //tm.start();
    //int file_count, thresh_count, air_count, run_count;
    //file_count = thresh_count = air_count = run_count = 0;
    QString path = settings->value(S_SCENE,"").toString();   // sceneEd->text();
    if (!util_isValidFGSceneryDir(path)) {
        outLog("on_test6_thread: NOT valid scenery directory ["+path+"]");
        return;
    }
    if (!loadair)
        loadair = new loadAirports(this);
    loadair->def_xr_flag = 0;   // clear this flag
    msg = "Scanning ["+path+"] for ICAO.threshold.xml files... on a thread...";
    outLog(msg);
    infoLabel5->setText(msg);
    setStatusMessage(msg);
    connect(loadair,SIGNAL(on_thread_done()),this,SLOT(on_test6_done()));
    loadair->readOnThread(path);
}

// catches emit on_thread_done() in loadAirports
void MainWindow::on_test6_done()
{
    QString msg;
    int air_count = loadair->air_count;
    int file_count = loadair->file_count;
    int thresh_count = loadair->thresh_count;
    int run_count = loadair->run_count;
    int time_ms = loadair->time_ms;
    if (air_count == file_count) {
        msg.sprintf("airports XML load:  %d (%d) threshold.xml, %d airports, with %d runways.",
                file_count, thresh_count, air_count, run_count);
    } else {
        msg.sprintf("airports XML load:  %d files, %d threshold.xml, %d airports, with %d runways.",
                file_count, thresh_count, air_count, run_count);
    }

    msg.append(", in "+getElapTimeStg(time_ms));
    QString path = settings->value(S_SCENE,"").toString();   // sceneEd->text();
    outLog("Scanned ["+path+"] for threshold.xml files, on a thread.");
    outLog(msg);
    infoLabel5->setText(msg);
    setStatusMessage(msg);
    testAct6->setText("Re-&Load &Apt XML");   // &A
}

void MainWindow::on_test7()
{
    bool ok = false;
    QString path = settings->value(S_FGAPTDAT,"").toString();
    QFile f(path);
    if (path.length() && f.exists()) {
        ok = true;
    } else {
        path = settings->value(S_ROOT,"").toString();   // sceneEd->text();
        if (util_isValidFGRootDir(path)) {
            path.append("/Airports/apt.dat.gz");
            QFile file(path);
            if (file.exists()) {
                ok = true;
            }
        }
    }
    if (ok) {
        QString msg("Loading ");
        msg.append(path);
        msg.append("... direct... moment...");
        infoLabel6->setText(msg);
        outLog(msg);
        setStatusMessage(msg);
        // update
        qApp->processEvents();  // see if this UPDATES the UI??? - appears to be ok
        // ===============================================================
        if (airdata) {
            airdata->clear_list(); // clear any previous list
        } else {
            airdata = new loadAptDat;
        }
        // ===============================================================
        if (airdata->loadDirect(path)) {    // DO THE WORK, ON THIS MAIN THREAD
            testAct7->setText("Re-&Load Apt.dat");   // &L
        }
        // ===============================================================
        msg.sprintf("Apt.dat load: %d airports, %d runways, ",
                    airdata->getAirportCount(),
                    airdata->getRunwayCount() );
        msg.append("in "+getElapTimeStg(airdata->getLoadTime()));
        outLog(msg);
        infoLabel6->setText(msg);
#ifdef USE_ALLOC_DIALOG
        if (movedialog)
            movedialog->done_list = false;  // reset to re-populate list
#endif // #ifdef USE_ALLOC_DIALOG
        testAct7->setText("Re-&Load Apt.dat");   // &L
    }
}

void MainWindow::on_test7_thread()
{
    bool ok = false;
    QString path = settings->value(S_FGAPTDAT,"").toString();
    QFile f(path);
    if (path.length() && f.exists()) {
        ok = true;
    } else {
        path = settings->value(S_ROOT,"").toString();   // sceneEd->text();
        if (util_isValidFGRootDir(path)) {
            path.append("/Airports/apt.dat.gz");
            QFile file(path);
            if (file.exists()) {
                ok = true;
                settings->setValue(S_FGAPTDAT,path);
                fg_apt_dat_file = path;
                infoLabel7->setText("FG Apt.dat: "+path);
                saveAptDatList(path);
            }
        }
    }
    if (ok) {
        QString msg("Loading ");
        msg.append(path);
        msg.append("... on a thread...");
        infoLabel6->setText(msg);
        outLog(msg);
        setStatusMessage(msg);
        if (airdata) {
            airdata->clear_list(); // clear any previous list
        } else {
            airdata = new loadAptDat;
        }
        connect(airdata,SIGNAL(load_done()),this,SLOT(on_test7_done()));
        airdata->loadOnThread(path);
    }
}

void MainWindow::on_test7_done()
{
    QString msg;
    int apcnt = airdata->getAirportCount();
    int runcnt = airdata->getRunwayCount();
    msg.sprintf("Apt.dat load: %d airports, %d runways, ",apcnt,runcnt);
    msg.append("in "+getElapTimeStg(airdata->getLoadTime()));
    outLog(msg);
    infoLabel6->setText(msg);
    setStatusMessage(msg);
#ifdef USE_ALLOC_DIALOG
    if (movedialog)
        movedialog->done_list = false;  // reset to re-populate list
#endif // #ifdef USE_ALLOC_DIALOG
    testAct7->setText("Re-&Load Apt.dat");   // &L
    test7_thread_done = true;
}

void MainWindow::on_test8()
{
    searchDialog d(this);
    d.exec();
}

void MainWindow::saveAptDatList(QString file)
{
    QFile f(file);
    if (file.length() && f.exists()) {
        if (!aptdat_list.contains(file)) {
            aptdat_list += file;
            settings->setValue(S_LISTFGAPTDAT,aptdat_list);
        }
    }
}


//     testAct9 = menuTest->addAction(tr("Set D&ata files..."),this,SLOT(543())); // &A
void MainWindow::on_test9()
{
    QString aptdat(fg_apt_dat_file);
    if (aptdat.length() == 0) {
        aptdat = settings->value(S_ROOT, "").toString();
        if (aptdat.length()) {
            aptdat.append("/Airports/apt.dat.gz");
        }
    }
    // QString title(tr("Get Existing apt.dat File"));
    getAptDat d(this);
    d.setup(aptdat,aptdat_list);
    d.exec();
    if (d.got_ok) {
        QString nfile(d.getFile());
        QFile f(nfile);
        if (nfile.length() && f.exists() && (nfile != fg_apt_dat_file)) {
            fg_apt_dat_file = nfile;                    // set NEw file
            infoLabel7->setText("FG Apt.dat: "+nfile);  // SHOW IT
            settings->setValue(S_FGAPTDAT,nfile);       // SAVE IT
            saveAptDatList(nfile);                      // and to LIST
        }
    }
}

// eof - mainwindow.cpp
