// load_nav.cpp

#include "test_config.h"
#include "load_nav.h"
#include "utilities/fgx_gzlib.h"
#include "utilities.h"
#include "cmp_apt_lists.h" // to check if apt ICAO exists in APT list

QString def_fgnav_file(DEF_FGNAV_FILE);
QString def_xpnav_file(DEF_XPNAV_FILE);

void set_fgnav_dat(QString file ) { def_fgnav_file = file; }
QString get_fgnav_dat() { return def_fgnav_file; }
void set_xpnav_dat(QString file ) { def_xpnav_file = file; }
QString get_xpnav_dat() { return def_xpnav_file; }


QString xnav_NDB("2"); // NDB (Non-Directional Beacon) Includes NDB component of Locator Outer Markers (LOM)
QString xnav_VOR("3"); // VOR (including VOR-DME and VORTACs) Includes VORs, VOR-DMEs and VORTACs
QString xnav_ILSLOC("4"); // Localiser component of an ILS (Instrument Landing System)
QString xnav_LDALOC("5"); // Localiser component of a localiser-only approach Includes for LDAs and SDFs
QString xnav_ILSGS("6"); // Glideslope component of an ILS Frequency shown is paired frequency, not the DME channel
QString xnav_OM("7"); // Outer markers (OM) for an ILS Includes outer maker component of LOMs
QString xnav_MM("8"); // Middle markers (MM) for an ILS
QString xnav_IM("9"); // Inner markers (IM) for an ILS
QString xnav_DME("12"); // DME, including the DME component of an ILS, VORTAC or VOR-DME Frequency display suppressed on X-Planes charts
QString xnav_NDBDME("13"); // Stand-alone DME, or the DME component of an NDB-DME Frequency will displayed on X-Planes charts
QString end("99");

typedef struct tagCODE2TYPE {
    int code;
    char * type;
}CODE2TYPE, *PCODE2TYPE;

CODE2TYPE code2type[] = {
    { 2, (char *)"NDB" }, // (Non-Directional Beacon) Includes NDB component of Locator Outer Markers (LOM)
    { 3, (char *)"VOR" }, // (including VOR-DME and VORTACs) Includes VORs, VOR-DMEs and VORTACs
    { 4, (char *)"LOC" }, // Localiser component of an ILS (Instrument Landing System)
    { 5, (char *)"LOC" }, // Localiser component of a localiser-only approach Includes for LDAs and SDFs
    { 6, (char *)"ILSGS" }, // Glideslope component of an ILS Frequency shown is paired frequency, not the DME channel
    { 7, (char *)"OM" }, // Outer markers (OM) for an ILS Includes outer maker component of LOMs
    { 8, (char *)"MM" }, // Middle markers (MM) for an ILS
    { 9, (char *)"IM" }, // Inner markers (IM) for an ILS
    { 12, (char *)"DME" }, // DME, including the DME component of an ILS, VORTAC or VOR-DME Frequency display suppressed on X-Planes charts
    { 13, (char *)"DME" }, // Stand-alone DME, or the DME component of an NDB-DME Frequency will displayed on X-Planes charts
    { 0, 0 }
};

char * get_type_4_code(int code)
{
    PCODE2TYPE pc = &code2type[0];
    while (pc->type) {
        if (pc->code == code)
            return pc->type;
        pc++;
    }
    return (char *)"unknonw code";
}

enum NavOff {
    NV_CODE = 0,
    NV_LAT = 1,
    NV_LON = 2,
    NV_ELEV = 3,
    NV_FREQ = 4,
    NV_RNG = 5,
    NV_VAR = 6,
    NV_ID = 7,
    NV_NAME = 8,
    NV_MIN = 9
};

/* ------------------------
   2 NDB - Non-directional beacon
Row code for an NDB
Latitude of NDB in decimal degrees
Longitude of NDB in decimal degrees
Elevation in feet above MSL
Frequency in KHz
Minimum reception range in nautical miles
Not used for NDBs
NDB identifier
NDB name

   3 VOR - VOR-DMEs and VORTACs
Row code for a VOR
Latitude of VOR in decimal degrees
Longitude of VOR in decimal degrees
Elevation in feet above MSL
Frequency in MHZ (multiplied by 100)
Minimum reception range in nautical miles
Slaved variation for VOR
VOR identifier
VOR name

   4-5 LOC - 4=ILS localizer, 5=stand-alone localizer (inc LOC, LDA & SDF)
Row code for a localizer associated with an ILS
Latitude of localiser in decimal degrees
Longitude of localiser in decimal degrees
Elevation in feet above MSL
Frequency in MHZ (multiplied by 100)
Minimum reception range in nautical miles
Localiser bearing in true degrees
Localiser identifier
Airport ICAO code
+Associated runway number
+Localiser name

   6 GlideSlope - Glideslope associated with an ILS
Row code for a glideslope
Latitude of glideslope aerial in decimal degrees
Longitude of glideslope aerial in decimal degrees
Elevation in feet above MSL
Frequency in MHZ (multiplied by 100)
Minimum reception range in nautical miles
Associated localiser bearing in true degrees prefixed by glideslope angle
Glideslope identifier
Airport ICAO code
+Associated runway number
+Name

   7,8,9 - Marker beacons OM,MM,IM
Row code for a middle marker
Latitude of marker in decimal degrees
Longitude of marker in decimal degrees
Elevation in feet above MSL
Not used
Not used
Associated localiser bearing in true degrees
Not used
+Airport ICAO code
+Associated runway number
+Name

   12, 13 DME Distance Measuring Equipment
Row code for a DME 12=Suppress frequency, 13=display frequency
Latitude of DME in decimal degrees Eight decimal places supported
Longitude of DME in decimal degrees Eight decimal places supported
Elevation in feet above MSL Integer
Frequency in MHZ (multiplied by 100) Integer -MHz multiplied by 100 (eg. 123.45MHz = 12345)
Minimum reception range in nautical miles Integer
DME bias in nautical miles. Default is 0.000
Identifier Up to four characters. Not unique.
Airport ICAO code (for DMEs associated with an ILS) Must be valid ICAO code
Associated runway number (for DMEs associated with an ILS) Up to three characters
DME name (all DMEs) DME-ILS if associated with ILS navaid name for VOR-DMEs, VORTACs &
      NDB-DMEs (eg. SEATTLE VORTAC DME in example data) For standalone DMEs just use DME name

   ------------------------ */
static int ndb_count, vor_count, ilsloc_count, ldaloc_count, ilsgs_count, om_count, mm_count, im_count, dme_count, ndbdme_count, unk_count;

typedef struct tagNAVDAT {
    int code;
    QString id, name, icao, rwy;
    double lat,lon,freq,var;
    int elev_ft, rang_nm;
    int flag;
}NAVDAT, * PNAVDAT;

typedef QList<PNAVDAT> NAV_LIST;
typedef NAV_LIST * PNAV_LIST;

int load_nav_file(QString file, QStringList & sl, PNAV_LIST pnl)
{
    QTime tm;
    QFile f(file);
    if (! f.exists() ) {
        outLog("ERROR: Can NOT locate file ["+file+"]");
        return 1;
    }
    fgx_gzHandle gzf; // opaque file gz handle
    tm.start();
    gzf = fgx_gzOpen(file);
    if (!gzf) {
        outLog("ERROR: Failed to open ["+file+"]");
        return 2;
    }
    ndb_count = vor_count = ilsloc_count = ldaloc_count = ilsgs_count = om_count =  mm_count = im_count = dme_count = ndbdme_count = unk_count = 0;
    outLog("Processing file ["+file+"]");
    //* ignore first line
    fgx_gzReadline(gzf);
    QString credits = fgx_gzReadline(gzf);

    int version = 999;
    if(credits.startsWith("810 Version")) {
        version = 810;
        outLog("Dealing with Version 810");
    } else {
        credits.chop(credits.length() - 12);
        outLog("Dealing with "+credits);
    }
    QString line, row_code;
    QStringList parts;
    QString id, name, icao, rwy;
    int pcnt;
    int line_counter = 2;
    double lat,lon,freq,var;
    int i, elev_ft, rang_nm, ok;
    while ( ! fgx_gzEof(gzf)) {
        line = fgx_gzReadline(gzf);
        line_counter++;
        line = line.trimmed();
        if (line.length() == 0)
            continue;
        row_code = line.section(' ',0, 0);
        parts = line.split(" ", QString::SkipEmptyParts);
        pcnt = parts.size();
        ok = 0;
        if (pcnt < NV_MIN) {
            outLog("WARNING: line ["+line+" skipped!");
            continue;
        }
        sl.append(line);
        // get the standard set
        lat = parts[NV_LAT].toDouble();
        lon = parts[NV_LON].toDouble();
        elev_ft = parts[NV_ELEV].toInt();
        freq = parts[NV_FREQ].toDouble();
        rang_nm = parts[NV_RNG].toInt();
        var = parts[NV_VAR].toDouble();
        id = parts[NV_ID];
        name = parts[NV_NAME];
        icao = "";
        rwy = "";
        if (row_code == xnav_NDB) { // 2 NDB (Non-Directional Beacon) Includes NDB component of Locator Outer Markers (LOM)
            ndb_count++;
            for (i = NV_MIN; i < pcnt; i++)
                name.append(" "+parts[i]);
            ok = 1;
        } else if (row_code == xnav_VOR) { // 3 VOR (including VOR-DME and VORTACs) Includes VORs, VOR-DMEs and VORTACs
            vor_count++;
            for (i = NV_MIN; i < pcnt; i++)
                name.append(" "+parts[i]);
            ok = 1;
        } else if (row_code == xnav_ILSLOC) { // 4 Localiser component of an ILS (Instrument Landing System)
            ilsloc_count++;
            icao = name;
            if (pcnt >= (NV_MIN+2)) {
                rwy = parts[NV_NAME+1];
                name = parts[NV_NAME+2];
                for (i = NV_NAME+3; i < pcnt; i++)
                    name.append(" "+parts[i]);
                ok = 1;
            }
        } else if (row_code == xnav_LDALOC) { // 5 Localiser component of a localiser-only approach Includes for LDAs and SDFs
            ldaloc_count++;
            icao = name;
            if (pcnt >= (NV_MIN+2)) {
                rwy = parts[NV_NAME+1];
                name = parts[NV_NAME+2];
                for (i = NV_NAME+3; i < pcnt; i++)
                    name.append(" "+parts[i]);
                ok = 1;
            }
        } else if (row_code == xnav_ILSGS) { // Glideslope component of an ILS Frequency shown is paired frequency, not the DME channel
            ilsgs_count++;
            icao = name;
            if (pcnt >= (NV_MIN+2)) {
                rwy = parts[NV_NAME+1];
                name = parts[NV_NAME+2];
                for (i = NV_NAME+3; i < pcnt; i++)
                    name.append(" "+parts[i]);
                ok = 1;
            }
        } else if (row_code == xnav_OM) { // Outer markers (OM) for an ILS Includes outer maker component of LOMs
            om_count++;
            icao = name;
            if (pcnt >= (NV_MIN+2)) {
                rwy = parts[NV_NAME+1];
                name = parts[NV_NAME+2];
                for (i = NV_NAME+3; i < pcnt; i++)
                    name.append(" "+parts[i]);
                ok = 1;
            }
        } else if (row_code == xnav_MM) { // Middle markers (MM) for an ILS
            mm_count++;
            icao = name;
            if (pcnt >= (NV_MIN+2)) {
                rwy = parts[NV_NAME+1];
                name = parts[NV_NAME+2];
                for (i = NV_NAME+3; i < pcnt; i++)
                    name.append(" "+parts[i]);
                ok = 1;
            }
        } else if (row_code == xnav_IM) { // Inner markers (IM) for an ILS
            im_count++;
            icao = name;
            if (pcnt >= (NV_MIN+2)) {
                rwy = parts[NV_NAME+1];
                name = parts[NV_NAME+2];
                for (i = NV_NAME+3; i < pcnt; i++)
                    name.append(" "+parts[i]);
                ok = 1;
            }
        } else if (row_code == xnav_DME) { // DME, including the DME component of an ILS, VORTAC or VOR-DME Frequency display suppressed on X-Planes charts
            dme_count++;
            icao = name;
            if (pcnt >= (NV_MIN+2)) {
                rwy = parts[NV_NAME+1];
                name = parts[NV_NAME+2];
                for (i = NV_NAME+3; i < pcnt; i++)
                    name.append(" "+parts[i]);
                ok = 1;
            }
        } else if (row_code == xnav_NDBDME) { // Stand-alone DME, or the DME component of an NDB-DME Frequency will displayed on X-Planes charts
            ndbdme_count++;
            icao = name;
            if (pcnt >= (NV_MIN+2)) {
                rwy = parts[NV_NAME+1];
                name = parts[NV_NAME+2];
                for (i = NV_NAME+3; i < pcnt; i++)
                    name.append(" "+parts[i]);
                ok = 1;
            }
        } else if (row_code == end) {
            break;
        } else {
            unk_count++;
        }
        if (ok) {
            PNAVDAT pnd = new NAVDAT;
            pnd->code = row_code.toInt();
            pnd->id = id;
            pnd->name = name;
            pnd->icao = icao;
            pnd->rwy = rwy;
            pnd->lat = lat;
            pnd->lon = lon;
            pnd->freq = freq;
            pnd->var = var;
            pnd->elev_ft = elev_ft;
            pnd->rang_nm = rang_nm;
            pnl->append(pnd);
        }
    }
    fgx_gzClose(gzf);
    QString msg;
    msg.sprintf("Done %d lines, found %d items, ndb=%d vor=%d ils=%d loc=%d gs=%d om=%d mm=%d im=%d dme=%d ndme=%d unk=%d",
                line_counter,sl.count(),
                ndb_count, vor_count, ilsloc_count, ldaloc_count, ilsgs_count, om_count, mm_count, im_count, dme_count, ndbdme_count, unk_count );
    int ms = tm.elapsed();
    outLog(msg+", in "+getElapTimeStg(ms));
    return 0;
}

QStringList fg_list, xp_list;
NAV_LIST fgnav_list, xpnav_list;

void clear_nav_lists()
{
    fg_list.clear();
    xp_list.clear();
    PNAVDAT pnd;
    while (!fgnav_list.isEmpty()) {
        pnd = fgnav_list.takeFirst();
        delete pnd;
    }
    while (!xpnav_list.isEmpty()) {
        pnd = xpnav_list.takeFirst();
        delete pnd;
    }
}

#define IS_MARKER(c) ((c == 7)||(c == 8)||(c == 9))

QString get_assoc_icao( PNAVDAT pnd )
{
    int code = pnd->code;
    QString icao("");
    if ((code == 2)||(code == 3)) { // is an NBD or VOR
        icao = "";
    } else if (IS_MARKER(code)) { // is marker beacom
        icao = pnd->icao;
    } else if ((code == 4)||(code == 5)||(code == 6)||(code == 12)||(code == 13)) {
        // LOC, GS, DME
        icao = pnd->icao;
    } // else ???
    return icao;
}

bool has_assoc_icao( int code )
{
    bool bret = false;
    if ((code == 2)||(code == 3)) { // is an NBD or VOR
        bret = false;
    } else if (IS_MARKER(code)) { // is marker beacom
        bret = true;
    } else if ((code == 4)||(code == 5)||(code == 6)||(code == 12)||(code == 13)) {
        // LOC, GS, DME
        bret = true;
    } // else ???
    return bret;
}

QString get_nav_string( PNAVDAT pnd, int flag = 0 );

QString get_nav_string( PNAVDAT pnd, int flag )
{
    char * cp = get_type_4_code(pnd->code);
    QString posn;
    QString msg;
    msg = cp;
    msg.append(" ["+pnd->id+"] ["+pnd->name+"] ");
    if (flag) {
        int code = pnd->code;
        if (code == 2) { // is an NBD
            code = (int)pnd->freq;
            posn.sprintf("%d KHz at %d ft", code, pnd->elev_ft);
        } else if ((code == 7)||(code == 8)||(code == 9)) { // is marker beacom
            posn.sprintf("%d ft ", pnd->elev_ft);
            posn.append(pnd->icao+" "+pnd->rwy);
        } else { // VOR, LOC, GS, DME
            posn.sprintf("%.2f at %d ft", (pnd->freq / 100.0), pnd->elev_ft);
            if ((code == 4)||(code == 5)||(code == 6)||(code == 12)||(code == 13)) {
                posn.append(" "+pnd->icao+" "+pnd->rwy);
            }
        }
        msg.append(posn+" ");
    }
    posn.sprintf("%f,%f", pnd->lat, pnd->lon);
    msg.append(posn);
    return msg;
}


void cmp_nav_files(QString file1, QStringList & sl1, PNAV_LIST pnl1,
                   QString file2, QStringList & sl2, PNAV_LIST pnl2)
{
    QString msg,num;
    double min_nav_dist = 10.0;
    double max_nav_dist = 200.0;
    int cnt1 = pnl1->count();
    int cnt2 = pnl2->count();
    msg.sprintf("Comp FG %d, %d lines",cnt1, sl1.count());
    msg.append(", file ["+file1+"]");
    outLog(msg);
    msg.sprintf("With XP %d, %d lines",cnt2, sl2.count());
    msg.append(", file ["+file2+"]");
    outLog(msg);
    int i, j, fnd;
    PNAVDAT pnd1, pnd2;
    int code1, code2;
    QString id1,id2;
    QString name1, name2;
    QString icao1, icao2;
    QString rwy1, rwy2;
    double lat1, lat2;
    double lon1, lon2;
    double freq1, freq2;
    double var1, var2;
    int elev_ft1, elev_ft2;
    int rang_nm1, rang_nm2;
    double dist, min_dist;
    int min_off;
    int not_found1, not_found2;
    outLog("FG NAV items not found in XP");
    not_found1 = 0;
    for (i = 0; i < cnt1; i++) {
        pnd1 = pnl1->at(i);
        pnd1->flag = 0;
    }
    for (j = 0; j < cnt2; j++) {
        pnd2 = pnl2->at(j);
        pnd2->flag = 0;
    }
    for (i = 0; i < cnt1; i++) {
        pnd1 = pnl1->at(i);
        code1 = pnd1->code;
        id1 = pnd1->id;
        name1 = pnd1->name;
        icao1 = pnd1->icao;
        rwy1 = pnd1->rwy;
        lat1 = pnd1->lat;
        lon1 = pnd1->lon;
        freq1 = pnd1->freq;
        var1 = pnd1->var;
        elev_ft1 = pnd1->elev_ft;
        rang_nm1 = pnd1->rang_nm;
        fnd = 0;
        min_dist = 999999.9;
        min_off = -1;
        for (j = 0; j < cnt2; j++) {
            pnd2 = pnl2->at(j);
            if (pnd2->flag)
                continue;
            code2 = pnd2->code;
            if (code1 == code2) {
                // same type of nav aid
                id2 = pnd2->id;
                if (id1 == id2) {
                    // same ID
                    name2 = pnd2->name;
                    icao2 = pnd2->icao;
                    rwy2 = pnd2->rwy;
                    lat2 = pnd2->lat;
                    lon2 = pnd2->lon;
                    freq2 = pnd2->freq;
                    var2 = pnd2->var;
                    elev_ft2 = pnd2->elev_ft;
                    rang_nm2 = pnd2->rang_nm;
                    dist = dist_est_km(lat1, lon1, lat2, lon2);
                    if ((dist < max_nav_dist)&&(dist < min_dist)) {
                        if ((code2 != 7)&&(code2 != 8)&&(code2 != 9)) {
                            min_dist = dist;
                            min_off = j;
                        }
                    }
                    if (dist < min_nav_dist) {
                        fnd = 1;
                        pnd2->flag = i+1;
                        pnd1->flag = j+1;
                        break;
                    }
                }
            }
        }
        if (fnd) {
            // could compare MORE
        } else {
            not_found1++;
            num.sprintf("FG:%d:%d: ", not_found1, i);
            msg = num+get_nav_string(pnd1,1)+" NF in XP";
            outLog(msg);

            if (min_off >= 0) {
                pnd2 = pnl2->at(min_off);
                code2 = pnd2->code;
                if ((code2 != 7)&&(code2 != 8)&&(code2 != 9)) {
                    // not a marker type
                    num.sprintf("At %.1f Km found ",min_dist);
                    msg = num+get_nav_string(pnd2,1);
                    outLog(msg);
                }
            }
        }
    }
    outLog("XP NAV items not found in FG...");
    not_found2 = 0;
    for (j = 0; j < cnt2; j++) {
        pnd2 = pnl2->at(j);
        if (pnd2->flag)
            continue; // aready matched
        code2 = pnd2->code;
        id2 = pnd2->id;
        name2 = pnd2->name;
        icao2 = pnd2->icao;
        rwy2 = pnd2->rwy;
        lat2 = pnd2->lat;
        lon2 = pnd2->lon;
        freq2 = pnd2->freq;
        var2 = pnd2->var;
        elev_ft2 = pnd2->elev_ft;
        rang_nm2 = pnd2->rang_nm;
        fnd = 0;
        min_dist = 999999.9;
        min_off = -1;
        for (i = 0; i < cnt1; i++) {
            pnd1 = pnl1->at(i);
            if (pnd1->flag)
                continue; // already matched
            code1 = pnd1->code;
            if (code1 == code2) {
                // same type of nav aid
                id1 = pnd1->id;
                if (id1 == id2) {
                    // same ID
                    name1 = pnd1->name;
                    icao1 = pnd1->icao;
                    rwy1 = pnd1->rwy;
                    lat1 = pnd1->lat;
                    lon1 = pnd1->lon;
                    freq1 = pnd1->freq;
                    var1 = pnd1->var;
                    elev_ft1 = pnd1->elev_ft;
                    rang_nm1 = pnd1->rang_nm;
                    dist = dist_est_km(lat1, lon1, lat2, lon2);
                    if ((dist < max_nav_dist)&&(dist < min_dist)) {
                        if ((code1 != 7)&&(code1 != 8)&&(code1 != 9)) {
                            min_dist = dist;
                            min_off = i;
                        }
                    }
                    if (dist < min_nav_dist) {
                        fnd = 1;
                        break;
                    }
                }
            }
        }
        if (fnd) {
            // could compare MORE
        } else {
            not_found2++;
            num.sprintf("XP:%d:%d: ", not_found2, j);
            outLog(num+get_nav_string(pnd2,1)+" NF in FG");
            if (min_off >= 0) {
                pnd1 = pnl1->at(min_off);
                code1 = pnd1->code;
                if ((code1 != 7)&&(code1 != 8)&&(code1 != 9)) {
                    // not a marker type
                    num.sprintf("At %.1f Km found ",min_dist);
                    outLog(num+get_nav_string(pnd1,1));
                }
            }
        }
    }
    double pct1 = (double)(cnt1 - not_found1) / (double)cnt1;
    int ipct1 = (int)((pct1 + 0.0005) * 1000.0);
    pct1 = (double)ipct1 / 10.0;
    double pct2 = (double)(cnt2 - not_found2) / (double)cnt2;
    int ipct2 = (int)((pct2 + 0.0005) * 1000.0);
    pct2 = (double)ipct2 / 10.0;
    msg.sprintf("NAV: FG NF %d of %d, %.1f %% SAME, XP NF %d of %d, %.1f %% SAME",
                not_found1, cnt1, pct1,
                not_found2, cnt2, pct2);
    outLog(msg,0x4001);
}


void load_nav_files()
{
    clear_nav_lists();
    QString def_fgnav = get_fgnav_dat();
    QString def_xpnav = get_xpnav_dat();
    load_nav_file(def_fgnav,fg_list,&fgnav_list);
    load_nav_file(def_xpnav,xp_list,&xpnav_list);
    cmp_nav_files(def_fgnav,fg_list,&fgnav_list,
                  def_xpnav,xp_list,&xpnav_list);

}

