// load_fix.cpp

#include "test_config.h"
#include "load_fix.h"
#include "utilities/fgx_gzlib.h"
#include "utilities.h"

QString def_fgfix_file(DEF_FGFIX_FILE);
QString def_xpfix_file(DEF_XPFIX_FILE);

QString get_fgfix_dat() { return def_fgfix_file; }
void set_fgfix_dat(QString file) { def_fgfix_file = file; }
QString get_xpfix_dat() { return def_xpfix_file; }
void set_xpfix_dat(QString file) { def_xpfix_file = file; }

static bool add_nearest_fix = false; // this does NOT seem to offer addition info

enum FixOff {
    FX_LAT = 0, // 37.428522 Latitude of NDB in decimal degrees Eight decimal places supported
    FX_LON = 1, // -097.419194 Longitude of NDB in decimal degrees Eight decimal places supported
    FX_NAME = 2, //  ACESI Name of fix Usually five characters.  Unique within an ICAO region
    FX_SIZE = 3
};

typedef struct tagFIXDAT {
    double lat;
    double lon;
    QString name;
    int flag;
}FIXDAT, * PFIXDAT;

typedef QList<PFIXDAT> FIX_LIST;
typedef FIX_LIST * PFIX_LIST;

FIX_LIST fgfix_list;
FIX_LIST xpfix_list;
QStringList fgfix_lines, xpfix_lines;

void clear_fix_lists()
{
    fgfix_lines.clear();
    xpfix_lines.clear();
    PFIXDAT pfd;
    while (!fgfix_list.isEmpty()) {
        pfd = fgfix_list.takeFirst();
        delete pfd;
    }
    while (!xpfix_list.isEmpty()) {
        pfd = xpfix_list.takeFirst();
        delete pfd;
    }
}

int load_fix_file(QString file, QStringList & sl, PFIX_LIST pfl)
{
    QTime tm;
    QFile f(file);
    if (! f.exists() ) {
        outLog("ERROR: Can NOT locate file ["+file+"]");
        return 1;
    }
    fgx_gzHandle gzf; // opaque file gz handle
    tm.start();
    gzf = fgx_gzOpen(file);
    if (!gzf) {
        outLog("ERROR: Failed to open ["+file+"]");
        return 2;
    }
    outLog("Processing file ["+file+"]");

    QString end("99");
    QString name;
    //* ignore first line
    fgx_gzReadline(gzf);
    QString credits = fgx_gzReadline(gzf);

    int version = 999;
    if(credits.startsWith("600 Version")) {
        version = 600;
        outLog("Dealing with Version 600");
    } else {
        credits.chop(credits.length() - 12);
        outLog("Dealing with "+credits);
    }

    QString line;
    QStringList parts;
    int line_counter = 2;
    int pcnt;
    while ( ! fgx_gzEof(gzf) ) {
        line = fgx_gzReadline(gzf);
        line_counter++;
        line = line.trimmed();
        if (line.length() == 0)
            continue;
        parts = line.split(" ", QString::SkipEmptyParts);
        pcnt = parts.size();
        name = parts[FX_LAT]; // 99 - end of file
        if (name == end)
            break;  // last line is "99"
        if (pcnt < FX_SIZE) {
            outLog("Discarding line ["+line+"]");
            continue;
        }
        PFIXDAT pfd = new FIXDAT;
        pfd->lat = name.toDouble(); // // 37.428522 Latitude of NDB in decimal degrees Eight decimal places supported
        pfd->lon = parts[FX_LON].toDouble(); // -097.419194 Longitude of NDB in decimal degrees Eight decimal places supported
        pfd->name = parts[FX_NAME]; //  ACESI Name of fix Usually five characters.  Unique within an ICAO region
        // FX_SIZE = 10
        pfl->append(pfd);
        sl.append(line);
    }
    fgx_gzClose(gzf);
    QString msg;
    msg.sprintf("Done %d lines, found %d items...", line_counter,sl.count());
    int ms = tm.elapsed();
    outLog(msg+", in "+getElapTimeStg(ms));

    return 0;
}

QString get_fix_string( PFIXDAT pfd )
{
    QString msg;
    msg.sprintf("%f %f ", pfd->lat, pfd->lon);
    msg.append(pfd->name);
    return msg;
}

typedef struct tagSAMENAME {
    QString name;
    int i, j;
    double dist;
    double lon, lat;
}SAMENAME, * PSAMENAME;

typedef QList<PSAMENAME> SAME_LIST;
typedef SAME_LIST * PSAME_LIST;

void clear_same_list( PSAME_LIST psl )
{
    PSAMENAME psn;
    while (! psl->isEmpty() ) {
        psn = psl->takeFirst();
        delete psn;
    }
}

void cmp_fix_files(QString file1, QStringList & sl1, PFIX_LIST pfl1,
                   QString file2, QStringList & sl2, PFIX_LIST pfl2)
{
    QString msg, tmp;
    SAME_LIST same;
    int cnt1 = pfl1->count();
    int cnt2 = pfl2->count();
    msg.sprintf("Comp FG %d, %d lines",cnt1, sl1.count());
    msg.append(", file ["+file1+"]");
    outLog(msg);
    msg.sprintf("With XP %d, %d lines",cnt2, sl2.count());
    msg.append(", file ["+file2+"]");
    outLog(msg);
    int i, j, fnd, s, scnt;
    QString name1, name2;
    PFIXDAT pfd1, pfd2;
    PSAMENAME psn;
    double lat1, lon1;
    double lat2, lon2;
    double min_fix_dist = 200.0; // Km
    int not_found1 = 0;
    int not_found2 = 0;
    double dist, min_dist;
    int min_off;
    for (i = 0; i < cnt1; i++) {
        pfd1 = pfl1->at(i);
        pfd1->flag = 0;
    }
    for (j = 0; j < cnt2; j++) {
        pfd2 = pfl2->at(j);
        pfd2->flag = 0;
    }
    outLog("Find FG fixes in XP...");
    for (i = 0; i < cnt1; i++) {
        pfd1 = pfl1->at(i);
        lat1 = pfd1->lat;
        lon1 = pfd1->lon;
        name1 = pfd1->name;
        fnd = 0;
        min_off = -1;
        min_dist = 999999999.9;
        clear_same_list( & same );
        for (j = 0; j < cnt2; j++) {
            pfd2 = pfl2->at(j);
            if (pfd2->flag)
                continue; // already matched
            name2 = pfd2->name;
            lat2 = pfd2->lat;
            lon2 = pfd2->lon;
            dist = dist_est_km(lat1, lon1, lat2, lon2);
            if (name1 == name2) {
                if (dist < min_fix_dist) {
                    fnd = 1;
                    pfd1->flag = j + 1;
                    pfd2->flag = i + 1;
                    break;
                } else {
                    psn = new SAMENAME;
                    psn->name = name1;
                    psn->i = i;
                    psn->j = j;
                    psn->dist = dist;
                    psn->lat = lat2;
                    psn->lon = lon2;
                    same.append(psn);
                }
            }
            if (dist < min_dist) {
                min_dist = dist;
                min_off = j;
            }
        }
        if (fnd) {
            // could check MORE
        } else {
            not_found1++;
            msg.sprintf("FG:%d:%d: ", not_found1, (i+1));
            msg.append(get_fix_string(pfd1)+" NF in XP");
            scnt = same.count();
            if (scnt) {
                tmp.sprintf(" (%d same name)", scnt);
                msg.append(tmp);
                for (s = 0; s < scnt; s++) {
                    psn = same.at(s);
                    tmp.sprintf(" %f %f at %.1f Km",
                                psn->lat, psn->lon, psn->dist);
                    msg.append(tmp);
                }
            }
            outLog(msg);
            if ( add_nearest_fix && (min_off >= 0) ) {
                tmp.sprintf(" at %.1f Km", min_dist);
                pfd2 = pfl2->at(min_off);
                msg = "Nearest XP "+get_fix_string(pfd2)+"! "+tmp;
                outLog(msg);
            }
        }
    }
    outLog("Find XP fixes in FG...");
    for (j = 0; j < cnt2; j++) {
        pfd2 = pfl2->at(j);
        if (pfd2->flag)
            continue; // already matched
        name2 = pfd2->name;
        lat2 = pfd2->lat;
        lon2 = pfd2->lon;
        fnd = 0;
        min_off = -1;
        min_dist = 999999999.9;
        clear_same_list( & same );
        for (i = 0; i < cnt1; i++) {
            pfd1 = pfl1->at(i);
            if (pfd1->flag)
                continue; // already matched
            lat1 = pfd1->lat;
            lon1 = pfd1->lon;
            name1 = pfd1->name;
            dist = dist_est_km(lat1, lon1, lat2, lon2);
            if (name1 == name2) {
                if (dist < min_fix_dist) {
                    fnd = 1;
                    pfd1->flag = j + 1;
                    pfd2->flag = i + 1;
                    break;
                } else {
                    psn = new SAMENAME;
                    psn->name = name1;
                    psn->i = i;
                    psn->j = j;
                    psn->dist = dist;
                    psn->lat = lat1;
                    psn->lon = lon1;
                    same.append(psn);
                }
            }
            if (dist < min_dist) {
                min_dist = dist;
                min_off = i;
            }
        }
        if (fnd) {
            // could check MORE
        } else {
            not_found2++;
            msg.sprintf("XP:%d:%d: ", not_found2, (j+1));
            msg.append(get_fix_string(pfd2)+" NF in FG");
            scnt = same.count();
            if (scnt) {
                tmp.sprintf(" (%d same name)", scnt);
                msg.append(tmp);
                for (s = 0; s < scnt; s++) {
                    psn = same.at(s);
                    tmp.sprintf(" %f %f at %.1f Km",
                                psn->lat, psn->lon, psn->dist);
                    msg.append(tmp);
                }
            }
            outLog(msg);
            if ( add_nearest_fix && (min_off >= 0) ) {
                tmp.sprintf(" at %.1f Km", min_dist);
                pfd1 = pfl1->at(min_off);
                msg = "Nearest FG "+get_fix_string(pfd1)+"! "+tmp;
                outLog(msg);
            }
        }
    }
    double pct1 = (double)(cnt1 - not_found1) / (double)cnt1;
    int ipct1 = (int)((pct1 + 0.0005) * 1000.0);
    pct1 = (double)ipct1 / 10.0;
    double pct2 = (double)(cnt2 - not_found2) / (double)cnt2;
    int ipct2 = (int)((pct2 + 0.0005) * 1000.0);
    pct2 = (double)ipct2 / 10.0;
    msg.sprintf("FIX: FG NF %d of %d, %.1f %% SAME, XP NF %d of %d, %.1f %% SAME",
                not_found1, cnt1, pct1,
                not_found2, cnt2, pct2);
    outLog(msg,0x4001);
    clear_same_list( & same );
}

void load_fix_files()
{
    clear_fix_lists();
    QString def_fgfix = get_fgfix_dat();
    QString def_xpfix = get_xpfix_dat();
    load_fix_file(def_fgfix,fgfix_lines,&fgfix_list);
    load_fix_file(def_xpfix,xpfix_lines,&xpfix_list);
    cmp_fix_files(def_fgfix,fgfix_lines,&fgfix_list,
                  def_xpfix,xpfix_lines,&xpfix_list);
}


// eof - load_fix.cpp
