/* ==================================================
   qt_test_gui project
   Created: Geoff R. McLane - Aug 2011
   License: GPL2 (or later)
   ================================================== */
// test_apt850.cpp

// load the latest apt.dat from X-Plane

#include "test_config.h"
#include "test_apt850.h"
#include "utilities.h"
#include "utilities/simgear/simgear.h"
#include "test_aptdat.h"

QString def_xpapt_file = DEF_APT850_FILE;
void set_xpapt_dat(QString file ) { def_xpapt_file = file; };
QString get_xpapt_dat() { return def_xpapt_file; };

/* -------------------------------------
1 Land airport header
16 Seaplane base header
17 Heliport header
  1    - Row code for an airport, seaplane base or heliport = 1, 16 or 17
  21 - Elevation of airport in feet above mean sea level (AMSL)
  1 - Control tower
  0 - Depreciated
  KBFI - Airport ICAO code. If no ICAO code exists, use FAA code (USA only) - Max 4 chars - unique.
  Boeing Field King Co  - Airport name. (up to 40 characters)

100 Runway
  100 - Row code for a land runway (the most common) 100
  29.87 - Width of runway in metres 2 decimal places. Must be >= 1.00
  1 - Code defining the surface type (concrete, asphalt, etc) Integer
  0  - Code defining a runway shoulder surface type 0=none, 1=asphalt, 2=concrete
  0.15 - Runway smoothness (not used by X-Plane yet) 0.00 (smooth) to 1.00 (very rough). Default is 0.25
  0  - Runway centre-line lights 0=no centerline lights, 1=centre line lights
  2  - Runway edge lighting (also implies threshold lights) 0=no edge lights, 2=medium intensity edge lights
  1 - Auto-generate distance-remaining signs (turn off if created manually) 0=no auto signs, 1=auto-generate signs
  The following fields are repeated for each end of the runway
  13L - Runway number (eg. 31R, 02). Leading zeros are required. 2-3 chars. suffixes: L, R or C (or blank)
  47.53801700 - Latitude of runway end (on runway centerline) in decimal degrees 8 decimal places supported
  -122.30746100 - Longitude of runway end (on runway centerline) in decimal degrees 8 decimal places supported
  73.15 - Length of displaced threshold in metres (this is included in runway length) be inside (between) ends
  0.00 - Length of overrun/blast-pad in metres (not included in runway length)
  2 - Code for runway markings (Visual, non-precision, precision) Integer
  0 - Code for approach lighting for this runway end Integer
  0 - Flag for runway touchdown zone (TDZ) lighting 0=no TDZ lighting, 1=TDZ lighting
  1 - Code for Runway End Identifier Lights (REIL) 0=no REIL, 1=omni-directional REIL, 2=unidirectional REIL

 101 Water runway
 102 Helipad
 110 Pavement (taxiway or ramp) header Must form a closed loop
 120 Linear feature (painted line or light string) header Can form closed loop or simple string
 130 Airport boundary header Must form a closed loop
 111 Node All nodes can also include a style (line or lights)
 112 Node with Bezier control point Bezier control points define smooth curves
 113 Node with implicit close of loop Implied join to first node in chain
 114 Node with Bezier control point, with implicit close of loop Implied join to first node in chain
 115 Node terminating a string (no close loop) No styles used
 116 Node with Bezier control point, terminating a string (no close loop) No styles used
 14 Airport viewpoint One or none for each airport
 15 Aeroplane startup location Zero, one or many for each airport
 18 Airport light beacon One or none for each airport
 19 Windsock Zero, one or many for each airport
 20 Taxiway sign (inc. runway distance-remaining signs) Zero, one or many for each airport
 21 Lighting object (VASI, PAPI, Wig-Wag, etc.) Zero, one or many for each airport
 50  56 Communication frequencies Zero, one or many for each airport

   ------------------------------------- */

enum Apt850 {
    A85_CODE = 0,
    A85_ELEV = 1, // feet AMSL
    A85_TWR = 2,  // Control tower
    A85_DEP = 3,  // Depreciated
    A85_ICAO = 4, // ICAO (or FAA code USA)
    A85_NAME = 5, // Begin of NAME
    A85_MIN = 6   // min of 6 parts
};

enum Rwy850 {
    R85_CODE = 0, // 100 - Row code for a land runway (the most common) 100
    R85_WIDTH = 1, //29.87 - Width of runway in metres 2 decimal places. Must be >= 1.00
    R85_SURF = 2, // 1 - Code defining the surface type (concrete, asphalt, etc) Integer
    R85_SHDR = 3, // 0  - Code defining a runway shoulder surface type 0=none, 1=asphalt, 2=concrete
    R85_SMTH = 4, // 0.15 - Runway smoothness (not used by X-Plane yet) 0.00 (smooth) to 1.00 (very rough). Default is 0.25
    R85_CLTS = 5, // 0  - Runway centre-line lights 0=no centerline lights, 1=centre line lights
    R85_ELTS = 6, // 2  - Runway edge lighting (also implies threshold lights) 0=no edge lights, 2=medium intensity edge lights
    R85_SIGN = 7, // 1 - Auto-generate distance-remaining signs (turn off if created manually) 0=no auto signs, 1=auto-generate signs
    // The following fields are repeated for each end of the runway
    R85_NUM1 = 8, // 13L - Runway number (eg. 31R, 02). Leading zeros are required. 2-3 chars. suffixes: L, R or C (or blank)
    R85_LAT1 = 9, // 47.53801700 - Latitude of runway end (on runway centerline) in decimal degrees 8 decimal places supported
    R85_LON1 = 10, // -122.30746100 - Longitude of runway end (on runway centerline) in decimal degrees 8 decimal places supported
    R85_DIS1 = 11, // 73.15 - Length of displaced threshold in metres (this is included in runway length) be inside (between) ends
    R85_OVR1 = 12, // 0.00 - Length of overrun/blast-pad in metres (not included in runway length)
    R85_MRK1 = 13, // 2 - Code for runway markings (Visual, non-precision, precision) Integer
    R85_ALTS1 = 14, // 0 - Code for approach lighting for this runway end Integer
    R85_TDZ1 = 15, // 0 - Flag for runway touchdown zone (TDZ) lighting 0=no TDZ lighting, 1=TDZ lighting
    R85_REIL1 = 16, // 1 - Code for Runway End Identifier Lights (REIL) 0=no REIL, 1=omni-directional REIL, 2=unidirectional REIL
    R85_NUM2 = 17, // 13L - Runway number (eg. 31R, 02). Leading zeros are required. 2-3 chars. suffixes: L, R or C (or blank)
    R85_LAT2 = 18, // 47.53801700 - Latitude of runway end (on runway centerline) in decimal degrees 8 decimal places supported
    R85_LON2 = 19, // -122.30746100 - Longitude of runway end (on runway centerline) in decimal degrees 8 decimal places supported
    R85_DIS2 = 20, // 73.15 - Length of displaced threshold in metres (this is included in runway length) be inside (between) ends
    R85_OVR2 = 21, // 0.00 - Length of overrun/blast-pad in metres (not included in runway length)
    R85_MRK2 = 22, // 2 - Code for runway markings (Visual, non-precision, precision) Integer
    R85_ALTS2 = 23, // 0 - Code for approach lighting for this runway end Integer
    R85_TDZ2 = 24, // 0 - Flag for runway touchdown zone (TDZ) lighting 0=no TDZ lighting, 1=TDZ lighting
    R85_REIL2 = 25, // 1 - Code for Runway End Identifier Lights (REIL) 0=no REIL, 1=omni-directional REIL, 2=unidirectional REIL
    R85_SIZE = 26   // size of runway space split
};

APT_LIST apt850_list;  // final AIRPORT list

PAPT_LIST get_apt850_list()
{
    return &apt850_list;
}


void clear_apt850_list()
{
    PAIRPORT2 pa2;
    while (!apt850_list.isEmpty()) {
        pa2 = apt850_list.takeFirst();
        while (!pa2->rways.isEmpty())
            delete pa2->rways.takeFirst();
        delete pa2;
    }
}

bool set_distances3( QString icao, PAPT_LIST pal, PAIRPORT2 *ppa )
{
    bool success = false;
    QTime tm;
    int i, found;
    int ind;
    double lat, lon, dist, min_dist;
    PAIRPORT2 pfnd, pa;
    QString msg;
    int max = pal->count();
    msg.sprintf("List of %d airports, ",max);
    msg.append("centering on "+icao);
    outLog(msg);
    tm.start();
    min_dist = 99999.0;
    ind = -1;
    *ppa = 0;
    for (i = 0; i < max; i++) {
        //pa = airportList.at(i);
        pa = pal->at(i);
        if (pa->icao == icao)
            break;
    }
    if ( i < max ) {
        found = i;
        //pfnd = airportList.at(found);
        pfnd = pal->at(found);
        lat = pfnd->lat;
        lon = pfnd->lon;
        for (i = 0; i < max; i++) {
            //pa = airportList.at(i);
            pa = pal->at(i);
            if (i == found) {
                pa->dist = 0.0;
                pa->rank = 0;
            } else {
                dist = dist_est_km(lat, lon, pa->lat, pa->lon);
                if (dist < 1.0)
                    dist = 1.0;
                pa->dist = dist;
                if (dist < min_dist) {
                    min_dist = dist;
                    ind = i;
                }
            }
        }
        success = true;
    }
    if (ind != -1)
        *ppa = pal->at(ind);
        //*ppa = airportList.at(ind);
    msg = getElapTimeStg(tm.elapsed());
    if (success)
        outLog("Airport sort done in "+msg);
    else
        outLog("FAILED, in "+msg);
    return success;
}

bool get_next_distance3( double mdist, PAPT_LIST pal, PAIRPORT2 *ppa )
{
    bool success = false;
    int i;
    int ind;
    double dist, min_dist;
    PAIRPORT2 pa;
    int max = pal->count();
    min_dist = 99999.0;
    ind = -1;
    *ppa = 0;
    for (i = 0; i < max; i++) {
        pa = pal->at(i);
        dist = pa->dist;
        if (dist > mdist) {
            if (dist < min_dist) {
                min_dist = dist;
                ind = i;
            }
        }
    }
    if (ind != -1) {
        *ppa = pal->at(ind);
        success = true;
    }
    return success;
}


void do_apt850_sort()
{
    PAIRPORT2 pa2;
    QString icao, name, path, line, posn;
    double lat, lon, dist;
    if (set_distances3("YGIL", &apt850_list, &pa2)) {
        icao = pa2->icao;
        lat = pa2->lat;
        lon = pa2->lon;
        name = pa2->name;
        path = pa2->path;
        dist = pa2->dist;
        posn.sprintf("%.2f Km, at %f,%f", dist, lat, lon);
        line = "nearest "+icao+" "+posn+" "+name;
        outLog(line);
        int i = 0;
        while ( (i < 5) && get_next_distance3(dist, &apt850_list, &pa2)) {
            i++;
            dist = pa2->dist;
            lat = pa2->lat;
            lon = pa2->lon;
            icao = pa2->icao;
            name = pa2->name;
            posn.sprintf("%.2f Km, at %f,%f", dist, lat, lon);
            line.sprintf("next %d: ", i);
            line.append(icao+" "+posn+" "+name);
            outLog(line);
        }
    }
}

static RWY_LIST _s_rwy_list;  // temporary RUNWAY list
void load_apt850( void )
{
    QTime tm;
    QString def_file = get_xpapt_dat();
    QFile f(def_file);
    if (!f.exists()) {
        outLog("ERROR: Failed to find ["+def_file+"! NO AIRPORT FILE DATA!");
        return;
    }
    tm.start();
    if (!f.open(QIODevice::ReadOnly)) {
        outLog("ERROR: Failed to open ["+def_file+"! NO AIRPORT FILE DATA!");
        return;
    }
    outLog("Processing file ["+def_file+"]");
    QString line, row_code, elevation, msg;
    QStringList parts;
    int pcnt, elev_ft, tower, bldgs, res, p;
    QString sp(" ");
    QString apt("1");
    QString rwy("100");
    QString one("1");
    QString end("99");
    QString airport_code, airport_name, rwy_number;
    double lat, lon, lat1, lon1, lat2, lon2;
    double az1, az2, s;
    int line_counter, rwy_cnt, got_apt, rwy_count, air_count, ms;
    double sum_lat, sum_lon;
    QRegExp rxICAOAirport("[A-Z]{4}");

    line_counter = 2; // done 2 lines
    rwy_cnt = 0;
    got_apt = 0;
    elev_ft = 0;
    rwy_count = 0;
    air_count = 0;
    sum_lat = 0.0;
    sum_lon = 0.0;
    _s_rwy_list.clear();
    clear_apt850_list();

    //* ignore first line
    f.readLine();
    QString credits = f.readLine();

    int version = 999;
    if(credits.startsWith("850 Version")) {
        version = 850;
        outLog("Dealing with Version 850");
    } else {
        credits.chop(credits.length() - 12);
        outLog("ERROR: Dealing with UNKNOWN version "+credits);
        f.close();
        return;
    }

    while (!f.atEnd()) {
        line = f.readLine();
        line = line.trimmed();
        line_counter++;
        row_code = line.section(' ',0, 0);
        parts = line.split(" ", QString::SkipEmptyParts);
        pcnt = parts.size();
        if (row_code == apt) {
            if (rwy_cnt && got_apt) {
                // get 'average' center of last airport
                lat = sum_lat / (double)rwy_cnt;
                lon = sum_lon / (double)rwy_cnt;
                PAIRPORT2 pa2 = new AIRPORT2;
                pa2->icao = airport_code;
                pa2->name = airport_name;
                pa2->elev = elev_ft;
                pa2->lat  = lat;
                pa2->lon  = lon;
                while (!_s_rwy_list.isEmpty())
                    pa2->rways.append(_s_rwy_list.takeFirst());
                apt850_list.append(pa2);
            }

            got_apt = 0;
            if (pcnt < A85_MIN) {
                outLog("BAD AIRPORT LINE ["+line+"]");
                continue;
            }
            elevation = parts[A85_ELEV];
            elev_ft = elevation.toInt();
            tower = (parts[A85_TWR] == one) ? 1 : 0;
            bldgs = (parts[A85_DEP] == one) ? 1 : 0;
            airport_code = parts[A85_ICAO];
            airport_name = parts[A85_NAME];
            for(p = A85_MIN; p < pcnt; p++){
                airport_name.append(sp+parts[p]);
            }
            airport_name = airport_name.trimmed();
            if (import_icao_only) {
                if( rxICAOAirport.exactMatch(airport_code) ) {
                    // airports[airport_code] = airport_name;
                    air_count++;
                    got_apt = 1;
                }
            } else {
                // airports[airport_code] = airport_name;
                air_count++;
                got_apt = 1;
            } /* if(is_icao) */

            // done airport init - restart runway stuff
            sum_lat = 0.0;
            sum_lon = 0.0;
            rwy_cnt = 0;
            // clear the runway list
            while (!_s_rwy_list.isEmpty())
                 delete _s_rwy_list.takeFirst();

        } else if (row_code == rwy) {
            if (pcnt != R85_SIZE) {
                outLog("BAD RUNWAY LINE ["+line+"]");
                continue;
            }
            if ( !got_apt )
                continue;   // no airport, so no runways !!!

            rwy_number = parts[R85_NUM1];
            lat1 = parts[R85_LAT1].toDouble();
            lat2 = parts[R85_LAT2].toDouble();
            lon1 = parts[R85_LON1].toDouble();
            lon2 = parts[R85_LON2].toDouble();
            res = _geo_inverse_wgs_84( lat1, lon1, lat2,
                        lon2, &az1, &az2, &s );
            lat = (lat1 + lat2) / 2.0;
            lon = (lon1 + lon2) / 2.0;
            PRUNWAY2 pr2 = new RUNWAY2;
            pr2->heading = az1; // **CHECK** maybe should be az2????
            pr2->lat = lat;
            pr2->lon = lon;
            pr2->len_ft = s * SG_METER_TO_FEET;
            pr2->rwy_no = rwy_number;
            _s_rwy_list.append(pr2);
            rwy_cnt++;
            // accumulate
            sum_lat += lat;
            sum_lon += lon;
            rwy_count++;

        } else if (row_code == end) {
            break;
        }
    }
    if (rwy_cnt && got_apt) {
        // get 'average' center of last airport
        lat = sum_lat / (double)rwy_cnt;
        lon = sum_lon / (double)rwy_cnt;
        PAIRPORT2 pa2 = new AIRPORT2;
        pa2->icao = airport_code;
        pa2->name = airport_name;
        pa2->elev = elev_ft;
        pa2->lat  = lat;
        pa2->lon  = lon;
        while (!_s_rwy_list.isEmpty())
            pa2->rways.append(_s_rwy_list.takeFirst());
        apt850_list.append(pa2);
    }

    f.close();

    msg.sprintf("Done %d lines, found %d airports (%d), %d runways",
                line_counter,air_count,
                apt850_list.count(),
                rwy_count);
    ms = tm.elapsed();
    outLog(msg+", in "+getElapTimeStg(ms));

    return;
}

// eof - test_apt850.cpp
