/* ==================================================
   qt_test_gui project
   Created: Geoff R. McLane - Aug 2011
   License: GPL2 (or later)
   ================================================== */
// test_aptdat.cpp

#include "test_config.h"
#include "test_aptdat.h"
#include "utilities/fgx_gzlib.h"
#include "utilities.h"

QString def_fgapt_dat = DEF_APT810_FILE;
QString get_fgapt_dat() { return def_fgapt_dat; }
void set_fgapt_dat(QString file) { def_fgapt_dat = file; }

/* ------------------------------------
   from : http://data.x-plane.com/file_specs/Apt810.htm
Code (apt.dat) Used for
1 Airport header data.
16 Seaplane base header data. No airport buildings or boundary fences will be rendered in X-Plane.
17 Heliport header data.  No airport buildings or boundary fences will be rendered in X-Plane.
10 Runway or taxiway at an airport.
14 Tower view location.
15 Ramp startup position(s)
18 Airport light beacons (usually "rotating beacons" in the USA).  Different colours may be defined.
19 Airport windsocks.
50 to 56 Airport ATC (Air Traffic Control) frequencies.

"10  52.908983  156.853257 xxx 179.98   546 0.0 0.0   124 111111  2 0 0 0.00 0";

Runways and taxiways Example Usage
 0: 10          Identifies this as a data line for a runway or taxiway segment.
 1: 35.044209   Latitude (in decimal degrees) of runway or taxiway segment center.
 2: -106.598557 Longitude (in decimal degrees) of runway or taxiway segment center.
 3: 08x         Runway number (eg 25x or 24R).  If there is no runway suffix (eg. L, R, C or "S"), then an x is used.  xxx identifies the entry as a taxiway.  Helipads at the same airport are numbered sequentially as "H1x", H2x".
 4: 90.439      True (not magnetic) heading of the runway in degrees.  Must be between 0.00 and 360.00.
 5: 13749       Runway or taxiway segment length in feet.

 0: 10          Identifies this as a data line for a runway or taxiway segment.
 1: 35.044209   Latitude (in decimal degrees) of runway or taxiway segment center.
 2: -106.598557 Longitude (in decimal degrees) of runway or taxiway segment center.
 3: 08x         Runway number (eg 25x or 24R).  If there is no runway suffix (eg. L, R, C or "S"), then an x is used.  xxx identifies the entry as a taxiway.  Helipads at the same airport are numbered sequentially as "H1x", H2x".
 4: 90.439      True (not magnetic) heading of the runway in degrees.  Must be between 0.00 and 360.00.
 5: 13749       Runway or taxiway segment length in feet.
 6: 1000.0000   Length of displaced threshold (1,000 feet) for runway 08 and for the reciprocal runway 26 (0 feet).  The length of the reciprocal runways displaced threshold is expressed as the fractional part of this number.  Take the runway 26 displaced threshold length  (in feet) and divide it by 10,000, then add it to the displaced threshold length for runway 08.  For example, for displaced threshold lengths of 543 feet and 1234 feet, the code would be 543.1234.
            Note that the displaced threshold length is included in the overall runway length but that the stopway length is excluded from the overall runway length.  This code should be 0.0000 for taxiway segments.  FYI, the displaced threshold is usually marked (in the real world) with long white arrows pointing toward the threshold.  The displaced threshold is not available for use by aeroplanes landing, but may be used for take-off (in practice, if you use these last few feet of the runway for take-off, you are probably in serious trouble!).
 7: 0.1000      Length of stopway/blastpad/over-run at the approach end of runway 08 (0 feet) and for runway 26 (1,000 feet), using the same coding structure defined above.  FYI, in the real world the stopway/blastpad/over-run is usually marked with large yellow chevrons, and aeroplane movements are not permitted.
 8: 150         Runway or taxiway segment width in feet.
 9: 252231      Runway or taxiway segment lighting codes.
            The first three digits ("252") define the lighting for the runway as seen when approached from the direction implied by the runway number (08 in our example).
            The final three ("231") define the lighting for the runway as seen when approached from the opposite end (26 in our example).
            In order, these codes represent:
              Runway end A (08): Visual approach path (VASI / PAPI etc.) lighting.  Here, code 2 corresponds to a VASI.
              Runway end A (08): Runway lighting. Here, code 5 corresponds to TDZ lighting, which also implies centre-line lighting, REIL and edge lighting.
              Runway end A (08): Approach lighting.  Here, code 2 corresponds to SSALS.
              Other runway end (26): Visual approach path (VASI / PAPI etc.) lighting. Here, code 2 corresponds to a VASI.
              Other runway end (26): Runway lighting. Here, code 3 corresponds to REIL, which also implies edge lighting.
              Other runway end (26): Approach lighting. Here, code 1 implies no approach lighting.
10: 02           Runway or taxiway surface code for the runway or taxiway segment.  The leading zero is optional - but I always use it to keep all the columns neatly lined up.
11: 0            Runway shoulder code.  These are only available in file version 701 and later.  Here, code 0 implies that there is no runway shoulder.
12: 3            Runway markings (the white painted markings on the surface of the runway.  Here, code 3 implies precision runway markings (ie. there is an associated precision approach for the runway, either an ILS or MLS).
13: 0.25         Runway smoothness.  Used to cause bumps when taxying or rolling along the runway in X-Plane.  It is on a scale of 0.0 to 1.0, with 0.0 being very smooth, and 1.0 being very, very rough.  X-Plane determines a baseline smoothness based upon the runway surface type, and then uses this factor to determine the 'quality' of the runway surface.  The default value is 0.25.
14: 1            Runway has 'distance remaining' signs (0=no signs, 1=show signs).  These are the white letters on a black background on little illuminated signs along a runway, indicating the number of thousands of feet of usable runway that remain.  They are inappropriate at small airports or on most dirt, gravel or grass runways.
15: 0300.0350    NEW for file version 810:  Visual glideslope angle for the VASI or PAPI at each end of the runway (3.00 degrees for runway 08 and 3.50 degrees for runway 26).  The angle for runway 08 is the whole part of this number divided by 100 (so "0300" becomes 3.00 degrees) and the angle for the reciprocal runway (26) is the fractional part of this number multiplied by 100 (so "0.0350" becomes 3.50 degrees).  This data is required for runways, but is NOT necessary for taxiways.
   ------------------------------------ */

// Runways and taxiways Example Usage
enum RwyOff {
    RW_CODE = 0, //10 Identifies this as a data line for a runway or taxiway segment.
    RW_LAT = 1,  //35.044209   Latitude (in decimal degrees) of runway or taxiway segment center.
    RW_LON = 2,  //-106.598557 Longitude (in decimal degrees) of runway or taxiway segment center.
    RW_NUM = 3,  //08x Runway number (eg 25x or 24R).  If there is no runway suffix (eg. L, R, C or "S"), then an x is used.  xxx identifies the entry as a taxiway.  Helipads at the same airport are numbered sequentially as "H1x", H2x".
    RW_HEAD = 4, //90.439 True (not magnetic) heading of the runway in degrees.  Must be between 0.00 and 360.00.
    RW_LEN = 5,  //13749 Runway or taxiway segment length in feet.
    RW_DISP = 6, //1000.0000   Length of displaced threshold (1,000 feet) for runway 08 and for the reciprocal runway 26 (0 feet).  The length of the reciprocal runways displaced threshold is expressed as the fractional part of this number.  Take the runway 26 displaced threshold length  (in feet) and divide it by 10,000, then add it to the displaced threshold length for runway 08.  For example, for displaced threshold lengths of 543 feet and 1234 feet, the code would be 543.1234.
    //  Note that the displaced threshold length is included in the overall runway length but that the stopway length is excluded from the overall runway length.  This code should be 0.0000 for taxiway segments.  FYI, the displaced threshold is usually marked (in the real world) with long white arrows pointing toward the threshold.  The displaced threshold is not available for use by aeroplanes landing, but may be used for take-off (in practice, if you use these last few feet of the runway for take-off, you are probably in serious trouble!).
    RW_STOP = 7, // 0.1000  Length of stopway/blastpad/over-run at the approach end of runway 08 (0 feet) and for runway 26 (1,000 feet), using the same coding structure defined above.  FYI, in the real world the stopway/blastpad/over-run is usually marked with large yellow chevrons, and aeroplane movements are not permitted.
    RW_WID = 8,  // 150     Runway or taxiway segment width in feet.
    RW_LITES = 9,// 252231  Runway or taxiway segment lighting codes.
    RW_SURF = 10,// 02      Runway or taxiway surface code for the runway or taxiway segment.  The leading zero is optional - but I always use it to keep all the columns neatly lined up.
    RW_SHLDR = 11, // 0  Runway shoulder code.  These are only available in file version 701 and later.  Here, code 0 implies that there is no runway shoulder.
    RW_MARKS = 12, // 3  Runway markings (the white painted markings on the surface of the runway.  Here, code 3 implies precision runway markings (ie. there is an associated precision approach for the runway, either an ILS or MLS).
    RW_SMOOTH = 13,// 0.25 Runway smoothness.  Used to cause bumps when taxying or rolling along the runway in X-Plane.  It is on a scale of 0.0 to 1.0, with 0.0 being very smooth, and 1.0 being very, very rough.  X-Plane determines a baseline smoothness based upon the runway surface type, and then uses this factor to determine the 'quality' of the runway surface.  The default value is 0.25.
    RW_SIGNS = 14, // 1 Runway has 'distance remaining' signs (0=no signs, 1=show signs).  These are the white letters on a black background on little illuminated signs along a runway, indicating the number of thousands of feet of usable runway that remain.  They are inappropriate at small airports or on most dirt, gravel or grass runways.
    // RW_GS = 15, // 0300.0350 NEW for file version 810:  Visual glideslope angle for the VASI or PAPI at each end of the runway (3.00 degrees for runway 08 and 3.50 degrees for runway 26).  The angle for runway 08 is the whole part of this number divided by 100 (so "0300" becomes 3.00 degrees) and the angle for the reciprocal runway (26) is the fractional part of this number multiplied by 100 (so "0.0350" becomes 3.50 degrees).  This data is required for runways,
    // but is NOT necessary for taxiways.
    RW_SIZE = 15
};

QHash<QString, QString> airports;

bool import_icao_only = false;

// bool AirportsData::import(QProgressDialog &progress, MainObject *mainObject, bool import_icao_only){
bool test_import() {
    //====================================
    // Step 1: Get a hash map of aircraft descriptions from aptdat
    //int estimated_lines = 1510000;
    //qDebug() << "IMPORT";
    //progress.setValue(0);
    //progress.setWindowTitle("Importing Apt Dat");
    //progress.setRange(0, estimated_lines);
    //progress.repaint();
    QString msg;
    QTime tm;
    int   ms;
    //QString zf = mainObject->X->fgroot("/Airports/apt.dat.gz");
    QString zf = "C:\\FGCVS\\flightgear\\data\\Airports\\apt.dat.gz";
    fgx_gzHandle gzf; // opaque file gz handle
    QFile f;
    if (!f.exists(zf)) {
        outLog("ERROR: Failed to find ["+zf+"! NO AIRPORT FILE DATA!");
        return true;
    }
    tm.start();
    gzf = fgx_gzOpen(zf);
    if (!gzf) {
        outLog("ERROR: Failed to open ["+zf+"]");
        return true;
    }
    outLog("Processing file ["+zf+"]");

    int air_count = 0;
    int line_counter = 0;

    QRegExp rxICAOAirport("[A-Z]{4}");

    //* ignore first line
    fgx_gzReadline(gzf);
    QString credits = fgx_gzReadline(gzf);

    int version = 999;
    if(credits.startsWith("810 Version")) {
        version = 810;
        outLog("Dealing with Version 810");
    } else {
        credits.chop(credits.length() - 12);
        outLog("Dealing with "+credits);
    }

     QString airport_code;
     QString airport_name;
     QString elevation;
     QString tower;

     line_counter = 2; // doen 2 lines
     while ( ! fgx_gzEof(gzf)) {
         QString line = fgx_gzReadline(gzf);

        //qDebug() << line;
        QString row_code = line.section(' ',0, 0);
        //qDebug() << row_code;
        QStringList parts = line.split(" ", QString::SkipEmptyParts);

        //**********************************************************************
        //*** Airport
        if(row_code == "1"){
            // http://data.x-plane.com/file_specs/Apt715.htm
            // 0 = airport code
            //  1 = elevation
            // 2 = has tower
            // 3 = not approp
            // 4 = code
            // 5+ description
            airport_code = parts[4];
            elevation = parts[1];
            tower =  parts[2] == "1" ? "1" : "";
            airport_name.clear();
            for(int p = 5; p < parts.size(); p++){ //** TODO WTF ?
                airport_name.append(parts[p]).append(" ");
            }
            if(import_icao_only){
                if( rxICAOAirport.exactMatch(airport_code) ){
                        airports[airport_code] = airport_name;
                        air_count++;
                    }
            }else{
                    airports[airport_code] = airport_name;
                    air_count++;
            } /* if(is_icao) */

        } /* if(row_code == "1") airport */
        //if (progress.wasCanceled()){
        //    progress.hide();
        //    fgx_gzClose(gzf);
        //    outLog("User abort of airport imports!");
        //    return true;
        //
        //}
        line_counter++;
        //if(line_counter % 1000 == 0){
        //    progress.setValue(line_counter);
        //    QString prog_text = QString("%1 of approx %2").arg(line_counter).arg(estimated_lines);
        //    progress.setLabelText(prog_text);
        //
        //    progress.repaint();
        //}
    } /* end while readline */

    fgx_gzClose(gzf);

    msg.sprintf("Done %d lines, found %d airport entries",line_counter,air_count);
    ms = tm.elapsed();
    outLog(msg+", in "+getElapTimeStg(ms));

    return false;
}

//typedef struct tagRUNWAY2 {
//    double lat,lon,heading;
//    int len_ft;
//    QString rwy_no;
//}RUNWAY2, * PRUNWAY2;
//typedef QList<PRUNWAY2> RWY_LIST;
//typedef struct tagAIRPORT2 {
//    QString icao;
//    double lat, lon;
//    int elev; // feet MSL
//    QString name;
//    QString path;
//    double dist;
//    RWY_LIST rways;
//    int rank;
//}AIRPORT2, * PAIRPORT2;
//typedef QList<PAIRPORT2> APT_LIST;

APT_LIST apt_list;  // final AIRPORT list
RWY_LIST rwy_list;  // temporary RUNWAY list

void clear_apt_list()
{
    PAIRPORT2 pa2;
    while (!apt_list.isEmpty()) {
        pa2 = apt_list.takeFirst();
        while (!pa2->rways.isEmpty())
            delete pa2->rways.takeFirst();
        delete pa2;
    }
}

PAPT_LIST get_apt_list() { return &apt_list; }

bool set_distances2( QString icao, PAIRPORT2 *ppa )
{
    bool success = false;
    QTime tm;
    int i, found;
    int ind;
    double lat, lon, dist, min_dist;
    PAIRPORT2 pfnd, pa;
    QString msg;
    //int max = airportList.count();
    int max = apt_list.count();
    msg.sprintf("List of %d airports, ",max);
    msg.append("centering on "+icao);
    outLog(msg);
    tm.start();
    min_dist = 99999.0;
    ind = -1;
    *ppa = 0;
    for (i = 0; i < max; i++) {
        //pa = airportList.at(i);
        pa = apt_list.at(i);
        if (pa->icao == icao)
            break;
    }
    if ( i < max ) {
        found = i;
        //pfnd = airportList.at(found);
        pfnd = apt_list.at(found);
        lat = pfnd->lat;
        lon = pfnd->lon;
        for (i = 0; i < max; i++) {
            //pa = airportList.at(i);
            pa = apt_list.at(i);
            if (i == found) {
                pa->dist = 0.0;
                pa->rank = 0;
            } else {
                if (pa->icao == "YSDU")
                    dist = 0;
                dist = dist_est_km(lat, lon, pa->lat, pa->lon);
                if (dist < 1.0)
                    dist = 1.0;
                pa->dist = dist;
                if (dist < min_dist) {
                    min_dist = dist;
                    ind = i;
                }
            }
        }
        success = true;
    }
    if (ind != -1)
        *ppa = apt_list.at(ind);
        //*ppa = airportList.at(ind);
    msg = getElapTimeStg(tm.elapsed());
    if (success)
        outLog("Airport sort done in "+msg);
    else
        outLog("FAILED, in "+msg);
    return success;
}

bool get_next_distance2( double mdist, PAIRPORT2 *ppa )
{
    bool success = false;
    int i;
    int ind;
    double dist, min_dist;
    PAIRPORT2 pa;
    int max = apt_list.count();
    min_dist = 99999.0;
    ind = -1;
    *ppa = 0;
    for (i = 0; i < max; i++) {
        pa = apt_list.at(i);
        dist = pa->dist;
        if (dist > mdist) {
            if (dist < min_dist) {
                min_dist = dist;
                ind = i;
            }
        }
    }
    if (ind != -1) {
        *ppa = apt_list.at(ind);
        success = true;
    }
    return success;
}


void do_apt_sort()
{
    PAIRPORT2 pa2;
    QString icao, name, path, line, posn;
    double lat, lon, dist;
    if (set_distances2("YGIL", &pa2)) {
        icao = pa2->icao;
        lat = pa2->lat;
        lon = pa2->lon;
        name = pa2->name;
        path = pa2->path;
        dist = pa2->dist;
        posn.sprintf("%.2f Km, at %f,%f", dist, lat, lon);
        line = "nearest "+icao+" "+posn+" "+name;
        outLog(line);
        int i = 0;
        while ( (i < 5) && get_next_distance2(dist, &pa2)) {
            i++;
            dist = pa2->dist;
            lat = pa2->lat;
            lon = pa2->lon;
            icao = pa2->icao;
            name = pa2->name;
            posn.sprintf("%.2f Km, at %f,%f", dist, lat, lon);
            line.sprintf("next %d: ", i);
            line.append(icao+" "+posn+" "+name);
            outLog(line);
        }
    }
}

bool test_import2()
{
    QString msg;
    QTime tm;
    int   ms;

    //QString zf = mainObject->X->fgroot("/Airports/apt.dat.gz");
    QString zf = get_fgapt_dat(); // "C:\\FGCVS\\flightgear\\data\\Airports\\apt.dat.gz";
    fgx_gzHandle gzf; // opaque file gz handle
    QFile f;
    if (!f.exists(zf)) {
        outLog("ERROR: Failed to find ["+zf+"! NO AIRPORT FILE DATA!");
        return true;
    }
    tm.start();
    gzf = fgx_gzOpen(zf);
    if (!gzf) {
        outLog("ERROR: Failed to open ["+zf+"]");
        return true;
    }
    outLog("Processing file ["+zf+"]");

    int air_count = 0;
    int line_counter = 0;
    int rwy_count = 0;
    QRegExp rxICAOAirport("[A-Z]{4}");

    //* ignore first line
    fgx_gzReadline(gzf);
    QString credits = fgx_gzReadline(gzf);

    int version = 999;
    if(credits.startsWith("810 Version")) {
        version = 810;
        outLog("Dealing with Version 810");
    } else {
        credits.chop(credits.length() - 12);
        outLog("Dealing with "+credits);
    }

    QString airport_code;
    QString airport_name;
    QString elevation;
    QString sp(" ");
    QString apt("1");
    QString rwy("10");
    QString one("1");
    QString txwy("xxx");
    QString end("99");

    QString rwy_number;
    int elev_ft, tower, bldgs, pcnt, rwy_cnt, p;
    double lat,lon,sum_lat,sum_lon, got_apt;
    line_counter = 2; // done 2 lines
    rwy_cnt = 0;
    got_apt = 0;
    elev_ft = 0;
    sum_lat = 0.0;
    sum_lon = 0.0;
    rwy_list.clear();
    clear_apt_list(); // clear any PREVIOUS
    while ( ! fgx_gzEof(gzf) ) {
        QString line = fgx_gzReadline(gzf);
        line_counter++;
        line = line.trimmed();
        if (line.isEmpty())
            continue;
        QString row_code = line.section(' ',0, 0);
        if (row_code == end)
            break; // last line

        QStringList parts = line.split(" ", QString::SkipEmptyParts);
        pcnt = parts.size();
        //**********************************************************************
        //*** Airport
        //if(row_code == "1"){
        if (row_code == apt) {
            if (rwy_cnt && got_apt) {
                // get 'average' center of last airport
                lat = sum_lat / (double)rwy_cnt;
                lon = sum_lon / (double)rwy_cnt;
                PAIRPORT2 pa2 = new AIRPORT2;
                pa2->icao = airport_code;
                pa2->name = airport_name;
                pa2->elev = elev_ft;
                pa2->lat  = lat;
                pa2->lon  = lon;
                while (!rwy_list.isEmpty())
                    pa2->rways.append(rwy_list.takeFirst());
                apt_list.append(pa2);
            }

            got_apt = 0;
            if (pcnt >= 6) {
                // http://data.x-plane.com/file_specs/Apt715.htm and 815
                // 0 = airport code = '10'
                // 1 = elevation (feet)
                // 2 = has tower
                // 3 = has building
                // 4 = ICAO code
                // 5+ description
                elevation = parts[1];
                elev_ft = elevation.toInt();
                tower = (parts[2] == one) ? 1 : 0;
                bldgs = (parts[3] == one) ? 1 : 0;
                airport_code = parts[4];
                airport_name = parts[5];
                for(p = 6; p < pcnt; p++){ //** TODO WTF ?
                    airport_name.append(sp+parts[p]);
                }
                airport_name = airport_name.trimmed();
                if (import_icao_only) {
                    if( rxICAOAirport.exactMatch(airport_code) ) {
                        airports[airport_code] = airport_name;
                        air_count++;
                        got_apt = 1;
                    }
                } else {
                    airports[airport_code] = airport_name;
                    air_count++;
                    got_apt = 1;
                } /* if(is_icao) */
            } else {
                // WHAT IS THIS???
                outLog("Discard apt: "+line);
            }
            // done airport init
            sum_lat = 0.0;
            sum_lon = 0.0;
            rwy_cnt = 0;
            // clear the runway list
            while (!rwy_list.isEmpty())
                 delete rwy_list.takeFirst();
        } /* if(row_code == "1") airport */
        else if (row_code == rwy) {
            // RUNWAY OR TAXIWAY
            // =================
            if (pcnt < RW_SIZE) {
                outLog("WARNING: skipped RWY ["+line+"]");
                continue;
            }
            // if (pcnt >= 6) {
            /* -----------------
                0  runway or taxiway segment = '10'
                1  Latitude (in decimal degrees)
                2  Longitude (in decimal degrees)
                3  Runway number (eg 25x or 24R).
                //  If there is no runway suffix (eg. L, R, C or "S"),
                //  then an x is used.  xxx identifies the entry as a taxiway.
                //  Helipads at the same airport are numbered sequentially as "H1x", H2x".
                4   Heading True (not magnetic) in degrees
                5   Length Runway or taxiway segment in feet.
               ----------------- */
                rwy_number = parts[RW_NUM];
                if ((rwy_number != txwy) && got_apt) {

                    lat = parts[RW_LAT].toDouble();
                    lon = parts[RW_LON].toDouble();

                    PRUNWAY2 pr2 = new RUNWAY2;
                    pr2->heading = parts[RW_HEAD].toDouble();
                    pr2->lat = lat;
                    pr2->lon = lon;
                    pr2->len_ft = parts[RW_LEN].toInt();
                    pr2->rwy_no = rwy_number;

                    rwy_list.append(pr2);
                    rwy_cnt++;
                    // accumulate
                    sum_lat += lat;
                    sum_lon += lon;
                    rwy_count++;
                }
            // }
        }

    } /* end while readline */
    // get any LAST runway list
    if (rwy_cnt && got_apt) {
        // get 'average' center of last airport
        lat = sum_lat / (double)rwy_cnt;
        lon = sum_lon / (double)rwy_cnt;
        PAIRPORT2 pa2 = new AIRPORT2;
        pa2->icao = airport_code;
        pa2->name = airport_name;
        pa2->elev = elev_ft;
        pa2->lat  = lat;
        pa2->lon  = lon;
        while (!rwy_list.isEmpty())
            pa2->rways.append(rwy_list.takeFirst());
        apt_list.append(pa2);
    }
    fgx_gzClose(gzf);

    msg.sprintf("Done %d lines, found %d airports (%d), %d runways",
                line_counter,air_count,
                apt_list.count(),
                rwy_count);
    ms = tm.elapsed();
    outLog(msg+", in "+getElapTimeStg(ms));

    return false; // no problems
}

void test_aptdat()
{
    bool res;
    //res = test_import();
    res = test_import2();
}

// eof - test_aptdat.cpp
