// load_apt85.cpp

// load the latest apt.dat from X-Plane
#include "test_config.h"
#include "test_apt850.h"
#include "utilities.h"
#include "utilities/simgear/simgear.h"
#include "test_aptdat.h"

/* -------------------------------------
1 Land airport header
16 Seaplane base header
17 Heliport header
  1    - Row code for an airport, seaplane base or heliport = 1, 16 or 17
  21 - Elevation of airport in feet above mean sea level (AMSL)
  1 - Control tower
  0 - Depreciated
  KBFI - Airport ICAO code. If no ICAO code exists, use FAA code (USA only) - Max 4 chars - unique.
  Boeing Field King Co  - Airport name. (up to 40 characters)

100 Runway
  100 - Row code for a land runway (the most common) 100
  29.87 - Width of runway in metres 2 decimal places. Must be >= 1.00
  1 - Code defining the surface type (concrete, asphalt, etc) Integer
  0  - Code defining a runway shoulder surface type 0=none, 1=asphalt, 2=concrete
  0.15 - Runway smoothness (not used by X-Plane yet) 0.00 (smooth) to 1.00 (very rough). Default is 0.25
  0  - Runway centre-line lights 0=no centerline lights, 1=centre line lights
  2  - Runway edge lighting (also implies threshold lights) 0=no edge lights, 2=medium intensity edge lights
  1 - Auto-generate distance-remaining signs (turn off if created manually) 0=no auto signs, 1=auto-generate signs
  The following fields are repeated for each end of the runway
  13L - Runway number (eg. 31R, 02). Leading zeros are required. 2-3 chars. suffixes: L, R or C (or blank)
  47.53801700 - Latitude of runway end (on runway centerline) in decimal degrees 8 decimal places supported
  -122.30746100 - Longitude of runway end (on runway centerline) in decimal degrees 8 decimal places supported
  73.15 - Length of displaced threshold in metres (this is included in runway length) be inside (between) ends
  0.00 - Length of overrun/blast-pad in metres (not included in runway length)
  2 - Code for runway markings (Visual, non-precision, precision) Integer
  0 - Code for approach lighting for this runway end Integer
  0 - Flag for runway touchdown zone (TDZ) lighting 0=no TDZ lighting, 1=TDZ lighting
  1 - Code for Runway End Identifier Lights (REIL) 0=no REIL, 1=omni-directional REIL, 2=unidirectional REIL

 101 Water runway
 102 Helipad
 110 Pavement (taxiway or ramp) header Must form a closed loop
 120 Linear feature (painted line or light string) header Can form closed loop or simple string
 130 Airport boundary header Must form a closed loop
 111 Node All nodes can also include a style (line or lights)
 112 Node with Bezier control point Bezier control points define smooth curves
 113 Node with implicit close of loop Implied join to first node in chain
 114 Node with Bezier control point, with implicit close of loop Implied join to first node in chain
 115 Node terminating a string (no close loop) No styles used
 116 Node with Bezier control point, terminating a string (no close loop) No styles used
 14 Airport viewpoint One or none for each airport
 15 Aeroplane startup location Zero, one or many for each airport
 18 Airport light beacon One or none for each airport
 19 Windsock Zero, one or many for each airport
 20 Taxiway sign (inc. runway distance-remaining signs) Zero, one or many for each airport
 21 Lighting object (VASI, PAPI, Wig-Wag, etc.) Zero, one or many for each airport
 50  56 Communication frequencies Zero, one or many for each airport

 EEK '10' looks undocumented, but it exists - MANY

   ------------------------------------- */


#define a85_Land      1 // Land airport header
#define a85_Sea      16 // Seaplane base header
#define a85_Hele     17 // Heliport header
//  1  15 1 0 VHXX [X] CLOSED Kai Tak
// 16   0 0 0 QANA [S] Anahola Bay
enum Apt850 {
    A85_CODE = 0,
    A85_ELEV = 1, // 15 feet AMSL
    A85_TWR = 2,  //  1 Control tower
    A85_DEP = 3,  //  0 Depreciated
    A85_ICAO = 4, // VHXX - ICAO (or FAA code USA)
    A85_NAME = 5, // '[X] CLOSED Kai Tak' Begin of NAME
    A85_MIN = 6   // min of 6 parts
};

#define a85_Runway  100 // Runway
// 100 54.86 2 0 0.00 1 3 0 13 22.32526300 114.19222700 563.88 60.05 3  0 0 2 31   22.30395000  114.21587400  220.98    0.00 3  1 0 2
enum Rwy850 {
    R85_CODE = 0, // 100 - Row code for a land runway (the most common) 100
    R85_WIDTH = 1, //29.87 - Width of runway in metres 2 decimal places. Must be >= 1.00
    R85_SURF = 2, // 1 - Code defining the surface type (concrete, asphalt, etc) Integer
    R85_SHDR = 3, // 0  - Code defining a runway shoulder surface type 0=none, 1=asphalt, 2=concrete
    R85_SMTH = 4, // 0.15 - Runway smoothness (not used by X-Plane yet) 0.00 (smooth) to 1.00 (very rough). Default is 0.25
    R85_CLTS = 5, // 0  - Runway centre-line lights 0=no centerline lights, 1=centre line lights
    R85_ELTS = 6, // 2  - Runway edge lighting (also implies threshold lights) 0=no edge lights, 2=medium intensity edge lights
    R85_SIGN = 7, // 1 - Auto-generate distance-remaining signs (turn off if created manually) 0=no auto signs, 1=auto-generate signs
    // The following fields are repeated for each end of the runway
    R85_NUM1 = 8, // 13L - Runway number (eg. 31R, 02). Leading zeros are required. 2-3 chars. suffixes: L, R or C (or blank)
    R85_LAT1 = 9, // 47.53801700 - Latitude of runway end (on runway centerline) in decimal degrees 8 decimal places supported
    R85_LON1 = 10, // -122.30746100 - Longitude of runway end (on runway centerline) in decimal degrees 8 decimal places supported
    R85_DIS1 = 11, // 73.15 - Length of displaced threshold in metres (this is included in runway length) be inside (between) ends
    R85_OVR1 = 12, // 0.00 - Length of overrun/blast-pad in metres (not included in runway length)
    R85_MRK1 = 13, // 2 - Code for runway markings (Visual, non-precision, precision) Integer
    R85_ALTS1 = 14, // 0 - Code for approach lighting for this runway end Integer
    R85_TDZ1 = 15, // 0 - Flag for runway touchdown zone (TDZ) lighting 0=no TDZ lighting, 1=TDZ lighting
    R85_REIL1 = 16, // 1 - Code for Runway End Identifier Lights (REIL) 0=no REIL, 1=omni-directional REIL, 2=unidirectional REIL
    R85_NUM2 = 17, // 13L - Runway number (eg. 31R, 02). Leading zeros are required. 2-3 chars. suffixes: L, R or C (or blank)
    R85_LAT2 = 18, // 47.53801700 - Latitude of runway end (on runway centerline) in decimal degrees 8 decimal places supported
    R85_LON2 = 19, // -122.30746100 - Longitude of runway end (on runway centerline) in decimal degrees 8 decimal places supported
    R85_DIS2 = 20, // 73.15 - Length of displaced threshold in metres (this is included in runway length) be inside (between) ends
    R85_OVR2 = 21, // 0.00 - Length of overrun/blast-pad in metres (not included in runway length)
    R85_MRK2 = 22, // 2 - Code for runway markings (Visual, non-precision, precision) Integer
    R85_ALTS2 = 23, // 0 - Code for approach lighting for this runway end Integer
    R85_TDZ2 = 24, // 0 - Flag for runway touchdown zone (TDZ) lighting 0=no TDZ lighting, 1=TDZ lighting
    R85_REIL2 = 25, // 1 - Code for Runway End Identifier Lights (REIL) 0=no REIL, 1=omni-directional REIL, 2=unidirectional REIL
    R85_SIZE = 26   // size of runway space split
};

#define a85_Water   101 // Water runway
// 16       0 0 0 QANA [S] Anahola Bay
// 101 1219.20 0 36   22.16029304 -159.29608200 18   22.17126496 -159.29608200
// 16       0 0 0 SP25 [S] Andros Town Harbor
// 101 3047.70 0 36   24.69577661 -077.73910500 18   24.72594939 -077.73910500
enum Water850 {
    W85_CODE = 0, // 101 Row code for a water runway 101
    W85_WID = 1,  //  49 Width of runway in metres Two decimal places recommended. Must be >= 1.00
    W85_BUOYS = 2,//   1 Flag for perimeter buoys 0=no buoys, 1=render buoys
    // The following fields are repeated for each end of the water runway
    W85_NUM1 = 3,  //  08 Runway number. Not rendered in X-Plane (its on water!) Valid suffixes are L, R or C (or blank)
    W85_LAT1 = 4,  // 35.04420911 Latitude of runway end (on runway centerline) in decimal degrees Eight decimal places supported
    W85_LON1 = 5,  // -106.59855711 Longitude of runway end (on runway centerline) in decimal degrees Eight decimal places supported
    W85_NUM2 = 6,  //  08 Runway number. Not rendered in X-Plane (its on water!) Valid suffixes are L, R or C (or blank)
    W85_LAT2 = 7,  // 35.04420911 Latitude of runway end (on runway centerline) in decimal degrees Eight decimal places supported
    W85_LON2 = 8,  // -106.59855711 Longitude of runway end (on runway centerline) in decimal degrees Eight decimal places supported
    W85_SIZE = 9
};

#define a85_Helepad 102 // Helipad
// 102 H1   43.30478555 -008.37734016  30.56   21.03   21.03   1 0   0 0.00 0
enum Hele850 {
    H85_CODE = 0,  // 102 Helipad Row code for a helipad
    H85_DESIG = 1, //  H1 Designator for a helipad. Must be unique at an airport. Usually H suffixed by an integer (eg. H1, H3)
    H85_LAT = 2,   // 47.53918248 Latitude of helipad centre in decimal degrees Eight decimal places supported
    H85_LON = 3,   // -122.30722302 Longitude of helipad centre in decimal degrees Eight decimal places supported
    H85_HEAD = 4,  // 2.00 Orientation (true heading) of helipad in degrees Two decimal places recommended
    H85_LEN = 5,   // 10.06  Helipad length in metres Two decimal places recommended (metres), must be >=1.00
    H85_WID = 6,   // 10.06  Helipad width in metres Two decimal places recommended (metres), must be >= 1.00
    H85_SURF = 7,  //     1  Helipad surface code Integer value for a Surface Type Code
    H85_MARK = 8,  //     0  Helipad markings 0 (other values not yet supported)
    H85_SHLDR = 9, //     0  Code defining a helipad shoulder surface type 0=no shoulder, 1=asphalt shoulder, 2=concrete shoulder
    H85_SMOOTH = 10, // 0.25 Helipad smoothness (not used by X-Plane yet) 0.00 (smooth) to 1.00 (very rough). Default is 0.25
    H85_ELITE = 11, //     0 Helipad edge lighting 0=no edge lights, 1=yellow edge lights
    H85_SIZE = 12
};

#define a85_Taxiway 110 // Pavement (taxiway or ramp) header Must form a closed loop
enum Tax850 {
    T85_CODE = 0, // 110 Row code for a pavement chunk header (must be followed by a set of nodes) 110
    T85_SURF = 1, // 1   Code defining the surface type (concrete, asphalt, etc) Integer value for a Surface Type Code
    T85_SMTH = 2, // 0.25 Runway smoothness (not used by X-Plane yet) 0.00 (smooth) to 1.00 (very rough). Default is 0.25
    T85_HEAD = 3, // 150.29 Orientation (true degrees) of pavement texture grain Two decimal places recommended
    T85_DESC = 4, // A2 Exit Description of pavement chunk (not used by X-Plane) Text string
    T85_MIN = 5
};

#define a85_TaxiOld 10 // Looks like a TAXIWAY
// 10 57.19970300 -002.19489200 xxx   4.57   120 0.0 0.0   120 161161  1 0 0 0.25 0
// Seems to share the OLD 810 format of runways and taxiways
enum Tax810 {
    TX8_CODE = 0, // 10 Row code for taxiway
    TX8_LAT = 1,  //35.044209   Latitude (in decimal degrees) of runway or taxiway segment center.
    TX8_LON = 2,  //-106.598557 Longitude (in decimal degrees) of runway or taxiway segment center.
    TX8_NUM = 3,  // xxx number xxx identifies the entry as a taxiway.  Helipads at the same airport are numbered sequentially as "H1x", H2x".
    TX8_HEAD = 4, //90.439 True (not magnetic) heading of the runway in degrees.  Must be between 0.00 and 360.00.
    TX8_LEN = 5,  //13749 taxiway segment length in feet.
    TX8_DISP = 6, //1000.0000   Length of displaced threshold (1,000 feet) for runway 08 and for the reciprocal runway 26 (0 feet).  The length of the reciprocal runways displaced threshold is expressed as the fractional part of this number.  Take the runway 26 displaced threshold length  (in feet) and divide it by 10,000, then add it to the displaced threshold length for runway 08.  For example, for displaced threshold lengths of 543 feet and 1234 feet, the code would be 543.1234.
    //  Note that the displaced threshold length is included in the overall runway length but that the stopway length is excluded from the overall runway length.  This code should be 0.0000 for taxiway segments.  FYI, the displaced threshold is usually marked (in the real world) with long white arrows pointing toward the threshold.  The displaced threshold is not available for use by aeroplanes landing, but may be used for take-off (in practice, if you use these last few feet of the runway for take-off, you are probably in serious trouble!).
    TX8_STOP = 7, // 0.1000  Length of stopway/blastpad/over-run at the approach end of runway 08 (0 feet) and for runway 26 (1,000 feet), using the same coding structure defined above.  FYI, in the real world the stopway/blastpad/over-run is usually marked with large yellow chevrons, and aeroplane movements are not permitted.
    TX8_WID = 8,  // 150     Runway or taxiway segment width in feet.
    TX8_LITES = 9,// 252231  Runway or taxiway segment lighting codes.
    TX8_SURF = 10,// 02      Runway or taxiway surface code for the runway or taxiway segment.  The leading zero is optional - but I always use it to keep all the columns neatly lined up.
    TX8_SHLDR = 11, // 0  Runway shoulder code.  These are only available in file version 701 and later.  Here, code 0 implies that there is no runway shoulder.
    TX8_MARKS = 12, // 3  Runway markings (the white painted markings on the surface of the runway.  Here, code 3 implies precision runway markings (ie. there is an associated precision approach for the runway, either an ILS or MLS).
    TX8_SMOOTH = 13,// 0.25 Runway smoothness.  Used to cause bumps when taxying or rolling along the runway in X-Plane.  It is on a scale of 0.0 to 1.0, with 0.0 being very smooth, and 1.0 being very, very rough.  X-Plane determines a baseline smoothness based upon the runway surface type, and then uses this factor to determine the 'quality' of the runway surface.  The default value is 0.25.
    TX8_SIGNS = 14, // 1 Runway has 'distance remaining' signs (0=no signs, 1=show signs).  These are the white letters on a black background on little illuminated signs along a runway, indicating the number of thousands of feet of usable runway that remain.  They are inappropriate at small airports or on most dirt, gravel or grass runways.
    // RW_GS = 15, // 0300.0350 NEW for file version 810:  Visual glideslope angle for the VASI or PAPI at each end of the runway (3.00 degrees for runway 08 and 3.50 degrees for runway 26).  The angle for runway 08 is the whole part of this number divided by 100 (so "0300" becomes 3.00 degrees) and the angle for the reciprocal runway (26) is the fractional part of this number multiplied by 100 (so "0.0350" becomes 3.50 degrees).  This data is required for runways,
    // but is NOT necessary for taxiways.
    TX8_SIZE = 15
};

// 112 - Row code for a node. First node must follow an appropriate header row
// 111 thru 116
#define a85_Node    111 // Node All nodes can also include a style (line or lights)
enum Node850 {
    N85_CODE = 0, // code for a NODE
    N85_LAT = 1,  // all nodes 111 thru 116
    N85_LON = 2,  // all nodes 111 thru 116
    N85_BLAT = 3, // [112, 114, 116 only] Latitude of Bezier control point in decimal degrees
    N85_BLON = 4, // Eight decimal places supported. Ignore for 111, 113, 115
    N85_TYPE = 5, // 3 [Not for 115 or 116] Code for painted line type on line segment starting at this node
    // Integer Line Type Code. Not for 115 or 116
    N85_LITE = 6, // 102 [Not for 115 or 116] Code for lighting on line segment starting at this node
    // Integer Line Type Code. Not for 115 or 116
    N85_MAX = 7
};

#define a85_Bezier  112 // Node with Bezier control point Bezier control points define smooth curves
#define a85_Close   113 // Node with implicit close of loop Implied join to first node in chain
// 110   1 0.00  30.90 Turn 22
// 111  43.30991619 -008.37128174
// 112  43.30991619 -008.37128174  43.30992779 -008.37130764 102
// 112  43.30973502 -008.37176974  43.30954225 -008.37195299 102
// 112  43.30916251 -008.37189523  43.30899583 -008.37195698
// 113  43.30916251 -008.37189523
// 110   1 0.00  30.90 Turn 04
// 111  43.29565137 -008.38299509
// 112  43.29565137 -008.38299509  43.29551768 -008.38317003 102
// 112  43.29536490 -008.38357237  43.29517391 -008.38374731 102
// 112  43.29490653 -008.38360736  43.29479406 -008.38354030
// 113  43.29490653 -008.38360736
// 110   1 0.00 120.90 Twy N

#define a85_Control 114 // Node with Bezier control point, with implicit close of loop Implied join to first node in chain
#define a85_Term    115 // Node terminating a string (no close loop) No styles used
#define a85_String  116 // Node with Bezier control point, terminating a string (no close loop) No styles used

#define a85_Linear  120 // Linear feature (painted line or light string) header Can form closed loop or simple string
#define a85_Bounds  130 // Airport boundary header Must form a closed loop

#define a85_View     14 // Airport viewpoint One or none for each airport
// 14 22.16348500 -159.30136100 0 0 Tower Viewpoint
enum View850 {
    V85_CODE = 0,
    V85_LAT = 1,
    V85_LON = 2,
    V85_UNK1 = 3,
    V85_UNK2 = 4,
    V85_DESC = 5,
    V85_MIN = 6
};

#define a85_Startup  15 // Aeroplane startup location Zero, one or many for each airport
// 15   22.16142100 -159.30091900  40.00 Anchor
enum Su850 {
    S85_CODE = 0,
    S85_LAT = 1,
    S85_LON = 2,
    S85_ALT = 3,
    S85_NAME = 4,
    S85_MIN = 5
};

#define a85_Beacon   18 // Airport light beacon One or none for each airport
// 18   36.96329190  127.03226689 4 BCN

#define a85_Windsock 19 // Windsock Zero, one or many for each airport
// 19   36.97040984  127.02064840 1 WS
// 19   36.95401518  127.04149243 1 WS
enum ws850 {
    WS85_CODE = 0,
    WS85_LAT = 1,
    WS85_LON = 2,
    WS85_LIT = 3, // Flag for windsock lighting 0=unlit, 1=illuminated
    WS85_NAME = 4, // OPTIONAL
    WS85_MIN = 5
};

#define a85_Sign     20 // Taxiway sign (inc. runway distance-remaining signs) Zero, one or many for each airport
#define a85_Light    21 // Lighting object (VASI, PAPI, Wig-Wag, etc.) Zero, one or many for each airport
// 21   43.29737198 -008.38037207  3  30.90   3.00 API PAPI-4R
// 21   43.29784944 -008.38145086  2  30.90   3.00 API PAPI-4L
enum Lite850 {
    L85_CODE = 0,
    L85_LAT = 1,
    L85_LON = 2,

};

#define a85_Com_min  50
#define a85_Com_max  56 // Communication frequencies Zero, one or many for each airport
// 50 10820 ATIS
// 53 11950 DESIDERIO GND
// 54 12250 DESIDERIO TWR
// 55 12790 OSAN APP/DEP
enum Comm850 {
    C85_CODE = 0,
    C85_FREQ = 1,
    C85_NAME = 2,
    C85_MIN = 3
};

#define a85_End      99 // The file is terminated by a 99:

void load_apt850( void )
{
    QTime tm;
    PAIRPORT2 pa2 = 0;
    QString def_file = get_xpapt_dat();
    QFile f(def_file);
    if (!f.exists()) {
        outLog("ERROR: Failed to find ["+def_file+"! NO AIRPORT FILE DATA!");
        return;
    }
    tm.start();
    if (!f.open(QIODevice::ReadOnly)) {
        outLog("ERROR: Failed to open ["+def_file+"! NO AIRPORT FILE DATA!");
        return;
    }
    outLog("Processing file ["+def_file+"]");
    QString line, row_code, elevation, msg;
    QStringList parts;
    int pcnt, elev_ft, tower, bldgs, res, p;
    QString sp(" ");
    QString one("1");
    QString airport_icao, airport_name, rwy_number, tmp;
    double lat, lon, lat1, lon1, lat2, lon2;
    double az1, az2, s;
    int line_counter, rwy_cnt, got_apt, rwy_count, air_count, ms;
    int code;
    double sum_lat, sum_lon;
    QRegExp rxICAOAirport("[A-Z]{4}");
    PRUNWAY2 pr2;

    line_counter = 2; // done 2 lines
    rwy_cnt = 0;
    got_apt = 0;
    elev_ft = 0;
    rwy_count = 0;
    air_count = 0;
    sum_lat = 0.0;
    sum_lon = 0.0;
    PAPT_LIST pal = get_apt850_list();

    clear_apt850_list(); // delete any previous LIST - release all memory

    //* ignore first line
    f.readLine();
    QString credits = f.readLine();

    int version = 999;
    if(credits.startsWith("850 Version")) {
        version = 850;
        outLog("Dealing with Version 850");
    } else {
        credits.chop(credits.length() - 12);
        outLog("ERROR: Dealing with UNKNOWN version "+credits);
        f.close();
        return;
    }

    while (!f.atEnd()) {
        line = f.readLine();
        line = line.trimmed();
        line_counter++;
        if (line.isEmpty())
            continue;
        row_code = line.section(' ',0, 0);
        code = row_code.toInt();
        if (code == 99)
            break;
        parts = line.split(" ", QString::SkipEmptyParts);
        pcnt = parts.size();
        //if (row_code == apt) {
        // Row codes 1, 16 and 17 share a common format (see below)
        // 17     803 0 0 02KY [H]  Boone National Guard
        // 102 H1   38.18340600 -084.89995100 360.00   30.48   30.48   1 2   0 0.25 1
        // 18   38.18340600 -084.90030000 3 BCN
        // 19   38.18326800 -084.90030000 0 WS
        if ((code == a85_Land)||(code == a85_Sea)||(code == a85_Hele)) {
            // Three types of AIRPORTS - Ladn, Sea and Hele - common format
            if (rwy_cnt && got_apt && pa2) {
                // get 'average' center of last airport
                lat = sum_lat / (double)rwy_cnt;
                lon = sum_lon / (double)rwy_cnt;
                pa2->lat  = lat;
                pa2->lon  = lon;
            }
            lat = -200.0;
            lon = -200.0;
            got_apt = 0;
            pa2 = 0; // clear the last airport
            // type of airport Land or Sea
            // BUT helepads can be alone, OR at an airport
            if (pcnt < A85_MIN) {
                outLog("BAD AIRPORT LINE ["+line+"]");
                continue;
            }

            elevation = parts[A85_ELEV];
            elev_ft = elevation.toInt();
            tower = (parts[A85_TWR] == one) ? 1 : 0;
            bldgs = (parts[A85_DEP] == one) ? 1 : 0;
            airport_icao = parts[A85_ICAO];
            airport_name = parts[A85_NAME];
            for(p = A85_MIN; p < pcnt; p++){
                airport_name.append(sp+parts[p]);
            }
            airport_name = airport_name.trimmed();
            if (import_icao_only) {
                if( rxICAOAirport.exactMatch(airport_icao) ) {
                    // airports[airport_icao] = airport_name;
                    air_count++;
                    got_apt = 1;
                }
            } else {
                // airports[airport_icao] = airport_name;
                air_count++;
                got_apt = 1;
            } /* if(is_icao) */

            if (got_apt) {
                pa2 = new AIRPORT2; // start a NEW airport
                pa2->code = code;
                pa2->icao = airport_icao;
                pa2->name = airport_name;
                pa2->elev = elev_ft;
                pa2->lat  = lat; // start with BAD values
                pa2->lon  = lon;
                pa2->rways.clear();
                pal->append(pa2);
            }
            // done airport init - restart runway stuff
            sum_lat = 0.0;
            sum_lon = 0.0;
            rwy_cnt = 0;
        //} else if (row_code == rwy) {
        } else if (code == a85_Runway) {
            if (pcnt != R85_SIZE) {
                outLog("BAD RUNWAY LINE ["+line+"]");
                continue;
            }
            if ( !got_apt || !pa2 )
                continue;   // no airport, so no runways !!!

            rwy_number = parts[R85_NUM1];
            lat1 = parts[R85_LAT1].toDouble();
            lat2 = parts[R85_LAT2].toDouble();
            lon1 = parts[R85_LON1].toDouble();
            lon2 = parts[R85_LON2].toDouble();
            res = _geo_inverse_wgs_84( lat1, lon1, lat2,
                        lon2, &az1, &az2, &s );
            lat = (lat1 + lat2) / 2.0;
            lon = (lon1 + lon2) / 2.0;
            pr2 = new RUNWAY2;
            pr2->lat1 = lat1;
            pr2->lon1 = lon1;
            pr2->lat2 = lat2;
            pr2->lon2 = lon2;
            pr2->i_heading = az1;
            pr2->i_len_ft = (int)(s * SG_METER_TO_FEET);
            pr2->rwy_no = rwy_number;
            if (pa2)
                pa2->rways.append(pr2);
            rwy_cnt++;
            // accumulate
            sum_lat += lat;
            sum_lon += lon;
            rwy_count++;
        } else if (code == a85_Water) {
            // treat like a RUNWAY
            // 16       0 0 0 QANA [S] Anahola Bay
            // 101 1219.20 0 36   22.16029304 -159.29608200 18   22.17126496 -159.29608200
            // 16       0 0 0 SP25 [S] Andros Town Harbor
            // 101 3047.70 0 36   24.69577661 -077.73910500 18   24.72594939 -077.73910500
            //  enum Water850 {
            if ( !got_apt || !pa2 )
                continue;   // no airport, so no runways !!!
            if ( pcnt < W85_SIZE) {
                outLog("WARNING: Water ["+line+"] IGNORED");
                continue;
            }
            pr2 = new RUNWAY2;
            pr2->i_len_ft = (int)(parts[W85_WID].toDouble() * SG_METER_TO_FEET);
            //  49 Width of runway in metres Two decimal places recommended. Must be >= 1.00
            // W85_BUOYS = 2,//   1 Flag for perimeter buoys 0=no buoys, 1=render buoys
            // The following fields are repeated for each end of the water runway
            rwy_number = parts[W85_NUM1]; //  08 Runway number. Not rendered in X-Plane (its on water!) Valid suffixes are L, R or C (or blank)
            //W85_NUM2 = 6,  //  08 Runway number. Not rendered in X-Plane (its on water!) Valid suffixes are L, R or C (or blank)
            lat1 = parts[W85_LAT1].toDouble();
            lon1 = parts[W85_LON1].toDouble();
            lat2 = parts[W85_LAT2].toDouble();
            lon2 = parts[W85_LON2].toDouble();
            res = _geo_inverse_wgs_84( lat1, lon1, lat2,
                        lon2, &az1, &az2, &s );
            lat = (lat1 + lat2) / 2.0;
            lon = (lon1 + lon2) / 2.0;
            pr2->rwy_no = rwy_number;
            pr2->lat1 = lat1;
            pr2->lon1 = lon1;
            pr2->lat2 = lat2;
            pr2->lon2 = lon2;
            pr2->i_heading = az1;
            pr2->i_len_ft = (int)(s * SG_METER_TO_FEET);
            if (pa2)
                pa2->rways.append(pr2);
            rwy_cnt++;
            // accumulate
            sum_lat += lat;
            sum_lon += lon;
            rwy_count++;
        } else if (code == a85_Helepad) {
            // treat like a RUNWAY
            if ( !got_apt || !pa2 )
                continue;   // no airport, so no runways !!!
            if (pcnt < H85_SIZE) {
                outLog("WARNING: Helepad ["+line+"] IGNORED");
                continue;
            }
            // 102 H1   43.30478555 -008.37734016  30.56   21.03   21.03   1 0   0 0.00 0
            // enum Hele850 {
            pr2 = new RUNWAY2;
            pr2->i_heading = parts[H85_HEAD].toDouble();  // 2.00 Orientation (true heading) of helipad in degrees Two decimal places recommended
            pr2->i_len_ft = (int)(parts[H85_LEN].toDouble() * SG_METER_TO_FEET);
            // 10.06  Helipad length in metres Two decimal places recommended (metres), must be >=1.00
            // H85_WID = 6,   // 10.06  Helipad width in metres Two decimal places recommended (metres), must be >= 1.00
            // H85_SURF = 7,  //     1  Helipad surface code Integer value for a Surface Type Code
            // H85_MARK = 8,  //     0  Helipad markings 0 (other values not yet supported)
            // H85_SHLDR = 9, //     0  Code defining a helipad shoulder surface type 0=no shoulder, 1=asphalt shoulder, 2=concrete shoulder
            // H85_SMOOTH = 10, // 0.25 Helipad smoothness (not used by X-Plane yet) 0.00 (smooth) to 1.00 (very rough). Default is 0.25
            // H85_ELITE = 11, //     0 Helipad edge lighting 0=no edge lights, 1=yellow edge lights
            rwy_number = parts[H85_DESIG];
            lat = parts[H85_LAT].toDouble();
            lon = parts[H85_LON].toDouble();
            pr2->lat1 = lat;
            pr2->lon1 = lon;
            pr2->lat2 = lat;
            pr2->lon2 = lon;
            pr2->rwy_no = rwy_number;
            if (pa2)
                pa2->rways.append(pr2);
            rwy_cnt++;
            // accumulate
            sum_lat += lat;
            sum_lon += lon;
            rwy_count++;
        } else {
            if ( !got_apt || !pa2 )
                continue;   // no airport, so no airport bits!!!
            if (code == a85_Taxiway) { // 110 // Pavement (taxiway or ramp) header Must form a closed loop
                // enum Tax850 {
            } else if (code == a85_TaxiOld) {
                // enum Tax810 {
            } else if (code == a85_Node) { //  111 // Node All nodes can also include a style (line or lights)
                //
            } else if (code == a85_Bezier) { //  112 // Node with Bezier control point Bezier control points define smooth curves
                //
            } else if (code == a85_Close) { // 113 // Node with implicit close of loop Implied join to first node in chain
                //
            } else if (code == a85_Control) { // 114 // Node with Bezier control point, with implicit close of loop Implied join to first node in chain
                //
            } else if (code == a85_Term) { // 115 // Node terminating a string (no close loop) No styles used
                //
            } else if (code == a85_String) { //  116 // Node with Bezier control point, terminating a string (no close loop) No styles used
                //
            } else if (code == a85_Linear) { //  120 // Linear feature (painted line or light string) header Can form closed loop or simple string
                //
            } else if (code == a85_Bounds) { //  130 // Airport boundary header Must form a closed loop
                //
            } else if (code == a85_View) { // 14 // Airport viewpoint One or none for each airport
                // 14 22.16348500 -159.30136100 0 0 Tower Viewpoint
                // enum View850 {
            } else if (code == a85_Startup) { // 15 // Aeroplane startup location Zero, one or many for each airport
                // 15   22.16142100 -159.30091900  40.00 Anchor
                // enum Su850 {
            } else if (code == a85_Beacon) { //  18 // Airport light beacon One or none for each airport
                //
            } else if (code == a85_Windsock) { // 19 // Windsock Zero, one or many for each airport
                // Windosck
                if (pcnt >= (WS85_MIN - 1)) {
                    PWINDSOCK pws = new WINDSOCK;
                    // 19   36.97040984  127.02064840 1 WS
                    // 19   36.95401518  127.04149243 1 WS
                    pws->code = code;
                    pws->lat = parts[WS85_LAT].toDouble();
                    pws->lon = parts[WS85_LON].toDouble();
                    pws->lit = parts[WS85_LIT].toInt(); // Flag for windsock lighting 0=unlit, 1=illuminated
                    if (pcnt >= WS85_MIN) {
                        pws->name = parts[WS85_NAME];
                    }
                    // WS85_MIN = 5
                    pa2->ws.append(pws);
                 }
            } else if (code == a85_Sign) { //  20 // Taxiway sign (inc. runway distance-remaining signs) Zero, one or many for each airport
            } else if (code == a85_Light) { // 21 // Lighting object (VASI, PAPI, Wig-Wag, etc.) Zero, one or many for each airport
                // 21   43.29737198 -008.38037207  3  30.90   3.00 API PAPI-4R
                // 21   43.29784944 -008.38145086  2  30.90   3.00 API PAPI-4L
            } else if ((code >= a85_Com_min) && (code <= a85_Com_max)) { // 56 // Communication frequencies Zero, one or many for each airport
                // Airport communication frequency
                // 50 10820 ATIS
                // 53 11950 DESIDERIO GND
                // 54 12250 DESIDERIO TWR
                // 55 12790 OSAN APP/DEP
                if (pcnt < C85_MIN) {
                    outLog("WARNING: Discarding COM line ["+line+"]");
                    continue;
                }
                PCOMITEM pci = new COMITEM;
                pci->code = code;
                pci->freq = parts[C85_FREQ].toDouble();
                pci->name = parts[C85_NAME];
                for (p = C85_MIN; p < pcnt; p++) {
                    pci->name.append(sp+parts[p]);
                }
                pa2->cl.append(pci);
            } else {
                tmp.sprintf("WARNING:Ln %d:",line_counter);
                outLog(tmp+" Uncased line ["+line+"]");
            }
        }
    }
    if (rwy_cnt && got_apt && pa2) {
        // get 'average' center of last airport
        lat = sum_lat / (double)rwy_cnt;
        lon = sum_lon / (double)rwy_cnt;
        pa2->lat  = lat;
        pa2->lon  = lon;
    }

    f.close();

    msg.sprintf("Done %d lines, found %d airports (%d), %d runways",
                line_counter,air_count,
                pal->count(),
                rwy_count);
    ms = tm.elapsed();
    outLog(msg+", in "+getElapTimeStg(ms));

    return;
}

// eof - load_apt850.cpp
