/* ==================================================
   qt_test_gui project
   Created: Geoff R. McLane - Aug 2011
   License: GPL2 (or later)
   ================================================== */
#include <QtGui>

#include "test_config.h"
#include "mainwindow.h"
#include "utilities.h"
#include "test_dom.h"
#include "test_sort.h"
#include "test_aptdat.h"
#include "test_apt850.h"
#include "cmp_apt_lists.h"
#include "load_nav.h"
#include "load_awy.h"
#include "load_fix.h"

QString test_url = "http://weather.noaa.gov/pub/data/observations/metar/stations/YSDU.TXT";

QList<PTEST> m_test_list; // stacked tests running

#ifdef BUILD_MENU_TEST
#include "example.cpp"
#else // !_BUILD_MENU_TEST

#include <QVBoxLayout>
#include <QMenuBar>
#include <QMenu>

//QPushButton {
//    border-width: 0px;
//}

MainWindow::MainWindow(QWidget *parent)
    : QMainWindow(parent)
{
    QSettings settings;

    util_setStdLogFile();

    restoreGeometry(settings.value("mainWindowGeometry").toByteArray());

    processCommands();

    doing_test = 0;
    test_count = 0;
    m_test_list.clear();

    //====================================================
    //** Main Central Widget and Layout
    //====================================================
    QWidget *widget = new QWidget;
    setCentralWidget(widget);

    QVBoxLayout *mainLayout = new QVBoxLayout();
    //mainLayout->setContentsMargins(10, 10, 10, 10);
    mainLayout->setContentsMargins(0, 0, 0, 0);
    mainLayout->setSpacing(0);
    widget->setLayout(mainLayout);

#ifdef USE_OWN_MENUBAR
    //== Setup top menu
    menuBar = new QMenuBar();
#endif

    //== File Menu
    menuFile = new QMenu(tr("&File"));
    exitAct = menuFile->addAction(QIcon(":/icon/quit"), tr("&Quit"), this, SLOT(on_quit()));
//#ifdef Q_OS_WIN
//    testAct->setShortcut(QKeySequence(tr("Ctrl+q"))); // THIS BOMBED???
//#else
    exitAct->setShortcuts(QKeySequence::Quit);
//#endif
    exitAct->setStatusTip(tr("Quit the application"));
    exitAct->setIconVisibleInMenu(true);

    //== Action Menu
    menuTest = new QMenu(tr("&Actions"));

#ifdef ADD_URL_TEST
    test2Act = menuTest->addAction(QIcon(":/icon/test2"), tr("&Get URL"), this, SLOT(on_test2()));
    test2Act->setStatusTip(tr("Test get URL"));
    test2Act->setIconVisibleInMenu(true);
#endif

    testAct = menuTest->addAction(QIcon(":/icon/test2"), tr("&Load APT"), this, SLOT(on_load_apt()));
    testAct->setShortcut(QKeySequence(tr("Ctrl+t")));
    testAct->setStatusTip(tr("Load FG/XP apt.dat, and compare"));
    testAct->setIconVisibleInMenu(true);

    test5Act = menuTest->addAction(QIcon(":/icon/test2"), tr("Load NAV"), this, SLOT(on_load_nav()));
    test5Act->setStatusTip(tr("Load FG/XP NAV files, and compare"));
    test5Act->setIconVisibleInMenu(true);

    test6Act = menuTest->addAction(QIcon(":/icon/test2"), tr("Load AWY"), this, SLOT(on_load_awy()));
    test6Act->setStatusTip(tr("Load FG/XP AWY files, and compare"));
    test6Act->setIconVisibleInMenu(true);

    test7Act = menuTest->addAction(QIcon(":/icon/test2"), tr("Load FIX"), this, SLOT(on_load_fix()));
    test7Act->setStatusTip(tr("Load FG/XP FIX files, and compare"));
    test7Act->setIconVisibleInMenu(true);

    test4Act = menuTest->addAction(QIcon(":/icon/test"), tr("DO all tests"), this, SLOT(on_test_all()));
    test4Act->setStatusTip(tr("Do all tests"));
    test4Act->setIconVisibleInMenu(true);

    menuTest->addSeparator();

    testAct = menuTest->addAction(QIcon(":/icon/test2"), tr("&Load apt810"), this, SLOT(on_test()));
    //testAct->setShortcut(QKeySequence(tr("Ctrl+t")));
    testAct->setStatusTip(tr("Load FG 810 apt.dat.gz"));
    testAct->setIconVisibleInMenu(true);

    test3Act = menuTest->addAction(QIcon(":/icon/test2"), tr("Load &apt850"), this, SLOT(on_test3()));
    test3Act->setStatusTip(tr("Load XP 850 apt.dat"));
    test3Act->setIconVisibleInMenu(true);

    //== Help Menu
    menuHelp = new QMenu(tr("&Help"));
    aboutAct = new QAction(tr("&About"), this);
    aboutAct->setStatusTip(tr("Show the application's About box"));
    connect(aboutAct, SIGNAL(triggered()), this, SLOT(about()));

    aboutQtAct = new QAction(tr("About &Qt"), this);
    aboutQtAct->setStatusTip(tr("Show the Qt library's About box"));
    connect(aboutQtAct, SIGNAL(triggered()), qApp, SLOT(aboutQt()));
    //connect(aboutQtAct, SIGNAL(triggered()), this, SLOT(aboutQt()));

    menuHelp->addAction(aboutAct);
    menuHelp->addAction(aboutQtAct);

    //helpAct = menuHelp->addAction(QIcon(":/icon/help"), tr("Help"), this, SLOT(on_help()));
    //helpAct->setIconVisibleInMenu(true);

#ifdef USE_OWN_MENUBAR
    // establish menu bar items
    menuBar->addMenu(menuFile);
    menuBar->addMenu(menuTest);
    menuBar->addMenu(menuHelp);
#else
    menuBar()->addMenu(menuFile);
    menuBar()->addMenu(menuTest);
    menuBar()->addMenu(menuHelp);
#endif

    // center - text box
    txtExtraArgs = new QPlainTextEdit();

#ifdef USE_OWN_STATUS
    // Bottom Statusbar
    statusBar = new StatusBar();
    statusBar->setMinimumHeight(30);
    //statusBar->setContentsMargins(10,0,0,0);
    statusBar->setContentsMargins(0,0,0,0);
#endif

    //= establish main layout
#ifdef USE_OWN_MENUBAR
    mainLayout->addWidget(menuBar);
#endif

    // get file names
    QString tmp;

    QGroupBox *fileGroup = new QGroupBox(tr("Input Files - green=ok, red=not valid"));

    // airport data
    FGaptLabel = new QLabel(tr("FG apt.dat:"));
    tmp = settings.value("files/fgapt",DEF_APT810_FILE).toString();
    FGaptLineEdit = new QLineEdit(tmp,this);
    connect(FGaptLineEdit,SIGNAL(editingFinished()),this,SLOT(on_fgapt_text()));
    FGaptBrowse = new QPushButton("...",this);
    connect(FGaptBrowse,SIGNAL(pressed()),this,SLOT(on_fgapt_browse()));
    on_fgapt_text();

    XPaptLabel = new QLabel(tr("XP apt.dat:"));
    tmp = settings.value("files/xpapt",DEF_APT850_FILE).toString();
    XPaptLineEdit = new QLineEdit(tmp,this);
    connect(XPaptLineEdit,SIGNAL(editingFinished()),this,SLOT(on_xpapt_text()));
    XPaptBrowse = new QPushButton("...",this);
    connect(XPaptBrowse,SIGNAL(pressed()),this,SLOT(on_xpapt_browse()));
    on_xpapt_text();

    // nav data
    FGnavLabel = new QLabel(tr("FG nav.dat:"));
    tmp = settings.value("files/fgnav",DEF_FGNAV_FILE).toString();
    FGnavLineEdit = new QLineEdit(tmp,this);
    connect(FGnavLineEdit,SIGNAL(editingFinished()),this,SLOT(on_fgnav_text()));
    FGnavBrowse = new QPushButton("...",this);
    connect(FGnavBrowse,SIGNAL(pressed()),this,SLOT(on_fgnav_browse()));
    on_fgnav_text();

    XPnavLabel = new QLabel(tr("XP nav.dat:"));
    tmp = settings.value("files/xpnav",DEF_XPNAV_FILE).toString();
    XPnavLineEdit = new QLineEdit(tmp,this);
    connect(XPnavLineEdit,SIGNAL(editingFinished()),this,SLOT(on_xpnav_text()));
    XPnavBrowse = new QPushButton("...",this);
    connect(XPnavBrowse,SIGNAL(pressed()),this,SLOT(on_xpnav_browse()));
    on_xpnav_text();

    // awy data
    FGawyLabel = new QLabel(tr("FG awy.dat:"));
    tmp = settings.value("files/fgawy",DEF_FGAWY_FILE).toString();
    FGawyLineEdit = new QLineEdit(tmp,this);
    connect(FGawyLineEdit,SIGNAL(editingFinished()),this,SLOT(on_fgawy_text()));
    FGawyBrowse = new QPushButton("...",this);
    connect(FGawyBrowse,SIGNAL(pressed()),this,SLOT(on_fgawy_browse()));
    on_fgawy_text();

    XPawyLabel = new QLabel(tr("XP awy.dat:"));
    tmp = settings.value("files/xpawy",DEF_XPAWY_FILE).toString();
    XPawyLineEdit = new QLineEdit(tmp,this);
    connect(XPawyLineEdit,SIGNAL(editingFinished()),this,SLOT(on_xpawy_text()));
    XPawyBrowse = new QPushButton("...",this);
    connect(XPawyBrowse,SIGNAL(pressed()),this,SLOT(on_xpawy_browse()));
    on_xpawy_text();

    // fix data
    FGfixLabel = new QLabel(tr("FG fix.dat:"));
    tmp = settings.value("files/fgfix",DEF_FGFIX_FILE).toString();
    FGfixLineEdit = new QLineEdit(tmp,this);
    connect(FGfixLineEdit,SIGNAL(editingFinished()),this,SLOT(on_fgfix_text()));
    FGfixBrowse = new QPushButton("...",this);
    connect(FGfixBrowse,SIGNAL(pressed()),this,SLOT(on_fgfix_browse()));
    on_fgfix_text();

    XPfixLabel = new QLabel(tr("XP fix.dat:"));
    tmp = settings.value("files/xpfix",DEF_XPFIX_FILE).toString();
    XPfixLineEdit = new QLineEdit(tmp,this);
    connect(XPfixLineEdit,SIGNAL(editingFinished()),this,SLOT(on_xpfix_text()));
    XPfixBrowse = new QPushButton("...",this);
    connect(XPfixBrowse,SIGNAL(pressed()),this,SLOT(on_xpfix_browse()));
    on_xpfix_text();

    QGridLayout *fileLayout = new QGridLayout;
    fileLayout->setContentsMargins(10, 0, 10, 0);
    fileLayout->setSpacing(0);

    int row = 0;
    fileLayout->addWidget(FGaptLabel,row,0);
    fileLayout->addWidget(FGaptLineEdit,row,1);
    fileLayout->addWidget(FGaptBrowse,row,2);
    row++;
    fileLayout->addWidget(XPaptLabel,row,0);
    fileLayout->addWidget(XPaptLineEdit,row,1);
    fileLayout->addWidget(XPaptBrowse,row,2);
    row++;
    fileLayout->addWidget(FGnavLabel,row,0);
    fileLayout->addWidget(FGnavLineEdit,row,1);
    fileLayout->addWidget(FGnavBrowse,row,2);
    row++;
    fileLayout->addWidget(XPnavLabel,row,0);
    fileLayout->addWidget(XPnavLineEdit,row,1);
    fileLayout->addWidget(XPnavBrowse,row,2);

    row++;
    fileLayout->addWidget(FGfixLabel,row,0);
    fileLayout->addWidget(FGfixLineEdit,row,1);
    fileLayout->addWidget(FGfixBrowse,row,2);
    row++;
    fileLayout->addWidget(XPfixLabel,row,0);
    fileLayout->addWidget(XPfixLineEdit,row,1);
    fileLayout->addWidget(XPfixBrowse,row,2);

    row++;
    fileLayout->addWidget(FGawyLabel,row,0);
    fileLayout->addWidget(FGawyLineEdit,row,1);
    fileLayout->addWidget(FGawyBrowse,row,2);
    row++;
    fileLayout->addWidget(XPawyLabel,row,0);
    fileLayout->addWidget(XPawyLineEdit,row,1);
    fileLayout->addWidget(XPawyBrowse,row,2);

    fileGroup->setLayout(fileLayout);
    mainLayout->addWidget(fileGroup);

#ifdef ADD_USER_OPTIONS

    checkIcao = new QCheckBox(this);
    checkIcao->setText("Load Airports with ICAO [A-Z{4}] only");
    connect(checkIcao, SIGNAL(clicked()), this, SLOT(on_check_icao()));
    import_icao_only = settings.value("check/icao",false).toBool();
    checkIcao->setCheckState(import_icao_only ? Qt::Checked : Qt::Unchecked);
    mainLayout->addWidget(checkIcao);

    checkAptNav = new QCheckBox(this);
    checkAptNav->setText("Show NAV items also per ICAO/Name, sorted");
    connect(checkAptNav, SIGNAL(clicked()), this, SLOT(on_check_aptnav()));
    nav_apt_check = settings.value("check/aptnav",false).toBool();
    checkAptNav->setCheckState(nav_apt_check ? Qt::Checked : Qt::Unchecked);
    mainLayout->addWidget(checkAptNav);

#endif // ADD_USER_OPTIONS

    //mainLayout->addSpacing(10);
    mainLayout->addWidget(txtExtraArgs);

    //mainLayout->addSpacing(10);
    QString message = tr("Ready");

#ifdef USE_OWN_STATUS
    mainLayout->addWidget(statusBar);
    setStatusBar(statusBar);
    statusBar->showMessage(message);
#else
    statusBar()->showMessage(message);
    QString msg = util_getTimestg();
    label = new QLabel(msg);
    //label->setFixedWidth(50);
    label->setFrameStyle(QFrame::Panel | QFrame::Raised);
    //label->setMargin(1);
    statusBar()->addPermanentWidget(label);

    timer = new QTimer(this);
    connect(timer, SIGNAL(timeout()), this, SLOT(set_time()));
    timer->start(1000);

#endif

    setWindowTitle(tr("Qt Test Gui"));
    //setMinimumSize(160, 160);
    //resize(480, 320);
    restoreState(settings.value("mainWindowState").toByteArray());

#ifdef USE_WORK_THREAD
    outLog("Allocating the worker...");
    worker = new workThread;
    connect(worker, SIGNAL(work_done(int,int)), this, SLOT(done_it(int,int)));
#endif

}

MainWindow::~MainWindow()
{
    clear_apt_list();
    clear_apt850_list();
    clear_nav_lists();
    clear_awy_lists();
    clear_fix_lists();
#ifdef USE_WORK_THREAD
    outLog("Destroying the worker...");
    if (worker)
        delete worker;
#endif

}

int MainWindow::processCommands()
{
    int i;
    QString msg;
    QStringList cmd_args = QCoreApplication::arguments();
    for (i = 0; i < cmd_args.count(); i++) {
        msg.sprintf("CMD %d: ", (i + 1));
        outLog(msg+cmd_args.at(i));
    }
    return 0;
}

// quit
void MainWindow::on_quit()
{
    close();
}
void MainWindow::about()
{
    QString msg;
    msg = "Version: ";
    msg.append(PGM_VERSION);
    msg.append(", dated ");
    msg.append(PGM_DATE);
    msg.append("\n");
    msg.append("Built: ");
    msg.append(__DATE__);
    msg.append(" at ");
    msg.append(__TIME__);
    msg.append("\n\n");
    msg.append("Qt Test Gui shows various aspects of Qt, and\n");
    msg.append("was extended to load and compare airport and\n");
    msg.append("navigation data from FlightGear and X-Plane.\n\n");
    msg.append("License: GPL v2 or later\n\n");
    msg.append("Source: http://geoffair.org/projects/qt_test_gui.htm");
    QMessageBox::about(this, tr("About Qt Test Gui"), msg);
}

void MainWindow::aboutQt()
{
    QMessageBox::aboutQt(this, "About Qt");
}

void MainWindow::saveSettings()
{
    QSettings settings;
    QString tmp;
    QFile f;
    settings.setValue("mainWindowGeometry", saveGeometry());
    settings.setValue("mainWindowState", saveState());

    tmp = FGaptLineEdit->text();
    if (f.exists(tmp))
        settings.setValue("files/fgapt", tmp);
    tmp = XPaptLineEdit->text();
    if (f.exists(tmp))
        settings.setValue("files/xpapt", tmp);

    tmp = FGnavLineEdit->text();
    if (f.exists(tmp))
        settings.setValue("files/fgnav", tmp);
    tmp = XPnavLineEdit->text();
    if (f.exists(tmp))
        settings.setValue("files/xpnav", tmp);

    tmp = FGfixLineEdit->text();
    if (f.exists(tmp))
        settings.setValue("files/fgfix", tmp);
    tmp = XPfixLineEdit->text();
    if (f.exists(tmp))
        settings.setValue("files/xpfix", tmp);

    tmp = FGawyLineEdit->text();
    if (f.exists(tmp))
        settings.setValue("files/fgawy", tmp);
    tmp = XPawyLineEdit->text();
    if (f.exists(tmp))
        settings.setValue("files/xpawy", tmp);
}

void MainWindow::closeEvent(QCloseEvent *event)
{
    saveSettings();
    outLog(util_getDateTimestg()+" - Application close",0x8001);
    event->accept();
}

void MainWindow::appendTestList( PTEST pt )
{
    test_count++;
    m_test_list.append(pt);
}

void MainWindow::setStatusMessage(QString msg)
{
    outLog("STATUS: "+msg);
#ifdef USE_OWN_STATUS
    statusBar->showMessage(msg);
#else
    statusBar()->showMessage(msg);
#endif
}


#ifndef USE_OWN_STATUS
void MainWindow::set_time()
{
    QString msg = util_getTimestg();
    if (doing_test)
        msg.append("*");
    label->setText(msg);
#ifndef USE_WORK_THREAD
#ifdef USE_CONCURRENT
    if (doing_test) {
        QString msg;
        msg = "Test ";
        if (future.isStarted())
            msg.append("started ");
        if (future.isRunning())
            msg.append("running ");
        if (future.isPaused())
            msg.append("paused ");

        if (future.isFinished()) {
            doing_test = false;
            msg.append("Ended in "+getElapTimeStg(tm.elapsed()));
        }
        setStatusMessage(msg);
    }
#endif // #ifdef USE_CONCURRENT
#endif // #ifndef USE_WORK_THREAD
}
#endif

// tests

// "Load FG apt.dat"
void MainWindow::on_test()
{
    QString msg;
#ifdef USE_WORK_THREAD
    // ===================================================
    // this is looking good
    doing_test++;
    PTEST pt = new TEST;
    pt->desc = "Load FG apt.dat.gz";
    appendTestList(pt);
    pt->tt.start();
    int jn = worker->work(test_aptdat);
    pt->test_num = jn;
    msg.sprintf("Load FG apt.dat.gz, as job %d to thread...",jn);
    setStatusMessage(msg);

    doing_test++;
    pt = new TEST;
    pt->desc = "do_apt_sort";
    pt->tt.start();
    appendTestList(pt);
    pt->test_num = worker->work(do_apt_sort);
    msg.sprintf("do_apt_sort, as job %d to thread...",pt->test_num);
    setStatusMessage(msg);

    // ===================================================
#else // !USE_WORK_THREAD
    if (doing_test) {
        outLog("Test is already running...");
        return; // do NOT start a second one ;=()
    }
    tm.start();
    // ===================================================
#ifdef USE_CONCURRENT
    // ***************************************************
    msg = "File!test to run concurrent";
    //QFuture<void> future = QtConcurrent::run(test_sort);
    setStatusMessage(msg);
    doing_test++;
    future = QtConcurrent::run(test_sort);
    // ***************************************************
#else // !USE_CONCURRENT
    // ***************************************************
    msg = "File!test to run direct";
    setStatusMessage(msg);
#ifdef USE_OWN_STATUS
    statusBar->update();
#else
    statusBar()->update();
#endif
    qApp->processEvents();
    //test_dom();
    test_sort();
    msg = "End File!test in "+getElapTimeStg(tm.elapsed());
    setStatusMessage(msg);
    // ***************************************************
#endif // USE_CONCURRENT y/n
    // ===================================================
#endif // USE_WORK_THREAD y/n

}

void MainWindow::on_test2()
{
    QTime tm;
    QString msg;
    http = new httpGet;
    connect(http, SIGNAL(http_done(QString,int)),this,SLOT(got_http(QString,int)));
    tm.start();
    http->getUrl(test_url);
    msg = "End File!test2 in "+getElapTimeStg(tm.elapsed());
    setStatusMessage(msg);
}

#ifdef USE_WORK_THREAD

// hmmm, since I now pass back a JOB NUMBER
// should perhaps search the list for this job number,
// but for the moment, will take just the first ;=()
void MainWindow::done_it(int jn, int ms)
{
    QString msg, tmp;
    Q_UNUSED(ms);
    if ( !m_test_list.isEmpty() ) {
        if (doing_test)
            doing_test--;
        PTEST pt = m_test_list.takeFirst();
        int tn = pt->test_num;
        msg = "Done '"+pt->desc+"'' in "+getElapTimeStg(pt->tt.elapsed());
        tmp.sprintf(", job %d (%d). ",jn,tn);
        msg.append(tmp);
        if (m_test_list.isEmpty()) {
            tmp.sprintf(" Last of %d",test_count);
        } else {
            tmp.sprintf(" Now doing %d of %d", (jn+1), test_count);
        }
        msg.append(tmp);
        delete pt;
    } else {
        msg = "INTERNAL PROBLEM: No tests in QUEUE!";
    }
    setStatusMessage(msg);
}

#endif // #ifdef USE_WORK_THREAD

void MainWindow::got_http(QString txt, int ms)
{
    QString msg;
    msg = "Done http in "+getElapTimeStg(ms);
    setStatusMessage(msg);
    msg.append(" got ["+txt+"]");
    txtExtraArgs->appendHtml(msg);
}


void MainWindow::on_test3()
{
#ifdef USE_WORK_THREAD
    QString msg;
    // ===================================================
    // this is looking good
    doing_test++;
    PTEST pt = new TEST;
    pt->desc = "Load_apt850";
    appendTestList(pt);
    pt->tt.start();
    pt->test_num = worker->work(load_apt850);
    msg.sprintf("load_apt850: Job %d, and job %d to thread...",test_count, pt->test_num);
    setStatusMessage(msg);

    doing_test++;
    pt = new TEST;
    pt->desc = "do_apt850_sort";
    pt->tt.start();
    appendTestList(pt);
    pt->test_num = worker->work(do_apt850_sort);
    msg.sprintf("do_apt850_sort: Job %d, and job %d to thread...",test_count, pt->test_num);
    setStatusMessage(msg);

#endif
}

void MainWindow::on_test_all()
{
    QString msg;
    on_load_apt();  // load FG/XP apt.dat, and COMPARE
    on_load_nav();  // load FG/XP nav.dat, and COMPARE
    on_load_awy();  // load FG/XP awy.dat, and COMPARE
    on_load_fix();  // load FG/XP fix.dat, and COMPARE
    msg.sprintf("Added ALL %d jobs to thread...",test_count);
    setStatusMessage(msg);
}

void MainWindow::on_load_apt()
{
    QString msg;
    on_test();      // load FG apt.dat, and sort on YGIL
    on_test3();     // load XP apt.dat, and sort on YGIL

    doing_test++;
    PTEST pt = new TEST;
    pt->desc = "cmp_apt_list";
    appendTestList(pt);
    pt->tt.start();
    pt->test_num = worker->work(cmp_apt_lists);
    msg.sprintf("cmp_apt_list: Test %d, as jobs %d to thread... load APTs and compare...",test_count, pt->test_num);
    setStatusMessage(msg);
}

void MainWindow::on_load_nav()
{
    QString msg;
    doing_test++;
    PTEST pt = new TEST;
    pt->desc = "Load NAV files";
    appendTestList(pt);
    pt->tt.start();
    pt->test_num = worker->work(load_nav_files);
    msg.sprintf("Load NAV files: test %d, as job %d to thread...",test_count,pt->test_num);
    setStatusMessage(msg);
}

void MainWindow::on_load_awy()
{
    QString msg;
    doing_test++;
    PTEST pt = new TEST;
    pt->desc = "Load AWY files";
    appendTestList(pt);
    pt->tt.start();
    pt->test_num = worker->work(load_awy_files);
    msg.sprintf("Load AWY files, test %d, as job %d to thread...",test_count,pt->test_num);
    setStatusMessage(msg);
}

void MainWindow::on_load_fix()
{
    QString msg;
    doing_test++;
    PTEST pt = new TEST;
    pt->desc = "Load FIX files";
    appendTestList(pt);
    pt->tt.start();
    pt->test_num = worker->work(load_fix_files);
    msg.sprintf("Load FIX files, test %d, as job %d to thread...",test_count,pt->test_num);
    setStatusMessage(msg);
}

QString MainWindow::browseFile(QString title, QString current, QString filter)
{
    QString file = QFileDialog::getOpenFileName(
          this, title, current, filter );
    return file;
}

void MainWindow::on_fgapt_text()
{
    QString tmp;
    QFile f;
    QString style;
    tmp = FGaptLineEdit->text();
    if (f.exists(tmp)) {
        style = "color:green";
        set_fgapt_dat(tmp);
    } else {
        style = "color:red";
    }
    FGaptLabel->setStyleSheet(style);
}

void MainWindow::on_fgapt_browse()
{
    QString current(FGaptLineEdit->text());
    QString title = "Select FG apt.dat[.gz] File";
    QString filter = "All (*.*)";
    QString nFile = browseFile(title,current,filter);
    QFile f;
    if (nFile.size() && f.exists(nFile)) {
        FGaptLineEdit->setText(nFile);
        on_fgapt_text();
    }
}
void MainWindow::on_xpapt_text()
{
    QString tmp;
    QFile f;
    QString style;
    tmp = XPaptLineEdit->text();
    if (f.exists(tmp)) {
        style = "color:green";
        set_xpapt_dat(tmp);
    } else {
        style = "color:red";
    }
    XPaptLabel->setStyleSheet(style);

}
void MainWindow::on_xpapt_browse()
{
    QString current(XPaptLineEdit->text());
    QString title = "Select XP apt.dat File";
    QString filter = "All (*.*)";
    QString nFile = browseFile(title,current,filter);
    QFile f;
    if (nFile.size() && f.exists(nFile)) {
        XPaptLineEdit->setText(nFile);
        on_xpapt_text();
    }

}

// nav dat
void MainWindow::on_fgnav_text()
{
    QString tmp;
    QFile f;
    QString style;
    tmp = FGnavLineEdit->text();
    if (f.exists(tmp)) {
        style = "color:green";
        set_fgnav_dat(tmp);
    } else {
        style = "color:red";
    }
    FGnavLabel->setStyleSheet(style);
}

void MainWindow::on_fgnav_browse()
{
    QString current(FGnavLineEdit->text());
    QString title = "Select FG nav.dat[.gz] File";
    QString filter = "All (*.*)";
    QString nFile = browseFile(title,current,filter);
    QFile f;
    if (nFile.size() && f.exists(nFile)) {
        FGnavLineEdit->setText(nFile);
        on_fgnav_text();
    }
}
void MainWindow::on_xpnav_text()
{
    QString tmp;
    QFile f;
    QString style;
    tmp = XPnavLineEdit->text();
    if (f.exists(tmp)) {
        style = "color:green";
        set_xpnav_dat(tmp);
    } else {
        style = "color:red";
    }
    XPnavLabel->setStyleSheet(style);

}
void MainWindow::on_xpnav_browse()
{
    QString current(XPnavLineEdit->text());
    QString title = "Select XP nav.dat File";
    QString filter = "All (*.*)";
    QString nFile = browseFile(title,current,filter);
    QFile f;
    if (nFile.size() && f.exists(nFile)) {
        XPnavLineEdit->setText(nFile);
        on_xpnav_text();
    }
}

// awy dat
void MainWindow::on_fgawy_text()
{
    QString tmp;
    QFile f;
    QString style;
    tmp = FGawyLineEdit->text();
    if (f.exists(tmp)) {
        style = "color:green";
        set_fgawy_dat(tmp);
    } else {
        style = "color:red";
    }
    FGawyLabel->setStyleSheet(style);
}

void MainWindow::on_fgawy_browse()
{
    QString current(FGawyLineEdit->text());
    QString title = "Select FG awy.dat[.gz] File";
    QString filter = "All (*.*)";
    QString nFile = browseFile(title,current,filter);
    QFile f;
    if (nFile.size() && f.exists(nFile)) {
        FGawyLineEdit->setText(nFile);
        on_fgawy_text();
    }
}
void MainWindow::on_xpawy_text()
{
    QString tmp;
    QFile f;
    QString style;
    tmp = XPawyLineEdit->text();
    if (f.exists(tmp)) {
        style = "color:green";
        set_xpawy_dat(tmp);
    } else {
        style = "color:red";
    }
    XPawyLabel->setStyleSheet(style);

}
void MainWindow::on_xpawy_browse()
{
    QString current(XPawyLineEdit->text());
    QString title = "Select XP awy.dat File";
    QString filter = "All (*.*)";
    QString nFile = browseFile(title,current,filter);
    QFile f;
    if (nFile.size() && f.exists(nFile)) {
        XPawyLineEdit->setText(nFile);
        on_xpawy_text();
    }
}

// fix dat
void MainWindow::on_fgfix_text()
{
    QString tmp;
    QFile f;
    QString style;
    tmp = FGfixLineEdit->text();
    if (f.exists(tmp)) {
        style = "color:green";
        set_fgfix_dat(tmp);
    } else {
        style = "color:red";
    }
    FGfixLabel->setStyleSheet(style);
}

void MainWindow::on_fgfix_browse()
{
    QString current(FGfixLineEdit->text());
    QString title = "Select FG fix.dat[.gz] File";
    QString filter = "All (*.*)";
    QString nFile = browseFile(title,current,filter);
    QFile f;
    if (nFile.size() && f.exists(nFile)) {
        FGfixLineEdit->setText(nFile);
        on_fgfix_text();
    }
}
void MainWindow::on_xpfix_text()
{
    QString tmp;
    QFile f;
    QString style;
    tmp = XPfixLineEdit->text();
    if (f.exists(tmp)) {
        style = "color:green";
        set_xpfix_dat(tmp);
    } else {
        style = "color:red";
    }
    XPfixLabel->setStyleSheet(style);

}
void MainWindow::on_xpfix_browse()
{
    QString current(XPfixLineEdit->text());
    QString title = "Select XP fix.dat File";
    QString filter = "All (*.*)";
    QString nFile = browseFile(title,current,filter);
    QFile f;
    if (nFile.size() && f.exists(nFile)) {
        XPfixLineEdit->setText(nFile);
        on_xpfix_text();
    }
}

void MainWindow::on_check_icao()
{
    QSettings settings;
    if (import_icao_only)
        import_icao_only = false;
    else
        import_icao_only = true;
    settings.setValue("check/icao",import_icao_only);
}

void MainWindow::on_check_aptnav()
{
    QSettings settings;
    if (nav_apt_check)
        nav_apt_check = false;
    else
        nav_apt_check = true;
    settings.setValue("check/aptnav",nav_apt_check);
}


#endif // BUILD_MENU_TEST y/n
// eof - mainwindow.cpp

